/**
 * Copyright (c) 2007 Novell, Inc.
 * All Rights Reserved.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; version 2.1 of the license.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; if not, contact Novell, Inc.
 *
 * To contact Novell about this file by physical or electronic mail,
 * you may find current contact information at www.novell.com
 */

/*
 * Copyright (c) 2007 Novell, Inc.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *		Daniel Sanders
 */

package org.eclipse.higgins.idas.cp.xmlfile;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import org.eclipse.higgins.idas.api.IContext;
import org.eclipse.higgins.idas.api.IAttribute;
import org.eclipse.higgins.idas.api.ISingleValuedAttribute;
import org.eclipse.higgins.idas.api.IHasAttributes;
import org.eclipse.higgins.idas.api.IEntity;

import org.eclipse.higgins.idas.api.IdASException;
import org.eclipse.higgins.idas.api.ContextNotOpenException;
import org.eclipse.higgins.idas.api.InvalidTypeException;
import org.eclipse.higgins.idas.api.NotImplementedException;
import org.eclipse.higgins.idas.api.NotSingleValuedAttributeException;
import org.eclipse.higgins.idas.api.NoSuchEntityException;

import org.eclipse.higgins.idas.api.model.IEntityModel;

/**
 * 
 * @author dsanders@novell.com
 * 
 */

public class NonsharedEntity implements IEntity, NonsharedAttrContainer
{
	private Log							_log = LogFactory.getLog( NonsharedEntity.class.getName());
	private NonsharedContext		_nonsharedContext;
	private long						_lContextSyncCount;
	private SharedEntity	_sharedEntity;
	private String						_szEntityID;
	private long						_lSyncCount = 0;

	NonsharedEntity(
		NonsharedContext		nonsharedCtx,
		SharedEntity	sharedEntity)
	{
		_nonsharedContext = nonsharedCtx;
		_lContextSyncCount = _nonsharedContext.getSyncCount();
		_sharedEntity = sharedEntity;
		_szEntityID = _sharedEntity.getEntityID();
	}
	
	// nonsharedAttrContainer methods
	
	public NonsharedContext getNonSharedContext()
	{
		return( _nonsharedContext);
	}
	
	public SharedAttrContainer getSharedContainer()
	{
		return( _sharedEntity);
	}
	
	public java.util.ArrayList getAttrList()
	{
		return( _sharedEntity.getAttrList());
	}
	
	public java.util.Map getAttrMap()
	{
		return( _sharedEntity.getAttrMap());
	}
	
	public String getContainerType()
	{
		return( "Entity");
	}
	
	public String getContainerID()
	{
		return( _szEntityID);
	}
	
	public long getSyncCount()
	{
		return( _lSyncCount);
	}
	
	public boolean sync(
		boolean	bForceSync) throws IdASException
	{
		boolean	bSynced = false;
	
		if (_nonsharedContext.sync( bForceSync) || _lContextSyncCount != _nonsharedContext.getSyncCount())
		{
			bForceSync = true;
		}

		if (bForceSync ||
			_sharedEntity == null ||
			_sharedEntity.isDeleted())
		{
			_sharedEntity = _nonsharedContext.getSharedContext().getEntity( _szEntityID);
			bSynced = true;
			_lSyncCount++;
			_lContextSyncCount = _nonsharedContext.getSyncCount();
		}
		if (_sharedEntity == null)
		{
			String	szErrMsg = "Entity '" + _szEntityID + "' no longer exists";
			
			_log.debug( szErrMsg);
			throw new NoSuchEntityException( szErrMsg);
		}
		return( bSynced);
	}
	
	// Methods of the IEntity interface
	
	public IContext getContext() throws IdASException
	{
		return( _nonsharedContext);
	}
	
	/**
	 */
	public String getEntityID() throws IdASException
	{
		return( _szEntityID);
	}
	
	/**
	 */
	public java.net.URI getEntityType() throws IdASException
	{
		java.net.URI	entityType = null;
		
		if (!_nonsharedContext.isOpen())
		{
			throw new ContextNotOpenException( "nonsharedEntity:getEntityType(): Context not open");
		}
		
		try
		{
			_nonsharedContext.lockSharedContext();
			sync( false);
			entityType = java.net.URI.create( _sharedEntity.getEntityType());
		}
		finally
		{
			_nonsharedContext.unlockSharedContext();
		}
		
		return( entityType);
	}
	
	/**
	 */
	public java.util.Iterator getAttributes() throws IdASException
	{
		java.util.Iterator	iter = null;
		
		if (!_nonsharedContext.isOpen())
		{
			throw new ContextNotOpenException( "nonsharedEntity:getAttributes(): Context not open");
		}
		
		try
		{
			_nonsharedContext.lockSharedContext();
			sync( false);
			iter = new AttrIterator( this);
		}
		finally
		{
			_nonsharedContext.unlockSharedContext();
		}
		
		return( iter);
	}

	/**
	 */
	public IAttribute getAttribute(
		java.net.URI attrID) throws IdASException
	{
		SharedAttribute	sharedAttr = null;
		
		if (!_nonsharedContext.isOpen())
		{
			throw new ContextNotOpenException( "nonsharedEntity:getAttribute(attrID): Context not open");
		}
		
		try
		{
			_nonsharedContext.lockSharedContext();
			sync( false);
			sharedAttr = _sharedEntity.getAttribute( attrID.toString());
		}
		finally
		{
			_nonsharedContext.unlockSharedContext();
		}
		return( sharedAttr == null ? null : new NonsharedAttribute( this, sharedAttr));
	}

	/**
	 */
	public IEntityModel getModel() throws IdASException
	{
		throw new NotImplementedException( "Not Implemented: nonsharedEntity:getModel()");
	}

	/**
	 */
	public void remove() throws IdASException
	{
		boolean	bStartedTrans = false;
		
		if (!_nonsharedContext.isOpen())
		{
			throw new ContextNotOpenException( "nonsharedEntity:remove(): Context not open");
		}
		
		try
		{
			bStartedTrans = _nonsharedContext.beginTrans();
			sync( bStartedTrans);
			_sharedEntity.remove();
			_sharedEntity = null;
		}
		finally
		{
			if (!bStartedTrans)
			{
				_nonsharedContext.unlockSharedContext();
			}
		}
	}

	/**
	 */
	public ISingleValuedAttribute getSingleValuedAttribute(
		java.net.URI	attrID) throws IdASException, NotSingleValuedAttributeException
	{
		SharedAttribute	sharedAttr = null;
		
		if (!_nonsharedContext.isOpen())
		{
			throw new ContextNotOpenException( "nonsharedEntity:getSingleValuedAttribute(): Context not open");
		}
		
		try
		{
			_nonsharedContext.lockSharedContext();
			sync( false);
			sharedAttr = _sharedEntity.getSingleValuedAttribute( attrID.toString());
		}
		finally
		{
			_nonsharedContext.unlockSharedContext();
		}
		return( sharedAttr == null ? null : new NonsharedAttribute( this, sharedAttr));
	}

	/**
	 */
	public IAttribute addAttribute(
		java.net.URI	attrID) throws IdASException, InvalidTypeException
	{
		boolean				bStartedTrans = false;
		SharedAttribute	sharedAttr = null;
		
		if (!_nonsharedContext.isOpen())
		{
			throw new ContextNotOpenException( "nonsharedEntity:addAttribute(attrID): Context not open");
		}
		
		try
		{
			bStartedTrans = _nonsharedContext.beginTrans();
			sync( bStartedTrans);
			sharedAttr = _sharedEntity.addAttribute( attrID.toString());
		}
		finally
		{
			if (!bStartedTrans)
			{
				_nonsharedContext.unlockSharedContext();
			}
		}
		return( sharedAttr == null ? null : new NonsharedAttribute( this, sharedAttr));
	}
	
	/**
	 */
	public IAttribute addAttribute(
		IAttribute copyFrom) throws IdASException
	{
		boolean				bStartedTrans = false;
		SharedAttribute	sharedAttr = null;
		
		if (!_nonsharedContext.isOpen())
		{
			throw new ContextNotOpenException( "nonsharedEntity:addAttribute(copyFrom): Context not open");
		}
		
		try
		{
			bStartedTrans = _nonsharedContext.beginTrans();
			sharedAttr = _sharedEntity.addAttribute( copyFrom);
		}
		finally
		{
			if (!bStartedTrans)
			{
				_nonsharedContext.unlockSharedContext();
			}
		}
		return( sharedAttr == null ? null : new NonsharedAttribute( this, sharedAttr));
	}

	/**
	 */
	public void removeAttribute(
		java.net.URI	attrID) throws IdASException
	{
		boolean	bStartedTrans = false;
		
		if (!_nonsharedContext.isOpen())
		{
			throw new ContextNotOpenException( "nonsharedEntity:removeAttribute(attrID): Context not open");
		}
		
		try
		{
			bStartedTrans = _nonsharedContext.beginTrans();
			sync( bStartedTrans);
			_sharedEntity.removeAttribute( attrID.toString());
		}
		finally
		{
			if (!bStartedTrans)
			{
				_nonsharedContext.unlockSharedContext();
			}
		}
	}

	/**
	 */
	public void removeAttributeValue(
		java.net.URI	attrID,
		Object			value) throws IdASException
	{
		boolean	bStartedTrans = false;
		
		if (!_nonsharedContext.isOpen())
		{
			throw new ContextNotOpenException( "nonsharedEntity:removeAttributeValue(attrID,value): Context not open");
		}
		
		try
		{
			bStartedTrans = _nonsharedContext.beginTrans();
			sync( bStartedTrans);
			_sharedEntity.removeAttributeValue( attrID.toString(), value);
		}
		finally
		{
			if (!bStartedTrans)
			{
				_nonsharedContext.unlockSharedContext();
			}
		}
	}

	/**
	 */
	public void removeAttributeValue(
		IAttribute	attr) throws IdASException
	{
		boolean	bStartedTrans = false;
		
		if (!_nonsharedContext.isOpen())
		{
			throw new ContextNotOpenException( "nonsharedEntity:removeAttributeValue(IAttribute): Context not open");
		}
		
		try
		{
			bStartedTrans = _nonsharedContext.beginTrans();
			sync( bStartedTrans);
			_sharedEntity.removeAttributeValue( attr);
		}
		finally
		{
			if (!bStartedTrans)
			{
				_nonsharedContext.unlockSharedContext();
			}
		}
	}

	/**
	 */
	public boolean equals(
		IHasAttributes	attributes) throws IdASException
	{
		boolean	bEqual = true;
		
		if (!_nonsharedContext.isOpen())
		{
			throw new ContextNotOpenException( "nonsharedEntity:equals(IHasAttributes): Context not open");
		}
		
		try
		{
			_nonsharedContext.lockSharedContext();
			sync( false);
			bEqual = _sharedEntity.equals( attributes);
		}
		finally
		{
			_nonsharedContext.unlockSharedContext();
		}
		return( bEqual);
	}
	
}

