/*******************************************************************************
 * Copyright (c) 2007 Parity Communications, Inc.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     Markus Sabadello - Initial API and implementation
 *******************************************************************************/
package org.eclipse.higgins.idas.registry.discovery;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.net.HttpURLConnection;
import java.net.MalformedURLException;
import java.net.URL;
import java.net.URLConnection;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import org.apache.xerces.parsers.DOMParser;
import org.openxri.util.DOMUtils;
import org.openxri.xml.XRDS;
import org.w3c.dom.Document;
import org.xml.sax.InputSource;

/**
 * A Discovery implementation that uses Yadis discovery to obtain an XRDS document
 * @author msabadello at parityinc dot net
 */
public class YadisDiscovery extends AbstractDiscovery {

	private static final String ACCEPT_HEADER = "Accept";
	private static final String CONTENT_TYPE_HEADER = "Content-Type";
	private static final String X_XRDS_LOCATION_HEADER = "X-XRDS-Location";
	private static final String XRDS_MIME_TYPE = "application/xrds+xml";
	private static final String HTML_MIME_TYPE = "text/html";
	private static final int MAX_X_XRDS_LOCATION = 15;

	private static final Pattern HTML_META_REGEX = Pattern.compile(".*?<META\\s+HTTP-EQUIV=\"" + X_XRDS_LOCATION_HEADER + "\"\\s+CONTENT=\"(.+?)\"\\s*>.*|.*?<META\\s+CONTENT=\"(.+?)\"\\s+HTTP-EQUIV=\"" + X_XRDS_LOCATION_HEADER + "\"\\s*>.*", Pattern.CASE_INSENSITIVE + Pattern.DOTALL);
	
	private URL url;

	public YadisDiscovery(URL url) {

		this.url = url;
	}

	public XRDS discoverXRDS() throws DiscoveryException {

		// do the Yadis protocol, starting with a HEAD

		return(this.yadisHead(this.url));
	}

	public XRDS yadisHead(URL url) throws DiscoveryException {

		return(this.yadisHead(url, true, 1));
	}

	private XRDS yadisHead(URL url, boolean fallbackToGet, int depth) throws DiscoveryException {

		// try to open connection

		URLConnection connection;

		try {

			connection = url.openConnection();
		} catch (IOException ex) {

			throw new DiscoveryException("Cannot open connection to " + url.toString(), ex);
		}

		if (! (connection instanceof HttpURLConnection)) throw new DiscoveryException("Yadis discovery only works with HTTP(S) URLs.");

		// prepare HEAD request

		try {

			((HttpURLConnection) connection).setRequestMethod("HEAD");
			((HttpURLConnection) connection).addRequestProperty(ACCEPT_HEADER, XRDS_MIME_TYPE);
		} catch (Exception ex) {

			throw new DiscoveryException("Protocol not supported.", ex);
		}

		// check if we got an X-XRDS-Location header; if yes do a GET on it

		if (connection.getHeaderField(X_XRDS_LOCATION_HEADER) != null) {

			URL newUrl;

			try {

				newUrl = new URL(connection.getHeaderField(X_XRDS_LOCATION_HEADER));
			} catch (MalformedURLException ex) {

				throw new DiscoveryException("URL found in " + X_XRDS_LOCATION_HEADER + " header is malformed.", ex);
			}

			// try to detect loops

			if (newUrl.equals(url)) throw new DiscoveryException("URL " + url + " points back to itself. Aborting.");
			if (depth > MAX_X_XRDS_LOCATION) throw new DiscoveryException("Aborting after " + MAX_X_XRDS_LOCATION + " " + X_XRDS_LOCATION_HEADER + " headers without finding an XRDS document. Aborting.");

			return(yadisGet(newUrl, depth+1));
		}

		// check if we got a Content-Type: application/xrds+xml; if yes proceed with GET

		if (XRDS_MIME_TYPE.equals(connection.getHeaderField(CONTENT_TYPE_HEADER))) {

			return(yadisGet(url, depth+1));
		}

		// fall back to GET

		if (fallbackToGet) {

			return(yadisGet(url, depth+1));
		}

		// failed to discover anything

		throw new DiscoveryException("Failed to discover any XRDS document from URL: " + url.toString());
	}

	public XRDS yadisGet(URL url) throws DiscoveryException {

		return(this.yadisGet(url, 1));
	}

	public XRDS yadisGet(URL url, int depth) throws DiscoveryException {

		// try to open connection

		URLConnection connection;

		try {

			connection = url.openConnection();
		} catch (IOException ex) {

			throw new DiscoveryException("Cannot open connection to " + url.toString(), ex);
		}

		if (! (connection instanceof HttpURLConnection)) throw new DiscoveryException("Yadis discovery only works with HTTP(S) URLs.");

		// prepare GET request

		try {

			((HttpURLConnection) connection).setRequestMethod("GET");
			((HttpURLConnection) connection).addRequestProperty(ACCEPT_HEADER, XRDS_MIME_TYPE);
		} catch (Exception ex) {

			throw new DiscoveryException("Protocol not supported.", ex);
		}

		// check if we got an X-XRDS-Location header; if yes do a GET on it

		if (connection.getHeaderField(X_XRDS_LOCATION_HEADER) != null) {

			URL newUrl;

			try {

				newUrl = new URL(connection.getHeaderField(X_XRDS_LOCATION_HEADER));
			} catch (MalformedURLException ex) {

				throw new DiscoveryException("URL found in " + X_XRDS_LOCATION_HEADER + " header is malformed.", ex);
			}

			// try to detect loops

			if (newUrl.equals(url)) throw new DiscoveryException("URL " + url + " points back to itself. Aborting.");
			if (depth > MAX_X_XRDS_LOCATION) throw new DiscoveryException("Aborting after " + MAX_X_XRDS_LOCATION + " " + X_XRDS_LOCATION_HEADER + " headers without finding an XRDS document. Aborting.");

			return(yadisGet(newUrl, depth+1));
		}

		// read document

		ByteArrayOutputStream buffer = new ByteArrayOutputStream();
		byte[] body;

		try {

			InputStream stream = connection.getInputStream();
			byte[] b = new byte[8192];
			int count;

			while ((count = stream.read(b)) > 0) buffer.write(b, 0, count);
			stream.close();

			body = buffer.toByteArray();
		} catch (IOException ex) {

			throw new DiscoveryException("Cannot read document", ex);
		}

		// check if it is an XRDS document

		if (XRDS_MIME_TYPE.equals(connection.getHeaderField(CONTENT_TYPE_HEADER))) {

			// parse XML document from input stream

			Document document; 

			try {

				DOMParser domParser = DOMUtils.getDOMParser();

				domParser.parse(new InputSource(new ByteArrayInputStream(body)));
				document = domParser.getDocument();
			} catch (Exception ex) {

				throw new DiscoveryException("Cannot parse XML.", ex);
			}

			// build XRDS object from XML document

			XRDS xrds;

			try {

				xrds = new XRDS(document.getDocumentElement(), true);
			} catch (Exception ex) {

				throw new DiscoveryException("Cannot parse XRDS.", ex);
			}

			// return it

			return(xrds);
		}

		// check if it is an HTML document

		if (HTML_MIME_TYPE.equals(connection.getHeaderField(CONTENT_TYPE_HEADER))) {

			// look for <meta> element with an http-equiv attribute equal to X-XRDS-Location

			URL newUrl;
			
			String document = new String(body);
			Matcher matcher = HTML_META_REGEX.matcher(document);

			if (matcher.matches() && matcher.groupCount() > 0) {
				
				try {

					newUrl = new URL(matcher.group(1));
				} catch (MalformedURLException ex) {

					throw new DiscoveryException("URL found in " + X_XRDS_LOCATION_HEADER + " header is malformed.", ex);
				}

				// try to detect loops

				if (newUrl.equals(url)) throw new DiscoveryException("URL " + url + " points back to itself. Aborting.");
				if (depth > MAX_X_XRDS_LOCATION) throw new DiscoveryException("Aborting after " + MAX_X_XRDS_LOCATION + " " + X_XRDS_LOCATION_HEADER + " headers without finding an XRDS document. Aborting.");

				return(yadisGet(newUrl, depth+1));
			}
		}

		// failed to discover anything

		throw new DiscoveryException("Failed to discover any XRDS document from URL: " + url.toString());
	}

	public URL getUrl() {
		return url;
	}
	
	public void setUrl(URL url) {
		this.url = url;
	}
	
	public String toString() {
		
		return(this.url.toString());
	}
}
