/*******************************************************************************
 * Copyright (c) 2007 Parity Communications, Inc.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     Markus Sabadello - Initial API and implementation
 *******************************************************************************/
package org.eclipse.higgins.idas.registry.contextid;

import java.io.File;
import java.io.InputStream;
import java.net.URL;
import java.util.Map;

import org.eclipse.higgins.idas.api.IContextId;
import org.eclipse.higgins.idas.api.IdASException;
import org.eclipse.higgins.idas.registry.IdASRegistry;
import org.eclipse.higgins.idas.registry.discovery.FileDiscovery;
import org.eclipse.higgins.idas.registry.discovery.ResolutionDiscovery;
import org.eclipse.higgins.idas.registry.discovery.StreamDiscovery;
import org.eclipse.higgins.idas.registry.discovery.YadisDiscovery;
import org.openxri.XRI;

/**
 * This is a helper class that can be used to obtain an IContextId in the following ways:
 * 1) From a local file (an XRDS document)
 * 2) From a XRI (which yields an XRDS document, using XRI resolution)
 * 3) From a URL (which yields an XRDS document, using Yadis discovery)
 * 4) From a stream (from which an XRDS document is read)
 * 5) From the ContextId list that was passed to IdASRegistry using the Configuration API.
 * 6) From a string, in which case the following will be tried in order: 1), 2), 3), 5)
 *  
 * In any case, the ContextId contains everything needed to instantiate a context. This is:
 * - One or more context types (used by IdASRegistry to find a suitable context factory).
 * - Context configuration.
 * - In the case of XRDS, optional URI(s) of the service endpoint that was used for the ContextId.
 * 
 * @author msabadello at parityinc dot net
 */
public class ContextIdFactory {

	public static IContextId fromString(String string) throws IdASException {

		// try the contextid list that was passed to IdASRegistry in its configure() methods

		try {

			return(fromConfiguration(string));
		} catch (Exception ex) { }

		// is the string a local file?

		try {

			return(fromFile(string));
		} catch (Exception ex) { }

		// try an XRI

		try {

			return(fromXRI(string));
		} catch (Exception ex) { }

		// try a URL

		try {

			return(fromUrl(string));
		} catch (Exception ex) {

		}

		// failed to construct a context ID from the string

		throw new IdASException("Not a valid context ID: " + string);
	}

	public static IContextId fromConfiguration(String contextId) throws IdASException {

		IdASRegistry registry = IdASRegistry.getInstance();

		Map contextIdConfiguration = registry.getContextIdConfiguration(contextId);

		return(new ConfigurationContextId(contextIdConfiguration));
	}

	public static IContextId fromFile(File file) throws IdASException {

		if (file == null) throw new NullPointerException();
		if (! file.exists()) throw new IdASException("File does not exist.");
		if (! file.isFile()) throw new IdASException("Not a regular file.");

		try {

			return(new XRDSContextId(new FileDiscovery(file)));
		} catch (Exception ex) {

			throw new IdASException("Cannot create context ID from file.");
		}
	}

	public static IContextId fromFile(String string) throws IdASException {

		File file = new File(string);

		return(fromFile(file));
	}

	public static IContextId fromXRI(XRI xri) throws IdASException {

		try {

			return(new XRDSContextId(new ResolutionDiscovery(xri)));
		} catch (Exception ex) {

			throw new IdASException("Cannot create context ID from XRI.", ex);
		}
	}

	public static IContextId fromXRI(String string) throws IdASException {

		XRI xri;

		try {

			xri = new XRI(string);
		} catch (Exception ex) {

			throw new IdASException("Cannot create XRI from string.", ex);
		}

		return(fromXRI(xri));
	}

	public static IContextId fromUrl(URL url) throws IdASException {

		try {

			return(new XRDSContextId(new YadisDiscovery(url)));
		} catch (Exception ex) {

			throw new IdASException("Cannot create context ID from URL.", ex);
		}
	}

	public static IContextId fromUrl(String string) throws IdASException {

		URL url;

		try {

			url = new URL(string);
		} catch (Exception ex) {

			throw new IdASException("Cannot create URL from string.", ex);
		}

		return(fromUrl(url));
	}

	public static IContextId fromStream(InputStream stream) throws IdASException {

		try {

			return(new XRDSContextId(new StreamDiscovery(stream)));
		} catch (Exception ex) {

			throw new IdASException("Cannot create context ID from stream.", ex);
		}
	}
}
