/*******************************************************************************
 * Copyright (c) 2007 Parity Communications, Inc.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     Markus Sabadello - Initial API and implementation
 *******************************************************************************/
package org.eclipse.higgins.idas.registry.discovery;

import java.util.List;

import org.openxri.XRI;
import org.openxri.resolve.Resolver;
import org.openxri.resolve.TrustType;
import org.openxri.resolve.exception.IllegalTrustTypeException;
import org.openxri.resolve.exception.PartialResolutionException;
import org.openxri.xml.SEPElement;
import org.openxri.xml.Service;
import org.openxri.xml.Tags;
import org.openxri.xml.XRD;
import org.openxri.xml.XRDS;

/**
 * A Discovery implementation that uses XRI resolution to obtain an XRDS document
 * @author msabadello at parityinc dot net
 */
public class ResolutionDiscovery extends AbstractDiscovery {

	private static final String DEFAULT_TRUST_TYPE = TrustType.TRUST_NONE;
	private static final boolean DEFAULT_FOLLOW_REFS = true;

	private XRI xri;
	private String trustType;
	private boolean followRefs;

	private Resolver resolver;

	public ResolutionDiscovery(XRI xri, String trustType, boolean followRefs) {

		this.xri = xri;
		this.trustType = trustType;
		this.followRefs = followRefs;

		this.resolver = createResolver();
	}

	public ResolutionDiscovery(XRI xri, String trustType) {

		this(xri, trustType, DEFAULT_FOLLOW_REFS);
	}

	public ResolutionDiscovery(XRI xri) {

		this(xri, DEFAULT_TRUST_TYPE, DEFAULT_FOLLOW_REFS);
	}

	public Resolver getResolver() {

		return resolver;
	}

	public void setResolver(Resolver resolver) {

		this.resolver = resolver;
	}

	public XRDS discoverXRDS() throws DiscoveryException {

		XRDS xrds;

		try {

			xrds = this.resolver.resolveAuthToXRDS(new XRI(this.xri), new TrustType(this.trustType), this.followRefs);
		} catch (PartialResolutionException ex) {

			throw new DiscoveryException("Cannot resolve XRI.", ex);
		} catch (IllegalTrustTypeException ex) {

			throw new DiscoveryException("Invalid trust type for XRI resolution.", ex);
		}

		return(xrds);
	}

	public Service discoverDefaultService() throws DiscoveryException {

		XRD xrd;

		try {

			xrd = this.resolver.resolveSEPToXRD(new XRI(this.xri), new TrustType(this.trustType), null, null, this.followRefs);
		} catch (PartialResolutionException ex) {

			throw new DiscoveryException("Cannot resolve XRI.", ex);
		} catch (IllegalTrustTypeException ex) {

			throw new DiscoveryException("Invalid trust type for XRI resolution.", ex);
		}

		List seps = xrd.getSelectedServices().getList();

		if (seps.size() < 1) {

			return(null);
		} else {

			return((Service) seps.get(0));
		}
	}

	protected Resolver createResolver() {

		Resolver resolver = new Resolver();

		XRD eqRoot = new XRD();
		Service eqAuthService = new Service();
		eqAuthService.addMediaType(Tags.CONTENT_TYPE_XRDS + ";trust=none", SEPElement.MATCH_ATTR_CONTENT, Boolean.FALSE);
		eqAuthService.addType(Tags.SERVICE_AUTH_RES);
		eqAuthService.addURI("http://equal.xri.net");
		eqRoot.addService(eqAuthService);

		XRD atRoot = new XRD();
		Service atAuthService = new Service();
		atAuthService.addMediaType(Tags.CONTENT_TYPE_XRDS + ";trust=none", SEPElement.MATCH_ATTR_CONTENT, Boolean.FALSE);
		atAuthService.addType(Tags.SERVICE_AUTH_RES);
		atAuthService.addURI("http://at.xri.net");
		atRoot.addService(atAuthService);

		XRD bangRoot = new XRD();
		Service bangAuthService = new Service();
		bangAuthService.addMediaType(Tags.CONTENT_TYPE_XRDS + ";trust=none", SEPElement.MATCH_ATTR_CONTENT, Boolean.FALSE);
		bangAuthService.addType(Tags.SERVICE_AUTH_RES);
		bangAuthService.addURI("http://bang.xri.net");
		bangRoot.addService(bangAuthService);

		resolver.setAuthority("=", eqRoot);
		resolver.setAuthority("@", atRoot);
		resolver.setAuthority("!", bangRoot);

		return(resolver);
	}

	public boolean isFollowRefs() {
		return followRefs;
	}

	public void setFollowRefs(boolean followRefs) {
		this.followRefs = followRefs;
	}

	public String getTrustType() {
		return trustType;
	}

	public void setTrustType(String trustType) {
		this.trustType = trustType;
	}

	public XRI getXri() {
		return xri;
	}

	public void setXri(XRI xri) {
		this.xri = xri;
	}
	
	public String toString() {
		
		return(this.xri.toString());
	}
}
