/*******************************************************************************
 * Copyright (c) 2007 Parity Communications, Inc.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     Markus Sabadello - Initial API and implementation
 *******************************************************************************/
package org.eclipse.higgins.idas.registry.contextid;

import java.net.URI;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Vector;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.eclipse.higgins.configuration.xrds.ConfigurationHandler;
import org.eclipse.higgins.idas.api.IContextId;
import org.eclipse.higgins.idas.api.IdASException;
import org.eclipse.higgins.idas.registry.discovery.DiscoveryException;
import org.eclipse.higgins.idas.registry.discovery.IDiscovery;
import org.openxri.xml.SEPElement;
import org.openxri.xml.SEPType;
import org.openxri.xml.SEPUri;
import org.openxri.xml.Service;

public class XRDSContextId implements IContextId {

	private static final Log log = LogFactory.getLog(XRDSContextId.class);
	
	protected IDiscovery discovery;
	protected Service sep;
	protected String[] types;
	protected URI[] uris;
	protected Map configuration;

	public XRDSContextId(IDiscovery discovery) {

		this.discovery = discovery;
		this.sep = null;
		this.types = null;
		this.uris = null;
		this.configuration = null;
	}

	public String[] getTypes() throws IdASException {

		if (this.sep == null) this.resolve();
		return(this.types);
	}

	public URI[] getUris() throws IdASException {

		if (this.sep == null) this.resolve();
		return(this.uris);
	}

	public Map getConfiguration() throws IdASException {

		if (this.sep == null) this.resolve();
		return(this.configuration);
	}

	protected void resolve() throws IdASException {

		// resolve the XRI with default parameters

		try {

			this.sep = this.discovery.discoverDefaultService();
			if (this.sep == null) throw new IdASException("No service endpoint found.");
		} catch (DiscoveryException ex) {

			throw new IdASException(ex);
		}

		// build array of types according to non-empty <Type> elements in the service endpoint

		List sepTypes = this.sep.getTypes();	// <SEPType>
		List types = new Vector();				// <String>

		if (sepTypes != null) for (Iterator i = sepTypes.iterator(); i.hasNext(); ) {

			SEPType type = (SEPType) i.next();
			if (type.getMatch() != null && ! type.getMatch().equals(SEPElement.MATCH_ATTR_CONTENT)) continue;
			types.add(type.getType());
		}

		this.types = (String[]) types.toArray(new String[types.size()]);

		// build array of uris according to <URI> elements in the service endpoint

		List sepUris = this.sep.getURIs();		// <SEPUri>
		List uris = new Vector();				// <URI>

		if (sepUris != null) for (Iterator i = sepUris.iterator(); i.hasNext(); ) {

			SEPUri uri = (SEPUri) i.next();
			uris.add(uri.getURI());
		}

		this.uris = (URI[]) uris.toArray(new URI[uris.size()]);

		// create configuration map from the service endpoint using the XRDS ConfigurationHandler

		ConfigurationHandler handler = new ConfigurationHandler();
		handler.setSEP(this.sep);

		try {

			handler.configure(null);
			this.configuration = handler.getSettings();
		} catch (Exception ex) {

			log.warn("Cannot read configuration from XRDS service endpoint.", ex);
			this.configuration = null;
		}
	}
	
	public String toString() {
		
		return(this.discovery.toString());
	}
	
	public int hashCode() {
		
		return(this.discovery.toString().hashCode());
	}
	
	public boolean equals(Object object) {
		
		if (object == this) return(true);
		if (object == null || ! (object instanceof IContextId)) return(false);
		
		IContextId other = (IContextId) object;
		
		return(this.toString().equals(other.toString()));
	}
}
