/*******************************************************************************
 * Copyright (c) 2006-2007 Novell, Inc..
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *    Jim Sermersheim - Initial cut
 *******************************************************************************/
package org.eclipse.higgins.idas.spi;

import java.net.URI;
import java.util.Iterator;
import java.util.Vector;

import org.apache.log4j.Logger;
import org.eclipse.higgins.idas.api.IAttribute;
import org.eclipse.higgins.idas.api.IComplexAttrValue;
import org.eclipse.higgins.idas.api.IContext;
import org.eclipse.higgins.idas.api.IHasMetadata;
import org.eclipse.higgins.idas.api.IMetadata;
import org.eclipse.higgins.idas.api.ISimpleAttrValue;
import org.eclipse.higgins.idas.api.IAttributeValue;
import org.eclipse.higgins.idas.api.IdASException;
import org.eclipse.higgins.idas.api.InvalidTypeException;
import org.eclipse.higgins.idas.api.NotImplementedException;
import org.eclipse.higgins.idas.api.model.IAttributeModel;
import org.eclipse.higgins.idas.api.model.IAttributeValueModel;

/**
 * Implements IAttribute using {@link BasicAttribute} and {@link BasicMetadata}
 */
public class BasicAttribute implements IAttribute, IAttributeValueContainer, IMetadataContainer {
	private Logger _log = Logger.getLogger(BasicAttribute.class.getName());
	private URI _attrID;
	private Vector _values;
	private IAttributeContainer _container;
	private IContext _context;
	private BasicMetadataSet _metaSetImpl;

	/**
	 * Note that the container's updateNotification is not called while 
	 * the values in copyFrom are being copied.
	 * @param copyFrom IAttribute from which to copy data from 
	 * @param container May be null. IAttributeContainer to send update notifications to. 
	 * @param context May be null. IContext instance from which a model may be obtained if needed.  
	 * @throws IdASException 
	 * 
	 */
	public BasicAttribute(
		IAttribute copyFrom, 
		IAttributeContainer container, 
		IContext context) throws IdASException {
		_context = context;
		_init(copyFrom.getAttrID(), copyFrom.getValues(), copyFrom.getMetadataSet(), container);
	}

	/**
	 * @param attrID the URI of the attribute's type
	 * @param value an IAttributeValue (either simple or complex) to store with the attribute
	 * @param container May be null. IAttributeContainer to send update notifications to. 
	 * @param context May be null. IContext instance from which a model may be obtained if needed.  
	 * @throws IdASException 
	 * 
	 */
	public BasicAttribute(
		URI attrID, 
		IAttributeValue value, 
		IAttributeContainer container, 
		IContext context) throws IdASException {
		_context = context;
		if (value != null) {
			Vector v = new Vector(1);
			v.add(value);
			_init(attrID, v.iterator(), null, container);
		} else {
			_init(attrID, null, null, container);
		}
	}

	/**
	 * @param attrID the URI of the attribute's type
	 * @param data an Object to be used as a value for attribute. 
	 * The Object is the data from which an {@link ISimpleAttrValue} 
	 * will be created. We do this by looking at the model for the attribute (sepcified in attrID)
	 * and associating that with the type of data object that is expected.
	 * If the value passed is of the wrong type, the constructor will fail.
	 * Note that this assumes the data will be used to construct a simple 
	 * value, not a complex value.  If one wishes to construct a BasicAttribute with a 
	 * complex value, then {@link #BasicAttribute(URI, IAttributeValue, IAttributeContainer)} is called.  
	 * @param container for purposes of notification of updates, the container of this attribute.  
	 * May be null when there is no containing object
	 * @param context May be null. IContext instance from which a model may be obtained if needed.  
	 * @throws IdASException 
	 * 
	 */
	public BasicAttribute(
		URI attrID, 
		Object data, 
		IAttributeContainer container, 
		IContext context) throws IdASException {
		_context = context;
		// Turn the value into an IAttribute 
		IAttributeValueModel valueModel = this.getModel().getValueModel();
		URI valueType = valueModel.getType();
		ISimpleAttrValue value = BasicContext.createSimpleValue(valueType, data, this);
		
		if (value != null) {
			Vector v = new Vector(1);
			v.add(value);
			_init(attrID, v.iterator(), null, container);
		} else {
			_init(attrID, null, null, container);
		}
	}

	/**
	 * Note that the container's updateNotification is not called while 
	 * the IValues in values are being added.
	 * 
	 * @param attrID
	 * @param values Contains {@link IAttributeValue}s
	 * @param container May be null. IAttributeContainer to send update notifications to. 
	 * @param context May be null. IContext instance from which a model may be obtained if needed.  
	 * @throws IdASException 
	 */
	public BasicAttribute(
		URI attrID, 
		Iterator values, 
		IAttributeContainer container, 
		IContext context) throws IdASException {
		_context = context;
		_init(attrID, values, null, container);
	}
	
	/**
	 * Note that the container's updateNotification is not called while 
	 * the passed values and metaSet are being added.
	 * @param attrID
	 * @param values Contains {@link IAttributeValue}s
	 * @throws IdASException 
	 */
	private void _init(
		URI attrID, 
		Iterator values, 
		Iterator metaSet, 
		IAttributeContainer container) throws IdASException {
		_attrID = attrID;
		_container = container;
		_values = new Vector();
		if (values != null) {
			while(values.hasNext()) {
				_addValue((IAttributeValue)values.next());
			}
		}
		_metaSetImpl = new BasicMetadataSet(metaSet, this);
	}

	private void _addValue(IAttributeValue val) throws IdASException {
		_values.add(val);
		_log.debug("Sending value add notification - " + val.getType().toString());
		AttributeValueNotification valueNotif = new AttributeValueNotification(val, AttributeValueNotification.UPDATE_ADD, null);
		this.updateNotification(valueNotif);
	}
	
	private boolean _containsValue (IAttributeValue value) throws IdASException {
		Iterator iter = this.getValues();
		IAttributeValue localVal;
		boolean bRet = false;
		while (iter.hasNext()) {
			localVal = (IAttributeValue)iter.next();
			if (localVal.equals(value)) {
				bRet = true;
				break;
			}
		}
		return bRet;
	}

	public URI getAttrID() throws IdASException {
		return _attrID;
	}

	public Iterator getValues() throws IdASException {
		/* TODO: Vector doesn't scale anyway, we need to fix this!
				However, the big deal here is the fact that we can't
				modify the thing we're iterating on, so, we're going
				with a clone temporarily. */
		return ((Vector)_values.clone()).iterator();
	}

	// Subclass must implement this as we have no model
	public IAttributeModel getModel() throws IdASException {
		if (_context != null)
			return (IAttributeModel)_context.getModel(this.getAttrID());
		else
			throw new NotImplementedException("No associated context to obtain model");
	}

	public IComplexAttrValue addComplexValue(URI type) throws IdASException, InvalidTypeException {
		IComplexAttrValue val = new BasicComplexValue(type, null, this);
		_addValue(val);
		return val;
	}

	public ISimpleAttrValue addSimpleValue(URI type, Object data) throws IdASException, InvalidTypeException {
		ISimpleAttrValue val = BasicContext.createSimpleValue(type, data, this);
		_addValue(val);
		return val;
	}

	public IAttributeValue addValue(URI type) throws IdASException, InvalidTypeException {
		IAttributeValueModel valueModel = this.getModel().getValueModel();
		IAttributeValue val;
		if (valueModel.isSimple())
			val = new BasicSimpleValue(type, null, this);
		else
			val = new BasicComplexValue(type, null, this);
		_addValue(val);
		return val;
	}

	public IAttributeValue addValue(IAttributeValue copyFrom) throws IdASException {
		IAttributeValue val;
		if (copyFrom.isSimple()) {
			ISimpleAttrValue simpleCopyFrom = (ISimpleAttrValue)copyFrom;
			val = new BasicSimpleValue(simpleCopyFrom.getType(), simpleCopyFrom.getData(), this);
		} else {
			IComplexAttrValue complexCopyFrom = (IComplexAttrValue)copyFrom;
			val = new BasicComplexValue(copyFrom.getType(), complexCopyFrom.getAttributes(), this);
		}
		_addValue(val);
		return val;
	}

	public void remove() throws IdASException {
		if (_container != null) {
			AttributeNotification attrNotif = new AttributeNotification(this, AttributeNotification.UPDATE_REMOVE, null, null);
			_log.debug("Sending attribute remove notification - " + attrNotif.getAction() + ", " + attrNotif.getAttr().getAttrID().toString());
			_container.updateNotification(attrNotif);
		}
	}

	public boolean isSingleValued() {
		return false;
	}

	/** 
	 * @see IHasMetadata#getMetadataSet()
	 */
	public Iterator getMetadataSet() throws IdASException {
		return _metaSetImpl.getMetadataSet();
	}

	public IMetadata getMetadata(URI metadataID) throws IdASException {
		return _metaSetImpl.getMetadata(metadataID);
	}

	public IMetadata addMetadata(URI type) throws IdASException, InvalidTypeException {
		return _metaSetImpl.addMetadata(type);
	}

	public IMetadata addMetadata(IMetadata copyFrom) throws IdASException {
		return _metaSetImpl.addMetadata(copyFrom);
	}

	public boolean equals(IHasMetadata metadataSet) throws IdASException {
		return _metaSetImpl.equals(metadataSet);
	}
	
	public void updateNotification(AttributeValueNotification attrValueNotif) throws IdASException {
		_log.debug("Received attribute value notification - " + attrValueNotif.getAction() + ", " + attrValueNotif.getAttributeValue().toString());
		_log.debug("_containter " + ((_container != null) ? _container.toString() : "null"));
		if (_container != null) {
			AttributeNotification attrNotif = new AttributeNotification(this, AttributeNotification.UPDATE_VALUE_NOTIFY,
				attrValueNotif, null);
			_container.updateNotification(attrNotif);
		}
		if (attrValueNotif.getAction() == AttributeValueNotification.UPDATE_REMOVE)
			_values.remove(attrValueNotif.getAttributeValue());
	}

	public void updateNotification(MetadataNotification metaNotif) throws IdASException {
		if (_container != null) {
			AttributeNotification attrNotif = new AttributeNotification(this, AttributeNotification.UPDATE_METADATA_NOTIFY,
				null, metaNotif);
			_container.updateNotification(attrNotif);
		}
	}

	public void setContainer(IAttributeContainer container) {
		_container = container;
	}
	
	public IAttributeContainer getContainer() {
		return _container;
	}

	public boolean equals(IAttribute attr) throws IdASException {
		boolean bRet;
		if (attr.getAttrID() != this.getAttrID())
			bRet = false;
		else if ((bRet = this.equals((IHasMetadata)attr)) == true) {
			Iterator iter = attr.getValues();
			// Compare each passed value
			while (iter.hasNext()) {
				if (this._containsValue((IAttributeValue)iter.next()) == false) {
					bRet = false;
					break;
				}					
			}
		}
		return bRet;
	}

}
