/*******************************************************************************
 * Copyright (c) 2006-2007 Novell, Inc..
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *    Jim Sermersheim - Initial cut
 *******************************************************************************/

package org.eclipse.higgins.idas.spi;

import java.net.URI;
import java.util.Hashtable;
import java.util.Iterator;
import java.util.Vector;

import org.apache.log4j.Logger;
import org.eclipse.higgins.idas.api.IAttribute;
import org.eclipse.higgins.idas.api.IAttributeValue;
import org.eclipse.higgins.idas.api.IContext;
import org.eclipse.higgins.idas.api.IHasAttributes;
import org.eclipse.higgins.idas.api.ISingleValuedAttribute;
import org.eclipse.higgins.idas.api.IdASException;
import org.eclipse.higgins.idas.api.InvalidTypeException;
import org.eclipse.higgins.idas.api.NotSingleValuedAttributeException;

/**
 * Basic implementation of {@link IHasAttributes}.
 * This class uses a {@link Hashtable} of {@link URI} and {@link IAttribute} to hold the attributes.
 * 
 */
public class BasicAttributeSet implements IHasAttributes, IAttributeContainer {
	private Logger _log = Logger.getLogger(BasicAttributeSet.class.getName());
	private Hashtable _attrs;
	private IAttributeContainer _attrContainer = this;
	private IContext _context;

	public BasicAttributeSet() {
		_attrs = new Hashtable();
	}

	/**
	 * 
	 * @param attrs Contains {@link IAttribute}s
	 * @param container May be null. IAttributeContainer to send update notifications to. 
	 * When null, this is the container.
	 * @param context May be null. IContext instance from which a model may be obtained if needed.  
	 */
	public BasicAttributeSet(
			Iterator attrs, 
			IAttributeContainer container,
			IContext context) throws IdASException {
		this();
		_context = context;
		if (container != null) {
			_attrContainer = container;
		}
		if (attrs != null) {
			while (attrs.hasNext()) {
				addAttribute((IAttribute)attrs.next());
			}
		}
	}

	public Iterator getAttributes() throws IdASException {
		return _attrs.values().iterator();
	}

	public IAttribute getAttribute(URI attrID) throws IdASException {
		return (IAttribute) _attrs.get(attrID);
	}
	
	public ISingleValuedAttribute getSingleValuedAttribute(URI attrID) throws IdASException, NotSingleValuedAttributeException {
		throw new NotSingleValuedAttributeException("BasicAttributeSet has no knowledge of the attributes' models, thus all attributes are treated as multi-value capable.");
	}	

	public IAttribute addAttribute(URI attrID) throws IdASException, InvalidTypeException {
		IAttribute attr = new BasicAttribute(attrID, (Iterator) null, this, _context);
		AttributeNotification attrNotif = new AttributeNotification(attr, AttributeNotification.UPDATE_ADD, null, null);
		this.updateNotification(attrNotif);
		return attr;
	}

	public IAttribute addAttribute(IAttribute copyFrom) throws IdASException {
		IAttribute attr = new BasicAttribute(copyFrom, this, _context);
		AttributeNotification attrNotif = new AttributeNotification(attr, AttributeNotification.UPDATE_ADD, null, null);
		this.updateNotification(attrNotif);
		return attr;
	}

	public void removeAttribute(URI attrID) throws IdASException {
		/* By creating a BasicAttribute and setting its container to this
		 * we can call remove on it and it will send us back the remove notification.
		 * In our remove notification, we will attempt to remove the actual attribute
		 * from our _attrs. If the attrID happens to point to an actual attr,
		 * it will be removed. Otherwise it will silently fail to remove the attr
		 * (which never existed in this set).  In either case, we get the notification
		 * which should be used by the underlying CP to do whatever it needs.
		 * 
		 * Note that 'new'ing up the BasicAttribute does *not* add the attr
		 * to our set.
		 */
		IAttribute attr = new BasicAttribute(attrID, (Iterator)null, this, _context);
		attr.remove();
	}

	public void removeAttributeValue(IAttribute attr) throws IdASException {
		BasicAttribute basicAttr;
		// First, let's see if this attribute exists in our set
		IAttribute localAttr = this.getAttribute(attr.getAttrID());
		if (localAttr != null) {
			/* We have this attr, so now we need to remove the values 
			 * (whether they exist or not)
			 */
			IAttributeValue passedVal, localVal;
			Iterator passedVals = attr.getValues();
			/* for each passed value, see if we have it.  If so,
			 * remove it locally, otherwise notify of its removal
			 * To notify, we'll store all the non-existant values
			 * into a vector of IAttributeValue and notify as if the
			 * attribute hadn't existed at all
			 */
			Vector nonExistantVals = new Vector();
			while (passedVals.hasNext()) {
				passedVal = (IAttributeValue)passedVals.next();
				Iterator localVals = localAttr.getValues();
				boolean bFoundLocalVal = false;
				while (localVals.hasNext()) {
					localVal = (IAttributeValue)localVals.next();
					if (localVal.equals(passedVal)) {
						localVal.remove();
						bFoundLocalVal = true;
						break;
					}
				}
				if (bFoundLocalVal == false) {
					// this passed value didn't exist, store for notification.
					nonExistantVals.add(passedVal);
				}
			}
			if (nonExistantVals.isEmpty() == false) {
				// set up basicAttr to contain only those values which we did not find
				basicAttr = new BasicAttribute(attr.getAttrID(), nonExistantVals.iterator(), null, _context);
			} else {
				// we got them all, nothing left to do
				return;
			}			
		} else {
			// set up basicAttr to contain the passed attr
			basicAttr = new BasicAttribute(attr, null, _context);
		}

		/* We need to do this only after creating the basic attribute and
		 * populating it with the values. Otherwise, the values will be seen
		 * as additions and those additions will be sent as notifications back
		 * to us. 
		 */
		basicAttr.setContainer(this);
		
		/* Now that we have the values to be deleted in a basic attribute 
		 * and it thinks this is it's container, we can remove the values. 
		 * This will cause remove notifications to be sent back to us.
		 */
		Iterator iter = basicAttr.getValues();
		IAttributeValue val;
		while (iter.hasNext()) {
			val = (IAttributeValue)iter.next();
			val.remove();
		}
	}

	public void removeAttributeValue(URI attrID, Object value) throws IdASException {
		IAttribute attr = new BasicAttribute(attrID, value, null, _context);
		this.removeAttributeValue(attr);		
	}


	public BasicAttribute addUnnotifiedAttribute(URI attrID) throws IdASException, InvalidTypeException {
		BasicAttribute attr = new BasicAttribute(attrID, (Iterator) null, null, _context);
		insertAttribute(attr);
		return attr;
	}

	public BasicAttribute addUnnotifiedAttribute(IAttribute copyFrom) throws IdASException {
		BasicAttribute attr = new BasicAttribute(copyFrom, null, _context);
		insertAttribute(attr);
		return attr;
	}

	/**
	 * Internal method which inserts the passed attribute.
	 * Does not make a copy.
	 * @param attr
	 * @throws IdASException
	 */
	private void insertAttribute(IAttribute attr) throws IdASException {
		_log.debug("Inserting attribute - " + attr.getAttrID());
		_attrs.put(attr.getAttrID(), attr);
	}

	private void _removeAttribute(URI attrID) throws IdASException {
		_log.debug("Removing attribute - " + attrID);
		_attrs.remove(attrID);
	}

	public void updateNotification(AttributeNotification attrNotif) throws IdASException {
		_log.debug("Received attribute update notification - " + attrNotif.getAction() + ", " + attrNotif.getAttr().getAttrID().toString());
		if (attrNotif.getAction().equals(AttributeNotification.UPDATE_ADD)) {
			insertAttribute(attrNotif.getAttr());
		} else if (attrNotif.getAction().equals(AttributeNotification.UPDATE_REMOVE)) {
			_removeAttribute(attrNotif.getAttr().getAttrID());
		}

		/* Other actions are simply passed to the next container if it's different from
		 * "this" container.
		 */
		if (_attrContainer != this) {
			_log.debug("Notifying containing container - " + _attrContainer);
			_attrContainer.updateNotification(attrNotif);
		}
	}

	public boolean equals(IHasAttributes attributes) throws IdASException {
		int localCount = 0, attrCount = 0;
		Iterator iter = attributes.getAttributes();
		while (iter.hasNext()) {
			IAttribute attr = (IAttribute)iter.next();
			if (this._contains(attr) == false) {
				return false;
			}
			attrCount++;
		}
		
		// everything we were given was found, make sure the sizes match
		iter = this.getAttributes();
		while (iter.hasNext()) {
			iter.next();
			localCount++;
		}
		return localCount == attrCount;
	}

	private boolean _contains(IAttribute attr) throws IdASException {
		Iterator iter = this.getAttributes();
		while (iter.hasNext()) {
			if (((IAttribute)iter.next()).equals(attr))
				return true;
		}
		return false;
	}

}
