/*******************************************************************************
 * Copyright (c) 2007 Novell, Inc..
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *   Duane Buss - Initial impl
 *******************************************************************************/
package org.eclipse.higgins.idas.spi;

import java.net.URI;
import java.util.Iterator;

import org.eclipse.higgins.idas.api.IAttribute;
import org.eclipse.higgins.idas.api.IContext;
import org.eclipse.higgins.idas.api.IDigitalSubject;
import org.eclipse.higgins.idas.api.IHasAttributes;
import org.eclipse.higgins.idas.api.IHasMetadata;
import org.eclipse.higgins.idas.api.IMetadata;
import org.eclipse.higgins.idas.api.ISingleValuedAttribute;
import org.eclipse.higgins.idas.api.IdASException;
import org.eclipse.higgins.idas.api.InvalidTypeException;
import org.eclipse.higgins.idas.api.NotImplementedException;
import org.eclipse.higgins.idas.api.NotSingleValuedAttributeException;
import org.eclipse.higgins.idas.api.model.IDigitalSubjectModel;

/**
 * Basic implementation of IDigitalSubject. The intent of this abstract class is
 * to be used by Context Provider writers as a superclass for IDigitalSubject
 * instances. The benefits of using this class are that you will pick up any
 * convenience methods as well as have the latest methods stubbed out such that
 * when new methods are added to IDigitalSubject and implemented here, the
 * subclass will still compile. <br>
 * The following methods are deliberatly not implemented in this abstract class
 * because they must be provided by the subclass :<br>
 * {@link  org.eclipse.higgins.idas.api.IDigitalSubject#remove() }<br>
 * {@link  IAttributeContainer#updateNotification(AttributeNotification)}<br>
 * {@link  IMetadataContainer#updateNotification(MetadataNotification)}<br>
 * 
 */
public abstract class BasicDigitalSubject implements IDigitalSubject,
	IAttributeContainer, IMetadataContainer
{

	private IContext _context;
	private String _subjectId;
	private URI _type;
	private BasicAttributeSet _attrs;
	private BasicMetadataSet _metadata;

	/**
	 * 
	 * @param context
	 * @throws IdASException
	 */
	public BasicDigitalSubject(
		IContext context) throws IdASException
	{
		this(context, null, null, null, null);
	}

	/**
	 * The prefered constructor
	 * 
	 * @param context
	 * @param type
	 * @param subjectId
	 * @throws IdASException
	 */
	public BasicDigitalSubject(
		IContext context,
		URI type,
		String subjectId) throws IdASException
	{
		this(context, type, subjectId, null, null);
	}

	/**
	 * The prefered constructor
	 * 
	 * @param context
	 * @param type
	 * @param subjectId
	 * @param attributes
	 * @param metadata
	 * @throws IdASException
	 */
	public BasicDigitalSubject(
		IContext context,
		URI type,
		String subjectId,
		Iterator attributes,
		Iterator metadata) throws IdASException
	{
		_context = context;
		_type = type;
		_subjectId = subjectId;

		_attrs = new BasicAttributeSet(attributes, this, _context);
		_metadata = new BasicMetadataSet(metadata, this);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.higgins.idas.api.IDigitalSubject#getContext()
	 */
	public IContext getContext() throws IdASException
	{
		return _context;
	}

	/*
	 * @see org.eclipse.higgins.idas.api.IDigitalSubject#getModel()
	 */
	public IDigitalSubjectModel getModel() throws IdASException
	{
		throw new NotImplementedException();
	}

	/*
	 * @see org.eclipse.higgins.idas.api.IDigitalSubject#getType()
	 */
	public URI getType() throws IdASException
	{
		return _type;
	}

	/*
	 * @see org.eclipse.higgins.idas.api.IDigitalSubject#getSubjectID()
	 */
	public String getSubjectID() throws IdASException
	{
		return _subjectId;
	}

	/*
	 * @see org.eclipse.higgins.idas.api.IHasMetadata#addMetadata(org.eclipse.higgins.idas.api.IMetadata)
	 */
	public IMetadata addMetadata(
		IMetadata copyFrom) throws IdASException
	{
		return _metadata.addMetadata(copyFrom);
	}

	/*
	 * @see org.eclipse.higgins.idas.api.IHasMetadata#addMetadata(java.net.URI)
	 */
	public IMetadata addMetadata(
		URI type) throws IdASException, InvalidTypeException
	{
		return _metadata.addMetadata(type);
	}

	/*
	 * @see org.eclipse.higgins.idas.api.IHasMetadata#getMetadata(java.net.URI)
	 */
	public IMetadata getMetadata(
		URI metadataID) throws IdASException
	{
		return _metadata.getMetadata(metadataID);
	}

	/*
	 * @see org.eclipse.higgins.idas.api.IHasMetadata#getMetadataSet()
	 */
	public Iterator getMetadataSet() throws IdASException
	{
		return _metadata.getMetadataSet();
	}

	/*
	 * @see org.eclipse.higgins.idas.api.IHasAttributes#addAttribute(org.eclipse.higgins.idas.api.IAttribute)
	 */
	public IAttribute addAttribute(
		IAttribute copyFrom) throws IdASException
	{
		return _attrs.addAttribute(copyFrom);
	}

	/*
	 * @see org.eclipse.higgins.idas.api.IHasAttributes#addAttribute(java.net.URI)
	 */
	public IAttribute addAttribute(
		URI type) throws IdASException, InvalidTypeException
	{
		return _attrs.addAttribute(type);
	}

	/*
	 * @see org.eclipse.higgins.idas.api.IHasAttributes#getAttribute(java.net.URI)
	 */
	public IAttribute getAttribute(
		URI attrID) throws IdASException
	{
		return _attrs.getAttribute(attrID);
	}
	
	/*
	 * @see org.eclipse.higgins.idas.api.IHasAttributes#getSingleValuedAttribute(java.net.URI)
	 */
	public ISingleValuedAttribute getSingleValuedAttribute(URI attrID) throws IdASException, NotSingleValuedAttributeException {
		return _attrs.getSingleValuedAttribute(attrID);
	}	
	

	/*
	 * @see org.eclipse.higgins.idas.api.IHasAttributes#getAttributes()
	 */
	public Iterator getAttributes() throws IdASException
	{
		return _attrs.getAttributes();
	}

	protected BasicAttributeSet getBasicAttributeSet()
	{
		return _attrs;
	}

	public void removeAttribute(URI attrID) throws IdASException {
		_attrs.removeAttribute(attrID);
	}

	public void removeAttributeValue(URI attrID, Object value)
	throws IdASException {
		_attrs.removeAttributeValue(attrID, value);
	}
	
	public void removeAttributeValue(IAttribute attr) throws IdASException {
		_attrs.removeAttributeValue(attr);
	}
	
	public boolean equals(IHasAttributes attributes) throws IdASException {
		return _attrs.equals(attributes);
	}

	protected BasicMetadataSet getBasicMetadataSet()
	{
		return _metadata;
	}

	public boolean equals(IHasMetadata metadataSet) throws IdASException {
		return _metadata.equals(metadataSet);
	}
	
}
