/*******************************************************************************
 * Copyright (c) 2006-2007 Novell, Inc..
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *    Jim Sermersheim - Initial cut
 *******************************************************************************/

package org.eclipse.higgins.idas.spi;

import org.eclipse.higgins.idas.api.ISimpleAttrValue;
import org.eclipse.higgins.idas.api.ITypedValue;
import org.eclipse.higgins.idas.api.IdASException;


public class BasicValueNormalizedString extends BasicValueString implements ISimpleAttrValue {
	
	public static final String WHITESPACE_RULE_PRESERVE = "preserve";
	public static final String WHITESPACE_RULE_REPLACE = "replace";
	public static final String WHITESPACE_RULE_COLLAPSE = "collapse";

	/**
	 * Constructs using a non-normalized string
	 * @param value Non-normalized string.
	 * @throws IdASException 
	 */
	public BasicValueNormalizedString(String value, IAttributeValueContainer container) throws IdASException {
		super();
		super.init(ITypedValue.NORMALIZEDSTRING_TYPE_URI, _normalize(value, WHITESPACE_RULE_REPLACE), container);
	}

	protected BasicValueNormalizedString() throws IdASException {
		// TODO Auto-generated constructor stub
	}

	/**
	 * Converts a string to a normalized string.
	 * Converts all tabs, line feeds, and carriage returns to spaces (http://www.w3.org/TR/xmlschema-2/#normalizedString)
	 * Converts occurrence of carriage return + line feed to a single space ( http://www.w3.org/TR/REC-xml/#sec-line-ends)
	 * 
	 * @param value String to be normalized
	 * @param whitespaceRule whiteSpace rule to be followed (http://www.w3.org/TR/xmlschema-2/#rf-whiteSpace)<br>
	 * One of {@link #WHITESPACE_RULE_PRESERVE}, {@link #WHITESPACE_RULE_REPLACE}, or {@link #WHITESPACE_RULE_COLLAPSE}
	 * @return the normalized string
	 */
	protected String _normalize(String value, String whitespaceRule)
	{
		if (whitespaceRule.equals(WHITESPACE_RULE_PRESERVE)) {
			return value;
		}

		int	iChar = 0,
			iLen = value.length();
		StringBuffer normVal = new StringBuffer(iLen);
		char c;
		
		for (; iChar < iLen; iChar++) {
			c = value.charAt(iChar);
			if (c != 0x09 && c != 0x0D && c != 0x0A) {
				normVal.append(c);
			}
			else {
				// Tread 0x0D,0x0A as a single space
				if ((c == 0x0D) && (value.charAt(iChar + 1) == 0x0A)) {
					iChar++;
				}
				if (whitespaceRule.equals(WHITESPACE_RULE_COLLAPSE)) {
					// No initial space
					if (normVal.length() == 0) {
						continue;
					}
					// Only single space
					if (normVal.charAt(normVal.length() - 1) == 0x20) {
						continue;
					}
				}
				normVal.append(' ');
			} 
		}
		// No trailing space
		if ((whitespaceRule.equals(WHITESPACE_RULE_COLLAPSE)) && (normVal.charAt(normVal.length() - 1) == 0x20)) {
			normVal.deleteCharAt(normVal.length() - 1);
		}
		return normVal.toString();
	}

}
