/*******************************************************************************
 * Copyright (c) 2006-2008 Novell, Inc..
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *    Jim Sermersheim
 *******************************************************************************/
package org.eclipse.higgins.idas.spi;

import java.net.URI;
import java.util.Iterator;
import java.util.Vector;

import org.apache.log4j.Logger;
import org.eclipse.higgins.idas.api.IComplexMetaValue;
import org.eclipse.higgins.idas.api.IContext;
import org.eclipse.higgins.idas.api.IHasMetadata;
import org.eclipse.higgins.idas.api.IMetadata;
import org.eclipse.higgins.idas.api.IMetadataValue;
import org.eclipse.higgins.idas.api.ISimpleMetaValue;
import org.eclipse.higgins.idas.api.IdASException;
import org.eclipse.higgins.idas.api.InvalidTypeException;
import org.eclipse.higgins.idas.api.NotImplementedException;
import org.eclipse.higgins.idas.api.model.IMetadataModel;

/**
 * Implements {@link IMetadata} 
 *
 */
public class BasicMetadata implements IMetadata  {
	private Logger _log = Logger.getLogger(BasicMetadata.class.getName());
	private URI _metaID;
	private Vector _values;
	private IMetadataContainer _container;
	private IContext _context;

	/**
	 * Note that the container's updateNotification is not called while 
	 * the values in copyFrom are being copied.
	 * @param copyFrom IMetadata from which to copy data from 
	 * @param container May be null. IMetadataContainer to send update notifications to. 
	 * @param context May be null. IContext instance from which a model may be obtained if needed.  
	 * @throws IdASException 
	 * 
	 */
	public BasicMetadata(
		IMetadata copyFrom, 
		IMetadataContainer container, 
		IContext context) throws IdASException {
		_context = context;
		_init(copyFrom.getMetaID(), copyFrom.getValues(), container);
	}

	/**
	 * @param metaID the URI of the metadata's type
	 * @param container May be null. IMetadataContainer to send update notifications to. 
	 * @throws IdASException 
	 * 
	 */
	public BasicMetadata(
		URI metaID, 
		IMetadataContainer container) throws IdASException {
		_context = null;
		_init(metaID, null, container);
	}

	/**
	 * @param metaID the URI of the metadata's type
	 * @param value an IMetadataValue (either simple or complex) to store with the metadata
	 * @param container May be null. IMetadataContainer to send update notifications to. 
	 * @param context May be null. IContext instance from which a model may be obtained if needed.  
	 * @throws IdASException 
	 * 
	 */
	public BasicMetadata(
		URI metaID, 
		IMetadataValue value, 
		IMetadataContainer container, 
		IContext context) throws IdASException {
		_context = context;
		if (value != null) {
			Vector v = new Vector(1);
			v.add(value);
			_init(metaID, v.iterator(), container);
		} else {
			_init(metaID, null, container);
		}
	}


	/**
	 * Note that the container's updateNotification is not called while 
	 * the IValues in values are being added.
	 * 
	 * @param metaID
	 * @param values Contains {@link IMetadataValue}s
	 * @param container May be null. IMetadataContainer to send update notifications to. 
	 * @param context May be null. IContext instance from which a model may be obtained if needed.  
	 * @throws IdASException 
	 */
	public BasicMetadata(
		URI metaID, 
		Iterator values, 
		IMetadataContainer container, 
		IContext context) throws IdASException {
		_context = context;
		_init(metaID, values, container);
	}
	
	/**
	 * Note that the container's updateNotification is not called while 
	 * the passed values are being added.
	 * @param metaID
	 * @param values Contains {@link IMetadataValue}s
	 * @throws IdASException 
	 */
	private void _init(
		URI metaID, 
		Iterator values, 
		IMetadataContainer container) throws IdASException {
		_metaID = metaID;
		_container = container;
		_values = new Vector();
		if (values != null) {
			while(values.hasNext()) {
				_addValue((IMetadataValue)values.next());
			}
		}
	}

	private void _addValue(IMetadataValue val) throws IdASException {
		_values.add(val);
		_log.debug("Sending value add notification - " + val.getModel().getType().toString());
		MetadataValueNotification valueNotif = new MetadataValueNotification(val, MetadataValueNotification.UPDATE_ADD, null);
		this.updateNotification(valueNotif);
	}
	
	private boolean _containsValue (IMetadataValue value) throws IdASException {
		Iterator iter = this.getValues();
		IMetadataValue localVal;
		boolean bRet = false;
		while (iter.hasNext()) {
			localVal = (IMetadataValue)iter.next();
			if (localVal.equals(value)) {
				bRet = true;
				break;
			}
		}
		return bRet;
	}

	public URI getMetaID() throws IdASException {
		return _metaID;
	}

	public Iterator getValues() throws IdASException {
		/* TODO: Vector doesn't scale anyway, we need to fix this!
				However, the big deal here is the fact that we can't
				modify the thing we're iterating on, so, we're going
				with a clone temporarily. */
		return ((Vector)_values.clone()).iterator();
	}

	// Subclass must implement this as we have no model
	public IMetadataModel getModel() throws IdASException {
		if (_context != null)
			return (IMetadataModel)_context.getContextModel().getModel(this.getMetaID());
		else
			throw new NotImplementedException("No associated context to obtain model");
	}

	public IComplexMetaValue addComplexValue(URI type) throws IdASException, InvalidTypeException {
		throw new NotImplementedException();
	}

	public ISimpleMetaValue addSimpleValue(URI type, Object data) throws IdASException, InvalidTypeException {
		throw new NotImplementedException();
	}

	public IMetadataValue addValue(URI type) throws IdASException, InvalidTypeException {
		throw new NotImplementedException();
	}

	public IMetadataValue addValue(IMetadataValue copyFrom) throws IdASException {
		throw new NotImplementedException();
	}

	public void remove() throws IdASException {
		if (_container != null) {
			MetadataNotification metaNotif = new MetadataNotification(this, MetadataNotification.UPDATE_REMOVE, null);
			_log.debug("Sending metadata remove notification - " + metaNotif.getAction() + ", " + metaNotif.getMeta().getMetaID().toString());
			_container.updateNotification(metaNotif);
		}
	}

	public boolean isSingleValued() {
		return false;
	}
	
	public void updateNotification(MetadataValueNotification metaValueNotif) throws IdASException {
		_log.debug("Received metadata value notification - " + metaValueNotif.getAction() + ", " + metaValueNotif.getMetadataValue().toString());
		_log.debug("_containter " + ((_container != null) ? _container.toString() : "null"));
		if (_container != null) {
			MetadataNotification metaNotif = new MetadataNotification(this, MetadataNotification.UPDATE_VALUE_NOTIFY,
				metaValueNotif);
			_container.updateNotification(metaNotif);
		}
		if (metaValueNotif.getAction() == MetadataValueNotification.UPDATE_REMOVE)
			_values.remove(metaValueNotif.getMetadataValue());
	}

	public void setContainer(IMetadataContainer container) {
		_container = container;
	}
	
	public IMetadataContainer getContainer() {
		return _container;
	}

	public boolean equals(IMetadata meta) throws IdASException {
		boolean bRet;
		if (meta.getMetaID() != this.getMetaID())
			bRet = false;
		else if ((bRet = this.equals((IHasMetadata)meta)) == true) {
			Iterator iter = meta.getValues();
			// Compare each passed value
			while (iter.hasNext()) {
				if (this._containsValue((IMetadataValue)iter.next()) == false) {
					bRet = false;
					break;
				}					
			}
		}
		return bRet;
	}
	
}
