/*******************************************************************************
 * Copyright (c) 2007-2008 Novell, Inc..
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *    Jim Sermersheim
 *******************************************************************************/
package org.eclipse.higgins.idas.spi;

import java.lang.reflect.Constructor;
import java.lang.reflect.InvocationTargetException;
import java.net.URI;
import java.util.Enumeration;
import java.util.HashMap;
import java.util.Hashtable;
import java.util.Iterator;
import java.util.Vector;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.eclipse.higgins.idas.api.ContextNotOpenException;
import org.eclipse.higgins.idas.api.ContextOpenException;
import org.eclipse.higgins.idas.api.IAttribute;
import org.eclipse.higgins.idas.api.IAuthNAttributesMaterials;
import org.eclipse.higgins.idas.api.IComplexAttrValue;
import org.eclipse.higgins.idas.api.IContext;
import org.eclipse.higgins.idas.api.IEntity;
import org.eclipse.higgins.idas.api.IExtension;
import org.eclipse.higgins.idas.api.IFilter;
import org.eclipse.higgins.idas.api.IFilterAttributeAssertion;
import org.eclipse.higgins.idas.api.IFilterEntityIDAssertion;
import org.eclipse.higgins.idas.api.IFilterEntityTypeAssertion;
import org.eclipse.higgins.idas.api.ISimpleAttrValue;
import org.eclipse.higgins.idas.api.ITypedValue;
import org.eclipse.higgins.idas.api.IdASException;
import org.eclipse.higgins.idas.api.InvalidEntityIDException;
import org.eclipse.higgins.idas.api.InvalidTypeException;
import org.eclipse.higgins.idas.api.NotImplementedException;
import org.eclipse.higgins.idas.api.EntityExistsException;
import org.eclipse.higgins.idas.api.UnhandledExtensionException;
import org.eclipse.higgins.idas.api.model.IContextModel;

/**
 * Basic implementation of IContext.  
 * The intent of this class is to be used by Context Provider 
 * writers as a superclass for IContext instances.  The benefits of 
 * using this class are that you will pick up any convenience methods
 * as well as have the latest methods stubbed out such that when new
 * methods are added to IContext and implemented here, the subclass
 * will still compile.  
 *
 */
public class BasicContext extends BasicAttributeSet implements IContext, IEntityContainer, IAttributeContainer
{
	private static Hashtable typeToBasicMap = null;

	public static ISimpleAttrValue createSimpleValue(URI type, Object data, IAttributeValueContainer container) throws IdASException {
		initTypeToBasicMap();
		Class valClass = (Class) typeToBasicMap.get(type);
		Class dataClass = data.getClass();
		Class ctorClasses[] = {dataClass, IAttributeValueContainer.class};
		Object ctorArgs[] = {data, container};
		Constructor ctor = null;
		if (valClass == null) {
			Class basicCtorClasses[] = {URI.class, Object.class, IAttributeValueContainer.class};
			Object basicCtorArgs[] = {type, data, container};
			valClass = BasicSimpleValue.class;
			ctorClasses = basicCtorClasses;
			ctorArgs = basicCtorArgs;
		}
		try {
			for (;;) {
				try {
					ctor = valClass.getConstructor(ctorClasses);
					break;
				} catch (NoSuchMethodException e) {
					dataClass = dataClass.getSuperclass();
					if (dataClass == null)
						throw e;
					ctorClasses[0] = dataClass;
				}
			}
			Object o = ctor.newInstance(ctorArgs);
			return (ISimpleAttrValue) o;
		} catch (SecurityException e) {
			throw new IdASException(e);
		} catch (NoSuchMethodException e) {
			throw new IdASException(e);
		} catch (IllegalArgumentException e) {
			throw new IdASException(e);
		} catch (InstantiationException e) {
			throw new IdASException(e);
		} catch (IllegalAccessException e) {
			throw new IdASException(e);
		} catch (InvocationTargetException e) {
			throw new IdASException(e);
		}
	}
	
	private static void initTypeToBasicMap() {
		if (typeToBasicMap != null)
			return;
		
		typeToBasicMap = new Hashtable();
		typeToBasicMap.put(ITypedValue.STRING_TYPE_URI, BasicValueString.class);
		typeToBasicMap.put(ITypedValue.NORMALIZEDSTRING_TYPE_URI, BasicValueNormalizedString.class); 
		typeToBasicMap.put(ITypedValue.BOOLEAN_TYPE_URI, BasicValueBoolean.class); 
		typeToBasicMap.put(ITypedValue.DECIMAL_TYPE_URI, BasicValueDecimal.class); 
		typeToBasicMap.put(ITypedValue.INTEGER_TYPE_URI, BasicValueInteger.class); 
		typeToBasicMap.put(ITypedValue.NONNEGATIVEINTEGER_TYPE_URI, BasicValueNonNegativeInteger.class); 
		typeToBasicMap.put(ITypedValue.POSITIVEINTEGER_TYPE_URI, BasicValuePositiveInteger.class); 
		typeToBasicMap.put(ITypedValue.NONPOSITIVEINTEGER_TYPE_URI, BasicValueNonPositiveInteger.class); 
		typeToBasicMap.put(ITypedValue.NEGATIVEINTEGER_TYPE_URI, BasicValueNegativeInteger.class); 
		typeToBasicMap.put(ITypedValue.FLOAT_TYPE_URI, BasicValueFloat.class); 
		typeToBasicMap.put(ITypedValue.DOUBLE_TYPE_URI, BasicValueDouble.class); 
		typeToBasicMap.put(ITypedValue.LONG_TYPE_URI, BasicValueLong.class); 
		typeToBasicMap.put(ITypedValue.INT_TYPE_URI, BasicValueInt.class); 
		typeToBasicMap.put(ITypedValue.SHORT_TYPE_URI, BasicValueShort.class); 
		typeToBasicMap.put(ITypedValue.BYTE_TYPE_URI, BasicValueByte.class); 
		typeToBasicMap.put(ITypedValue.UNSIGNEDLONG_TYPE_URI, BasicValueUnsignedLong.class); 
		typeToBasicMap.put(ITypedValue.UNSIGNEDINT_TYPE_URI, BasicValueUnsignedInt.class); 
		typeToBasicMap.put(ITypedValue.UNSIGNEDSHORT_TYPE_URI, BasicValueUnsignedShort.class); 
		typeToBasicMap.put(ITypedValue.UNSIGNEDBYTE_TYPE_URI, BasicValueUnsignedByte.class); 
		typeToBasicMap.put(ITypedValue.BASE64BINARY_TYPE_URI, BasicValueBase64Binary.class); 
		typeToBasicMap.put(ITypedValue.HEXBINARY_TYPE_URI, BasicValueHexBinary.class); 
		typeToBasicMap.put(ITypedValue.DATETIME_TYPE_URI, BasicValueDateTime.class); 
		typeToBasicMap.put(ITypedValue.TIME_TYPE_URI, BasicValueTime.class); 
		typeToBasicMap.put(ITypedValue.DATE_TYPE_URI, BasicValueDate.class); 
		typeToBasicMap.put(ITypedValue.GYEARMONTH_TYPE_URI, BasicValueGYearMonth.class); 
		typeToBasicMap.put(ITypedValue.GYEAR_TYPE_URI, BasicValueGYear.class); 
		typeToBasicMap.put(ITypedValue.GMONTHDAY_TYPE_URI, BasicValueGMonthDay.class); 
		typeToBasicMap.put(ITypedValue.GDAY_TYPE_URI, BasicValueGDay.class); 
		typeToBasicMap.put(ITypedValue.GMONTH_TYPE_URI, BasicValueGMonth.class); 
		typeToBasicMap.put(ITypedValue.ANYURI_TYPE_URI, BasicValueAnyURI.class); 
		typeToBasicMap.put(ITypedValue.TOKEN_TYPE_URI, BasicValueToken.class); 
		typeToBasicMap.put(ITypedValue.LANGUAGE_TYPE_URI, BasicValueLanguage.class); 
		typeToBasicMap.put(ITypedValue.NMTOKEN_TYPE_URI, BasicValueNMTOKEN.class); 
		typeToBasicMap.put(ITypedValue.NAME_TYPE_URI, BasicValueName.class); 
		typeToBasicMap.put(ITypedValue.NCNAME_TYPE_URI, BasicValueNCName.class);
	}

	public static boolean isSimpleType(URI type) {
		initTypeToBasicMap();
		return typeToBasicMap.containsKey(type);
	}

	private boolean bIsOpen = false;
	private IContextModel contextModel;
	private HashMap entities = new HashMap();
	private Log	 log = LogFactory.getLog(BasicContext.class.getName());
	/**
	 * This is a hashtable where the keys are entityIDs and the values are vectors of EntityNotifications
	 */
	private Hashtable updateTable = new Hashtable();

	public BasicContext() throws IdASException {
		super();
		contextModel = new BasicContextModel();
	}
	
	public IEntity addEntity(IEntity copyFrom) throws IdASException, EntityExistsException {
		if (!isOpen(null))
			throw new ContextNotOpenException();
		
		IEntity entity = this.addEntity(copyFrom.getModel().getType(), copyFrom.getEntityID());
		Iterator attrs = copyFrom.getAttributes();
		while (attrs.hasNext()) {
			entity.addAttribute((IAttribute)attrs.next());
		}
		return entity;
	}

	public IEntity addEntity(URI type, String entityID) throws IdASException, InvalidTypeException, InvalidEntityIDException, EntityExistsException {
		if (!isOpen(null))
			throw new ContextNotOpenException();
//TODO handle cases where entityID == null
		IEntity entity = new BasicEntity((BasicContext)this, type, entityID);
		EntityNotification eNotif = new EntityNotification(entity, EntityNotification.UPDATE_ADD, null);
		addEntityNotification(entityID, eNotif);
		IAttribute eid = entity.addAttribute(URI.create("http://www.eclipse.org/higgins/ontologies/2008/6/higgins#entityId"));
		eid.addSimpleValue(ISimpleAttrValue.STRING_TYPE_URI, entityID);		
		return entity;
	}

	private void addEntityNotification(String entityID, EntityNotification notif) {
		Vector updateNotifs = (Vector)updateTable.get(entityID);
		if (updateNotifs == null)
			updateNotifs = new Vector();
		updateNotifs.add(notif);
		updateTable.put(entityID, updateNotifs);
	}

	public void applyUpdates() throws IdASException {
		if (!isOpen(null))
			throw new ContextNotOpenException();
		Enumeration enu = updateTable.elements();
		while (enu.hasMoreElements()) {
			Vector v = (Vector)enu.nextElement();
			Iterator notifs = v.iterator();
			while (notifs.hasNext()) {
				EntityNotification eNotif = (EntityNotification)notifs.next();
				if (eNotif.getAction().equals(EntityNotification.UPDATE_ADD)) {
					IEntity entity = eNotif.getEntity();
					entities.put(entity.getEntityID(), entity);
				} else if (eNotif.getAction().equals(EntityNotification.UPDATE_REMOVE)) {
					entities.remove(eNotif.getEntity().getEntityID());
				} else if (eNotif.getAction().equals(EntityNotification.UPDATE_ATTR_NOTIFY)) {
					// No action needed as the in-mem entity should already hold the attribute(s)
				} else {
					throw new IdASException("Invalid Entity Notification Action: " + eNotif.getAction());
				}
			}
		}
		updateTable.clear();

		// Subclasses should override this. Otherwise all updates only happen to in-memory entities and attributes
	}

	public IAttribute buildAttribute(URI attrID) throws IdASException {
		return new BasicAttribute(attrID, (Iterator)null, null, this);
	}

	public IFilterAttributeAssertion buildAttributeAssertion() throws IdASException {
		return new BasicFilterAttributeAssertion();
	}

	public IAuthNAttributesMaterials buildAuthNAttributesMaterials() throws IdASException {
		return new BasicAuthNAttributesMaterials(null, this);
	}

	public IComplexAttrValue buildComplexAttrValue(URI type) throws IdASException, InvalidTypeException {
		throw new NotImplementedException();
	}

	public IFilterEntityIDAssertion buildEntityIDAssertion() throws IdASException {
		return new BasicFilterEntityIDAssertion();
	}

	public IFilterEntityTypeAssertion buildEntityTypeAssertion() throws IdASException {
		return new BasicFilterEntityTypeAssertion();
	}

	public IFilter buildFilter() throws IdASException {
		return new BasicFilter();
	}

	public ISimpleAttrValue buildSimpleAttrValue(URI type, Object data) throws IdASException {
		return createSimpleValue(type, data, null);
	}
	
	public void cancelUpdates() throws IdASException {
		if (!isOpen(null))
			throw new ContextNotOpenException();
		updateTable.clear();
	}

	/* (non-Javadoc)
	 * @see org.eclipse.higgins.idas.api.IContext#close()
	 */
	public void close() throws IdASException {
		if (!isOpen(null))
			throw new ContextNotOpenException();

		entities.clear();
		updateTable.clear();
		setOpen(false);
	}

	public void setOpen(boolean b) {
		bIsOpen = b;
	}

	/* (non-Javadoc)
	 * @see org.eclipse.higgins.idas.api.IContext#exportData(java.lang.String, java.lang.String)
	 */
	public String exportData(String filter, String representationFormat)
			throws IdASException {
		throw new NotImplementedException();
	}

	/* (non-Javadoc)
	 * @see org.eclipse.higgins.idas.api.IContext#getContextID()
	 */
	public URI getContextID() throws IdASException {
		throw new NotImplementedException();
	}

	/* (non-Javadoc)
	 * @see org.eclipse.higgins.idas.api.IContext#getContextModel()
	 */
	public IContextModel getContextModel() throws IdASException {
		return contextModel;
	}

	/* (non-Javadoc)
	 * @see org.eclipse.higgins.idas.api.IContext#getEntities(org.eclipse.higgins.idas.api.IFilter)
	 */
	public Iterator getEntities(IFilter filter)
			throws IdASException {
		if (!isOpen(null))
			throw new ContextNotOpenException();
		return this.getEntities(filter, null);
	}

	/* (non-Javadoc)
	 * @see org.eclipse.higgins.idas.api.IContext#getEntities(org.eclipse.higgins.idas.api.IFilter, java.lang.Iterable)
	 */
	public Iterator getEntities(IFilter filter,
			Iterator attrSelectionList) throws IdASException {
		if (!isOpen(null))
			throw new ContextNotOpenException();
		if (filter != null) {
			throw new NotImplementedException("BasicContext does not yet support filters");
		}
		// Ignore attrSelectionList for now
		return entities.values().iterator();
	}

	/* (non-Javadoc)
	 * @see org.eclipse.higgins.idas.api.IContext#getEntity(java.lang.String)
	 */
	public IEntity getEntity(String entityID) throws IdASException {
		if (!isOpen(null))
			throw new ContextNotOpenException();
		return this.getEntity(entityID, null);
	}

	/* (non-Javadoc)
	 * @see org.eclipse.higgins.idas.api.IContext#getEntity(java.lang.String, java.lang.Iterable)
	 */
	public IEntity getEntity(String entityID,
			Iterator attrSelectionList) throws IdASException {
		if (!isOpen(null))
			throw new ContextNotOpenException();
		// Ignore attrSelectionList for now
		IEntity entity = (IEntity) entities.get(entityID);
		return entity;
	}

	/* (non-Javadoc)
	 * @see org.eclipse.higgins.idas.api.IContext#getSchema()
	 */
	public String getSchema() throws IdASException {
		throw new NotImplementedException();
	}

	protected Hashtable getUpdateList()
	{
		return updateTable;
	}

	/* (non-Javadoc)
	 * @see org.eclipse.higgins.idas.api.IContext#importData(java.lang.String, java.lang.String)
	 */
	public void importData(String filter, String representationFormat)
			throws IdASException {
		throw new NotImplementedException();
	}

	/* (non-Javadoc)
	 * @see org.eclipse.higgins.idas.api.IContext#isOpen(java.lang.Object)
	 */
	public boolean isOpen(Object identity) throws IdASException {
		return bIsOpen;
	}

	/* (non-Javadoc)
	 * @see org.eclipse.higgins.idas.api.IContext#open(java.lang.Object, IExtension)
	 */
	public String open(
		Object authentication, 
		IExtension[] extensions) throws IdASException, ContextOpenException {
		if (isOpen(null))
			throw new ContextOpenException();
		
		if (extensions != null) {
			for (int index = 0; index < extensions.length; index++) {
				if (extensions[index].failIfUnsupported() == true) {
					throw new UnhandledExtensionException(extensions[index].getClass().getName() + " not supported");
				}
			}
		}

		setOpen(true);

		// TODO: return entityID
		return null;
	}
	/* (non-Javadoc)
	 * @see org.eclipse.higgins.idas.api.IContext#open(java.lang.Object)
	 */
	public String open(Object authentication) throws IdASException,
			ContextOpenException {
		return this.open(authentication, null);
	}

	public void setComponentSetting(String key, Object value) throws IdASException {
		setComponentSetting(key, value, false);	
	}

	public void setComponentSetting(String key, Object value,
			boolean failUnsupported) throws IdASException {
		if (failUnsupported)
			throw new NotImplementedException("Setting type: " + key + "not implmented.");
	}
	
	/* (non-Javadoc)
	 * @see org.eclipse.higgins.idas.api.IContext#setSchema(java.lang.String)
	 */
	public void setSchema(String schema) throws IdASException {
		throw new NotImplementedException();
	}
	
	public void updateNotification(EntityNotification entityNotif) throws IdASException {
		String entityID = entityNotif.getEntity().getEntityID();
		log.debug("Received entity update notifcation - " + entityID);
		addEntityNotification(entityID, entityNotif);
	}

	/* (non-Javadoc)
	 * @see org.eclipse.higgins.idas.api.IContext#verifyEntityAttributes(java.lang.String, java.lang.Iterable)
	 */
	public boolean verifyEntityAttributes(String entityID,
			Iterator attributes) throws IdASException {
		throw new NotImplementedException();
	}

}
