/*******************************************************************************
 * Copyright (c) 2006-2008 Novell, Inc..
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *    Jim Sermersheim - Initial cut
 *******************************************************************************/

package org.eclipse.higgins.idas.spi;

import java.net.URI;

import org.eclipse.higgins.idas.api.IAttributeValue;
import org.eclipse.higgins.idas.api.IHasAttributes;
import org.eclipse.higgins.idas.api.ISimpleAttrValue;
import org.eclipse.higgins.idas.api.IdASException;
import org.eclipse.higgins.idas.api.model.IAttributeSimpleValueModel;

public abstract class AbstractSimpleValue extends BasicAttributeSet implements ISimpleAttrValue {
	private Object data;
	private IAttributeValueContainer container;
	private IAttributeSimpleValueModel model;

	public AbstractSimpleValue() throws IdASException {
	}

	/**
	 * Note that the container's updateNotification is not called while 
	 * the value in data is being added.
	 * @param type
	 * @param data
	 * @param container
	 */
	protected void init(URI type, Object data, IAttributeValueContainer container) {
		this.data = data;
		this.container = container;
		this.model = new BasicSimpleValueModel(type);
	}

	/**
	 * Default implementation of getLexical calls toString() on the backing datatype.
	 */
	public String getLexical() throws IdASException {
		Object data = this.getData();
		if (data == null)
			throw new IdASException("Data not yet set on this object. DataType=" + this.getDataType() + ", Object=" + this.toString()); 
		return data.toString();
	}

	/**
	 * Default implementation of getCanonical calls toString() on the backing datatype.
	 */
	public String getCanonical() throws IdASException {
		Object data = this.getData();
		if (data == null)
			throw new IdASException("Data not yet set on this object. DataType=" + this.getDataType() + ", Object=" + this.toString()); 
		return data.toString();
	}

	/**
	 * returns true
	 */
	public boolean isSimple() throws IdASException {
		return true;
	}

	public URI getDataType() throws IdASException {
		IAttributeSimpleValueModel model = this.getModel();
		if (model == null)
			throw new IdASException("Model not yet set on this object. DataType=" + this.getDataType() + ", Object=" + this.toString()); 
		return model.getType();
	}
	
	public URI getValueType() throws IdASException {
		IAttributeSimpleValueModel model = this.getModel();
		if (model == null)
			throw new IdASException("Model not yet set on this object. DataType=" + this.getDataType() + ", Object=" + this.toString()); 
		return model.getType();
	}
	
	/**
	 * Returns the data object backing this value
	 */
	public Object getData() throws IdASException {
		return data;
	}
	
	/**
	 * @param data The backing data object to be set.
	 */
	public void setData(Object data) throws IdASException {
		if (container != null) {
			AttributeValueNotification attrValueNotif = new AttributeValueNotification(this, AttributeValueNotification.UPDATE_SET_DATA, this.data);
			container.updateNotification(attrValueNotif);
		}
		this.data = data;
	}
	
	public void remove() throws IdASException {
		if (container != null) {
			AttributeValueNotification attrValueNotif = new AttributeValueNotification(this, AttributeValueNotification.UPDATE_REMOVE, null);
			container.updateNotification(attrValueNotif);
		}
	}
	
	public boolean equals(IAttributeValue value) throws IdASException {
		if (value.isSimple() == false)
			return false;

		ISimpleAttrValue val = (ISimpleAttrValue)value;
		if (this.getValueType() != val.getValueType())
			return false;
		if (this.equals((IHasAttributes)value) == false)
			return false;
		return this.getCanonical().equals(val.getCanonical());
	}
	
	public IAttributeSimpleValueModel getModel() throws IdASException {
		return model;
	}
	
}
