 /*******************************************************************************
 * Copyright (c) 2006-2008 Novell, Inc..
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *    Jim Sermersheim - Initial cut
 *    Duane Buss - Minor tweaks and fixes
 *******************************************************************************/
package org.eclipse.higgins.idas.spi;

import java.net.URI;
import java.util.Iterator;
import java.util.Vector;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.eclipse.higgins.idas.api.IAttribute;
import org.eclipse.higgins.idas.api.IComplexAttrValue;
import org.eclipse.higgins.idas.api.IContext;
import org.eclipse.higgins.idas.api.IExtension;
import org.eclipse.higgins.idas.api.IHasAttributes;
import org.eclipse.higgins.idas.api.ISimpleAttrValue;
import org.eclipse.higgins.idas.api.IAttributeValue;
import org.eclipse.higgins.idas.api.ISingleValuedAttribute;
import org.eclipse.higgins.idas.api.ITypedValue;
import org.eclipse.higgins.idas.api.IdASException;
import org.eclipse.higgins.idas.api.InvalidTypeException;
import org.eclipse.higgins.idas.api.NotImplementedException;
import org.eclipse.higgins.idas.api.NotSingleValuedAttributeException;
import org.eclipse.higgins.idas.api.UnhandledExtensionException;
import org.eclipse.higgins.idas.api.model.IAttributeModel;
import org.eclipse.higgins.idas.api.model.IAttributeSimpleValueModel;
import org.eclipse.higgins.idas.api.model.IAttributeValueModel;
import org.eclipse.higgins.idas.api.model.IContextModel;
import org.eclipse.higgins.idas.api.model.IModel;
import org.eclipse.higgins.idas.common.DereferenceAttrValuesExtension;

/**
 * Implements IAttribute using {@link BasicAttribute} and {@link BasicAttributeSet}
 */
public class BasicAttribute implements IAttribute, IAttributeValueContainer, IAttributeContainer {
	private Log	 log = LogFactory.getLog(BasicAttribute.class.getName());
	private URI _attrID;
	private Vector _values;
	private IAttributeContainer _container;
	private IContext _context;
	private BasicAttributeSet _metaAttrs;

	/**
	 * Note that the container's updateNotification is not called while 
	 * the values in copyFrom are being copied.
	 * @param copyFrom IAttribute from which to copy data from 
	 * @param container May be null. IAttributeContainer to send update notifications to. 
	 * @param context May be null. IContext instance from which a model may be obtained if needed.  
	 * @throws IdASException 
	 * 
	 */
	public BasicAttribute(
		IAttribute copyFrom, 
		IAttributeContainer container, 
		IContext context) throws IdASException {
		_init(copyFrom.getAttrID(), copyFrom.getValues(), copyFrom.getAttributes(), container, context);
	}

	/**
	 * @param attrID the URI of the attribute's type
	 * @param value an IAttributeValue (either simple or complex) to store with the attribute
	 * @param container May be null. IAttributeContainer to send update notifications to. 
	 * @param context May be null. IContext instance from which a model may be obtained if needed.  
	 * @throws IdASException 
	 * 
	 */
	public BasicAttribute(
		URI attrID, 
		IAttributeValue value, 
		IAttributeContainer container, 
		IContext context) throws IdASException {
		if (value != null) {
			Vector v = new Vector(1);
			v.add(value);
			_init(attrID, v.iterator(), null, container, context);
		} else {
			_init(attrID, null, null, container, context);
		}
	}

	/**
	 * @param attrID the URI of the attribute's type
	 * @param data an Object to be used as a value for attribute. 
	 * The Object is the data from which an {@link ISimpleAttrValue} 
	 * will be created. We do this by looking at the model for the attribute (specified in attrID)
	 * and associating that with the type of data object that is expected.
	 * If the value passed is of the wrong type, the constructor will fail.
	 * Note that this assumes the data will be used to construct a simple 
	 * value, not a complex value.  If one wishes to construct a BasicAttribute with a 
	 * complex value, then {@link #BasicAttribute(URI, IAttributeValue, IAttributeContainer, IContext)} is called.  
	 * @param container for purposes of notification of updates, the container of this attribute.  
	 * May be null when there is no containing object
	 * @param context May be null. IContext instance from which a model may be obtained if needed.  
	 * @throws IdASException 
	 * 
	 */
	public BasicAttribute(
		URI attrID, 
		Object data, 
		IAttributeContainer container, 
		IContext context) throws IdASException {
		// Turn the value into an IAttribute 
		IAttributeValueModel valueModel = this.getModel().getValueModel();
		URI valueType = valueModel.getType();
		ISimpleAttrValue value = BasicContext.createSimpleValue(valueType, data, this);
		
		if (value != null) {
			Vector v = new Vector(1);
			v.add(value);
			_init(attrID, v.iterator(), null, container, context);
		} else {
			_init(attrID, null, null, container, context);
		}
	}

	/**
	 * Note that the container's updateNotification is not called while 
	 * the IValues in values are being added.
	 * 
	 * @param attrID
	 * @param values Contains {@link IAttributeValue}s
	 * @param container May be null. IAttributeContainer to send update notifications to. 
	 * @param context May be null. IContext instance from which a model may be obtained if needed.  
	 * @throws IdASException 
	 */
	public BasicAttribute(
		URI attrID, 
		Iterator values, 
		IAttributeContainer container, 
		IContext context) throws IdASException {
		_init(attrID, values, null, container, context);
	}
	
	/**
	 * Note that the container's updateNotification is not called while 
	 * the passed values and metaSet are being added.
	 * @param attrID
	 * @param values Contains {@link IAttributeValue}s
	 * @throws IdASException 
	 */
	private void _init(
		URI attrID, 
		Iterator values, 
		Iterator metaSet, 
		IAttributeContainer container,
		IContext context) throws IdASException {
		_attrID = attrID;
		_container = container;
		_values = new Vector();
		_context = context;
		if (values != null) {
			while(values.hasNext()) {
				addValue((IAttributeValue)values.next());
			}
		}
		_metaAttrs = new BasicAttributeSet(metaSet, this, context);
	}

	private void _addValue(IAttributeValue val) throws IdASException {
		_values.add(val);
		log.debug("Sending value add notification - " + val.getValueType());
		AttributeValueNotification valueNotif = new AttributeValueNotification(val, AttributeValueNotification.UPDATE_ADD, null);
		this.updateNotification(valueNotif);
	}
	
	private boolean _containsValue (IAttributeValue value) throws IdASException {
		Iterator iter = this.getValues();
		IAttributeValue localVal;
		boolean bRet = false;
		while (iter.hasNext()) {
			localVal = (IAttributeValue)iter.next();
			if (localVal.equals(value)) {
				bRet = true;
				break;
			}
		}
		return bRet;
	}

	public URI getAttrID() throws IdASException {
		return _attrID;
	}

	public Iterator getValues() throws IdASException {
		return getValues((DereferenceAttrValuesExtension)null);
	}
	// Subclass must implement this as we have no model
	private IModel getTypeModel( URI type) throws IdASException {
		IContextModel ctxModel;
		IModel valueModel = null;
		
		if ( type != null) {
			if (_context != null)
				ctxModel = _context.getContextModel();
			else
				ctxModel = new BasicContextModel();
			valueModel = ctxModel.getModel( type);
			if (valueModel == null) {
				throw new IdASException("Context model (" + ctxModel.getClass() + ") returned no value model for " + type);
			}
		}
		return valueModel; 
	}
	
	// Subclass must implement this as we have no model
	public IAttributeModel getModel() throws IdASException {
		if (_context != null)
			return (IAttributeModel)_context.getContextModel().getModel(this.getAttrID());
		else
			throw new NotImplementedException("No associated context to obtain model");
	}

	public IComplexAttrValue addComplexValue(URI type) throws IdASException, InvalidTypeException {
		IComplexAttrValue val = new BasicComplexValue(type, null, this);
		_addValue(val);
		return val;
	}

	public ISimpleAttrValue addSimpleValue(URI type, Object data) throws IdASException, InvalidTypeException {
		ISimpleAttrValue val = BasicContext.createSimpleValue(type, data, this);
		_addValue(val);
		return val;
	}

	public IAttributeValue addValue(URI type) throws IdASException, InvalidTypeException {
		IModel valueModel = this.getTypeModel(type);
		
		IAttributeValue val;
		if (valueModel instanceof IAttributeSimpleValueModel)
			val = new BasicSimpleValue(type, null, this);
		else
			val = new BasicComplexValue(type, null, this);
		_addValue(val);
		return val;
	}

	public IAttributeValue addValue(IAttributeValue copyFrom) throws IdASException {
		IAttributeValue val;
		if (copyFrom.isSimple()) {
			ISimpleAttrValue simpleCopyFrom = (ISimpleAttrValue)copyFrom;
			val = new BasicSimpleValue(simpleCopyFrom.getModel().getType(), simpleCopyFrom.getData(), this);
		} else {
			IComplexAttrValue complexCopyFrom = (IComplexAttrValue)copyFrom;
			val = new BasicComplexValue(complexCopyFrom.getModel().getType(), complexCopyFrom.getAttributes(), this);
		}
		_addValue(val);
		return val;
	}

	public void remove() throws IdASException {
		if (_container != null) {
			AttributeNotification attrNotif = new AttributeNotification(this, AttributeNotification.UPDATE_REMOVE, null, null);
			log.debug("Sending attribute remove notification - " + attrNotif.getAction() + ", " + attrNotif.getAttr().getAttrID().toString());
			_container.updateNotification(attrNotif);
		}
	}

	public boolean isSingleValued() throws IdASException {
		IAttributeModel attrModel = null;
		attrModel = this.getModel();
		if ( attrModel != null && attrModel.getMaxCardinality() == 1)
			return true;
		
		return false;
	}

	public void updateNotification(AttributeValueNotification attrValueNotif) throws IdASException {
		log.debug("Received attribute value notification - " + attrValueNotif.getAction() + ", " + attrValueNotif.getAttributeValue().toString());
		log.debug("_containter " + ((_container != null) ? _container.toString() : "null"));
		if (_container != null) {
			AttributeNotification attrNotif = new AttributeNotification(this, AttributeNotification.UPDATE_VALUE_NOTIFY,
				attrValueNotif, null);
			_container.updateNotification(attrNotif);
		}
		if (attrValueNotif.getAction() == AttributeValueNotification.UPDATE_REMOVE)
			_values.remove(attrValueNotif.getAttributeValue());
	}

	public void setContainer(IAttributeContainer container) {
		_container = container;
	}
	
	public IAttributeContainer getContainer() {
		return _container;
	}

	public boolean equals(IAttribute attr) throws IdASException {
		boolean bRet;
		if (attr.getAttrID() != this.getAttrID())
			bRet = false;
		else if ((bRet = this.equals((IHasAttributes)attr)) == true) {
			Iterator iter = attr.getValues();
			// Compare each passed value
			while (iter.hasNext()) {
				if (this._containsValue((IAttributeValue)iter.next()) == false) {
					bRet = false;
					break;
				}					
			}
		}
		return bRet;
	}

	public IAttribute addAttribute(URI attrID) throws IdASException,
			InvalidTypeException {
		return _metaAttrs.addAttribute(attrID);
	}

	public IAttribute addAttribute(IAttribute copyFrom) throws IdASException {
		return _metaAttrs.addAttribute(copyFrom);
	}

	public boolean equals(IHasAttributes attributes) throws IdASException {
		return _metaAttrs.equals(attributes);
	}

	public IAttribute getAttribute(URI attrID) throws IdASException {
		return _metaAttrs.getAttribute(attrID);
	}

	public Iterator getAttributes() throws IdASException {
		return _metaAttrs.getAttributes();
	}

	public ISingleValuedAttribute getSingleValuedAttribute(URI attrID)
			throws IdASException, NotSingleValuedAttributeException {
		return _metaAttrs.getSingleValuedAttribute(attrID);
	}

	public void removeAttribute(URI attrID) throws IdASException {
		_metaAttrs.removeAttribute(attrID);
	}

	public void removeAttributeValue(URI attrID, Object value)
			throws IdASException {
		_metaAttrs.removeAttributeValue(attrID, value);
	}

	public void removeAttributeValue(IAttribute attr) throws IdASException {
		_metaAttrs.removeAttributeValue(attr);
	}

	public void updateNotification(AttributeNotification attrNotif)
			throws IdASException {
		_metaAttrs.updateNotification(attrNotif);
	}

	public Iterator getValues(IExtension[] extensions) throws IdASException {
		DereferenceAttrValuesExtension ext = null;
		if (extensions == null) {
			return getValues(ext);
		}
		// We only handle one extension
		for (int index = 0; index < extensions.length; index++) {
			if (extensions[index].getID().compareTo(URI.create(DereferenceAttrValuesExtension.id)) != 0) {
				if (extensions[index].failIfUnsupported() == true) {
					throw new UnhandledExtensionException(extensions[index].getID() + "not supported");
				}
			}
			if (ext != null) {
				throw new UnhandledExtensionException("multiple occurances of " + DereferenceAttrValuesExtension.id + " not supported");
			}
			if (extensions[index] instanceof DereferenceAttrValuesExtension) {
				ext = (DereferenceAttrValuesExtension)extensions[index];
			} else {
// TODO: DereferenceAttrValuesExtension should implements IDereferenceAttrValuesExtension and that's what we should check for.  Just in case someone wants to impl their own DereferenceAttrValuesExtension
				throw new IdASException("Invalid extension");
			}
		}
		return getValues(ext);
	}

	private Iterator getValues(DereferenceAttrValuesExtension ext) throws IdASException {
		/* TODO: Vector doesn't scale anyway, we need to fix this!
		However, the big deal here is the fact that we can't
		modify the thing we're iterating on, so, we're going
		with a clone temporarily. */
		Vector vals = ((Vector)_values.clone());
		
		if (ext != null) {
			// dereference any ref vals
			for (int index = 0; index < vals.size(); index++) {
				IAttributeValue val = (IAttributeValue)vals.elementAt(index);
				if (ext.shouldDereference(this._attrID, val.getValueType())) {
					val = dereferenceValue(val);
					vals.setElementAt(val, index);
				}
			}
		}		
		return vals.iterator();
	}

	/**
	 * Using the passed attribute value, dereferences it and returns the resulting dereferenced 
	 * entity or literal as an attribute value
	 * @param val value to be dereferenced
	 * @return dereferenced attribute
	 * @throws IdASException
	 */
	private IAttributeValue dereferenceValue(IAttributeValue val) throws IdASException {
//TODO make this deref absolute entity UDIs
//TODO make this deref attribute UDIs (relative and absolute)
		URI valType = val.getValueType();
		IAttributeValue valOut;
		
		if (valType.compareTo(ITypedValue.RELATIVEENTITYUDI_TYPE_URI) == 0) {
			if (val.isSimple() == false) {
				throw new InvalidTypeException("Attribute values of type " + ITypedValue.RELATIVEENTITYUDI_ATTR_VALUE_TYPE_URI_STR + " must be simple types. " + val.getClass() + " is complex");
			}
			ISimpleAttrValue simpleVal = (ISimpleAttrValue)val;
			if (_context == null) {
				throw new IdASException("Can't dereference this value because context is unknown at this time: " + simpleVal.getLexical());
			}
			
			// Here's where we dereference relative entity UDIs
			valOut = new ComplexValueEntityWrapper(_context.getEntity(simpleVal.getCanonical())); 
		} else {
			throw new UnhandledExtensionException("Cannot dereference values of type: " + valType);
		}

		return valOut;
	}

}
