/*******************************************************************************
 * Copyright (c) 2007 Novell, Inc..
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *   Duane Buss - Initial impl
 *******************************************************************************/
package org.eclipse.higgins.idas.spi;

import java.net.URI;
import java.util.Iterator;

import org.eclipse.higgins.idas.api.IAttribute;
import org.eclipse.higgins.idas.api.IContext;
import org.eclipse.higgins.idas.api.IEntity;
import org.eclipse.higgins.idas.api.IHasAttributes;
import org.eclipse.higgins.idas.api.ISingleValuedAttribute;
import org.eclipse.higgins.idas.api.IdASException;
import org.eclipse.higgins.idas.api.InvalidTypeException;
import org.eclipse.higgins.idas.api.NotSingleValuedAttributeException;
import org.eclipse.higgins.idas.api.model.IEntityModel;
import org.eclipse.higgins.idas.api.model.IdASModelException;

/**
 * Basic implementation of IEntity. The intent of this abstract class is
 * to be used by Context Provider writers as a superclass for IEntity
 * instances. The benefits of using this class are that you will pick up any
 * convenience methods as well as have the latest methods stubbed out such that
 * when new methods are added to IEntity and implemented here, the
 * subclass will still compile. <br>
 * 
 */
public class BasicEntity implements IEntity, IAttributeContainer
{
	private IEntityContainer container = null;
	private IContext context;
	private String entityId;
	private URI type;
	private BasicAttributeSet attrs;
	private IEntityModel model;

	/**
	 * 
	 * @param context
	 * @throws IdASException
	 */
	public BasicEntity(
		IContext context) throws IdASException
	{
		this(context, null, null, null);
	}

	/**
	 * The preferred constructor
	 * 
	 * @param context
	 * @param type
	 * @param entityId
	 * @throws IdASException
	 */
	public BasicEntity(
		IContext context,
		URI type,
		String entityId) throws IdASException
	{
		this(context, type, entityId, null);
	}

	/**
	 * The preferred constructor
	 * 
	 * @param context
	 * @param type
	 * @param entityId
	 * @param attributes
	 * @throws IdASException
	 */
	public BasicEntity(
		IContext context,
		URI type,
		String entityId,
		Iterator attributes) throws IdASException
	{
		this.context = context;
		this.type = type;
		this.entityId = entityId;
		this.attrs = new BasicAttributeSet(attributes, this, context);
		model = new BasicEntityModel(type);
	}

	public BasicEntity(
			BasicContext containerCtx,
			URI type,
			String entityId
		) throws IdASException
	{
		this((IContext)containerCtx, type, entityId);
		container = containerCtx;
	}
	
	/**
	 */
	public BasicEntity(
		BasicContext containerCtx,
		URI type,
		String entityID,
		Iterator attributes) throws IdASException
	{
		this((IContext)containerCtx, type, entityID, attributes);
		container = containerCtx;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.higgins.idas.api.IEntity#getContext()
	 */
	public IContext getContext() throws IdASException
	{
		return context;
	}

	/*
	 * @see org.eclipse.higgins.idas.api.IEntity#getModel()
	 */
	public IEntityModel getModel() throws IdASException
	{
		if (model == null) {
			throw new IdASModelException("No model exists for this context");
		}
		return model;
	}

	/*
	 * @see org.eclipse.higgins.idas.api.IEntity#getType()
	 */
	public URI getEntityType() throws IdASException
	{
		return type;
	}

	/*
	 * @see org.eclipse.higgins.idas.api.IEntity#getEntityID()
	 */
	public String getEntityID() throws IdASException
	{
		return entityId;
	}

	/*
	 * @see org.eclipse.higgins.idas.api.IHasAttributes#addAttribute(org.eclipse.higgins.idas.api.IAttribute)
	 */
	public IAttribute addAttribute(
		IAttribute copyFrom) throws IdASException
	{
		return attrs.addAttribute(copyFrom);
	}

	/*
	 * @see org.eclipse.higgins.idas.api.IHasAttributes#addAttribute(java.net.URI)
	 */
	public IAttribute addAttribute(
		URI type) throws IdASException, InvalidTypeException
	{
		return attrs.addAttribute(type);
	}

	/*
	 * @see org.eclipse.higgins.idas.api.IHasAttributes#getAttribute(java.net.URI)
	 */
	public IAttribute getAttribute(
		URI attrID) throws IdASException
	{
		return attrs.getAttribute(attrID);
	}
	
	/*
	 * @see org.eclipse.higgins.idas.api.IHasAttributes#getSingleValuedAttribute(java.net.URI)
	 */
	public ISingleValuedAttribute getSingleValuedAttribute(URI attrID) throws IdASException, NotSingleValuedAttributeException {
		return attrs.getSingleValuedAttribute(attrID);
	}	
	

	/*
	 * @see org.eclipse.higgins.idas.api.IHasAttributes#getAttributes()
	 */
	public Iterator getAttributes() throws IdASException
	{
		return attrs.getAttributes();
	}

	protected BasicAttributeSet getBasicAttributeSet()
	{
		return attrs;
	}

	public void removeAttribute(URI attrID) throws IdASException {
		attrs.removeAttribute(attrID);
	}

	public void removeAttributeValue(URI attrID, Object value)
	throws IdASException {
		attrs.removeAttributeValue(attrID, value);
	}
	
	public void removeAttributeValue(IAttribute attr) throws IdASException {
		attrs.removeAttributeValue(attr);
	}
	
	public boolean equals(IHasAttributes attributes) throws IdASException {
		return attrs.equals(attributes);
	}

	public void remove() throws IdASException {
		EntityNotification entityNotif = new EntityNotification(this, EntityNotification.UPDATE_REMOVE, null);
		if (container != null) {
			container.updateNotification(entityNotif);
		}
	}

	public void updateNotification(AttributeNotification attrNotif)
			throws IdASException {
		// Subclasses should override this. Otherwise all updates only happen to in-memory attributes
		if (container != null) {
			container.updateNotification(new EntityNotification(this, EntityNotification.UPDATE_ATTR_NOTIFY,
				attrNotif));
		}		
	}

	protected void setContainer(IEntityContainer container) {
		this.container = container;
	}
	
	protected void setEntityID (String entityId) {
		this.entityId = entityId;
	}
	
	protected void setType (URI type) {
		this.type = type;
	}
	
	protected void setAttrs (BasicAttributeSet attrs) {
		this.attrs = attrs;
	}

	protected void setModel(IEntityModel model) {
		this.model = model;
	}
}
