/*******************************************************************************
 * Copyright (c) 2008 Parity Communications, Inc.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     Markus Sabadello - Initial API and implementation
 *******************************************************************************/
package org.eclipse.higgins.idas.udi;

import java.net.URI;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

import org.eclipse.higgins.configuration.api.IConfigurableComponent;
import org.eclipse.higgins.configuration.api.IConfigurableComponentFactory;
import org.eclipse.higgins.idas.api.IAttribute;
import org.eclipse.higgins.idas.api.IContext;
import org.eclipse.higgins.idas.api.IContextFactory;
import org.eclipse.higgins.idas.api.IContextId;
import org.eclipse.higgins.idas.api.IEntity;
import org.eclipse.higgins.idas.api.IdASException;
import org.eclipse.higgins.idas.registry.IdASRegistry;

public class UDIResolver implements IConfigurableComponent, IConfigurableComponentFactory {

	private static UDIResolver impl = null;

	private List udiFactoryList;

	public UDIResolver() {

		this.udiFactoryList = new ArrayList();
	}

	public static synchronized UDIResolver getInstance() {

		if (impl == null) impl = new UDIResolver();
		return(impl);
	}

	public IConfigurableComponent getNewInstance() {

		return(new IdASRegistry());
	}

	public IConfigurableComponent getSingletonInstance() {

		return(getInstance());
	}

	public void configure(Map mapGlobalSettings, String strComponentName, Map mapComponentSettings) throws Exception {

		List udiFactoryInstances = (List) mapComponentSettings.get("UDIFactoryInstancesList");

		this.udiFactoryList.clear();

		for (Iterator i = udiFactoryInstances.iterator(); i.hasNext(); ) {

			String udiFactoryString = (String) i.next();
			IUDIFactory udiFactory = (IUDIFactory) mapComponentSettings.get(udiFactoryString);
			if (udiFactory == null) udiFactory = (IUDIFactory) mapGlobalSettings.get(udiFactoryString);

			if (udiFactory == null) throw new NullPointerException("UDI Factory " + udiFactoryString + " not found in configuration.");

			this.udiFactoryList.add(udiFactory);
		}
	}

	public List getUdiFactoryList() {

		return(this.udiFactoryList);
	}

	/**
	 * Parse a string into an IContextUDI by trying all registered IUDIFactorys
	 * until one succeeds.
	 */
	public IContextUDI parseContextUDI(String contextUDIStr) throws IdASException {

		IContextUDI contextUDI = null;

		for (Iterator i = this.udiFactoryList.iterator(); i.hasNext(); ) {

			IUDIFactory udiFactory = (IUDIFactory) i.next();

			contextUDI = udiFactory.parseContextUDI(contextUDIStr);
			if (contextUDI != null) break;
		}

		return(contextUDI);
	}

	/**
	 * Parse a string into an IEntityUDI by trying all registered IUDIFactorys
	 * until one succeeds.
	 */
	public IEntityUDI parseEntityUDI(String entityUDIStr) throws IdASException {

		IEntityUDI entityUDI = null;

		for (Iterator i = this.udiFactoryList.iterator(); i.hasNext(); ) {

			IUDIFactory udiFactory = (IUDIFactory) i.next();

			entityUDI = udiFactory.parseEntityUDI(entityUDIStr);
			if (entityUDI != null) break;
		}

		return(entityUDI);
	}

	/**
	 * Parse a string into an IAttributeUDI by trying all registered IUDIFactorys
	 * until one succeeds.
	 */
	public IAttributeUDI parseAttributeUDI(String attributeUDIStr) throws IdASException {

		IAttributeUDI attributeUDI = null;

		for (Iterator i = this.udiFactoryList.iterator(); i.hasNext(); ) {

			IUDIFactory udiFactory = (IUDIFactory) i.next();

			attributeUDI = udiFactory.parseAttributeUDI(attributeUDIStr);
			if (attributeUDI != null) break;
		}

		return(attributeUDI);
	}

	/**
	 * Resolve an IContextUDI to an IContext by retrieving the IContextUDIMetadata
	 */
	public IContext resolve(IContextUDI contextUDI) throws IdASException {

		IContextUDIMetadata contextUDIMetadata = contextUDI.getContextUDIMetadata();

		return(this.resolve(contextUDIMetadata));
	}

	/**
	 * Resolve an IEntityUDI to an IEntity by retrieving the IEntityUDIMetadata
	 */
	public IEntity resolve(IEntityUDI entityUDI, IAuthnNMaterialsProxy authnMaterialsProxy) throws IdASException {

		IEntityUDIMetadata entityUDIMetadata = entityUDI.getEntityUDIMetadata();

		return(this.resolve(entityUDIMetadata, authnMaterialsProxy));
	}

	/**
	 * Resolve an IAttributeUDI to an IAttribute by retrieving the IAttributeUDIMetadata
	 */
	public IAttribute resolve(IAttributeUDI attributeUDI, IAuthnNMaterialsProxy authnMaterialsProxy) throws IdASException {

		IAttributeUDIMetadata attributeUDIMetadata = attributeUDI.getAttributeUDIMetadata();

		return(this.resolve(attributeUDIMetadata, authnMaterialsProxy));
	}

	/**
	 * With the IContextUDIMetadata, we have everything we need to instantiate the IContext.
	 */
	private IContext resolve(IContextUDIMetadata contextUDIMetadata) throws IdASException {

		IdASRegistry idasRegistry = IdASRegistry.getInstance();

		String[] types = contextUDIMetadata.getTypes();
		Map configuration = contextUDIMetadata.getConfiguration();

		IContextFactory contextFactory = idasRegistry.getContextFactory(types);
		if (contextFactory == null) throw new IdASException("No Context Provider found for types " + join(types, " ") + ".");

		IContext context = contextFactory.createContext(new ContextUDIWrapper(types, configuration));
		if (context == null) throw new IdASException("Cannot create Context.");

		return(context);
	}

	/**
	 * With the IEntityUDIMetadata, we have everything we need to instantiate the IContext and the IEntity.
	 */
	private IEntity resolve(IEntityUDIMetadata entityUDIMetadata, IAuthnNMaterialsProxy authnMaterialsProxy) throws IdASException {

		IContextUDIMetadata contextUDIMetadata = entityUDIMetadata.getContextUDIMetadata();
		String entityID = entityUDIMetadata.getEntityID();

		IContext context = this.resolve(contextUDIMetadata);
		context.open(authnMaterialsProxy.buildAuthNAttributesMaterials(context));

		IEntity entity = context.getEntity(entityID);
		if (entity == null) throw new IdASException("Cannot get Entity.");

		return(entity);
	}

	/**
	 * With the IAttributeUDIMetadata, we have everything we need to instantiate the IContext, the IEntity, and the IAttribute.
	 */
	private IAttribute resolve(IAttributeUDIMetadata attributeUDIMetadata, IAuthnNMaterialsProxy authnMaterialsProxy) throws IdASException {

		IEntityUDIMetadata entityUDIMetadata = attributeUDIMetadata.getEntityUDIMetadata();
		URI attributeID = attributeUDIMetadata.getAttributeID();

		IEntity entity = this.resolve(entityUDIMetadata, authnMaterialsProxy);
		if (entity == null) throw new IdASException("Cannot get Entity.");

		IAttribute attribute = entity.getAttribute(attributeID);
		if (entity == null) throw new IdASException("Cannot get Attribute.");

		return(attribute);
	}

	private static class ContextUDIWrapper implements IContextId {

		private String[] types;
		private Map configuration;

		private ContextUDIWrapper(String[] types, Map configuration) {

			this.types = types;
			this.configuration = configuration;
		}

		public Map getConfiguration() throws IdASException {

			return(this.configuration);
		}

		public String[] getTypes() throws IdASException {

			return(this.types);
		}

		public URI[] getUris() throws IdASException {

			return(new URI[0]);
		}
	}
	
    public static String join(Object[] array, String delimiter) {

    	StringBuffer buffer = new StringBuffer();

    	Iterator i = Arrays.asList(array).iterator();

    	while (i.hasNext()) {
        
    		buffer.append(i.next());
            if (i.hasNext()) buffer.append(delimiter);
        }

    	return(buffer.toString());
    }

}
