/*******************************************************************************
 * Copyright (c) 2008 Parity Communications, Inc.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     Markus Sabadello - Initial API and implementation
 *******************************************************************************/
package org.eclipse.higgins.idas.udi.impl.xri;

import java.io.UnsupportedEncodingException;
import java.net.URLDecoder;

import org.eclipse.higgins.idas.api.IdASException;
import org.eclipse.higgins.idas.udi.IContextUDIMetadata;
import org.eclipse.higgins.idas.udi.IEntityUDIMetadata;
import org.openxri.XRI;
import org.openxri.XRISegment;
import org.openxri.resolve.Resolver;

public class XRIEntityUDIMetadata implements IEntityUDIMetadata {

	private String entityUDI;
	private Resolver resolver;
	private boolean https;
	private boolean saml;
	private boolean cid;
	private boolean refs;

	private IContextUDIMetadata contextUDIMetadata;
	private String entityID;

	XRIEntityUDIMetadata(String entityUDI, Resolver resolver, boolean https, boolean saml, boolean cid, boolean refs) {

		this.entityUDI = entityUDI;
		this.resolver = resolver;
		this.https = https;
		this.saml = saml;
		this.cid = cid;
		this.refs = refs;

		this.contextUDIMetadata = null;
		this.entityID = null;
	}

	private void resolve() throws IdASException {

		// if the second segment is empty (i.e. there's a //, then we can
		// extract an explicit relative Entity ID

		XRI xri = new XRI(this.entityUDI);
		int splitIndex = -1;

		if (xri.getXRIPath().getNumSegments() > 1 && xri.getXRIPath().getSegmentAt(0).toString().equals("")) splitIndex = 0;
		if (xri.getXRIPath().getNumSegments() > 2 && xri.getXRIPath().getSegmentAt(1).toString().equals("")) splitIndex = 1;

		if (splitIndex != -1) {

			// split XRI Entity UDI into Context UDI and relative Entity ID

			String contextUDI = xri.getAuthorityPath().toString();
			for (int i=0; i<splitIndex; i++) contextUDI += "/" + xri.getXRIPath().getSegmentAt(i).toString();

			this.contextUDIMetadata = new XRIContextUDIMetadata(contextUDI, this.resolver, this.https, this.saml, this.cid, this.refs);
			XRISegment segment = xri.getXRIPath().getSegmentAt(splitIndex + 1);

			// if the relative Entity ID is a cross-reference, get its value

			if (segment.getSubSegmentAt(0).getXRef() != null) {

				this.entityID = segment.getSubSegmentAt(0).getXRef().toString();
				this.entityID = this.entityID.substring(1, this.entityID.length() - 1);
			} else {

				this.entityID = segment.toString();
			}

			// url decode the entity ID
			
			try {

				this.entityID = URLDecoder.decode(this.entityID, "UTF-8");
			} catch (UnsupportedEncodingException ex) {

				throw new IdASException(ex);
			}
		} else {

			this.contextUDIMetadata = new XRIContextUDIMetadata(this.entityUDI, this.resolver, this.https, this.saml, this.cid, this.refs);
			this.entityID = this.entityUDI;
		}
	}

	public IContextUDIMetadata getContextUDIMetadata() throws IdASException {

		if (this.contextUDIMetadata == null) this.resolve();

		return(this.contextUDIMetadata);
	}

	public String getEntityID() throws IdASException {

		if (this.entityID == null) this.resolve();

		return(this.entityID);
	}
}
