/*******************************************************************************
 * Copyright (c) 2007 Parity Communications, Inc.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     Valery Kokhan - Initial API and implementation
 *******************************************************************************/

package org.eclipse.higgins.iss.cardspace;

import java.net.URI;
import java.security.cert.X509Certificate;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

import javax.security.auth.callback.CallbackHandler;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.eclipse.higgins.configuration.xml.ConfigurationHandler;
import org.eclipse.higgins.icard.ICard;
import org.eclipse.higgins.icard.ICardProvider;
import org.eclipse.higgins.icard.IClaim;
import org.eclipse.higgins.icard.IInformationCard;
import org.eclipse.higgins.icard.IManagedInformationCard;
import org.eclipse.higgins.icard.IPersonalInformationCard;
import org.eclipse.higgins.icard.IPolicy;
import org.eclipse.higgins.icard.ISimpleClaim;
import org.eclipse.higgins.icard.auth.ICredential;
import org.eclipse.higgins.icard.auth.ISelfIssuedCredential;
import org.eclipse.higgins.icard.auth.ITokenServiceCredential;
import org.eclipse.higgins.icard.auth.IUsernamePasswordCredential;
import org.eclipse.higgins.icard.policy.ICardSpacePolicy;
import org.eclipse.higgins.icard.provider.cardspace.common.utils.PPIDHelper;
import org.eclipse.higgins.icard.registry.ICardRegistry;
import org.eclipse.higgins.iss.CredentialContainer;
import org.eclipse.higgins.iss.IICardSelector;
import org.eclipse.higgins.iss.IIdentityToken;
import org.eclipse.higgins.iss.SelectionANDofORs;
import org.eclipse.higgins.iss.SelectionANDofORsElm;
import org.eclipse.higgins.iss.UCTelm;
import org.eclipse.higgins.iss.UserChoiceTree;
import org.eclipse.higgins.iss.UserChoiceTree_ANDofORs;
import org.eclipse.higgins.iss.UserChoiceTree_OR;
import org.eclipse.higgins.sts.api.IConstants;
import org.eclipse.higgins.sts.api.IElement;
import org.eclipse.higgins.sts.api.ISTSResponse;
import org.eclipse.higgins.sts.api.ISecurityTokenService;
import org.eclipse.higgins.sts.client.TokenRequestFactory;
import org.eclipse.higgins.sts.common.Element;
import org.eclipse.higgins.sts.common.STSResponse;
import org.eclipse.higgins.sts.spi.IBase64Extension;
import org.eclipse.higgins.sts.utilities.XMLHelper;

public class CardSpaceSelector implements IICardSelector {
	protected static final Log log = LogFactory.getLog(CardSpaceSelector.class);
	
	static String PPID_CLAIM = "http://schemas.xmlsoap.org/ws/2005/05/identity/claims/privatepersonalidentifier";
		
	protected ICardSpacePolicy policy;
	protected CallbackHandler handler;

	protected ICardRegistry registry = ICardRegistry.getInstance();
	
	public CardSpaceSelector(CallbackHandler handler, ICardSpacePolicy policy) {
		this.handler = handler;
		this.policy = policy;
	}
	
	public IPolicy getPolicy() {
		return policy;
	}

	public CallbackHandler getCallbackHandler() {
		return handler;
	}

	public UserChoiceTree getUserChoice() throws Exception {
		log.info("CardSpaceICardSelector::getUserChoice()");
		log.info("policy: " + policy);
		UserChoiceTree res;
		List list = new ArrayList();
		List l = new ArrayList();
		Class[] iCardTypes = policy.getICardTypes();
		for (Iterator itr = registry.getICardProviders(); itr.hasNext(); ) {
			ICardProvider p = (ICardProvider) itr.next();
			boolean hasRequiredTypes = false;
			for (int i = 0; i < iCardTypes.length; i++) {
				Class[] supportedTypes = p.getSupportedTypes();
				for (int j = 0; j < supportedTypes.length; j++) {
					if (iCardTypes[i].isAssignableFrom(supportedTypes[j])) {
						hasRequiredTypes = true;
						break;
					}
				}
				if (hasRequiredTypes) {
					break;
				}
			}
			if (hasRequiredTypes) {
//				List l = new ArrayList();
				try {
					for(Iterator cards = p.getICards(handler, policy); cards.hasNext(); ) {
						ICard card = (ICard) cards.next();
						CredentialContainer cc = new CredentialContainer(card, card.getCUID().toString());
						l.add(cc);
					}
				} catch (Exception e) {
					log.error("CardSpaceICardSelector::getUserChoice()", e);
					//e.printStackTrace();
				}
				if (l.size() > 0) {
					UCTelm elm = new UCTelm(policy, l);
					list.add(new UserChoiceTree_OR(elm));
				}
			}
		}
		res = new UserChoiceTree_ANDofORs(list); 
		return res;
	}

	public IIdentityToken getIdentityToken(SelectionANDofORs selection) throws STSFaultException, Exception {
		// extract Infocard out of the selection (we assume that is the first
		// leave from the leave)
		SelectionANDofORsElm elm = (SelectionANDofORsElm) selection.getElements().get(0);
		String uid = elm.getUUID();

		IInformationCard infocard = (IInformationCard) registry.getICardByCUID(handler, uid);
//		URI uriTokenService = null;
		URI uriRP = null; 
		try { 
//			uriTokenService = new URI(infocard.getEndpointAddress());		
			uriRP = URI.create( XMLHelper.escapedString(selection.action) );
		} catch (Exception e) {
//			throw new Exception("Either the issuer ("+infocard.getEndpointAddress() +") or the Relying Party ("+selection.action+") URI is malformed.");
		}

		boolean selfIssued = infocard.isSelfIssued();
		
		final ConfigurationHandler ch = getConfigurationHandler( selfIssued );
		Map mapGlobalSettings = ch.getSettings();
		final IBase64Extension base64Extension = (org.eclipse.higgins.sts.spi.IBase64Extension)mapGlobalSettings.get("Base64Extension");
		final IConstants constants = new org.eclipse.higgins.sts.common.Constants();
		final TokenRequestFactory factoryRequest = new TokenRequestFactory();
		final ISTSResponse stsResponse = new STSResponse();
		org.eclipse.higgins.sts.api.ISTSRequest stsRequest = null;

		IElement elemClaims = makeClaims(policy, infocard, selection, base64Extension);
		
		log.info("@@@@@@@@@ X509 CERT is " + selection.sslCert);
		
		//if (!selfIssued) {
		if (infocard instanceof IManagedInformationCard) {
			IManagedInformationCard managed = (IManagedInformationCard) infocard;
			final ISecurityTokenService stsManaged = (ISecurityTokenService)mapGlobalSettings.get("SecurityTokenServiceClient");
			final ISecurityTokenService stsBinding = (ISecurityTokenService)mapGlobalSettings.get("SecurityTokenServiceBinding");
			//URI	uriMetadataService = new URI(infocard.getMetadataAddress());
			//log.info( " TrustURI: " + uriTokenService + " and MexURI: " + uriMetadataService);

			/*
			mapGlobalSettings.put("MetadataServiceURI", uriMetadataService);
			final IMetadataExchangeService mex = (IMetadataExchangeService)mapGlobalSettings.get
			("MetadataExchangeServiceClient");
			final IMetadataExchangeService mexBinding = (IMetadataExchangeService)mapGlobalSettings.get
			("MetadataExchangeServiceBinding");
			mexBinding.configure(mapGlobalSettings, "MetadataExchangeService", null);

			final org.eclipse.higgins.sts.api.IMEXResponse mexResponse = mex.getMetadata
				(mapGlobalSettings, null, null, null, constants, uriMetadataService);
			if (null != mexResponse.getFault()) {
				final org.eclipse.higgins.sts.api.IFault fault = mexResponse.getFault();
				//log.error("Returning MEX Fault: " + fault.getDetail());
				throw new Exception("MEX Fault: " + fault.getDetail());
			}			
            log.trace(mexResponse.getMetadata().getAs(String.class));
			*/
			
			ICredential credential = selection.getCredential();
			URI uriTokenService = null;
			URI	uriMetadataService = null;
			X509Certificate endpointCert = null;
			
			if (credential instanceof ITokenServiceCredential) {
				ITokenServiceCredential tsc = (ITokenServiceCredential) credential;
				uriTokenService = tsc.getAddress();
				endpointCert = tsc.getCertificate();
				uriMetadataService = tsc.getMetadataAddress();
			} else {
				throw new IllegalArgumentException();
			}
			
			// check auth method to remote STS
//			if (infocard.getTokenServiceCredentialType() == IInformationCard.TOKENSERVICE_SELFISSUEDSAML) {
			if (credential instanceof ISelfIssuedCredential) {
				ISelfIssuedCredential sic = (ISelfIssuedCredential) credential;
				log.info( "%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%");
				log.info( "Create request for managed card with self issued saml as authentification");
				//String ppid = infocard.getTokenServiceCredentialValue();
				String ppid = sic.getPPID();
				byte[] bppid = base64Extension.decode(ppid);
//				X509Certificate EndpointCert = infocard.getEndpointX509Certificate();
				log.info("CERT is " + endpointCert);
				PPIDCardSpacePolicy ppp = new PPIDCardSpacePolicy(endpointCert, bppid);
				IICardSelector subselector = new CardSpaceSelector(handler,ppp);
				UserChoiceTree_ANDofORs uct = (UserChoiceTree_ANDofORs)subselector.getUserChoice();
				SelectionANDofORs subselection = uct.getDefaultSelection();
				if (subselection.isEmpty()) {
					throw new Exception("Cannot find the Personal card used to authenticate for this managed card.");
				}
				subselection.action = infocard.getIssuer();
				subselection.sslCert = endpointCert;
				IIdentityToken subtok = subselector.getIdentityToken(subselection);
								
				log.info("\n\n\n SSS AUTH: " + subtok.getAs(String.class) + "\n\n\n");
				
				stsRequest = factoryRequest.createManagedRequest(
						uriRP, 
						new X509Certificate[] {selection.sslCert}, 
						uriTokenService, 
						java.net.URI.create("urn:oasis:names:tc:SAML:1.0:assertion"), 
						infocard, 
						elemClaims, 
						(IElement)subtok.getAs(IElement.class), 
						//ie2,
						base64Extension);
				
				log.info("%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%" );
				log.info( "create request for managed card with self issued saml as authentification done ");						

			} else if (credential instanceof IUsernamePasswordCredential) {
				IUsernamePasswordCredential unpc = (IUsernamePasswordCredential) credential;

				// username password authentication		
				log.info( "%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%");
				log.info( "Create request for managed card with username/password as authentification");
				stsRequest = factoryRequest.createManagedRequest
					(uriRP,
					new X509Certificate[] {selection.sslCert},
					uriTokenService,
					java.net.URI.create("urn:oasis:names:tc:SAML:1.0:assertion"),
					infocard,
					elemClaims,
//					selection.username,
//					selection.password,
					unpc.getUsername(),
					unpc.getPassword(),
					base64Extension
					);
				log.info("%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%% ");
				log.info( "create request for managed card with username/password as authentification done");
			} else {
				throw new Exception("Unsupported user credential " + credential);
			}
			
			mapGlobalSettings.put("TokenServiceTrustURI", uriTokenService); 
			stsBinding.configure(mapGlobalSettings, "SecurityTokenService", null);			

			log.info( "Invoke STS for managed card.");
			stsManaged.invoke (mapGlobalSettings, null, null, null, constants, stsRequest, stsResponse);

			log.info("%%%% END OF MANAGED %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%% done ");
						
		} else if (infocard instanceof IPersonalInformationCard) {
			IPersonalInformationCard pesonal = (IPersonalInformationCard) infocard;
			
			// self signed card
			final ISecurityTokenService stsPersonal = (ISecurityTokenService)mapGlobalSettings.get
			("SecurityTokenService");

			log.info( "Create request for self signed card");
			stsRequest = factoryRequest.createPersonalRequest
				(uriRP,
				new X509Certificate[] {selection.sslCert},
				java.net.URI.create("urn:oasis:names:tc:SAML:1.0:assertion"),
				infocard,
				elemClaims,
				base64Extension);
			log.info( "Invoke STS for self signed card.");
			stsPersonal.invoke(mapGlobalSettings, "SecurityTokenService", null, null, constants, stsRequest, stsResponse);
		}
		
		if (null != stsResponse.getFault()) {
			log.error( "Returning STS Fault: " + stsResponse.getFault().getDetail()  );
			throw new STSFaultException( stsResponse.getFault());
		}

		String encodedSSL = base64Extension.encode(selection.sslCert.getEncoded());
		
		return new IdentityToken(stsResponse, encodedSSL);		
	}
	
	
	//	 turn claims from policy into one string which is needed to set up
	// elemClaims
	// TODO: only temporary function
	private IElement makeClaims(ICardSpacePolicy pol, IInformationCard infocard, SelectionANDofORs selection, final IBase64Extension base64Extension) throws Exception {
		IElement elemClaims = new Element();

		String begin = "<wst:Claims xmlns:ic=\"http://schemas.xmlsoap.org/ws/2005/05/identity\" xmlns:wst=\"http://schemas.xmlsoap.org/ws/2005/02/trust\" wst:Dialect=\"http://schemas.xmlsoap.org/ws/2005/05/identity\">";
		String end = "</wst:Claims>";
		StringBuffer middle = new StringBuffer();
		X509Certificate rp = selection.sslCert;
		List requiredClaims = pol.getRequiredClaims();
		makeClaimTags(requiredClaims, middle, infocard, rp, base64Extension);
		if (selection.optionalClaims) {
			List optionalClaims = pol.getOptionalClaims();
			makeClaimTags(optionalClaims, middle, infocard, rp, base64Extension);
		}

		elemClaims.set(begin + middle + end);

		return elemClaims;
	}	

	public void makeClaimTags(List claims, StringBuffer middle, IInformationCard infocard, X509Certificate rp, final IBase64Extension base64Extension) {
		boolean selfIssued = infocard.isSelfIssued();
		for (int i = 0; i < claims.size(); i++) {
			String claim = (String)claims.get(i);
			if (selfIssued) {
				// place value into the RST
				try {
					// abs: this code should go into the infocard class--infocard should
					// compute ppid differently when it is a self-issued card.  not refactoring until
					// after Burton catalyst due to regression risk
					if (PPID_CLAIM.equalsIgnoreCase(claim)) {
//						byte[] ppid =  infocard.computePPI(rp);
						byte[] ppid = PPIDHelper.computeClaimValuePPID(infocard, rp);
						String sppid = base64Extension.encode(ppid);
						middle.append("<ic:ClaimType Uri=\"")
							.append(claim).append("\">")
							.append(sppid).append("</ic:ClaimType>");
					} else {
//						String val = infocard.getClaimValue(claim);
						IClaim c = infocard.getClaim(claim);
						String val = null;
						if (c instanceof ISimpleClaim) {
							ISimpleClaim sc = (ISimpleClaim) c;
							List l = sc.getValues();
							if (l.size() > 0) {
								val = (String) l.get(0);
							}
						}
						if (val!=null) {
							middle.append("<ic:ClaimType Uri=\"")
							.append(claim).append("\">")
							.append(val).append("</ic:ClaimType>");
						}
					}
				} catch (Exception e) {	
					e.printStackTrace();
				}
			} else {
				middle.append("<ic:ClaimType Uri=\"" + claim + "\"/>");
			}
		}
	}
	
	public ConfigurationHandler getConfigurationHandler(boolean selfIssued) throws Exception {
		String strConfigurationBase = System.getProperty("org.eclipse.higgins.sts.conf");
		if (null == strConfigurationBase) {
			throw new Exception("CardspaceSelector: org.eclipse.higgins.sts.conf System property not found");
		}

		final org.eclipse.higgins.configuration.xml.ConfigurationHandler configurationHandler = new org.eclipse.higgins.configuration.xml.ConfigurationHandler();
		configurationHandler.setConfigurationBase(strConfigurationBase);
		
		configurationHandler.setFileName("ClientConfiguration.xml");

		if (!configurationHandler.configure(null)) {
			log.error("Not Initialized!");
			throw new Exception("CardspaceSelector: cannot initialize configHandler");
		}

		return configurationHandler;
	}
	
}
