/*******************************************************************************
 * Copyright (c) 2007 Parity Communications, Inc.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     Valery Kokhan - Initial API and implementation
 *******************************************************************************/

package org.eclipse.higgins.iss.cardspace;

import java.net.URI;
import java.security.KeyStore;
import java.security.MessageDigest;
import java.security.PrivateKey;
import java.security.cert.CertStore;
import java.security.cert.CertStoreException;
import java.security.cert.CertificateException;
import java.security.cert.X509Certificate;
import java.util.ArrayList;
import java.util.Enumeration;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

import javax.security.auth.callback.Callback;
import javax.security.auth.callback.CallbackHandler;

import org.apache.commons.lang.StringEscapeUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.xml.security.utils.Base64;
import org.eclipse.higgins.configuration.xml.ConfigurationHandler;
import org.eclipse.higgins.icard.ICard;
import org.eclipse.higgins.icard.ICardProvider;
import org.eclipse.higgins.icard.IClaim;
import org.eclipse.higgins.icard.IInformationCard;
import org.eclipse.higgins.icard.IManagedInformationCard;
import org.eclipse.higgins.icard.IPersonalInformationCard;
import org.eclipse.higgins.icard.IPolicy;
import org.eclipse.higgins.icard.ISimpleClaim;
import org.eclipse.higgins.icard.auth.ICredential;
import org.eclipse.higgins.icard.auth.IPinCodeCredential;
import org.eclipse.higgins.icard.auth.ISelfIssuedCredential;
import org.eclipse.higgins.icard.auth.ITokenServiceCredential;
import org.eclipse.higgins.icard.auth.IUsernamePasswordCredential;
import org.eclipse.higgins.icard.auth.IX509V3Credential;
import org.eclipse.higgins.icard.common.auth.callback.X509V3Callback;
import org.eclipse.higgins.icard.policy.ICardSpacePolicy;
import org.eclipse.higgins.icard.provider.cardspace.common.PPIDCardSpacePolicy;
import org.eclipse.higgins.icard.provider.cardspace.common.PersonalCard;
import org.eclipse.higgins.icard.provider.cardspace.common.STSFaultException;
import org.eclipse.higgins.icard.provider.cardspace.common.TokenService;
import org.eclipse.higgins.icard.registry.ICardRegistry;
import org.eclipse.higgins.iss.CredentialContainer;
import org.eclipse.higgins.iss.IICardSelector;
import org.eclipse.higgins.iss.IIdentityToken;
import org.eclipse.higgins.iss.ISSException;
import org.eclipse.higgins.iss.SelectionANDofORs;
import org.eclipse.higgins.iss.SelectionANDofORsElm;
import org.eclipse.higgins.iss.UCTelm;
import org.eclipse.higgins.iss.UserChoiceTree;
import org.eclipse.higgins.iss.UserChoiceTree_ANDofORs;
import org.eclipse.higgins.iss.UserChoiceTree_OR;
import org.eclipse.higgins.keystore.IKeyStoreService;
import org.eclipse.higgins.keystore.registry.KeyStoreRegistry;
import org.eclipse.higgins.sts.api.IConstants;
import org.eclipse.higgins.sts.api.IElement;
import org.eclipse.higgins.sts.api.ISTSResponse;
import org.eclipse.higgins.sts.api.ISecurityTokenService;
import org.eclipse.higgins.sts.client.PPIDHelper;
import org.eclipse.higgins.sts.client.TokenRequestFactory;
import org.eclipse.higgins.sts.common.BinarySecurityToken;
import org.eclipse.higgins.sts.common.Element;
import org.eclipse.higgins.sts.common.STSResponse;
import org.eclipse.higgins.sts.utilities.XMLHelper;

public class CardSpaceSelector implements IICardSelector {
	protected static final Log log = LogFactory.getLog(CardSpaceSelector.class);
	
	static String PPID_CLAIM = "http://schemas.xmlsoap.org/ws/2005/05/identity/claims/privatepersonalidentifier";
	static String URI_BASE64 = "http://docs.oasis-open.org/wss/2004/01/oasis-200401-wss-soap-message-security-1.0#Base64Binary";
	static String URI_WSS_X509 = "http://docs.oasis-open.org/wss/2004/01/oasis-200401-wss-x509-token-profile-1.0#X509v3";
	static String URI_THUMBPRINT = "http://docs.oasis-open.org/wss/2004/xx/oasis-2004xx-wss-soap-message-security-1.1#ThumbprintSHA1";
	
	protected ICardSpacePolicy policy;
	protected CallbackHandler handler;
	protected String stsConfigurationBase;

	protected ICardRegistry registry = ICardRegistry.getInstance();
	
	public CardSpaceSelector(String stsConfigurationBase, CallbackHandler handler, ICardSpacePolicy policy) {
		this.handler = handler;
		this.policy = policy;
		this.stsConfigurationBase = stsConfigurationBase;
	}
	
	public IPolicy getPolicy() {
		return policy;
	}

	public CallbackHandler getCallbackHandler() {
		return handler;
	}

	public UserChoiceTree getUserChoice() throws Exception {
		log.trace("CardSpaceICardSelector::getUserChoice()");
		log.trace("policy: " + policy);
		UserChoiceTree res;
		List orTreeList = new ArrayList();
//		List l = new ArrayList();
		Class[] iCardTypes = policy.getICardTypes();
		List cardList = new ArrayList();
		for (Iterator itr = registry.getICardProviders(); itr.hasNext(); ) {
			ICardProvider p = (ICardProvider) itr.next();
			boolean hasRequiredTypes = false;
			for (int i = 0; i < iCardTypes.length; i++) {
				Class[] supportedTypes = p.getSupportedTypes();
				for (int j = 0; j < supportedTypes.length; j++) {
					if (iCardTypes[i].isAssignableFrom(supportedTypes[j])) {
						hasRequiredTypes = true;
						break;
					}
				}
				if (hasRequiredTypes) {
					break;
				}
			}
			if (hasRequiredTypes) {
				try {
					for(Iterator cards = p.getICardsByPolicy(handler, policy); cards.hasNext(); ) {
						ICard card = (ICard) cards.next();
						CredentialContainer cc = new CredentialContainer(card, card.getCUID().toString());
						cardList.add(cc);
					}
				} catch (Exception e) {
					log.error("CardSpaceICardSelector::getUserChoice()", e);
					//e.printStackTrace();
				}
			}
		}
		//if (cardList.size() > 0) {
		UCTelm elm = new UCTelm(policy, cardList);
		orTreeList.add(new UserChoiceTree_OR(elm));
		//}
		res = new UserChoiceTree_ANDofORs(orTreeList); 
		return res;
	}

	public IIdentityToken getIdentityToken(SelectionANDofORs selection) throws STSFaultException, Exception {
		// extract Infocard out of the selection (we assume that is the first
		// leave from the leave)
		SelectionANDofORsElm elm = (SelectionANDofORsElm) selection.getElements().get(0);
		String uid = elm.getUUID();

		IInformationCard infocard = (IInformationCard) registry.getICardByCUID(handler, uid);
//		URI uriTokenService = null;
		URI uriRP = null;
		String urlRP = null;
		try { 
//			uriTokenService = new URI(infocard.getEndpointAddress());
			urlRP = XMLHelper.escapedString(selection.action);
			uriRP = URI.create(urlRP);
		} catch (Exception e) {
//			throw new Exception("Either the issuer ("+infocard.getEndpointAddress() +") or the Relying Party ("+selection.action+") URI is malformed.");
		}

		boolean selfIssued = infocard.isSelfIssued();
		
		final ConfigurationHandler ch = getConfigurationHandler( selfIssued );
		Map mapGlobalSettings = ch.getSettings();
		final IConstants constants = new org.eclipse.higgins.sts.common.Constants();
		final TokenRequestFactory factoryRequest = new TokenRequestFactory();
		final ISTSResponse stsResponse = new STSResponse();
		org.eclipse.higgins.sts.api.ISTSRequest stsRequest = null;
		
		//Get the client side certificate store from configuration
		final KeyStore ksTrusted = (KeyStore)mapGlobalSettings.get("TrustedKeyStore");
		java.util.Collection collCertificates = new java.util.ArrayList();
		java.util.Enumeration enumCertificates = ksTrusted.aliases();
		while (enumCertificates.hasMoreElements()){
			String strAlias = (String)enumCertificates.nextElement();
			java.security.cert.Certificate cert = ksTrusted.getCertificate(strAlias);
			collCertificates.add(cert);
		}
		java.security.cert.CollectionCertStoreParameters collCSP = 
			new java.security.cert.CollectionCertStoreParameters(collCertificates);/**/
		java.security.cert.CertStore certStoreTrust = KeyStoreRegistry.getInstance().getSecurityService().getCertStore(handler); 
			//java.security.cert.CertStore.getInstance("Collection", collCSP);
		if (selection.sslCertChain != null && selection.sslCertChain.length > 0 )
			log.trace("@@@@@@@@@ X509 CERT is " + (selection.sslCertChain)[0]);
		
		//if (!selfIssued) {
		if (infocard instanceof IManagedInformationCard) {
			IManagedInformationCard managed = (IManagedInformationCard) infocard;
			X509Certificate[] rpCert = selection.sslCertChain;
			if (managed.isStrongRecipientIdentityRequired()) {
				if (rpCert == null || rpCert.length == 0)
					throw new RequireStrongRecipientIdentityEception("This card requires a cryptographically protected Relying Party identity");
			}
			IElement elemClaims = makeClaims(policy, infocard, selection, certStoreTrust, urlRP);
			log.trace(elemClaims.getAs(String.class));
			final ISecurityTokenService stsManaged = (ISecurityTokenService)mapGlobalSettings.get("SecurityTokenService");
			final ISecurityTokenService stsBinding = (ISecurityTokenService)mapGlobalSettings.get("SecurityTokenServiceBinding");
			//URI	uriMetadataService = new URI(infocard.getMetadataAddress());
			//log.trace( " TrustURI: " + uriTokenService + " and MexURI: " + uriMetadataService);

			/*
			mapGlobalSettings.put("MetadataServiceURI", uriMetadataService);
			final IMetadataExchangeService mex = (IMetadataExchangeService)mapGlobalSettings.get
			("MetadataExchangeServiceClient");
			final IMetadataExchangeService mexBinding = (IMetadataExchangeService)mapGlobalSettings.get
			("MetadataExchangeServiceBinding");
			mexBinding.configure(mapGlobalSettings, "MetadataExchangeService", null);

			final org.eclipse.higgins.sts.api.IMEXResponse mexResponse = mex.getMetadata
				(mapGlobalSettings, null, null, null, constants, uriMetadataService);
			if (null != mexResponse.getFault()) {
				final org.eclipse.higgins.sts.api.IFault fault = mexResponse.getFault();
				//log.error("Returning MEX Fault: " + fault.getDetail());
				throw new Exception("MEX Fault: " + fault.getDetail());
			}			
            log.trace(mexResponse.getMetadata().getAs(String.class));
			*/
			
			ICredential credential = selection.getCredential();
			URI uriTokenService = null;
			URI	uriMetadataService = null;
			X509Certificate endpointCert = null;
			
			if (credential instanceof ITokenServiceCredential) {
				ITokenServiceCredential tsc = (ITokenServiceCredential) credential;
				uriTokenService = tsc.getAddress();
				endpointCert = tsc.getCertificate();
				uriMetadataService = tsc.getMetadataAddress();
			} /*else {
				throw new IllegalArgumentException();
			}*/
			
			// check auth method to remote STS
//			if (infocard.getTokenServiceCredentialType() == IInformationCard.TOKENSERVICE_SELFISSUEDSAML) {
			if (credential instanceof ISelfIssuedCredential) {
				ISelfIssuedCredential sic = (ISelfIssuedCredential) credential;
				log.trace( "%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%");
				log.trace( "Create request for managed card with self issued saml as authentification");
				//String ppid = infocard.getTokenServiceCredentialValue();
				String ppid = sic.getPPID();
				
				byte[] bppid = org.apache.commons.codec.binary.Base64.decodeBase64(ppid.getBytes());
//				X509Certificate EndpointCert = infocard.getEndpointX509Certificate();
				log.info("CERT inside the card is " + endpointCert);
				/*
				
				java.security.KeyStore ks = java.security.KeyStore.getInstance("JKS");
				String strJREHome = System.getProperty("java.home");
				String strCACertsLocation = strJREHome + "/lib/security/cacerts";
			    ks.load(new java.io.FileInputStream(strCACertsLocation), "changeit".toCharArray());
			    java.security.cert.X509CertSelector certSelector = new java.security.cert.X509CertSelector();
			    certSelector.setCertificate(endpointCert);
			    java.security.cert.PKIXBuilderParameters pkixbuilderparams = new java.security.cert.PKIXBuilderParameters(ks, certSelector);
			    java.security.cert.CertPathBuilder cpb = java.security.cert.CertPathBuilder.getInstance
			    	(java.security.cert.CertPathBuilder.getDefaultType());
			    java.security.cert.PKIXCertPathBuilderResult cpbr = (java.security.cert.PKIXCertPathBuilderResult)cpb.build(pkixbuilderparams);
			    java.security.cert.CertPath cp = cpbr.getCertPath();
			    java.util.List listCerts = cp.getCertificates();
			    log.trace("Certificate Chain length is " + listCerts.size() + " plus 1 root");
			    
			    */				
				IKeyStoreService securityService = KeyStoreRegistry.getInstance().getSecurityService();
				X509Certificate[] chainOriginal = new X509Certificate[]{endpointCert};
				X509Certificate[] chain = null;
				if (chainOriginal != null) {
					try {
						chain = securityService.getValidateCertificateChain(handler, chainOriginal, uriTokenService);
					} catch (CertificateException e) {
						log.error(e, e);
					} catch (CertStoreException e) {
						log.error(e, e);
					}
				}
				if (chain == null)
					chain = chainOriginal;
				PPIDCardSpacePolicy ppp = new PPIDCardSpacePolicy(chain, certStoreTrust, bppid, urlRP);
				IICardSelector subselector = new CardSpaceSelector(stsConfigurationBase, handler, ppp);
				UserChoiceTree_ANDofORs uct = (UserChoiceTree_ANDofORs)subselector.getUserChoice();
				SelectionANDofORs subselection = uct.getDefaultSelection();
				if (subselection.isEmpty()) {
//					System.out.println("Cannot find the Personal card used to authenticate for this managed card.");
					throw new ISSException("Cannot find the Personal card used to authenticate for this managed card.");					
				}
				subselection.action = infocard.getIssuer();
				//subselection.sslCertChain = new X509Certificate[] {endpointCert};
				subselection.sslCertChain = chain; 
				
				subselection.setCredential(sic.getPinCodeCredential());
				IIdentityToken subtok = subselector.getIdentityToken(subselection);
				IElement elemSubTok = (IElement)subtok.getAs(IElement.class);	
				org.eclipse.higgins.sts.api.ISAML11Token samlSubTok = new org.eclipse.higgins.sts.common.SAML11Token();
				samlSubTok.fromElement(elemSubTok);
				
				log.info("\n\n\n SSS AUTH: " + subtok.getAs(String.class) + "\n\n\n");
				
				stsRequest = factoryRequest.createManagedRequest(
						uriRP, 
						selection.sslCertChain, 
						certStoreTrust,
						uriTokenService, 
						java.net.URI.create("urn:oasis:names:tc:SAML:1.0:assertion"), 
						infocard, 
						elemClaims, 
						samlSubTok);
				
				log.trace("%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%" );
				log.trace( "create request for managed card with self issued saml as authentification done ");						

			} else if (credential instanceof IUsernamePasswordCredential) {
				IUsernamePasswordCredential unpc = (IUsernamePasswordCredential) credential;
				String username = unpc.getUsername();
				String password = unpc.getPassword();

//				System.out.println("username : " + username);
//				System.out.println("password : " + password);
				
				TokenService ts = (TokenService)managed.getTokenServices().get(0);
				uriTokenService = ts.getEndpointReference().getAddress();

				// username password authentication		
				log.trace( "%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%");
				log.trace( "Create request for managed card with username/password as authentification");
				stsRequest = factoryRequest.createManagedRequest
					(uriRP,
					selection.sslCertChain,
					certStoreTrust,
					uriTokenService,
					java.net.URI.create("urn:oasis:names:tc:SAML:1.0:assertion"),
					infocard,
					elemClaims,
//					selection.username,
//					selection.password,
					username,
					password);
				log.trace("%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%% ");
				log.trace( "create request for managed card with username/password as authentification done");
			} else if (credential instanceof IX509V3Credential) {
				Callback[] callbacks = credential.getCallbacks();
				X509V3Callback x5cb = null;
				for(int i=0; i<callbacks.length; i++){
					if(callbacks[i] instanceof X509V3Callback){
						x5cb = (X509V3Callback) callbacks[i];
					}
				}
				if(x5cb==null){
					throw new ISSException("Cannot find certificate identifier from the i-card");
				}
				
				String value_type = x5cb.getValueType();
				String sha1_digest = x5cb.getEncoded_key();
				
				X509Certificate client_cert = null;
				String client_cert_alias = "";
				
				if(URI_THUMBPRINT.equals(value_type)){										
					Enumeration aliases = ksTrusted.aliases();
					if(aliases!=null){
						while(aliases.hasMoreElements()){
							String alias = (String) aliases.nextElement();
							if(ksTrusted.isKeyEntry(alias)){
								X509Certificate cert = (X509Certificate) ksTrusted.getCertificate(alias);
								byte[] cert_der = cert.getEncoded();								
								MessageDigest md = MessageDigest.getInstance("SHA");
								md.update(cert_der);
								byte[] digest = md.digest();
								String cert_digest = Base64.encode(digest);
								if(sha1_digest.equals(cert_digest)){
									client_cert = cert;
									client_cert_alias = alias;
									break;
								}
							}
						}
					}
				}
				
				if(client_cert==null){
					throw new ISSException("Cannot find client certificate to access this IDP");
				}
								
				PrivateKey client_pk = (PrivateKey) mapGlobalSettings.get(client_cert_alias);
				if(client_pk==null){
					throw new ISSException("configure error: cannot find your privatey key");
				}
					
				String cert_token = Base64.encode(client_cert.getEncoded());					
				BinarySecurityToken bst = new BinarySecurityToken();
				bst.setEncodedValue(cert_token);
				bst.setEncodingType(new URI(URI_BASE64));
				bst.setValueType(new URI(URI_WSS_X509));
					
				TokenService ts = (TokenService)managed.getTokenServices().get(0);
				uriTokenService = ts.getEndpointReference().getAddress();
					
				//X509 V3 certificate authentication
				log.trace( "%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%");
				log.trace( "Create request for managed card with X509 v3 certificate as authentification");
				stsRequest = factoryRequest.createManagedRequest(
					uriRP,
					selection.sslCertChain,
					certStoreTrust,
					uriTokenService,
					java.net.URI.create("urn:oasis:names:tc:SAML:1.0:assertion"),
					infocard,
					elemClaims,
					client_pk,
					bst);
			} else {
				throw new ISSException("Unsupported user credential " + credential);
			}
			
			mapGlobalSettings.put("TokenServiceTrustURI", uriTokenService); 
			stsBinding.configure(mapGlobalSettings, "SecurityTokenService", null, null, null);			

			log.trace( "Invoke STS for managed card.");
			stsManaged.invoke (mapGlobalSettings, null, null, null, constants, stsRequest, stsResponse);

			log.trace("%%%% END OF MANAGED %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%% done ");
						
		} else if (infocard instanceof IPersonalInformationCard) {			
			if(infocard instanceof PersonalCard){
				PersonalCard pcard = (PersonalCard) infocard;
				if(pcard.getPinStatus()==PersonalCard.LOCKED){
					IPinCodeCredential ipc = null;					
					ICredential cred = selection.getCredential();
					if((cred!=null) && (cred instanceof IPinCodeCredential)){
						ipc = (IPinCodeCredential)cred;
						pcard.unlock(ipc);
					} else {
						throw new ISSException("The personal card is PIN code locked.");
					}
				}
			}
			
			IPersonalInformationCard pesonal = (IPersonalInformationCard) infocard;
			ICardProvider provider = pesonal.getProvider();
			ICredential credential = selection.getCredential();
			if (credential!=null)
				pesonal = (IPersonalInformationCard) provider.getICardByCUID(handler, pesonal.getCUID(), credential);
			else
				pesonal = (IPersonalInformationCard) provider.getICardByCUID(handler, pesonal.getCUID());
			
			IElement elemClaims = makeClaims(policy, pesonal, selection, 
					certStoreTrust, urlRP);
			log.trace(elemClaims.getAs(String.class));
			// self signed card
			final ISecurityTokenService stsPersonal = (ISecurityTokenService)mapGlobalSettings.get
			("SecurityTokenService");

			log.trace( "Create request for self signed card");
			stsRequest = factoryRequest.createPersonalRequest
				(uriRP,
				selection.sslCertChain,
				certStoreTrust,
				java.net.URI.create("urn:oasis:names:tc:SAML:1.0:assertion"),
				pesonal,
				elemClaims);
			log.trace( "Invoke STS for self signed card.");
			stsPersonal.invoke(mapGlobalSettings, "SecurityTokenService", null, null, constants, stsRequest, stsResponse);
		}
		
		if (null != stsResponse.getFault()) {
			log.error( "Returning STS Fault: " + stsResponse.getFault().getDetail()  );
			throw new STSFaultException( stsResponse.getFault());
		}

		String encodedSSL = null;
		if (selection.sslCertChain != null) {
			X509Certificate first_cert = (selection.sslCertChain)[0];
			encodedSSL = new String
				(org.apache.commons.codec.binary.Base64.encodeBase64
					(first_cert.getEncoded()));
		}
		
		return new IdentityToken(stsResponse, encodedSSL);		
	}
	
	
	//	 turn claims from policy into one string which is needed to set up
	// elemClaims
	// TODO: only temporary function
	private IElement makeClaims(ICardSpacePolicy pol, IInformationCard infocard, 
			SelectionANDofORs selection, CertStore certStore, String rpURL) throws Exception {
		IElement elemClaims = new Element();

		String begin = "<wst:Claims xmlns:ic=\"http://schemas.xmlsoap.org/ws/2005/05/identity\" xmlns:wst=\"http://schemas.xmlsoap.org/ws/2005/02/trust\" wst:Dialect=\"http://schemas.xmlsoap.org/ws/2005/05/identity\">";
		String end = "</wst:Claims>";
		StringBuffer middle = new StringBuffer();
		X509Certificate[] rp_chain = selection.sslCertChain;
		List requiredClaims = pol.getRequiredClaims();
		makeClaimTags(requiredClaims, middle, infocard, rp_chain, 
				certStore, rpURL);
		if (pol.getOptionalClaims()!=null && pol.getOptionalClaims().size() > 0) {
			List optionalClaims = pol.getOptionalClaims();
			makeClaimTags(optionalClaims, middle, infocard, rp_chain, 
					certStore, rpURL);
		}

		elemClaims.set(begin + middle + end);

		return elemClaims;
	}	

	public void makeClaimTags(List claims, StringBuffer middle, IInformationCard infocard, 
			X509Certificate[] rp, CertStore certStore, String rpURL) {
		boolean selfIssued = infocard.isSelfIssued();
		for (int i = 0; i < claims.size(); i++) {
			String claim = (String)claims.get(i);
			if (selfIssued) {
				// place value into the RST
				try {
					// abs: this code should go into the infocard class--infocard should
					// compute ppid differently when it is a self-issued card.  not refactoring until
					// after Burton catalyst due to regression risk
					if (PPID_CLAIM.equalsIgnoreCase(claim)) {
//						byte[] ppid =  infocard.computePPI(rp);
						byte[] ppid = PPIDHelper.computeClaimValuePPID
							(infocard, rp, certStore, rpURL);
						if (ppid != null )
						{
						String sppid = new String
							(org.apache.commons.codec.binary.Base64.encodeBase64
								(ppid));
						middle.append("<ic:ClaimType Uri=\"")
							  .append(claim)
							  .append("\">")
							  .append(StringEscapeUtils.escapeXml(sppid))
							  .append("</ic:ClaimType>");
						}
					} else {
//						String val = infocard.getClaimValue(claim);
						IClaim c = infocard.getClaim(claim);
						String val = null;
						if (c instanceof ISimpleClaim) {
							ISimpleClaim sc = (ISimpleClaim) c;
							List l = sc.getValues();
							if (l.size() > 0) {
								val = (String) l.get(0);
							}
						}
						if (val!=null) {
							middle.append("<ic:ClaimType Uri=\"")
								  .append(claim)
								  .append("\">")
							      .append(StringEscapeUtils.escapeXml(val))
							      .append("</ic:ClaimType>");
						}
					}
				} catch (Exception e) {	
					log.error(e, e);
				}
			} else {
				if(infocard.getSupportedClaimTypesUris().contains(claim)){
					middle.append("<ic:ClaimType Uri=\"" + claim + "\"/>");
				}				
			}
		}
	}
	
	public ConfigurationHandler getConfigurationHandler(boolean selfIssued) throws Exception {
		if (null == stsConfigurationBase) {
			stsConfigurationBase = System.getProperty("org.eclipse.higgins.sts.conf");
		}
		if (null == stsConfigurationBase) {
			throw new ISSException("CardspaceSelector: org.eclipse.higgins.sts.conf System property not found");
		}
		String stsConfigurationFile = null;
		String stsConfigurationPersonalFile = System.getProperty("org.eclipse.higgins.sts.conf.personal.file");
		String stsConfigurationManagedFile = System.getProperty("org.eclipse.higgins.sts.conf.managed.client.file");
		if (selfIssued) {
			if (stsConfigurationPersonalFile == null)
				stsConfigurationFile = "PersonalConfiguration.xml";
			else
				stsConfigurationFile = stsConfigurationPersonalFile;
		}
		else {
			if (stsConfigurationManagedFile == null)
				stsConfigurationFile = "ClientConfiguration.xml";
			else
				stsConfigurationFile = stsConfigurationManagedFile;
		}	
		ConfigurationHandler configurationHandler = new ConfigurationHandler();
		configurationHandler.setConfigurationBase(stsConfigurationBase);
		
		configurationHandler.setFileName(stsConfigurationFile);

		if (!configurationHandler.configure(null)) {
			log.error("Not Initialized!");
			throw new ISSException("CardspaceSelector: cannot initialize configHandler");
		}

		return configurationHandler;
	}
	
}
