package org.eclipse.higgins.keystore.common;

import java.io.BufferedInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.net.URI;
import java.net.URISyntaxException;
import java.security.GeneralSecurityException;
import java.security.InvalidAlgorithmParameterException;
import java.security.NoSuchAlgorithmException;
import java.security.cert.CertPath;
import java.security.cert.CertPathValidator;
import java.security.cert.CertPathValidatorException;
import java.security.cert.CertPathValidatorResult;
import java.security.cert.CertStore;
import java.security.cert.CertStoreException;
import java.security.cert.CertificateException;
import java.security.cert.CertificateFactory;
import java.security.cert.CollectionCertStoreParameters;
import java.security.cert.PKIXParameters;
import java.security.cert.TrustAnchor;
import java.security.cert.X509CertSelector;
import java.security.cert.X509Certificate;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Hashtable;
import java.util.Iterator;
import java.util.List;

import javax.security.auth.callback.CallbackHandler;
import javax.security.auth.x500.X500Principal;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.eclipse.higgins.keystore.IKeyStoreService;
import org.eclipse.higgins.keystore.common.X509.AccessDescription;
import org.eclipse.higgins.keystore.common.X509.AuthorityInformationAccessExtension;
import org.eclipse.higgins.keystore.common.X509.ObjectIdentifier;
import org.eclipse.higgins.keystore.common.X509.PolicyInformation;
import org.eclipse.higgins.keystore.common.X509.URIName;
import org.eclipse.higgins.keystore.common.X509.X509HigginsCertificate;

public abstract class KeyStoreService implements IKeyStoreService {

	private Log log = LogFactory.getLog(KeyStoreService.class);
	
	private static Hashtable signsTable = new Hashtable();
	
	public X509Certificate[] getValidateCertificateChain(CallbackHandler handler,X509Certificate[] certPath, URI uri, CertStore certStore) throws CertificateException, CertStoreException {
		X509Certificate[] chain;
		try {
			chain = getParent(certPath, certStore);
			if (validateCertificateChain(handler,chain, uri, certStore))
				return chain;
		} catch (GeneralSecurityException e) {
			log.error(e, e);
			CertificateException exc = new CertificateException(e.getMessage());
			exc.setStackTrace(e.getStackTrace());
			throw exc;
		} catch (IOException e) {
			log.error(e, e);
			CertificateException exc = new CertificateException(e.getMessage());
			exc.setStackTrace(e.getStackTrace());
			throw exc;
		}
		return null;
	}
	
	protected X509Certificate [] getNormalizedCertificateChain(X509Certificate [] certPath, CertStore certStore) throws CertificateException, CertStoreException
	{
		X509Certificate[] chain;
		try {
			chain = getParent(certPath, certStore);
			return chain;
		} catch (GeneralSecurityException e) {
			log.error(e, e);
			CertificateException exc = new CertificateException(e.getMessage());
			exc.setStackTrace(e.getStackTrace());
			throw exc;
		} catch (IOException e) {
			log.error(e, e);
			CertificateException exc = new CertificateException(e.getMessage());
			exc.setStackTrace(e.getStackTrace());
			throw exc;
		}
	}
	
	public boolean extendendValidationCertificateChain(CallbackHandler handler,X509Certificate[] certChain,URI uri, CertStore certStore) {
		boolean result = false;		
		result = validateCertificateChain(handler,certChain,uri,certStore);
		if (result)
		{
			result=extendendValidationCertificate(certChain);
		}
		return result;
	}
	
	protected boolean extendendValidationCertificate(X509Certificate[] certChain) {
		boolean result = false;
		X509Certificate rootCert=certChain[certChain.length-1];
		X509HigginsCertificate rootHigginsCert=null;		
		
		if (rootCert instanceof X509HigginsCertificate)
			rootHigginsCert = (X509HigginsCertificate)rootCert;
		else
			rootHigginsCert = new X509HigginsCertificate(rootCert);
		String policyOID =  (String)signsTable.get(rootHigginsCert.getFingerprintSHA1());
		if (policyOID == null)
			return false;
		
		result = policyValidationCertificate(certChain[0],policyOID,false);
        
		if (!result)
			return false;
		
		for (int i=1;i<certChain.length-1;i++)
		{
			result = policyValidationCertificate(certChain[i],policyOID,true);	        
			if (!result)
				return false;
		}
		return true;
	}
	
	protected boolean uriValidationCertificate(X509Certificate cert, URI uri) throws URISyntaxException {
		X509HigginsCertificate higginsCert=null;
		if (cert instanceof X509HigginsCertificate)
			higginsCert = (X509HigginsCertificate)cert;
		else
			higginsCert = new X509HigginsCertificate(cert);
		
		String certHost=higginsCert.getTBSCertificateObject().getSubject().getAttributeValueByName("CN").toLowerCase();
		String host=uri.getHost();
		if (host != null)
			host = host.toLowerCase();
		else
			host = uri.toString();
		if (certHost.startsWith("*."))
		{
			certHost=certHost.substring(2);
			if (host.endsWith(certHost))
				return true;
		}
		else if (host.equals(certHost))
				return true;
		return false;
	}
	
	protected boolean policyValidationCertificate(X509Certificate cert,String policyOID, boolean isAnyPolicy ) {
		X509HigginsCertificate higginsCert=null;
		if (cert instanceof X509HigginsCertificate)
        	higginsCert = (X509HigginsCertificate)cert;
  		else
  			higginsCert = new X509HigginsCertificate(cert);
        
		if (higginsCert.getCertificatePolicies() == null)
        	return false;
		
		for (Iterator iter = higginsCert.getCertificatePolicies().getCertificatePolicies().iterator();iter.hasNext();)
        {
        	PolicyInformation policy = (PolicyInformation)iter.next();
        	if ( policy.getPolicyIdentifier().equals(policyOID))
        		return true;
        	if (isAnyPolicy && policy.getPolicyIdentifier().equals(ObjectIdentifier.id_ce_anyPolicy))
        		return true;
        }
		return false;
	}

	public boolean validateCertificateChain(CallbackHandler handler,X509Certificate[] certChain, URI uri, CertStore certStore) {

		try {
			if (certChain == null )
				return false;
			if (uri != null)
			{
				if (!uriValidationCertificate(certChain[0],uri))
					return false;
			}
			// Create the parameters for the validator
			HashSet trustAnchors = new HashSet();
			Collection colSert=certStore.getCertificates(null);			
			for (Iterator iter=colSert.iterator();iter.hasNext();)
				trustAnchors.add(new TrustAnchor((X509Certificate)iter.next(), null));
            
            PKIXParameters params = new PKIXParameters(trustAnchors);

			// Disable CRL checking since we are not supplying any CRLs
			params.setRevocationEnabled(false);

			// create list which contains only last element
			List ll = Arrays.asList(certChain);
			CertificateFactory certFact = CertificateFactory.getInstance("X.509");

			CertPath certPath = certFact.generateCertPath(ll);

			// Create the validator and validate the path
			CertPathValidator certPathValidator = CertPathValidator.getInstance(CertPathValidator.getDefaultType());
			CertPathValidatorResult result = certPathValidator.validate(certPath, params);

			if (result != null)
				return true;
			else
				return false;
		} catch (CertPathValidatorException e) {
			log.debug(e, e);
		} catch (Exception e) {
			log.error(e, e);
		}
		return false;
	}

	protected X509Certificate[] getParent(X509Certificate[] certificates, CertStore certStore) throws InvalidAlgorithmParameterException, NoSuchAlgorithmException, IOException, CertStoreException {
		List path = new ArrayList();
		List parentPath = null;
		X509Certificate cert = findLeafFromCertificates(certificates);
		List inputList = Arrays.asList(certificates);
		CollectionCertStoreParameters collCSP = new CollectionCertStoreParameters(inputList);
		CertStore tempStore = CertStore.getInstance("Collection", collCSP);		
		X509CertSelector selector = new X509CertSelector();
		path.add(cert);
		while (!isRootCertificate(cert)) {
			parentPath = getParent(cert, certStore, path);
			if (parentPath != null)
				break;
			selector.setSubject(cert.getIssuerX500Principal().getName());
			Collection collCertificates = tempStore.getCertificates(selector);
			if (!collCertificates.isEmpty()) {
				cert = (X509Certificate) collCertificates.iterator().next();
				path.add(cert);
			} else
				break;
		}

		if (parentPath != null) {
			X509Certificate[] chain = new X509Certificate[parentPath.size()];
			parentPath.toArray(chain);
			return chain;
		}
		return null;
	}

	protected List getParent(X509Certificate certificate, CertStore certStore, List path) {
		if (certificate != null) {
			X509Certificate cert = extendExternalPath(certificate);
			if (cert == null)
				cert = extendLocalPath(certificate, certStore);

			if (cert != null) {
				path.add(cert);
				if (!isRootCertificate(cert))
					return getParent(cert, certStore, path);
				else
					return path;
			}
		}
		return null;
	}

	protected X509Certificate extendExternalPath(X509Certificate certificate) {
		X509HigginsCertificate certHiggins = null;
		if (certificate instanceof X509HigginsCertificate)
			certHiggins = (X509HigginsCertificate)certificate;
		else
			certHiggins = new X509HigginsCertificate(certificate);
		AuthorityInformationAccessExtension extention = certHiggins.getAuthorityInformationAccess();
		if (extention != null) {
			URI uri = null;
			for (Iterator iter = extention.getAccessDescriptions().iterator(); iter.hasNext();) {
				AccessDescription accessDescription = (AccessDescription) iter.next();
				if (accessDescription.getAccessMethod().equals(ObjectIdentifier.id_ad_caIssuers)) {
					if (accessDescription.getAccessLocation().getValue() instanceof URIName) {
						uri = ((URIName) accessDescription.getAccessLocation().getValue()).getURI();
					}
				}
			}
			if (uri != null) {
				InputStream stream = null;
				try {
					stream = uri.toURL().openStream();

					BufferedInputStream bis = new BufferedInputStream(stream);
					CertificateFactory cf = CertificateFactory.getInstance("X.509");
					X509Certificate cert = (X509Certificate) cf.generateCertificate(bis);
					return new X509HigginsCertificate(cert);
				} catch (Exception e) {
					log.error(e, e);
				} finally {
					if (stream != null)
						try {
							stream.close();
						} catch (IOException e) {
							log.error(e, e);
						}
				}
			}
		}
		return null;
	}

	protected X509Certificate extendLocalPath(X509Certificate certificate, CertStore certStore) {
		try {
			X509Certificate parentCert = null;
			X509CertSelector selector = new X509CertSelector();
			selector.setSubject(certificate.getIssuerX500Principal().getName(X500Principal.CANONICAL));

			if (certStore != null) {
				Collection list = certStore.getCertificates(selector);

				if (!list.isEmpty())
				{
					parentCert = (X509Certificate) list.toArray()[0];
					return new X509HigginsCertificate(parentCert);
				}
			}

		} catch (Exception e) {
			log.error(e, e);
		}
		return null;
	}

	protected boolean isRootCertificate(X509Certificate cert) {

		X500Principal issuer_dn = null;
		X500Principal subject_dn = null;
		issuer_dn = cert.getIssuerX500Principal();
		subject_dn = cert.getSubjectX500Principal();
		return issuer_dn.equals(subject_dn);
	}

	public X509Certificate findLeafFromCertificates(X509Certificate [] listCerts) {
		if (0 == listCerts.length)
			return null;
		if (1 == listCerts.length)
			return (X509Certificate) listCerts[0];

		// Find the leaf certificate (the one that no other certificate points to as its issuer
		HashMap issuerMap = new HashMap();
		ArrayList subjectList = new ArrayList();

		for (int iCert = 0; iCert < listCerts.length; iCert++) {
			X509Certificate x509Certificate = (X509Certificate) listCerts[iCert];
			String strSubject = x509Certificate.getSubjectX500Principal().getName();
			String strIssuer = x509Certificate.getIssuerX500Principal().getName();
			issuerMap.put(strIssuer, strSubject);
			subjectList.add(strSubject);
		}
		for (int iCert = 0; iCert < subjectList.size(); iCert++) {
			String strSubject = (String) subjectList.get(iCert);
			if (null == issuerMap.get(strSubject)) {
				// log.trace("Found LEaf Certificate for Subject: " + strSubject);
				return (X509Certificate) listCerts[iCert];
			}
		}
		return null;
	}
	
	public static CertStore joinCertStore (CertStore certStore1, CertStore certStore2)
	{
		Collection certificates = new ArrayList();
		if (certStore1 == null)
			return certStore2;
		if (certStore2 == null)
			return certStore1;
		try {
			certificates.addAll(certStore1.getCertificates(null));
		} catch (CertStoreException e1) {
			// TODO Auto-generated catch block
			e1.printStackTrace();
		}
		try {
			certificates.addAll(certStore2.getCertificates(null));
		} catch (CertStoreException e1) {
			// TODO Auto-generated catch block
			e1.printStackTrace();
		}
		CollectionCertStoreParameters collCSP = new CollectionCertStoreParameters(certificates);
		try {
			return CertStore.getInstance("Collection", collCSP);
		} catch (InvalidAlgorithmParameterException e) {
			// TODO Auto-generated catch block
			e.printStackTrace();
		} catch (NoSuchAlgorithmException e) {
			// TODO Auto-generated catch block
			e.printStackTrace();
		}
		return null;
	}
	
	static
	{		
		// companies names from http://cabforum.org/forum.html and http://en.wikipedia.org/wiki/Extended_Validation_Certificate
		
//		A-Trust GmbH 
//		Certum 
		signsTable.put("66 31 bf 9e f7 4f 9e b6 c9 d5 a6 0c ba 6a be d1 f7 bd ef 7b", "1.3.6.1.4.1.6449.1.2.1.5.1");// Comodo CA Ltd 
		signsTable.put("5f 43 e5 b1 bf f8 78 8c ac 1c c7 ca 4a 9a c6 22 2b cc 34 c6", "1.3.6.1.4.1.6334.1.100.1");// Cybertrust 
		signsTable.put("5f b7 ee 06 33 e2 59 db ad 0c 4c 9a e6 d3 8f 1a 61 c7 dc 25", "2.16.840.1.114412.2.1");// DigiCert, Inc.
		signsTable.put("f0 bb 01 d1 d8 31 98 51 6a 4f f1 6d 8a b5 ef ab 74 c7 cc a3", "2.16.528.1.1001.1.1.1.12.6.1.1.1");// DigiNotar				
//		Echoworx Corporation 
		signsTable.put("b3 1e b1 b7 40 e3 6c 84 02 da dc 37 d4 4d f5 d4 67 49 52 f9", "2.16.840.1.114028.10.1.2");// Entrust, Inc. 
		signsTable.put("32 3c 11 8e 1b f7 b8 b6 52 54 e2 e2 10 0d d6 02 90 37 f0 96", "1.3.6.1.4.1.14370.1.6");// GeoTrust, Inc.
//		Getronics PinkRoccade
		signsTable.put("75 e0 ab b6 13 85 12 27 1c 04 f8 5f dd de 38 e4 b7 24 2e fe", "1.3.6.1.4.1.4146.1.1");// GlobalSign 
		signsTable.put("27 96 ba e6 3f 18 01 e2 77 26 1b a0 d7 77 70 02 8f 20 ee e4", "2.16.840.1.114413.1.7.23.3");// GoDaddy.com, Inc. 
//		IdenTrust, Inc. 
//		ipsCA, IPS Certification Authority s.l. 
//		Izenpe S.A. 
		signsTable.put("74 f8 a3 c3 ef e7 b3 90 06 4b 83 90 3c 21 64 60 20 e5 df ce", "1.3.6.1.4.1.782.1.2.1.8.1");// Network Solutions, LLC 
		signsTable.put("ca 3a fb cf 12 40 36 4b 44 b2 16 20 88 80 48 39 19 93 7c f7", "1.3.6.1.4.1.8024.0.2.100.1.2");// QuoVadis Ltd. (https://www.quovadis.bm) 
//		RSA Security, Inc. 
//		SECOM Trust Systems CO., Ltd. 
//		Skaitmeninio sertifikavimo centras (SSC)
		signsTable.put("ad 7e 1c 28 b0 64 ef 8f 60 03 40 20 14 c3 d0 e3 37 0e b5 8a", "2.16.840.1.114414.1.7.23.3");// Starfield Technologies
//		SwissSign AG
//		TC TrustCenter GmbH
//		TDC Certification Authority
		signsTable.put("87 82 c6 c3 04 35 3b cf d2 96 92 d2 59 3e 7d 44 d9 34 ff 11", "2.16.840.1.114404.1.1.2.4.1");// Trustwave
		signsTable.put("91 c6 d6 ee 3e 8a c8 63 84 e5 48 c2 99 29 5c 75 6c 81 7b 81", "2.16.840.1.113733.1.7.48.1");// Thawte, Inc. 
//		Trustis Limited 
		signsTable.put("4e b6 d5 78 49 9b 1c cf 5f 58 1e ad 56 be 3d 9b 67 44 a5 e5", "2.16.840.1.113733.1.7.23.6");// VeriSign, Inc. 
//		Wells Fargo Bank, N.A.		
	}
}
