/**
 * Copyright (c) 2008 Parity Communications, Inc. 
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     Artem Verkhovets - initial API and implementation
 */
package org.eclipse.higgins.keystore.common.X509;
import java.math.BigInteger;
import java.security.InvalidKeyException;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.security.NoSuchProviderException;
import java.security.Principal;
import java.security.PublicKey;
import java.security.SignatureException;
import java.security.cert.CertificateEncodingException;
import java.security.cert.CertificateException;
import java.security.cert.CertificateExpiredException;
import java.security.cert.CertificateNotYetValidException;
import java.security.cert.X509Certificate;
import java.util.Date;
import java.util.Set;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

/**
 *  
 * @author Artem Verkhovets
 *
 */
public class X509HigginsCertificate extends X509Certificate {

	/**
	 * 
	 */
	private static final long serialVersionUID = 2282425799629719567L;

	private Log log = LogFactory.getLog(X509HigginsCertificate.class);
	
	private X509Certificate instance = null;	
	private CertificatePoliciesExtension certificatePolicies;
	private AuthorityInformationAccessExtension authorityInformationAccess;
	private String fingerprintSHA1;
	private TBSCertificate tBSCertificateObject=null;
	
	public String getFingerprintSHA1() {
		return fingerprintSHA1;
	}

	public AuthorityInformationAccessExtension getAuthorityInformationAccess() {
		return authorityInformationAccess;
	}

	public CertificatePoliciesExtension getCertificatePolicies() {
		return certificatePolicies;
	}

	public X509HigginsCertificate(X509Certificate certificate) {
		instance = certificate;
		initialization();
	}

	public void checkValidity() throws CertificateExpiredException, CertificateNotYetValidException {
		instance.checkValidity();

	}

	public void checkValidity(Date date) throws CertificateExpiredException, CertificateNotYetValidException {
		instance.checkValidity(date);

	}

	public int getBasicConstraints() {
		return instance.getBasicConstraints();
	}

	public Principal getIssuerDN() {
		return instance.getIssuerDN();
	}

	public boolean[] getIssuerUniqueID() {

		return instance.getIssuerUniqueID();
	}

	public boolean[] getKeyUsage() {
		return instance.getKeyUsage();
	}

	public Date getNotAfter() {

		return instance.getNotAfter();
	}

	public Date getNotBefore() {

		return instance.getNotBefore();
	}

	public BigInteger getSerialNumber() {

		return instance.getSerialNumber();
	}

	public String getSigAlgName() {

		return instance.getSigAlgName();
	}

	public String getSigAlgOID() {

		return instance.getSigAlgOID();
	}

	public byte[] getSigAlgParams() {

		return instance.getSigAlgParams();
	}

	public byte[] getSignature() {

		return instance.getSignature();
	}

	public Principal getSubjectDN() {

		return instance.getSubjectDN();
	}

	public boolean[] getSubjectUniqueID() {

		return instance.getSubjectUniqueID();
	}

	public byte[] getTBSCertificate() throws CertificateEncodingException {

		return instance.getTBSCertificate();
	}

	public int getVersion() {

		return instance.getVersion();
	}

	public byte[] getEncoded() throws CertificateEncodingException {

		return instance.getEncoded();
	}

	public PublicKey getPublicKey() {

		return instance.getPublicKey();
	}

	public String toString() {

		return instance.toString();
	}

	public void verify(PublicKey key) throws CertificateException, NoSuchAlgorithmException, InvalidKeyException, NoSuchProviderException, SignatureException {
		instance.verify(key);

	}

	public void verify(PublicKey key, String sigProvider) throws CertificateException, NoSuchAlgorithmException, InvalidKeyException, NoSuchProviderException, SignatureException {
		instance.verify(key, sigProvider);

	}

	public Set getCriticalExtensionOIDs() {
		return instance.getCriticalExtensionOIDs();
	}

	public byte[] getExtensionValue(String oid) {
		return instance.getExtensionValue(oid);
	}

	public Set getNonCriticalExtensionOIDs() {
		return instance.getNonCriticalExtensionOIDs();
	}

	public boolean hasUnsupportedCriticalExtension() {
		return instance.hasUnsupportedCriticalExtension();
	}
	
	public TBSCertificate getTBSCertificateObject() {
		return tBSCertificateObject;
	}
	
	private void initialization()
	{
		calculateFingerprintSHA1();
		
		try {
			tBSCertificateObject = new TBSCertificate(this.instance.getTBSCertificate());
		} catch (CertificateEncodingException e) {
			log.error(e,e);
		}
		if (instance == null)
			return;
		
		byte [] certificatePoliciesByteArray=instance.getExtensionValue(ObjectIdentifier.id_ce_certificatePolicies);
		Set criticalExtention = instance.getCriticalExtensionOIDs();
		if (criticalExtention == null)
			return ;
		
		boolean critical=false;
		if (criticalExtention == null)
			critical=criticalExtention.contains(ObjectIdentifier.id_ce_certificatePolicies);
		
		if (certificatePoliciesByteArray!= null && certificatePoliciesByteArray.length>0)
			certificatePolicies = new CertificatePoliciesExtension(certificatePoliciesByteArray,critical);
		
		byte [] authorityInformationAccessByteArray=instance.getExtensionValue(ObjectIdentifier.id_pe_authorityInfoAccess);
		
		critical=false;		
		if (criticalExtention == null)
			critical=criticalExtention.contains(ObjectIdentifier.id_pe_authorityInfoAccess);
		if (authorityInformationAccessByteArray != null && authorityInformationAccessByteArray.length>0)
			authorityInformationAccess = new AuthorityInformationAccessExtension(authorityInformationAccessByteArray,critical);	
		
	}
	
    private void calculateFingerprintSHA1(){
    	if (this.instance==null)
    		return;
    	
	    MessageDigest sha1=null;
	    try {
	        sha1 = MessageDigest.getInstance("SHA1");
	    }
	    catch (NoSuchAlgorithmException e) {
	    }
	    if (sha1 == null)
	    	return;
	    
	    sha1.reset();
	    byte[] result=null;
		try {
			result = sha1.digest(this.instance.getEncoded());
		} catch (CertificateEncodingException e) {
			log.error(e,e);
		}
		if (result == null)
			return;
		StringBuffer builder = new StringBuffer();
		for (int i = 0;i < result.length;i++)
		{
			String strValue=Integer.toHexString(0xFF&result[i]).toLowerCase();
			if (strValue.length()==1)
				builder.append("0");
			builder.append(strValue);
			if (i != result.length-1)
				builder.append(" ");
		}
		this.fingerprintSHA1 = builder.toString();
	}
}
