/**
 * Copyright (c) 2008 Parity Communications, Inc. 
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     Artem Verkhovets - initial API and implementation
 */
package org.eclipse.higgins.keystore.common.utils;

import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.io.UnsupportedEncodingException;
import java.security.cert.CertificateParsingException;
import java.io.InputStream;
import java.math.BigInteger;
import java.util.ArrayList;
import java.util.List;

public class Asn1Node {

	private int tag;

	private byte[] value;

	private List collection;

	public static Asn1Node getInstance(byte[] value) {
		try {
			InputStream inputBuffer = new ByteArrayInputStream(value);
			return getInstance(inputBuffer);
		} catch (Exception exc) {

		}
		return null;
	}

	public static Asn1Node getInstance(InputStream stream) {
		try {
			return Asn1Node.GetAsn1Node(stream, new AllReadedByte());
		} catch (Exception exc) {

		}
		return null;
	}

	private Asn1Node() {
		this.tag = 0;
		this.value = new byte[0];
		this.collection = new ArrayList();
	}

	public int getTag() {
		return tag;
	}

	public void setTag(int tag) {
		this.tag = tag;
	}

	public boolean isUniversal() {
		return (tag & 0xc0) == 0;
	}

	public boolean isApplication() {
		return (tag & 0xc0) == 64;
	}

	public boolean isContextSpecific() {
		return (tag & 0xc0) == 128;
	}

	public boolean isContextSpecific(byte byte0) {
		if (!isContextSpecific())
			return false;
		else
			return (tag & 0x1f) == byte0;
	}

	boolean isPrivate() {
		return (tag & 0xc0) == 192;
	}

	public boolean isConstructed() {
		return (tag & 0x20) == 32;
	}

	public boolean isConstructed(byte byte0) {
		if (!isConstructed())
			return false;
		else
			return (tag & 0x1f) == byte0;
	}

	private static Asn1Node GetAsn1Node(InputStream stream, AllReadedByte allreadbytes) throws CertificateParsingException, IOException {
		Asn1Node node = new Asn1Node();
		AllReadedByte length = new AllReadedByte();
		if (GetTagValue(stream, node, allreadbytes) == Z_ERROR) {
			return null;
		} else {
			if (node.getTag() == 0) {
				return null;
			} else {
				if (GetLenghtValue(stream, length, allreadbytes) == Z_ERROR) {
					return null;
				} else {
					if (length.getValue() > 0) {
						// Length is known
						AllReadedByte readbytes = new AllReadedByte();
						if (node.isConstructed()) {
							do {
								Asn1Node node1 = Asn1Node.GetAsn1Node(stream, readbytes);
								if (node1 != null) {
									node.collection.add(node1);
								} else {
									break;
								}
								if (length.getValue() <= readbytes.getValue())
									break;
							} while (true);
						} else {
							byte[] temp = new byte[length.getValue()];
							for (int i = 0; i < length.getValue(); i++) {
								temp[i] = (byte) stream.read();
								readbytes.Plus();
							}
							node.value = temp;
						}
						allreadbytes.Plus(readbytes);
					} else {
						if (length.getValue() == -1) {
							// Length isn't known
							AllReadedByte readbytes = new AllReadedByte();
							do {
								Asn1Node node1 = Asn1Node.GetAsn1Node(stream, readbytes);
								if (node1 != null) {
									node.collection.add(node1);
								} else {
									break;
								}
							} while (true);
						} else {
							node.value = new byte[0];
						}
					}
				}
				return node;
			}
		}
	}

	private static int GetTagValue(InputStream stream, Asn1Node node, AllReadedByte readbytes) throws CertificateParsingException, IOException {
		int result;
		int ch;
		int tagno = 0;
		ch = stream.read();
		readbytes.Plus();
		if (ch == 0) {
			ch = stream.read();
			if (ch != 0) {
				result = Z_ERROR;
			} else {
				tagno = 0;
				result = Z_OK;
				readbytes.Plus();
			}
		} else {
			/*
			 * +-+-+---------+ |c|f| number | +-+-+---------+
			 */
			tagno = ch;

			/*
			 * +-+-+-+-+-+-+-+ +-+--------+ +-+------------+ +-+---------+ |c|f|1|1|1|1|1| |1| | ... |1| | |0| | +-+-+-+-+-+-+-+ +-+--------+ +-+------------+ +-+---------+
			 */
			if (tagno == 0x1F) {
				/* long form representation */
				tagno = 0;
				while (true) {
					ch = stream.read();
					if (ch == -1) {
						result = Z_ERROR;
						break;
					} else {
						readbytes.Plus();
						if (ch < 128) {
							/* the last octet */
							tagno <<= 7;
							tagno += ch;
							result = Z_OK;
							break;
						}
						ch &= 0x7F; /* turn off the msb in an octet */
						tagno <<= 7;
						tagno += ch;
					}
				}
			} else {
				result = Z_OK;
			}
		}
		node.setTag(tagno);
		return result;
	}

	private static int GetLenghtValue(InputStream stream, AllReadedByte length, AllReadedByte readbytes) throws CertificateParsingException, IOException {
		int ch;
		int nocs;
		int i = 0;
		int len = 0;
		int result;
		length.setValue(0);
		ch = stream.read();
		if (ch == -1) {
			result = Z_ERROR;
		} else {
			readbytes.Plus();
			/*
			 * +-----------+ |0| length | +-----------+
			 */
			if ((ch & 0x80) == 0) {
				result = Z_OK;
				length.setValue(ch);
			} else {
				/*
				 * +---------------+ +---------------+ +---------------+ |1|0|0|0|0|0|0|0| ... |0|0|0|0|0|0|0|0| |0|0|0|0|0|0|0|0| +---------------+ +---------------+ +---------------+
				 */
				/* Indefinite length form is used. */
				if (ch == 0x80) {
					result = Z_OK;
					length.setValue(-1);
				} else {
					/*
					 * +------------+ +----------+ +----------+ |1| #-octets | | octet | ... | octet | +------------+ +----------+ +----------+
					 */
					/* Long, definite length form. */
					nocs = ch & 0x7F;
					if (nocs > 4) {
						result = Z_ERROR;
						length.setValue(0);
					} else {
						i = nocs;
						while (i > 0) {
							ch = stream.read();
							if (ch == -1) {
								result = Z_ERROR;
								break;
							} else {
								readbytes.Plus();
								len <<= 8;
								len += ch;
							}
							i--;
						}
						result = Z_OK;
						length.setValue(len);
					}
				}
			}
		}
		return result;
	}

	public List getCollection() {
		return collection;
	}

	public void setCollection(ArrayList collection) {
		this.collection = collection;
	}

	public byte[] getValue() {
		return value;
	}

	public void setValue(byte[] value) {
		this.value = new byte[value.length];
		System.arraycopy(value, 0, this.value, 0, value.length);
	}

	/**
	 * @return
	 * @throws CertificateParsingException
	 */
	public String getValueAsString() throws UnsupportedEncodingException {
		switch (getTag()) {
		case Asn1Node.tag_UTF8String:
			return getValueAsUTF8String();
		case Asn1Node.tag_PrintableString:
			return getValueAsPrintableString();
		case Asn1Node.tag_T61String:
			return getValueAsT61String();
		case Asn1Node.tag_IA5String:
			return getValueAsIA5String();
		case Asn1Node.tag_GeneralString:
			return getValueAsGeneralString();
		case Asn1Node.tag_BMPString:
			return getValueAsBMPString();
		default:
			return null;
		}
	}

	public String getValueAsIA5String() throws UnsupportedEncodingException {
		if(tag != tag_IA5String)
            throw new UnsupportedEncodingException((new StringBuffer()).append("Value is not a IA5 string. Tag=").append(tag).toString());
        else
            return new String(value, "ASCII");	
	}

	public String getValueAsBMPString() throws UnsupportedEncodingException{
		if(tag != tag_BMPString)
            throw new UnsupportedEncodingException((new StringBuffer()).append("Value is not a BMP string. Tag=").append(tag).toString());
        else
            return new String(value, "UnicodeBigUnmarked");
	}

	public String getValueAsGeneralString() throws UnsupportedEncodingException {
		if(tag != tag_GeneralString)
            throw new UnsupportedEncodingException((new StringBuffer()).append("Value is not a GeneralString string. Tag=").append(tag).toString());
        else
            return new String(value, "ASCII");
	}

	public String getValueAsT61String() throws UnsupportedEncodingException{
		if(tag != tag_T61String)
            throw new UnsupportedEncodingException((new StringBuffer()).append("Value is not a T61 string. Tag=").append(tag).toString());
        else
            return new String(value, "ISO-8859-1");
	}

	public String getValueAsPrintableString() throws UnsupportedEncodingException {
        if(tag != tag_PrintableString)
            throw new UnsupportedEncodingException((new StringBuffer()).append("Value is not a string. Tag=").append(tag).toString());
        else
            return new String(value, "ASCII");
	}

	public String getValueAsUTF8String() throws UnsupportedEncodingException {
		if(tag != tag_UTF8String)
            throw new UnsupportedEncodingException((new StringBuffer()).append("Value is not a UTF8 string. Tag=").append(tag).toString());
        else
            return new String(value, "UTF8");
	}

	public int getInteger() throws CertificateParsingException {
		if (tag != 2)
			throw new CertificateParsingException((new StringBuffer()).append("getInteger, not an int ").append(tag).toString());
		else {
			BigInteger biginteger = getBigInteger();
			if (biginteger.compareTo(BigInteger.valueOf(0xffffffff80000000L)) < 0)
				throw new CertificateParsingException("Integer below minimum valid value");
			if (biginteger.compareTo(BigInteger.valueOf(0x7fffffffL)) > 0)
				throw new CertificateParsingException("Integer exceeds maximum valid value");
			else
				return biginteger.intValue();
		}
	}

	public BigInteger getBigInteger() throws CertificateParsingException {
		if (tag != 2)
			throw new CertificateParsingException((new StringBuffer()).append("getBigInteger, not an int ").append(tag).toString());
		else {
			byte temp[] = new byte[value.length];
			System.arraycopy(value, 0, temp, 0, value.length);
			return new BigInteger(temp);
		}
	}

	public static final int Type_None = -1;

	public static final int Primitive = 0;

	public static final int Constructed = 1;

	public static final int Class_None = -1;

	public static final int Universal = 0;

	public static final int Application = 1;

	public static final int Ber_Context = 2;

	public static final int Private = 3;

	public static final int Z_ERROR = 0;

	public static final int Z_OK = 1;

	public static final byte tag_Boolean = 1;

	public static final byte tag_Integer = 2;

	public static final byte tag_BitString = 3;

	public static final byte tag_OctetString = 4;

	public static final byte tag_Null = 5;

	public static final byte tag_ObjectId = 6;

	public static final byte tag_Enumerated = 10;

	public static final byte tag_UTF8String = 12;

	public static final byte tag_PrintableString = 19;

	public static final byte tag_T61String = 20;

	public static final byte tag_IA5String = 22;

	public static final byte tag_UtcTime = 23;

	public static final byte tag_GeneralizedTime = 24;

	public static final byte tag_GeneralString = 27;

	public static final byte tag_UniversalString = 28;

	public static final byte tag_BMPString = 30;

	public static final byte tag_Sequence = 48;

	public static final byte tag_SequenceOf = 48;

	public static final byte tag_Set = 49;

	public static final byte tag_SetOf = 49;
}
