/**
 * Copyright (c) 2008 Parity Communications, Inc. 
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     Artem Verkhovets - initial API and implementation
 */
package org.eclipse.higgins.keystore.common.X509;

import java.io.ByteArrayInputStream;
import java.security.cert.CertificateParsingException;
import java.io.IOException;
import java.io.InputStream;
import java.io.Serializable;

import org.eclipse.higgins.keystore.common.utils.Asn1Node;

/**
 * 
 * @author Artem Verkhovets
 * 
 */
public final class ObjectIdentifier implements Serializable {

	private static final long serialVersionUID = 7309080005146431409L;

	protected int components[];

	protected int componentLen;

	private volatile transient String stringForm;

	public ObjectIdentifier(String s) throws CertificateParsingException {
		byte byte0 = 46;
		int i = 0;
		int j = 0;
		for (componentLen = 0; (j = s.indexOf(byte0, i)) != -1; componentLen++)
			i = j + 1;

		componentLen++;
		components = new int[componentLen];
		i = 0;
		int k = 0;
		String s1 = null;
		while ((j = s.indexOf(byte0, i)) != -1) {
			s1 = s.substring(i, j);
			if ((components[k++] = Integer.valueOf(s1).intValue()) < 0)
				throw new CertificateParsingException("oid components must be nonnegative");
			i = j + 1;
		}
		s1 = s.substring(i);
		components[k] = Integer.valueOf(s1).intValue();
		if (components[0] < 0 || components[0] > 2)
			throw new CertificateParsingException("First oid component is invalid ");
		if (components[1] < 0 || components[1] > 39) {
			throw new CertificateParsingException("Second oid component is invalid ");
		} else {
			stringForm = s;
			return;
		}
	}

	public ObjectIdentifier(int ai[]) throws CertificateParsingException {
		if (ai.length >= 1 && (ai[0] < 0 || ai[0] > 2))
			throw new CertificateParsingException("First oid component is invalid ");
		if (ai.length >= 2 && (ai[1] < 0 || ai[1] > 39)) {
			throw new CertificateParsingException("Second oid component is invalid ");
		} else {
			components = (int[]) (int[]) ai.clone();
			componentLen = ai.length;
			toString();
			return;
		}
	}

	public ObjectIdentifier(byte[] value) throws CertificateParsingException {
		try {
			InputStream inputBuffer = new ByteArrayInputStream(value);
			initFromEncoding(inputBuffer, 0);
		} catch (IOException exc) {
			throw new CertificateParsingException(exc.getMessage());
		}
		toString();
	}

	public ObjectIdentifier(InputStream inputBuffer) throws CertificateParsingException {
		try {
			initFromEncoding(inputBuffer, 0);
		} catch (IOException exc) {
			throw new CertificateParsingException(exc.getMessage());
		}
		toString();
	}

	public ObjectIdentifier(Asn1Node node) throws CertificateParsingException {
		if (node.getTag() != Asn1Node.tag_ObjectId)
			return;
		try {
			InputStream inputBuffer = new ByteArrayInputStream(node.getValue());
			initFromEncoding(inputBuffer, 0);
		} catch (IOException exc) {
			throw new CertificateParsingException(exc.getMessage());
		}
		toString();
	}

	private void initFromEncoding(InputStream inputBuffer, int i) throws IOException, CertificateParsingException {
		boolean flag = true;
		components = new int[5];
		componentLen = 0;
		while (inputBuffer.available() > i) {
			int j = getComponent(inputBuffer);
			if (j < 0)
				throw new CertificateParsingException("Component values must be nonnegative");
			if (flag) {
				byte byte0;
				if (j < 40)
					byte0 = 0;
				else if (j < 80)
					byte0 = 1;
				else
					byte0 = 2;
				int k = j - byte0 * 40;
				if (k > 39)
					throw new CertificateParsingException("Invalid second component");
				components[0] = byte0;
				components[1] = k;
				componentLen = 2;
				flag = false;
			} else {
				if (componentLen >= components.length) {
					int ai[] = new int[components.length + 5];
					System.arraycopy(components, 0, ai, 0, components.length);
					components = ai;
				}
				components[componentLen++] = j;
			}
		}
		if (inputBuffer.available() != i)
			throw new CertificateParsingException("Malformed input data");
		else
			return;
	}

	private static int getComponent(InputStream inputBuffer) throws IOException, CertificateParsingException {
		int j = 0;
		int i = 0;
		for (; j < 4; j++) {
			i <<= 7;
			int k = inputBuffer.read();
			i |= k & 0x7f;
			if ((k & 0x80) == 0)
				return i;
		}

		throw new CertificateParsingException("X509.OID, component value too big");
	}

	public boolean equals(Object obj) {
		if (this == obj)
			return true;
		if (!(obj instanceof ObjectIdentifier || obj instanceof String))
			return false;

		if (obj instanceof String)
			return stringForm.equals(obj);

		ObjectIdentifier objectidentifier = (ObjectIdentifier) obj;
		if (componentLen != objectidentifier.componentLen)
			return false;
		for (int i = 0; i < componentLen; i++)
			if (components[i] != objectidentifier.components[i])
				return false;

		return true;
	}

	public int hashCode() {
		int i = componentLen;
		for (int j = 0; j < componentLen; j++)
			i += components[j] * 37;

		return i;
	}

	public String toString() {
		String s = stringForm;
		if (s == null) {
			StringBuffer stringbuffer = new StringBuffer(componentLen * 4);
			for (int i = 0; i < componentLen; i++) {
				if (i != 0)
					stringbuffer.append('.');
				stringbuffer.append(components[i]);
			}

			s = stringbuffer.toString();
			stringForm = s;
		}
		return s;
	}

	public static final String id_ce = "2.5.29";

	public static final String id_ce_certificatePolicies = id_ce + ".32";

	public static final String id_ce_anyPolicy = id_ce_certificatePolicies + ".0";

	public static final String id_pkix = "1.3.6.1.5.5.7";

	public static final String id_qt = id_pkix + ".2";

	public static final String id_qt_cps = id_qt + ".1";

	public static final String id_qt_unotice = id_qt + ".2";

	public static final String id_pe = id_pkix + ".1";

	public static final String id_pe_authorityInfoAccess = id_pe + ".1";

	public static final String id_ad = id_pkix + ".48";

	public static final String id_ad_caIssuers = id_ad + ".2";

	public static final String id_ad_ocsp = id_ad + ".1";

	public static final String id_at = "2.5.4";

	public static final String id_at_commonName = id_at + ".3";

	public static final String id_at_countryName = id_at + ".6";

	public static final String id_at_localityName = id_at + ".7";

	public static final String id_at_stateOrProvinceName = id_at + ".8";

	public static final String id_at_organizationName = id_at + ".10";

	public static final String id_at_organizationalUnitName = id_at + ".11";
}
