/*******************************************************************************
 * Copyright (c) 2006-2007 Parity Communications, Inc.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     Valery Kokhan - Initial API and implementation
 *******************************************************************************/

package org.eclipse.higgins.registry;

import java.security.AccessController;
import java.security.PrivilegedAction;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;

import javax.imageio.spi.ServiceRegistry;

import sun.misc.ServiceConfigurationError;

/**
 * This implementation of
 * {@link org.eclipse.higgins.registry.IRegistryExtension} uses
 * {@link javax.imageio.spi.ServiceRegistry} mechanism to lookup for available
 * service providers.
 * 
 * @see org.eclipse.higgins.registry.IRegistryExtension
 * @see javax.imageio.spi.ServiceRegistry
 * 
 * @param <E>
 *            the service provider type to search for
 */
public class IIORegistryExtension implements
		IRegistryExtension {

	private Class type;

	public IIORegistryExtension(Class type) {
		this.type = type;
	}

	/* (non-Javadoc)
	 * @see org.eclipse.higgins.registry.IRegistryExtension#getServiceProviders()
	 */
	public Iterator getServiceProviders() {
		List list = new ArrayList(getInstalledProviders());
		list.addAll(getRuntimeProviders());
		return list.iterator();
	}

	/**
	 * Searches for implementations of a particular service class using the
	 * given class loader.
	 * 
	 * <p>
	 * This method transforms the name of the given service class into a
	 * provider-configuration filename as described in the class comment and
	 * then uses the <code>getResources</code> method of the given class
	 * loader to find all available files with that name. These files are then
	 * read and parsed to produce a list of provider-class names. The iterator
	 * that is returned uses the given class loader to look up and then
	 * instantiate each element of the list.
	 * 
	 * <p>
	 * Because it is possible for extensions to be installed into a running Java
	 * virtual machine, this method may return different results each time it is
	 * invoked.
	 * 
	 * @param providerClass
	 *            a <code>Class</code>object indicating the class or
	 *            interface of the service providers being detected.
	 * 
	 * @param loader
	 *            the class loader to be used to load provider-configuration
	 *            files and instantiate provider classes, or <code>null</code>
	 *            if the system class loader (or, failing that the bootstrap
	 *            class loader) is to be used.
	 * 
	 * @return An <code>Iterator</code> that yields provider objects for the
	 *         given service, in some arbitrary order. The iterator will throw
	 *         an <code>Error</code> if a provider-configuration file violates
	 *         the specified format or if a provider class cannot be found and
	 *         instantiated.
	 * 
	 * @exception IllegalArgumentException
	 *                if <code>providerClass</code> is <code>null</code>.
	 */
	private static Iterator providers(Class providerClass,
			ClassLoader loader) {
		if (providerClass == null) {
			throw new IllegalArgumentException("providerClass == null!");
		}
		return ServiceRegistry.lookupProviders(providerClass, loader);
	}

	/**
	 * Locates and incrementally instantiates the available providers of a given
	 * service using the context class loader. This convenience method is
	 * equivalent to:
	 * 
	 * <pre>
	 * ClassLoader cl = Thread.currentThread().getContextClassLoader();
	 * return providers(service, cl);
	 * </pre>
	 * 
	 * @param providerClass
	 *            a <code>Class</code>object indicating the class or
	 *            interface of the service providers being detected.
	 * 
	 * @return An <code>Iterator</code> that yields provider objects for the
	 *         given service, in some arbitrary order. The iterator will throw
	 *         an <code>Error</code> if a provider-configuration file violates
	 *         the specified format or if a provider class cannot be found and
	 *         instantiated.
	 * 
	 * @exception IllegalArgumentException
	 *                if <code>providerClass</code> is <code>null</code>.
	 * 
	 * @see #providers(java.lang.Class, java.lang.ClassLoader)
	 */
	private static Iterator providers(Class providerClass) {
		ClassLoader cl = Thread.currentThread().getContextClassLoader();
		return providers(providerClass, cl);
	}

	/**
	 * Locates and incrementally instantiates the available providers of a given
	 * service using the extension class loader. This convenience method simply
	 * locates the extension class loader, call it <tt>extClassLoader</tt>,
	 * and then does
	 * 
	 * <pre>
	 * return providers(providerClass, extClassLoader);
	 * </pre>
	 * 
	 * If the extension class loader cannot be found then the system class
	 * loader is used; if there is no system class loader then the bootstrap
	 * class loader is used.
	 * 
	 * @param providerClass
	 *            The service's abstract service class
	 * 
	 * @return An <tt>Iterator</tt> that yields provider objects for the given
	 *         service, in some arbitrary order. The iterator will throw a
	 *         <tt>ServiceConfigurationError</tt> if a provider-configuration
	 *         file violates the specified format or if a provider class cannot
	 *         be found and instantiated.
	 * 
	 * @exception IllegalArgumentException
	 *                if <code>providerClass</code> is <code>null</code>.
	 * 
	 * @see #providers(java.lang.Class, java.lang.ClassLoader)
	 */
	private static Iterator installedProviders(Class type) {
		ClassLoader cl = ClassLoader.getSystemClassLoader();
		ClassLoader prev = null;
		while (cl != null) {
			prev = cl;
			cl = cl.getParent();
		}
		return providers(type, prev);
	}

	/**
	 * Returns all available service providers found on the system classpath (<i>e.g.</i>,
	 * the <code>jre/lib/ext</code> directory in Sun's implementation of JDK).
	 */
	private List getInstalledProviders() {
		final List list = new ArrayList();
		/*
		 * We need load installed providers from lib/ext directory in the
		 * privileged mode in order to be able read corresponding jar files even
		 * if file read capability is restricted (like the applet context case).
		 */
		PrivilegedAction doRegistration = new PrivilegedAction() {
			public Object run() {
				Iterator providers = installedProviders(type);
				while (providers.hasNext()) {
					//IServiceProvider p = (IServiceProvider) providers.next();
					Object o = providers.next();
					if (!(o instanceof IServiceProvider)) {
						System.err.println("WARNING: object=\"" + o + "\" is not instance of IServiceProvider");
					}
					// registry.registerServiceProvider(p, type);
					list.add(o);
				}
				return this;
			}
		};

		AccessController.doPrivileged(doRegistration);
		return list;
	}

	/**
	 * Returns all available service providers found on the application class
	 * path, using the default <code>ClassLoader</code>.
	 */
	private List getRuntimeProviders() {
		List list = new ArrayList();

		Iterator providers = providers(type);
		while (providers.hasNext()) {
			try {
				//IServiceProvider p = (IServiceProvider) providers.next();
				Object o = providers.next();
				if (!(o instanceof IServiceProvider)) {
					System.err.println("WARNING: object=\"" + o + "\" is not instance of IServiceProvider");
				}
				// registry.registerServiceProvider(p, type);
				list.add(o);
			} catch (ServiceConfigurationError err) {
				if (System.getSecurityManager() != null) {
					// In the applet case, we will catch the error so
					// registration of other plugins can proceed
					err.printStackTrace();
				} else {
					// In the application case, we will throw the
					// error to indicate app/system misconfiguration
					throw err;
				}
			}
		}
		return list;
	}

}
