/*******************************************************************************
 * Copyright (c) 2007 IBM Corporation.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *    Bruce Rich (IBM Corporation) - initial API and implementation
 *******************************************************************************/ 

package org.eclipse.higgins.rp.icard.token.identity;

import java.io.Serializable;
import java.security.MessageDigest;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;
import java.util.Vector;
import org.eclipse.higgins.sts.api.IElement;
import org.w3c.dom.Element;
import org.w3c.dom.NodeList;




public class DigitalIdentity implements Serializable, IIterable {
	
	private static final boolean beTolerantOfNamespace = true;  

	private String ppid;
	//private List attrList = Collections.synchronizedList(new LinkedList());
	private Map fqnToAttrs = new HashMap(); // support multivalued entries, so will map to Vector
	
	private static final long serialVersionUID = 1L;
	private static final String PPID_SUFFIX = "privatepersonalidentifier";
	private static final String PPIDS = "http://schemas.xmlsoap.org/ws/2005/05/identity/claims/privatepersonalidentifier";
	
	public String getPpid() {
		return ppid;
	}
	
	public synchronized void addAttribute(Attribute attr) {
		String fqn = attr.getFQN();
		//this.attrList.add(attr);
		Vector v1 = (Vector)this.fqnToAttrs.get(fqn), v2 = null;
		if (v1 == null)
			v2 = new Vector();
		else
			v2 = v1;
		v2.add(attr);
		if (v2 != v1)
			this.fqnToAttrs.put(fqn, v2);
		if ((PPIDS.equals(fqn)) ||
			(beTolerantOfNamespace && (fqn.endsWith(PPID_SUFFIX)))) {
			if (this.ppid != null)
				throw new RuntimeException("multiple PPIDs not supported");
			this.ppid = attr.getValue();			
		}
	}
	
	public String toString() {
		StringBuffer sb = new StringBuffer();
		sb.append("DigitalIdentity :\n");
		Iterator iter = getIterator();
		while (iter.hasNext()) 
			sb.append("\t").append(((Attribute)(iter.next())).toString()).append("\n");
		
		return sb.toString();
	}
	
	public String[] toStrings() {
		Vector ans = new Vector();
		Iterator i = getIterator();
		while (i.hasNext())
			ans.add(i.next().toString());
		return (String[])ans.toArray(new String[]{});
	}
	
	public static DigitalIdentity fromXml(IElement input) throws Exception {
		
		DigitalIdentity di = new DigitalIdentity();
		// presume that the element is the SAML assertion
		Element domE = (Element) input.getAs(org.w3c.dom.Element.class);
		String ns = domE.getNamespaceURI();
		// ok, have uri for assertion ns
		// look for Attribute in same ns, or allow multiple namespaces?
		//ns = null;
		NodeList nl = domE.getElementsByTagNameNS(ns, "Attribute");
		// iterate through, adding all attributes to this digital identity
		for (int i=0; i<nl.getLength(); i++) {
			Attribute attr = processAttribute((Element)nl.item(i));
			di.addAttribute(attr);
		}
		if (di.ppid == null)
			throw new RuntimeException("XML element does not contain "+PPIDS);
		return di;
	}
	
	/**
	 * Create a DigitalIdentity from an email address
	 * @param emailAddress
	 * @return
	 * @throws Exception
	 */
	public static DigitalIdentity fromEmailAddress(String emailAddress) throws Exception {
		String empty = "";
		String utf8 = "utf8";
		DigitalIdentity di = new DigitalIdentity();
		if ((emailAddress == null) || (empty.equals(emailAddress)))
				throw new NullPointerException("need emailAddress");
		// gin up a PPID from a simple email address
		byte[] bytes = emailAddress.getBytes(utf8);
		MessageDigest md = MessageDigest.getInstance("SHA5");
		byte[] digBytes = md.digest(bytes);
		String thisPpid = org.apache.xml.security.utils.Base64.encode(digBytes);
		//Attribute tempAttr = new Attribute(BaseDigitalIdentity.EMAILADDRESS, BaseDigitalIdentity.BASE, emailAddress);
		//di.fqnToAttrs.put(tempAttr.getFQN(), tempAttr);	
		//di.attrList.add(tempAttr);
		//di.addAttribute(tempAttr);
		//tempAttr = new Attribute(BaseDigitalIdentity.PRIVATEPERSONALIDENTIFIER, BaseDigitalIdentity.BASE, thisPpid);
		//di.fqnToAttrs.put(tempAttr.getFQN(), tempAttr);
		//di.attrList.add(tempAttr);
		//di.addAttribute(tempAttr);
		return di;
	}

	private static Attribute processAttribute(Element attrE) {
		
		String name = attrE.getAttribute("AttributeName");
		String namespace = attrE.getAttribute("AttributeNamespace");
		String value = org.eclipse.higgins.sts.utilities.XMLHelper.getTextContent
			((Element)attrE.getElementsByTagNameNS(attrE.getNamespaceURI(), "AttributeValue").item(0));
		return new Attribute(name, namespace, value);
	}

	public Iterator getIterator() {
		//return this.fqnToAttrs.entrySet().iterator();
		//return this.attrList.iterator();
		return new DigitalIdentityIterator();
	}
	
	/**
	 * Provide a mechanism to iterate through the FQNs and values.
	 * In theory, we should check for mods while iterating through
	 * @author Bruce Rich (brich@us.ibm.com)
	 *
	 */
	class DigitalIdentityIterator implements Iterator {
		
		Iterator mapEntry = fqnToAttrs.entrySet().iterator();
		int vectorCursor = 0;
		String currentFqn = null;
		Vector currentV = null;
		
		public boolean hasNext() {
			// have to see if there are any keys and any new values in present key
			boolean ans = false;
			if ((mapEntry.hasNext()) ||
				((currentV != null) && (vectorCursor < currentV.size())))
				ans = true;
			return ans;
		}

		public Object next() {
			Object ans = null;
			while (true) {
				if (!hasNext())
					break;
				if ((currentV == null) || (vectorCursor >= currentV.size())) {
					if (!mapEntry.hasNext())
						break;
					currentV = (Vector)((Map.Entry)mapEntry.next()).getValue();
					vectorCursor = 0; // start at beginning of new vector
				}
				// see if can get attr from current
				if ((currentV != null) &&
					(vectorCursor < currentV.size())) {
					ans = currentV.get(vectorCursor++);
					break;
				}
				
				break;
			}
			if (ans == null)
				throw new java.util.NoSuchElementException();
			return ans;
		}

		public void remove() {
			throw new UnsupportedOperationException();			
		}
		
	}
	
}

