/*******************************************************************************
 * Copyright (c) 2007 IBM Corporation.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *    Bruce Rich (IBM Corporation) - initial API and implementation
 *******************************************************************************/ 

package org.eclipse.higgins.rp.servlet.server;

import java.io.IOException;
import java.util.List;
import java.util.StringTokenizer;
import java.util.Vector;

import javax.servlet.ServletContext;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpSession;

import org.eclipse.higgins.rp.util.RPClaimTypeImpl;

public class AttributeHelper {

    public static String getSingleValuedAttribute(HttpServletRequest req, String attrname)
    {
        String result = "";
        
        Object o = req.getAttribute(attrname);
        if (o == null) {
        	// defer to session, if not found in request scope
        	HttpSession session = req.getSession(false); // don't force a session creation
        	if (session != null) 
        		o = session.getAttribute(attrname);
        }
        if ( o instanceof String )
        {
            result = (String) o;
        }
        else if ( o instanceof String[] )
        {
            String[] strs = (String[]) o;
            if ( strs != null && strs.length > 0 )
            {
                result = strs[0];
            }
        }
        
        return htmlEncode(result);
    }

    public static String[] getMultiValuedAttribute(HttpServletRequest req, String attrname)
    {
        String[] result = {};
        
        Object o = req.getAttribute(attrname);
        if (o == null) {
        	// defer to session, if not found in request scope
        	HttpSession session = req.getSession(false); // don't force a session creation
        	if (session != null) 
        		o = session.getAttribute(attrname);
        }
        if ( o instanceof String )
        {
            result = new String[]{ (String) o };
        }
        else if ( o instanceof String[] )
        {
            result = (String[]) o;
        }
        
        return htmlEncode(result);
    }
    
	/**
	 * Parse out both required and optional space-separated claim names and return just the suffix of the claim names
	 * @param request the servletrequest whose context will be examined to find configured claims
	 * @return an array of claim suffixes
	 */
	public static String[] getAllSimpleStringClaims(HttpServletRequest request) {
		ServletContext sCtxt = request.getSession().getServletContext();
		String claimsS = (String) sCtxt.getInitParameter(CxtConstants.REQUIRED_CLAIMS) + " " + (String) sCtxt.getInitParameter(CxtConstants.OPTIONAL_CLAIMS);	
		return getSuffixes(claimsS, " ");
	}
	
	/**
	 * Parse out space-separated claim names and return just the suffix of the claim name
	 * @param request the servletrequest whose context will be examined to find configured claims
	 * @param optional whether to process the optional or required claims
	 * @return an array of claim suffixes
	 */
	public static String[] getSimpleStringClaims(HttpServletRequest request, boolean optional) {
		String claimsKeyS = optional ? CxtConstants.OPTIONAL_CLAIMS : CxtConstants.REQUIRED_CLAIMS;
		String claimsS = (String) request.getSession().getServletContext().getInitParameter(claimsKeyS);	
		return getSuffixes(claimsS, " ");
	}
	
	/**
	 * Parse out URLs and return just the suffix of the URL string
	 * @param urlS a string of concatenated URLs
	 * @param separatorS the chars that act as separators
	 * @param optional whether to process the optional or required claims
	 * @return an array of suffixes from individual URL strings
	 */
	public static String[] getSuffixes(String urlS, String separatorS) {
		Vector v = new Vector();
		StringTokenizer st = new StringTokenizer(urlS, separatorS);
		String thisS;
		while (st.hasMoreTokens()) {
			thisS = st.nextToken();
			thisS = thisS.substring(thisS.lastIndexOf('/')+1);
			v.add(thisS);
		}		
		return (String[])v.toArray(new String[] {});
	}
	
	/**
	 * Parse either the required claims or the optional claims and returns a string
	 * delineated by whatever marker is passed in.
	 * @param request the servletrequest whose context will be examined to find configured claims
	 * @param startMarker the tag to start an entry with
	 * @param endMarker the tag to end an entry with
	 * @param optional whether to process the optional or required claims
	 * @return a list item string of claims
	 */
	public static String getSimpleStringClaimsAsMarkup(
			HttpServletRequest request, String startMarker, String endMarker, boolean optional) {
		String claimsKeyS = optional ? CxtConstants.OPTIONAL_CLAIMS : CxtConstants.REQUIRED_CLAIMS;
		String claimsS = (String) request.getSession().getServletContext().getInitParameter(claimsKeyS);	
		StringBuffer sb = new StringBuffer(claimsS.length());
		StringTokenizer st = new StringTokenizer(claimsS, " ");
		while (st.hasMoreTokens()) {
			sb.append(startMarker).append(st.nextToken()).append(endMarker);
		}
		return sb.toString();
	}

    // -------------------------------------
    // getValues: returns values 1..length
    // -------------------------------------
    public static String[] getValues(String[][] list, int index) {
    	// the new array is going to be one entry shorter than the indicated array was, which might mean 0 entries
    	// if it were already empty, then we need to be careful to just return an empty array
    	int newLen = list[index].length;
    	if (newLen > 0)
    		newLen--; // we'll return one less than current
        String[] tmpStr = new String[newLen];
        
        for (int j=1;j<list[index].length;j++){
            tmpStr[j-1]=list[index][j];
        }
  
        return tmpStr;
    }
    
    // -------------------------------------
    // getValueString: returns values 1..length 
    //     as single string separated by commas
    // -------------------------------------
    public static String getValueString(String[][] list, int index)
    {
        String tmpStr = "";
        
        for (int j=1;j<list[index].length;j++){
        
            tmpStr+=list[index][j];
            
            if ( j < list[index].length-1 ) {
                tmpStr+=", ";
            }
        }
        
        return tmpStr;
    }

    /**
     * Html-encode to guard against CSS and/or SQL injection attacks
     * @param pText string that may contain special characters like &, <, >, "
     * @return encoded string with offending characters replaced with innocuous content 
     * like <code>&amp</code>, <code>&gt</code>, <code>&lt</code> or <code>&quot</code>.
     */
    public static String htmlEncode(String pText)
    {
        StringTokenizer tokenizer = new StringTokenizer(pText, "&<>\"", true);
        int tokenCount = tokenizer.countTokens();

        /* no encoding's needed */
        if (tokenCount == 1)
            return pText;

        /*
         * text.length + (tokenCount * 6) gives buffer large enough so no
         * addition memory would be needed and no costly copy operations would
         * occur
         */
        StringBuffer buffer = new StringBuffer(pText.length() + tokenCount * 6);
        while (tokenizer.hasMoreTokens()) {
            String token = tokenizer.nextToken();
            if (token.length() == 1) {
                switch (token.charAt(0)) {
                case '&':
                    buffer.append("&amp;");
                    break;
                case '<':
                    buffer.append("&lt;");
                    break;
                case '>':
                    buffer.append("&gt;");
                    break;
                case '"':
                    buffer.append("&quot;");
                    break;
                default:
                    buffer.append(token);
                }
            } else {
                buffer.append(token);
            }
        }
        return buffer.toString();
    }

    static String[] htmlEncode(String[] strs)
    {
        String[] result = null;

        if (strs != null) {
            result = new String[strs.length];

            for (int i = 0; i < strs.length; i++) {
                result[i] = htmlEncode(strs[i]);
            }
        }

        return result;

    }
	/**
	 * Parse out space-separated claim names and return xhtml-style add claimType entries
	 * @param request the servlet whose context will be examined to find configured claims
	 * @param response the response stream, not currently used
	 * @param optional whether to process the optional or required claims
	 * @return
	 * @throws IOException
	 */
	public static String writeXClaims(List claimList, boolean optional) throws IOException {
		StringBuffer sb = new StringBuffer();
	
		
		String claimsS = (String) RPClaimTypeImpl.claimListToString( claimList );
		StringTokenizer st = new StringTokenizer(claimsS, " ");
		while (st.hasMoreTokens()) {
			sb.append("<ic:add claimType=\""+st.nextToken()+ "\" optional=\"" + Boolean.toString(optional) + "\" />\n");
		}
		return sb.toString();
	}
}
