/*******************************************************************************
 * Copyright (c) 2007 IBM Corporation.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *    Paula Austel, Suresh Chari (IBM Corporation) - initial implementation
 *******************************************************************************/ 

package org.eclipse.higgins.rp.servlet.impl;

import java.io.IOException;
import java.io.InputStream;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.net.URISyntaxException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Enumeration;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Properties;

import javax.servlet.ServletContext;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.eclipse.higgins.rp.AuthProtocolHandler;
import org.eclipse.higgins.rp.Constants;
import org.eclipse.higgins.rp.RelyingPartyEnabler;
import org.eclipse.higgins.rp.servlet.server.CxtConstants;
import org.eclipse.higgins.rp.util.RPClaimTypeImpl;

/**
 * A relying party factory which returns a protocol handler based on the input
 * policy. Policy is simply a HashMap<String,List<String>> which can include
 * entries for the keys "TokenTypes" "RequiredClaims", "OptionalClaims", Need to
 * standardize on a set of names.
 * 
 * This factory reads in a set of configuration parameters which document the
 * available protocol handlers and the classes implementing them.
 * 
 * Given a policy it identifies the set of protocol handlers which implement the
 * various token types.
 */
public class RelyingPartyEnablerImpl implements 
		RelyingPartyEnabler
{
	public static final String HANDLER_INIT_METHOD_NAME = "init";
    public static final String HANDLER_TOKEN_NAME_METHOD_NAME = "getSupportedTokenName";

	private List _registeredHandlers = null;
	private HashMap _tokenNames = null;
	public static final String propertiesSuffix = "Properties";	
	protected static Log log =
        LogFactory.getLog(RelyingPartyEnablerImpl.class.getName());
	
	private static List formClaimsList( List strLst ){
		
		List lst = new ArrayList();
		try {
			for (int i = 0; i < strLst.size(); ++i)
				lst.add(new RPClaimTypeImpl((String)strLst.get(i)));
		} catch (URISyntaxException exc){
			exc.printStackTrace();
		}
		return lst;
	}
	
	private static Map getProtocolParamsMap( ServletContext ctxt, String name ){
		Map _mp = new HashMap();
		String propertiesFileName = ctxt.getInitParameter(name+propertiesSuffix);
		Properties properties = new Properties();
		InputStream propStream = ctxt.getResourceAsStream(propertiesFileName);
		if(null == propStream){
			log.error("Cannot open properties file: " + ctxt.getRealPath(propertiesFileName));
		} else {
			try {
				properties.load(propStream);
			} catch (IOException e) {
				e.printStackTrace();
			}
			//properties.list(System.out);
			Enumeration keys = properties.propertyNames();

			while (keys.hasMoreElements())
			{
				String key = (String) keys.nextElement();
				String keyvalue = properties.getProperty(key);
				_mp.put( key, keyvalue );
			}
		}
			
		return _mp;
	}
	
	public RelyingPartyEnablerImpl(ServletContext sCtxt ){

			String param = sCtxt.getInitParameter( CxtConstants.AUTH_PROTOCOL_HANDLERS );
			_registeredHandlers = (List)( param==null? null: Arrays.asList(param.split(" ")));
			
			_tokenNames = new HashMap();

			for (int i = 0; i < _registeredHandlers.size(); ++i) {
				try {
					Class hdlrClass = Class.forName((String)_registeredHandlers.get(i));

					Method initMethod = hdlrClass.getMethod(
							HANDLER_INIT_METHOD_NAME,
							new Class[] { java.util.Map.class });

					Method tokenNameMethod = hdlrClass.getMethod(
							HANDLER_TOKEN_NAME_METHOD_NAME, (Class[]) null);

					Object hdlrObject = hdlrClass.newInstance();

					String tokenName = (String) tokenNameMethod.invoke(
							hdlrObject, (Object[]) null);
					_tokenNames.put((String)_registeredHandlers.get(i), tokenName);
					Map mp = getProtocolParamsMap(sCtxt,
							tokenName);
					if (!mp.isEmpty())
						initMethod.invoke(hdlrObject, new Object[] { mp });

				} catch (NoSuchMethodException ex1) {
					sCtxt.log("Method could not be found in subclass ");
					ex1.printStackTrace();
					throw new RuntimeException();
				} catch (InvocationTargetException ex2) {
					ex2.printStackTrace();
					throw new RuntimeException();
				} catch (ClassNotFoundException ex3) {
					sCtxt.log(" Class could not be found ");
					ex3.printStackTrace();
					throw new RuntimeException();
				} catch (IllegalAccessException ex4) {
					sCtxt.log("Illegal Access occured ");
					ex4.printStackTrace();
					throw new RuntimeException();
				} catch (InstantiationException ex5) {
					sCtxt.log("Error occured in instantiation ");
					ex5.printStackTrace();
					throw new RuntimeException();
				}
			}
		}

	public  AuthProtocolHandler createNewSession(Map policy) {
		return findBestProtocolHandler( null, policy );
	}

	public AuthProtocolHandler createNewSession(String userAgent, Map policy) {
		return findBestProtocolHandler( userAgent, policy );
	}
	
	private AuthProtocolHandler findBestProtocolHandler(String userAgent,
			Map policy) {

		List tokenTypes = (List)policy.get(Constants.TOKENTYPES);
		List _matchingHandlers = new ArrayList();
		AuthProtocolHandler result = null;

		try {
			for (int i = 0; i < _registeredHandlers.size(); ++i)
				if (tokenTypes.contains(_tokenNames.get((String)_registeredHandlers.get(i)))) {
					AuthProtocolHandler apHdlr = (AuthProtocolHandler) Class
							.forName((String)_registeredHandlers.get(i)).newInstance();
					_matchingHandlers.add(apHdlr);

				}
		} catch (InstantiationException exc1) {
			exc1.printStackTrace();
			return null;
		} catch (ClassNotFoundException exc2) {
			exc2.printStackTrace();
			return null;
		} catch (IllegalAccessException exc3) {
			exc3.printStackTrace();
			return null;
		}

		if (_matchingHandlers.size() == 1)
			result = (AuthProtocolHandler)_matchingHandlers.get(0);
		else
			// TODO: This is the case when we wish to accept more than 1 authentication token
			// TODO: Currently we return a Null. Watch this space for composite handlers which handle multiple
			// TODO: tokens
			result = null;

		// insert the list of required claims and optional claims from the policy to the
		// protocol handler

		List tmp = (List)policy.get(Constants.REQUIREDCLAIMS);
		if (tmp != null)
			result.setRequiredClaims(formClaimsList(tmp));

		tmp = (List)policy.get(Constants.OPTIONALCLAIMS);
		if (tmp != null)
			result.setOptionalClaims(formClaimsList(tmp));

		return result;

	}
}
