/*******************************************************************************
 * Copyright (c) 2007 IBM Corporation.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *    Paula Austel (IBM Corporation) - initial implementation
 *******************************************************************************/
package org.eclipse.higgins.rp.servlet.server;

import java.io.IOException;
import java.net.HttpURLConnection;
import java.util.HashMap;
import java.util.Map;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import javax.servlet.http.HttpSession;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.eclipse.higgins.rp.AuthProtocolHandler;

/**
 * Login
 * 
 * This helper method will be called from the application after it has received
 * an authentication token. The method calls the authenticate method of the
 * protocol handler to continue processing the authentication. The
 * authentication method of the protocol handler keeps track of authentication
 * state and knows when to transition between states.
 * 
 */
public class Login {

	protected static Log log = LogFactory.getLog(Login.class.getName());

	/**
	 * Utility routine which continues authentication after POST. Fetches the
	 * protocol handler from session state, checks consistency and calls the
	 * handler
	 * 
	 * @param req
	 *            current request object
	 * @param resp
	 *            current response object
	 * @throws IOException
	 */
	public static void ContinueAuthentication(HttpServletRequest req,
			HttpServletResponse resp) throws IOException {
		// Continue Processing of the protocol handler after POST.

		String authSession = req
				.getParameter(AuthProtocolHandler.AUTH_SESSION_PARAM);
		if (authSession == null)
			throw new RuntimeException(
					" internal authentication protocol handler error ");

		Map _mp = (Map) req.getSession().getAttribute(authSession);
		if (_mp != null) {
			AuthProtocolHandler protocolHandler = (AuthProtocolHandler) _mp
					.get(CxtConstants.PROTOCOL_HANDLER_URI);
			if (protocolHandler != null) {
				Map paramsMap = (Map) req.getParameterMap();

				Map hdrsMap = Invoker.extractRequestHeaders(req);
				protocolHandler.authenticate(hdrsMap, paramsMap);
				return;
			}
		}
		// If _mp is null or the protocolHandler is null
		resp.sendError(HttpURLConnection.HTTP_INTERNAL_ERROR,
				"This page cannot be accessed directly");

	}

	/**
	 * Restarts a new Authentication session with the assumption that the
	 * protected resource is '/' Used when the login page is bookmarked etc.
	 * 
	 * @param req
	 *            current request object
	 * @param resp
	 *            current response object
	 */
	public static void newAuthSession(HttpServletRequest req,
			HttpServletResponse resp) {

		// start a new context
		String authSession = AuthNFilter.getNextAuthSession();
		// Save the request and response to the session
		Map _mp = new HashMap();
		_mp.put(CxtConstants.PROTECTED_RESOURCE, "/");
		_mp.put(CxtConstants.AUTHENTICATION_STATE_URI, CxtConstants.INIT_STATE);
		log.info("Saving Original URI to session: /");
		_mp.put(CxtConstants.REQUEST, req);
		_mp.put(CxtConstants.RESPONSE, resp);

		HttpSession httpsession = req.getSession();
		httpsession.setAttribute(authSession, _mp);
		Invoker.invoke(authSession, httpsession);
	}

	/**
	 * Utility routine which detects if the current request is a valid request
	 * i.e. it has a authSession which is valid, there is a map associated with
	 * this authSession identifier in the httpsession and the map contains a
	 * handler object. If any of these conditions is not true it returns false
	 * and restarts another authentication session with "/" as the protected
	 * resource. If all the conditions are true then it returns true and the
	 * application page can process the request as desired.
	 * 
	 * @param req
	 *            current request object
	 * @param resp
	 *            current response object
	 * @return true if there is a authSession parameter in request AND the http
	 *         session contains a map for this authSession AND that map contains
	 *         a protocol handler
	 */
	public static boolean handleInvalidRequest(HttpServletRequest req,
			HttpServletResponse resp) {

		String authSession = req
				.getParameter(AuthProtocolHandler.AUTH_SESSION_PARAM);
		if (authSession != null) {
			Map _mp = (Map) req.getSession().getAttribute(authSession);
			if (_mp != null) {
				AuthProtocolHandler protocolHandler = (AuthProtocolHandler) _mp
						.get(CxtConstants.PROTOCOL_HANDLER_URI);
				if (protocolHandler != null)
					return true;
			}
		}

		// If we reach here then we have an invalid request. Just start a new
		// authSession with "/" as the protected resource
		newAuthSession(req, resp);
		return false;
	}

}
