/*
 * Copyright (c) 2007 Parity Communications, Inc. 
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     Sergei Yakovlev - initial API and implementation
 */

package org.eclipse.higgins.rpps.webservices;

import java.rmi.Remote;
import java.rmi.RemoteException;

import org.eclipse.higgins.rpps.core.tobj.CardInformationTO;
import org.eclipse.higgins.rpps.core.tobj.CardUsageTO;
import org.eclipse.higgins.rpps.core.tobj.CategoryTO;
import org.eclipse.higgins.rpps.core.tobj.ExtraTokenResponseTO;
import org.eclipse.higgins.rpps.core.tobj.FormatDescriptorTO;
import org.eclipse.higgins.rpps.core.tobj.ICardTO;
import org.eclipse.higgins.rpps.core.tobj.ICardTemplateTO;
import org.eclipse.higgins.rpps.core.tobj.ICardsAndCategoryTO;
import org.eclipse.higgins.rpps.core.tobj.ResponseMessage;
import org.eclipse.higgins.rpps.core.tobj.TokenResponseTO;
import org.eclipse.higgins.rpps.core.tobj.UIResponseCardTO;
import org.eclipse.higgins.rpps.core.tobj.UIResponseTO;
import org.eclipse.higgins.rpps.core.tobj.UITokenServiceCredentialTO;
import org.eclipse.higgins.rpps.core.tobj.UserProfileTO;

/**
 * This is the service endpoint interface for the RPPSServiceweb service. Created 18.01.2007 10:34:54
 * 
 * @author Sergei Yakovlev
 */

public interface RPPSServiceSEI extends Remote {

	/**
	 * Web service operation creating a new card
	 * 
	 * @param userId the id to identify the user.
	 * @param password the password to identify the user.
	 * @param id the i-card id
	 * @param properties the i-card properties
	 * @return a new card cuid
	 */
	public String createICardByProperty(String userId, String password, String id, String properties) throws RemoteException;

	/**
	 * Web service operation creating a new card
	 * 
	 * @param userId the id to identify the user.
	 * @param password the password to identify the user.
	 * @param id the i-card id
	 * @param properties the i-card properties
	 * @return a new card cuid
	 */
	public String createICardByArrayProperties(String userId, String password, String id, String[] properties) throws RemoteException;

	/**
	 * Web service operation creating a new card
	 * 
	 * @param userId the id to identify the user.
	 * @param password the password to identify the user.
	 * @param templete the i-card template
	 * @return a new card cuid
	 */
	public ICardTO createICardByTemplate(String userId, String password, ICardTemplateTO template) throws RemoteException;

	/**
	 * Web service operation creating a new card for HBX
	 * 
	 * @param userId the id to identify the user.
	 * @param password the password to identify the user.
	 * @param providerID the providerID to identify the card provider.
	 * @param cardType the cardType to identify the type of card.
	 * @param description the card's description
	 * @param name the card's name
	 * @param claimarray the card's claims
	 * @return a new card cuid
	 */
	public String createICardFromHBX(String userId, String password, String cardname) throws RemoteException;

	/**
	 * 
	 * @param userId the id to identify the user.
	 * @param password the password to identify the user.
	 * @param cardName name of new card
	 * @param cuid cuid of old card
	 * @return a new card
	 * @throws RemoteException
	 */
	public ICardTO createDuplicateICard(String userId, String password, String cardName, String cuid, String pinCode) throws RemoteException;

	/**
	 * Web service operation creating a new card
	 * 
	 * @param userId the id to identify the user.
	 * @param password the password to identify the user.
	 * @return a template list
	 */
	public ICardTemplateTO[] getICardCreationTemplate(String userId, String password) throws RemoteException;

	/**
	 * Web service operation to acquire the i-card from RP site.
	 * 
	 * @param userId the id to identify the user.
	 * @param password the password to identify the user.
	 * @param subscribeURL the URL of the site from where i-card will be acquired
	 * @return a new card id
	 */
	// public String acquireICard(String userId, String password, String subscribeURL) throws RemoteException;
	/**
	 * Web service operation Locate the URIICard-type I-Card whose URI matches 'cref'. If no such card exists, create it. Find the Digital Subject whose unique identifier is 'cuid'. On the uniqueIdentifier attribute value itself: (1) if 'success' is true, add/update to "now" the the lastVerifiedFromSource metadata property's timestamp and (2) add/update the lastVerifiedAttempt metadata property's timestamp to "now".
	 * 
	 * @param userId the id to identify the user.
	 * @param password the password to identify the user.
	 * @param host the host name of the RP site where user is verified
	 * @param username the user name to login to the RP site
	 * @param success the verifying result
	 */
	public void connectionNotify(String userId, String password, String host, String username, boolean success) throws RemoteException;

	/**
	 * Gets a form filling script for the specified site&user
	 * 
	 * @param page the page url where user lands on
	 * @param username the user name
	 * @param form the HTML form name
	 * @return a form filling script
	 */
	public String getFormFillingScript(String page, String username, String form) throws RemoteException;

	/**
	 * Gets an HTML scraping script for the specified site
	 * 
	 * @param page the page url
	 * @return an HTML scraping script
	 */
	public String getHtmlScrapingScript(String page) throws RemoteException;

	/**
	 * Sets an HTML scraping script for the specified site
	 * 
	 * @param page the page url
	 * @param script the script body
	 */
	public void setHtmlScrapingScript(String page, String script) throws RemoteException;

	/**
	 * The RP site invokes this method to verify that user really has an account on the specified site.
	 * 
	 * @param userId the id to identify the user.
	 * @param password the password to identify the user.
	 * @param host The verified site url.
	 * @param username The username to login onto verified site.
	 * @return The result of verification process.
	 */
	public boolean connectionVerify(String userId, String password, String host, String username) throws RemoteException;

	/**
	 * Web service operation. This method takes as input a policy---for example, one that has been retrieved from the Relying Party Agent (called here the 'Requester')---as well as a description of the Requester. The UserInterfaceResponse object should contain all information necessary to display both (a) a user interface which shares the same card metaphor, sequences of experiences and decision points (the ceremony) of CardSpace and (b) other interface paradigms to be developed in the future
	 * 
	 * @param userId the id to identify the user.
	 * @param password the password to identify the user.
	 * @param url the URL to the html page with <object> tag
	 * @param target
	 * @param sslCert the SSL certificate
	 * @param policyType the RP Security Policy type
	 * @param policy the RP Agent's Relying Party Security Policy
	 * @return the UIResponseTO transfer object.
	 * @throws RemoteException
	 */
	public UIResponseTO getUI(String userId, String password, String url, String target, String sslCert, String policytype, String policy) throws RemoteException;

	/**
	 * Web service operation. Gets the card list of the specified user.
	 * 
	 * @param userId the id to identify the user.
	 * @param password the password to identify the user.
	 * @param cuid the card's cuid
	 * @param userCredintial
	 * @return the card list
	 * @throws java.rmi.RemoteException
	 */
	public ICardTO getICardByIdAndTokenServiceCredential(String userId, String password, String cuid, UITokenServiceCredentialTO userCredential) throws java.rmi.RemoteException;

	public UIResponseCardTO getICardClaims(String userId, String password, String cuid, String policy, String typeofCredential, String[] credentialKey, String[] credentialValue) throws java.rmi.RemoteException;

	/**
	 * Web service operation. Gets the card list of the specified user.
	 * 
	 * @param userId the id to identify the user.
	 * @param password the password to identify the user.
	 * @param cuid the card's cuid.
	 * @return the card list
	 * @throws java.rmi.RemoteException
	 */
	public ICardTO getICardById(String userId, String password, String cuid) throws java.rmi.RemoteException;

	/**
	 * Web service operation. Gets the card list of the specified user.
	 * 
	 * @param userId the id to identify the user.
	 * @param password the password to identify the user.
	 * @return the card list
	 * @throws java.rmi.RemoteException
	 */
	public ICardTO[] getICards(String userId, String password) throws java.rmi.RemoteException;

	/**
	 * Web service operation. Gets the card list and category list of the specified user.
	 * 
	 * @param userId
	 * @param password
	 * @return the card and category lists
	 * @throws RemoteException
	 */
	public ICardsAndCategoryTO getICardsAndCategoriesByUser(String userId, String password) throws RemoteException;

	/**
	 * Web service operation
	 * 
	 * @param userId the id to identify the user.
	 * @param password the password to identify the user.
	 * 
	 * 
	 */
	public ICardTO updateICard(String userId, String password, ICardTO icardTO) throws java.rmi.RemoteException;

	/**
	 * Web service operation. Imports card from a file.
	 * 
	 * @param userId the id to identify the user.
	 * @param password the password to identify the user.
	 * @param crd the card as a byte array.
	 * @return list of cards
	 * @throws java.rmi.RemoteException
	 */
	public ICardTO[] importICardAsByteArray(String userId, String password, byte[] crd) throws java.rmi.RemoteException;

	/**
	 * Web service operation. Imports card from a file.
	 * 
	 * @param userId the id to identify the user.
	 * @param password the password to identify the user.
	 * @param crd the card as a string.
	 * @return list of cards
	 * @throws java.rmi.RemoteException
	 */
	public void importICardsAsString(String userId, String password, String crd) throws java.rmi.RemoteException;

	public ICardTO[] importICards(String userId, String password, byte[] crd, String formatID, UITokenServiceCredentialTO credential) throws RemoteException;

	public FormatDescriptorTO checkImportStream(String userId, String password, byte[] crd) throws RemoteException;

	public byte[] exportICards(String userId, String password, String formatID, String[] cards, UITokenServiceCredentialTO credential) throws RemoteException;

	public ICardTO[] getICardsByFormat(String userId, String password, String formatID) throws RemoteException;

	public FormatDescriptorTO[] getOutputFormats(String userId, String password) throws RemoteException;

	/**
	 * Web service operation. Deletes specified card.
	 * 
	 * @param userId the id to identify the user.
	 * @param password the password to identify the user.
	 * @param cuid the i-card CUID
	 * @throws java.rmi.RemoteException
	 */
	public void deleteICard(String userId, String password, String cuid) throws java.rmi.RemoteException;

	/**
	 * Web service operation. Deletes all user's card.
	 * 
	 * @param userId the id to identify the user.
	 * @param password the password to identify the user.
	 * @param cuid the i-card CUID
	 * @throws java.rmi.RemoteException
	 */
	public void deleteAllICard(String userId, String password) throws java.rmi.RemoteException;

	/**
	 * Web service operation. Gets the global i-name registered to the Higgins service (e.g. "=parity").
	 * 
	 * @return the global i-name.
	 * @throws java.rmi.RemoteException
	 */
	public String getHigginsServiceGlobalIName() throws java.rmi.RemoteException;

	/**
	 * Web service operation. Registers new i-name.
	 * 
	 * @param iName the i-name we want to register.
	 * @throws java.rmi.RemoteException
	 */
	// public void registerIName(String iName) throws java.rmi.RemoteException;
	/**
	 * Web service operation. The test method that returns inputed string array.
	 * 
	 * @param strArray the string array to be returned.
	 * @return the string array inputed as a parameter.
	 * @throws java.rmi.RemoteException
	 */
	public String[] testStringArray(String[] strArray) throws java.rmi.RemoteException;

	/**
	 * Web service operation. Makes xmlToken usage claim list.
	 * 
	 * @param sslCert the SSL certificate
	 * @param claimNameArray the array which contains claim names
	 * @param claimValueArray the array which contains claim values
	 * @return the security token as a String
	 * @throws RemoteException
	 */
	public TokenResponseTO getTokenByClaims(String sslCert, String claimNameArray, String claimValueArray) throws RemoteException;

	/**
	 * Web service operation. Given the Relying Party Agent (aka 'Requester')'s policy, identity information about the requester, and the set of one or more selections that the user has just made in the ISS Web UI (see userInterfaceRequest), AND presuming that the protocol involved in interacting with the RPA requires a security token, request the token that corresponds to the user's selection(s).
	 * 
	 * @param userId the id to identify the user.
	 * @param password the password to identify the user.
	 * @param policy the RP Agent's Relying Party Security Policy
	 * @param policytype the RP Security Policy type
	 * @param sslCert the SSL certificate
	 * @param cuids the user-selected subset of the UIDs (handles) contained in the UserInterfaceResponse object returned from an earlier invocation of 'userInterfaceRequest'
	 * @param typeofCredential the type of selected credential
	 * @param credentialKey the names of fields of selected credential
	 * @param credentialValue the value of fields of selected credential
	 * @return the security token as a String
	 * @throws RemoteException
	 */
	public TokenResponseTO getTokenObject(String userId, String password, String policy, String policytype, String sslCert, String[] cuids, String typeofCredential, String[] credentialKey, String[] credentialValue) throws RemoteException;

	/**
	 * Web service operation. Makes security token usage saved users information
	 * 
	 * @param userId the id to identify the user.
	 * @param password the password to identify the user.
	 * @param policy the RP Agent's Relying Party Security Policy
	 * @param policytype the RP Security Policy type
	 * @param sslCert the SSL certificate
	 * @param url the url of site which needs token
	 * @return the security token as a String
	 * @throws RemoteException
	 */
	public TokenResponseTO getTokenByUrl(String userId, String password, String policy, String policytype, String sslCert, String url) throws RemoteException;

	/**
	 * Web service operation. Makes security token usage saved users information
	 * 
	 * @param userId the id to identify the user.
	 * @param password the password to identify the user.
	 * @param policy the RP Agent's Relying Party Security Policy
	 * @param policytype the RP Security Policy type
	 * @param sslCert the SSL certificate
	 * @param url the url of site which needs token
	 * @return the security token as a String with extra card's information
	 * @throws RemoteException
	 */
	public ExtraTokenResponseTO getTokenExtraByUrl(String userId, String password, String policy, String policytype, String sslCert, String url) throws RemoteException;

	/**
	 * Return templates for creating personal card
	 * 
	 * @param userId the id to identify the user.
	 * @param password the password to identify the user.
	 * @return a template list
	 */
	public ICardTemplateTO getPersonalICardCreationTemplate(String userId, String password) throws RemoteException;

	/**
	 * 
	 * @param handler
	 * @param userIdentifier
	 * @return
	 * @throws Exception
	 */
	public UserProfileTO getUserProfile(String userId, String password) throws RemoteException;

	/**
	 * 
	 * @param handler
	 * @param userProfile
	 * @throws Exception
	 */
	public void addUserProfileFromHBX(String userId, String password) throws RemoteException;

	/**
	 * 
	 * @param handler
	 * @param userProfile
	 * @throws Exception
	 */
	public ResponseMessage addUserProfile(String userId, String password, UserProfileTO userProfile) throws RemoteException;

	/**
	 * 
	 * @param handler
	 * @param userIdentifier
	 * @throws Exception
	 */
	public ResponseMessage deleteUserProfile(String userId, String password) throws RemoteException;

	public boolean deleteUrlCardRelation(String userId, String password, String cuid, String url) throws RemoteException;

	/**
	 * 
	 * @param handler
	 * @param userIdentifier
	 * @return
	 * @throws Exception
	 */
	public boolean userIdentifierExists(String userId) throws RemoteException;

	/**
	 * 
	 * @param handler
	 * @param userProfile
	 * @throws Exception
	 */
	public UserProfileTO modifyUserProfile(String userId, String password, UserProfileTO userProfile) throws RemoteException;

	/**
	 * 
	 * @param userId
	 * @param password
	 * @param cuid
	 * @param pinCode
	 * @return icard
	 * @throws RemoteException
	 */
	public ICardTO setPinCode(String userId, String password, String cuid, String pinCode) throws RemoteException;

	/**
	 * 
	 * @param userId
	 * @param password
	 * @param cuid
	 * @param oldPinCode
	 * @return icard
	 * @throws RemoteException
	 */
	public ICardTO resetPinCode(String userId, String password, String cuid, String oldPinCode) throws RemoteException;

	public ICardTO editPinCode(String userId, String password, String cuid, String oldPinCode, String newPinCode) throws RemoteException;

	public CategoryTO[] getCategory(String userId, String password) throws RemoteException;

	public CategoryTO addCategory(String userId, String password, CategoryTO category) throws RemoteException;

	public void deleteCategory(String userId, String password, String categoryId) throws RemoteException;

	public CategoryTO modifyCategory(String userId, String password, CategoryTO category) throws RemoteException;

	public ResponseMessage modifyCategories(String userId, String password, CategoryTO[] categories) throws RemoteException;

	public CardInformationTO getCardInformation(String userId, String password, String cuid, int size) throws RemoteException;

	public ICardTO clearCardCredential(String userId, String password, String cuid) throws RemoteException;

	public boolean setCardCredential(String userId, String password, String cuid, UITokenServiceCredentialTO tokenCredential) throws RemoteException;

	// public CardInformationTO addUrlCardRelation(String userId, String password, String cuid, String url) throws RemoteException;

	// public CardInformationTO deleteUrlCardRelation(String userId, String password, String cuid, String url) throws RemoteException;

	// public CardInformationTO deleteUrlsCardRelation(String userId, String password, String cuid, String[] urls) throws RemoteException;

	public boolean clearCardHistory(String userId, String password, String cuid) throws RemoteException;

	public CardUsageTO[] getCardHistory(String userId, String password, String cuid, int startFrom, int capacity, String orderBy, String orderDirection) throws RemoteException;

	public String[] getHBXUIList(String version, String windowsName) throws RemoteException;
}
