/*
 * Copyright (c) 2007 Parity Communications, Inc. 
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     Sergei Yakovlev - initial API and implementation
 *     Artem Verkhovets - initial API and implementation
 */

package org.eclipse.higgins.rpps.webservices;

import java.rmi.Remote;
import java.rmi.RemoteException;

import org.eclipse.higgins.rpps.core.tobj.CardInformationTO;
import org.eclipse.higgins.rpps.core.tobj.ICardResponseTO;
import org.eclipse.higgins.rpps.core.tobj.CardUsageTO;
import org.eclipse.higgins.rpps.core.tobj.CategoryTO;
import org.eclipse.higgins.rpps.core.tobj.ExtraTokenResponseTO;
import org.eclipse.higgins.rpps.core.tobj.FormatDescriptorTO;
import org.eclipse.higgins.rpps.core.tobj.ICardTO;
import org.eclipse.higgins.rpps.core.tobj.ICardTemplateTO;
import org.eclipse.higgins.rpps.core.tobj.ICardUpdateResponseTO;
import org.eclipse.higgins.rpps.core.tobj.ICardUpdateTO;
import org.eclipse.higgins.rpps.core.tobj.ICardsAndCategoryTO;
import org.eclipse.higgins.rpps.core.tobj.MapTO;
import org.eclipse.higgins.rpps.core.tobj.ResponseMessage;
import org.eclipse.higgins.rpps.core.tobj.TokenResponseTO;
import org.eclipse.higgins.rpps.core.tobj.UIResponseCardTO;
import org.eclipse.higgins.rpps.core.tobj.UIResponseTO;
import org.eclipse.higgins.rpps.core.tobj.UITokenServiceCredentialTO;
import org.eclipse.higgins.rpps.core.tobj.UserProfileTO;

/**
 * This is the service endpoint interface for the RPPSServiceweb service. Created 18.01.2007 10:34:54
 * 
 * @author Sergei Yakovlev
 * @author Artem Verkhovets
 */

public interface RPPSServiceSEI extends Remote {

	/**
	 * Web service operation. This method creating a new card by template
	 * 
	 * @param userId the id to identify the user.
	 * @param password the password to identify the user.
	 * @param templete the i-card template as ICardTemplateTO
	 * @return a new card as ICardTO
	 * @throws RemoteException
	 * @see ICardTO
	 * @see ICardTemplateTO
	 */
	public ICardTO createICardByTemplate(String userId, String password, ICardTemplateTO template) throws RemoteException;

	/**
	 * Web service operation. This method creating a new card for HBX
	 * 
	 * @param userId the id to identify the user.
	 * @param password the password to identify the user.
	 * @param name the card's name
	 * @return a new card cuid
	 * @throws RemoteException
	 */
	public String createICardFromHBX(String userId, String password, String cardname) throws RemoteException;

	/**
	 * Web service operation. This method create duplicate of personal card only 
	 * 
	 * @param userId the id to identify the user.
	 * @param password the password to identify the user.
	 * @param cardName name of new card
	 * @param cuid cuid of old card
	 * @param pinCode pinCode for retrieve claims of personal card if that is pin protected. Otherwise null 
	 * @return a new card as ICardTO
	 * @throws RemoteException
	 * @see ICardTO
	 */
	public ICardTO createDuplicateICard(String userId, String password, String cardName, String cuid, String pinCode) throws RemoteException;

	/**
	 * Web service operation. This method return list of card creation template as array of ICardTemplateTO
	 * 
	 * @param userId the id to identify the user.
	 * @param password the password to identify the user.
	 * @return a template list as array of ICardTemplateTO
	 * @throws RemoteException
	 * @see ICardTemplateTO
	 */
	public ICardTemplateTO[] getICardCreationTemplate(String userId, String password) throws RemoteException;

	/**
	 * Web service operation. This method takes as input a policy---for example, one that has been retrieved from the Relying Party Agent (called here the 'Requester')---as well as a description of the Requester. The UserInterfaceResponse object should contain all information necessary to display both (a) a user interface which shares the same card metaphor, sequences of experiences and decision points (the ceremony) of CardSpace and (b) other interface paradigms to be developed in the future
	 * 
	 * @param userId the id to identify the user.
	 * @param password the password to identify the user.
	 * @param url the URL to the page with <object> tag
	 * @param target
	 * @param sslCert the SSL certificate
	 * @param policyType the RP Security Policy type
	 * @param policy the RP Agent's Relying Party Security Policy
	 * @return the UIResponseTO transfer object.
	 * @throws RemoteException
	 * @see UIResponseTO
	 */
	public UIResponseTO getUI(String userId, String password, String url, String target, String sslCert, String policytype, String policy) throws RemoteException;

	/**
	 * Web service operation. This method gets the card with retrieved claims
	 * 
	 * @param userId the id to identify the user.
	 * @param password the password to identify the user.
	 * @param cuid the card's cuid
	 * @param userCredential filled card credential for retrieve claims
	 * @return the card as ICardTO
	 * @throws RemoteException
	 * @see ICardTO
	 */
	public ICardTO getICardByIdAndTokenServiceCredential(String userId, String password, String cuid, UITokenServiceCredentialTO userCredential) throws RemoteException;

	/**
	 * Web service operation. Created for some demo.
	 * 
	 *  
	 * @param userId the id to identify the user.
	 * @param password the password to identify the user.
	 * @param cuid the card's cuid.
	 * @param policy
	 * @param typeofCredential
	 * @param credentialKey
	 * @param credentialValue
	 * @return UIResponseCardTO
	 * @throws RemoteException
	 * @deprecated
	 */
	public UIResponseCardTO getICardClaims(String userId, String password, String cuid, String policy, String typeofCredential, String[] credentialKey, String[] credentialValue) throws RemoteException;

	/**
	 * Web service operation. This method gets the card list of the specified user.
	 * 
	 * @param userId the id to identify the user.
	 * @param password the password to identify the user.
	 * @param cuid the card's cuid.
	 * @return the card list
	 * @throws RemoteException
	 */
	public ICardTO getICardById(String userId, String password, String cuid) throws RemoteException;

	/**
	 * Web service operation. This method gets ICardResponseTO which contains the card list and card relation list of the specified user.
	 * 
	 * @param userId the id to identify the user.
	 * @param password the password to identify the user.
	 * @return ICardResponseTO
	 * @throws RemoteException
	 */
	public ICardResponseTO getICards(String userId, String password) throws RemoteException;

	/**
	 * Web service operation. This method gets cards as array of ICardTO by CUIDs array.
	 * 
	 * @param userId the id to identify the user.
	 * @param password the password to identify the user.
	 * @param cuids the card CUIDs.
	 * @return array of ICardTO.
	 * @throws RemoteException
	 */
	public ICardTO [] getICardsByCUIDs(String userId, String password,String [] cuids) throws RemoteException;
	
	/**
	 * Web service operation. This method gets the card list and category list of the specified user.
	 * 
	 * @param userId the id to identify the user.
	 * @param password the password to identify the user.
	 * @return ICardsAndCategoryTO which contain the cards and the categorys lists
	 * @throws RemoteException
	 */
	public ICardsAndCategoryTO getICardsAndCategoriesByUser(String userId, String password) throws RemoteException;

	/**
	 * Web service operation
	 * 
	 * @param userId the id to identify the user.
	 * @param password the password to identify the user.
	 * @throws RemoteException
	 */
	public ICardTO updateICard(String userId, String password, ICardTO icardTO) throws RemoteException;
	
	/**
	 * Web service operation
	 * 
	 * @param userId the id to identify the user.
	 * @param password the password to identify the user.
	 * @throws RemoteException
	 */
	public ICardUpdateResponseTO [] updateICards(String userId, String password, ICardTO [] icardTOs) throws RemoteException;

	/**
	 * Web service operation. Imports card from a file.
	 * 
	 * @param userId the id to identify the user.
	 * @param password the password to identify the user.
	 * @param crd the card as a byte array.
	 * @return list of cards
	 * @throws RemoteException
	 */
	public ICardResponseTO importICardAsByteArray(String userId, String password, byte[] crd) throws RemoteException;

	/**
	 * Web service operation. Imports card from a file.
	 * 
	 * @param userId the id to identify the user.
	 * @param password the password to identify the user.
	 * @param crd the card as a string.
	 * @return list of cards
	 * @throws RemoteException
	 */
	public void importICardsAsString(String userId, String password, String crd) throws RemoteException;
	/**
	 * Web service operation. 
	 * 
	 * @param userId the id to identify the user.
	 * @param password the password to identify the user.
	 * @param crd
	 * @param formatID
	 * @param credential
	 * @return
	 * @throws RemoteException
	 */
	public ICardResponseTO importICards(String userId, String password, byte[] crd, String formatID, UITokenServiceCredentialTO credential) throws RemoteException;
	/**
	 * Web service operation. 
	 * 
	 * @param userId the id to identify the user.
	 * @param password the password to identify the user.
	 * @param crd
	 * @return
	 * @throws RemoteException
	 */
	public FormatDescriptorTO checkImportStream(String userId, String password, byte[] crd) throws RemoteException;
	/**
	 * Web service operation. 
	 * 
	 * @param userId the id to identify the user.
	 * @param password the password to identify the user.
	 * @param formatID
	 * @param cards
	 * @param credential
	 * @return
	 * @throws RemoteException
	 */
	public byte[] exportICards(String userId, String password, String formatID, String[] cards, UITokenServiceCredentialTO credential) throws RemoteException;
	/**
	 * Web service operation.
	 * 
	 * @param userId the id to identify the user.
	 * @param password the password to identify the user.
	 * @param formatID
	 * @return
	 * @throws RemoteException
	 */
	public ICardTO[] getICardsByFormat(String userId, String password, String formatID) throws RemoteException;
	/**
	 * Web service operation. 
	 * 
	 * @param userId the id to identify the user.
	 * @param password the password to identify the user.
	 * @return
	 * @throws RemoteException
	 */
	public FormatDescriptorTO[] getOutputFormats(String userId, String password) throws RemoteException;

	/**
	 * Web service operation. Deletes specified card.
	 * 
	 * @param userId the id to identify the user.
	 * @param password the password to identify the user.
	 * @param cuid the i-card CUID
	 * @throws RemoteException
	 */
	public void deleteICard(String userId, String password, String cuid) throws RemoteException;

	/**
	 * Web service operation. Deletes all user's card.
	 * 
	 * @param userId the id to identify the user.
	 * @param password the password to identify the user.
	 * @param cuid the i-card CUID
	 * @throws RemoteException
	 */
	public void deleteAllICard(String userId, String password) throws RemoteException;

	/**
	 * Web service operation. Gets the global i-name registered to the Higgins service (e.g. "=parity").
	 * 
	 * @return the global i-name.
	 * @throws RemoteException
	 */
	public String getHigginsServiceGlobalIName() throws RemoteException;

	/**
	 * Web service operation. The test method that returns inputed string array.
	 * 
	 * @param strArray the string array to be returned.
	 * @return the string array inputed as a parameter.
	 * @throws RemoteException
	 * @deprecated
	 */
	public String[] testStringArray(String[] strArray) throws RemoteException;

	/**
	 * Web service operation. Makes xmlToken usage claim list.
	 * 
	 * @param sslCert the SSL certificate
	 * @param claimNameArray the array which contains claim names
	 * @param claimValueArray the array which contains claim values
	 * @return the security token as a String
	 * @throws RemoteException
	 */
	public TokenResponseTO getTokenByClaims(String sslCert, String claimNameArray, String claimValueArray) throws RemoteException;

	/**
	 * Web service operation. Given the Relying Party Agent (aka 'Requester')'s policy, identity information about the requester, and the set of one or more selections that the user has just made in the ISS Web UI (see userInterfaceRequest), AND presuming that the protocol involved in interacting with the RPA requires a security token, request the token that corresponds to the user's selection(s).
	 * 
	 * @param userId the id to identify the user.
	 * @param password the password to identify the user.
	 * @param policy the RP Agent's Relying Party Security Policy
	 * @param policytype the RP Security Policy type
	 * @param sslCert the SSL certificate
	 * @param cuids the user-selected subset of the UIDs (handles) contained in the UserInterfaceResponse object returned from an earlier invocation of 'userInterfaceRequest'
	 * @param typeofCredential the type of selected credential
	 * @param credentialKey the names of fields of selected credential
	 * @param credentialValue the value of fields of selected credential
	 * @return the security token as a String
	 * @throws RemoteException
	 */
	public TokenResponseTO getTokenObject(String userId, String password, String policy, String policytype, String sslCert, String[] cuids, String typeofCredential, String[] credentialKey, String[] credentialValue) throws RemoteException;

	/**
	 * Web service operation. Makes security token usage saved users information
	 * 
	 * @param userId the id to identify the user.
	 * @param password the password to identify the user.
	 * @param policy the RP Agent's Relying Party Security Policy
	 * @param policytype the RP Security Policy type
	 * @param sslCert the SSL certificate
	 * @param url the url of site which needs token
	 * @return the security token as a String
	 * @throws RemoteException
	 */
	public TokenResponseTO getTokenByUrl(String userId, String password, String policy, String policytype, String sslCert, String url) throws RemoteException;

	/**
	 * Web service operation. Makes security token usage saved users information
	 * 
	 * @param userId the id to identify the user.
	 * @param password the password to identify the user.
	 * @param policy the RP Agent's Relying Party Security Policy
	 * @param policytype the RP Security Policy type
	 * @param sslCert the SSL certificate
	 * @param url the url of site which needs token
	 * @return the security token as a String with extra card's information
	 * @throws RemoteException
	 */
	public ExtraTokenResponseTO getTokenExtraByUrl(String userId, String password, String policy, String policytype, String sslCert, String url) throws RemoteException;

	/**
	 * Web service operation.
	 * 
	 * @param userId the id to identify the user.
	 * @param password the password to identify the user.
	 * @return
	 * @throws Exception
	 */
	public UserProfileTO getUserProfile(String userId, String password) throws RemoteException;

	/**
	 * Web service operation.
	 * 
	 * @param userId the id to identify the user.
	 * @param password the password to identify the user.
	 * @throws Exception
	 * @deprecated
	 */
	public void addUserProfileFromHBX(String userId, String password) throws RemoteException;

	/**
	 * Web service operation.
	 * 
	 * @param userId the id to identify the user.
	 * @param password the password to identify the user.
	 * @return 
	 * @throws Exception
	 */
	public ResponseMessage addUserProfile(String userId, String password, UserProfileTO userProfile) throws RemoteException;

	/**
	 * Web service operation.
	 * 
	 * @param userId the id to identify the user.
	 * @param password the password to identify the user.
	 * @return
	 * @throws Exception
	 */
	public ResponseMessage deleteUserProfile(String userId, String password) throws RemoteException;
	/**
	 * Web service operation.
	 * 
	 * @param userId the id to identify the user.
	 * @param password the password to identify the user.
	 * @param cuid
	 * @param url
	 * @return
	 * @throws RemoteException
	 */
	public boolean deleteUrlCardRelation(String userId, String password, String cuid, String url) throws RemoteException;

	/**
	 * Web service operation.
	 * 
	 * @param userId the id to identify the user.
	 * @return
	 * @throws Exception
	 */
	public boolean userIdentifierExists(String userId) throws RemoteException;

	/**
	 * Web service operation.
	 * 
	 * @param userId the id to identify the user.
	 * @param password the password to identify the user.
	 * @param userProfile
	 * @throws Exception
	 */
	public UserProfileTO modifyUserProfile(String userId, String password, UserProfileTO userProfile) throws RemoteException;

	/**
	 * Web service operation.
	 * 
	 * @param userId the id to identify the user.
	 * @param password the password to identify the user.
	 * @param cuid
	 * @param pinCode
	 * @return icard
	 * @throws RemoteException
	 */
	public ICardTO setPinCode(String userId, String password, String cuid, String pinCode) throws RemoteException;

	/**
	 * Web service operation.
	 * 
	 * @param userId the id to identify the user.
	 * @param password the password to identify the user.
	 * @param cuid
	 * @param oldPinCode
	 * @return icard
	 * @throws RemoteException
	 */
	public ICardTO resetPinCode(String userId, String password, String cuid, String oldPinCode) throws RemoteException;
	/**
	 * Web service operation.
	 * 
	 * @param userId the id to identify the user.
	 * @param password the password to identify the user.
	 * @param cuid
	 * @param oldPinCode
	 * @param newPinCode
	 * @return
	 * @throws RemoteException
	 */
	public ICardTO editPinCode(String userId, String password, String cuid, String oldPinCode, String newPinCode) throws RemoteException;
	/**
	 * Web service operation.
	 * 
	 * @param userId the id to identify the user.
	 * @param password the password to identify the user.
	 * @return
	 * @throws RemoteException
	 */
	public CategoryTO[] getCategory(String userId, String password) throws RemoteException;
	/**
	 * Web service operation.
	 * 
	 * @param userId the id to identify the user.
	 * @param password the password to identify the user.
	 * @param category
	 * @return
	 * @throws RemoteException
	 */
	public CategoryTO addCategory(String userId, String password, CategoryTO category) throws RemoteException;
	/**
	 * Web service operation.
	 * 
	 * @param userId the id to identify the user.
	 * @param password the password to identify the user.
	 * @param categoryId
	 * @throws RemoteException
	 */
	public void deleteCategory(String userId, String password, String categoryId) throws RemoteException;
	/**
	 * Web service operation. 
	 * 
	 * @param userId the id to identify the user.
	 * @param password the password to identify the user.
	 * @param category
	 * @return
	 * @throws RemoteException
	 */
	public CategoryTO modifyCategory(String userId, String password, CategoryTO category) throws RemoteException;
	/**
	 * Web service operation.
	 * 
	 * @param userId the id to identify the user.
	 * @param password the password to identify the user.
	 * @param categories
	 * @return
	 * @throws RemoteException
	 */
	public ResponseMessage modifyCategories(String userId, String password, CategoryTO[] categories) throws RemoteException;
	/**
	 * Web service operation. 
	 * 
	 * @param userId the id to identify the user.
	 * @param password the password to identify the user.
	 * @param cuid
	 * @param size
	 * @return
	 * @throws RemoteException
	 */
	public CardInformationTO getCardInformation(String userId, String password, String cuid, int size) throws RemoteException;
	/**
	 * Web service operation.
	 * 
	 * @param userId the id to identify the user.
	 * @param password the password to identify the user.
	 * @param cuid
	 * @return
	 * @throws RemoteException
	 */
	public ICardTO clearCardCredential(String userId, String password, String cuid) throws RemoteException;
	/**
	 * Web service operation.
	 * 
	 * @param userId the id to identify the user.
	 * @param password the password to identify the user.
	 * @param cuid
	 * @param tokenCredential
	 * @return
	 * @throws RemoteException
	 */
	public boolean setCardCredential(String userId, String password, String cuid, UITokenServiceCredentialTO tokenCredential) throws RemoteException;
	/**
	 * Web service operation.
	 * 
	 * @param userId the id to identify the user.
	 * @param password the password to identify the user.
	 * @param cuid
	 * @return
	 * @throws RemoteException
	 */
	public boolean clearCardHistory(String userId, String password, String cuid) throws RemoteException;
	/**
	 * Web service operation.
	 * 
	 * @param userId the id to identify the user.
	 * @param password the password to identify the user.
	 * @param cuid
	 * @param url
	 * @return
	 * @throws RemoteException
	 */
	public boolean deleteCardHistoryRecord(String userId, String password,String cuid, String url) throws RemoteException;
	/**
	 * Web service operation.
	 * 
	 * @param userId the id to identify the user.
	 * @param password the password to identify the user.
	 * @param cuid
	 * @param startFrom
	 * @param capacity
	 * @param orderBy
	 * @param orderDirection
	 * @return
	 * @throws RemoteException
	 */
	public CardUsageTO[] getCardHistory(String userId, String password, String cuid, int startFrom, int capacity, String orderBy, String orderDirection) throws RemoteException;
	/**
	 * Web service operation.
	 * 
	 * @param version
	 * @param windowsName
	 * @return
	 * @throws RemoteException
	 * @deprecated
	 */
	public String[] getHBXUIList(String version, String windowsName) throws RemoteException;
	/**
	 * Web service operation.
	 * 
	 * @param userId the id to identify the user.
	 * @param password the password to identify the user.
	 * @param claimTypes
	 * @param claimValues
	 * @param cardUpdate
	 * @return
	 * @throws RemoteException
	 */
	public ICardUpdateResponseTO [] updateClaimsAcrossCards( String userId, String password,String [] claimTypes, String [] claimValues, ICardUpdateTO [] cardUpdate ) throws RemoteException;
	/**
	 * Web service operation.
	 * 
	 * @param userId the id to identify the user.
	 * @param password the password to identify the user.
	 * @return
	 * @throws RemoteException
	 */
	public MapTO [] getClaimValuesMRU(String userId, String password) throws RemoteException;	
	/**
	 * Web service operation.
	 * 
	 * @param userId the id to identify the user.
	 * @param password the password to identify the user.
	 * @param claimType
	 * @param claimValue
	 * @throws RemoteException
	 */
	public void addClaimValuesMRU(String userId, String password,String claimType, String claimValue) throws RemoteException;
	/**
	 * Web service operation.
	 * 
	 * @param userId the id to identify the user.
	 * @param password the password to identify the user.
	 * @param claimType
	 * @param claimValue
	 * @throws RemoteException
	 */
	public void removeClaimValuesMRU(String userId, String password,String claimType, String claimValue) throws RemoteException;
}
