/*
 * Copyright (c) 2006 Parity Communications, Inc. 
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     Sergei Yakovlev - initial API and implementation
 */

package org.eclipse.higgins.rpps.webservices;

import java.awt.image.BufferedImage;
import java.io.ByteArrayOutputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.LineNumberReader;
import java.util.ArrayList;
import java.util.Locale;
import java.util.StringTokenizer;

import javax.servlet.ServletContext;
import javax.servlet.http.HttpServlet;
import javax.xml.namespace.QName;
import javax.xml.rpc.soap.SOAPFaultException;

import org.apache.axis.MessageContext;
import org.apache.axis.transport.http.HTTPConstants;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.eclipse.higgins.rpps.bean.CaptchaServiceSingleton;
import org.eclipse.higgins.rpps.core.RppsException;
import org.eclipse.higgins.rpps.core.impl.RppsService;
import org.eclipse.higgins.rpps.core.tobj.CaptchaTO;
import org.eclipse.higgins.rpps.core.tobj.CardInformationTO;
import org.eclipse.higgins.rpps.core.tobj.CardUsageTO;
import org.eclipse.higgins.rpps.core.tobj.CategoryTO;
import org.eclipse.higgins.rpps.core.tobj.ExtraTokenResponseTO;
import org.eclipse.higgins.rpps.core.tobj.FormatDescriptorTO;
import org.eclipse.higgins.rpps.core.tobj.ICardResponseTO;
import org.eclipse.higgins.rpps.core.tobj.ICardTO;
import org.eclipse.higgins.rpps.core.tobj.ICardTemplateTO;
import org.eclipse.higgins.rpps.core.tobj.ICardUpdateResponseTO;
import org.eclipse.higgins.rpps.core.tobj.ICardUpdateTO;
import org.eclipse.higgins.rpps.core.tobj.ICardsAndCategoryTO;
import org.eclipse.higgins.rpps.core.tobj.MapTO;
import org.eclipse.higgins.rpps.core.tobj.ResponseMessage;
import org.eclipse.higgins.rpps.core.tobj.StsFaultException;
import org.eclipse.higgins.rpps.core.tobj.TokenResponseTO;
import org.eclipse.higgins.rpps.core.tobj.UIResponseCardTO;
import org.eclipse.higgins.rpps.core.tobj.UIResponseTO;
import org.eclipse.higgins.rpps.core.tobj.UITokenServiceCredentialTO;
import org.eclipse.higgins.rpps.core.tobj.UserProfileTO;
import org.eclipse.higgins.userprofile.entity.Captcha;

import com.sun.image.codec.jpeg.JPEGCodec;
import com.sun.image.codec.jpeg.JPEGImageEncoder;

/**
 * This is the implementation bean class for the RPPSService web service. Created 18.01.2007 10:34:54
 * 
 * @author Sergei Yakovlev
 * @author Artem Verkhovets
 */
public class RPPSServiceImpl implements RPPSServiceSEI {
	private static final Log LOG = LogFactory.getLog(RPPSServiceImpl.class);

	// private TimeProductivity timeProductivity = new TimeProductivity(true);

	/**
	 * Breaks a string into array elements separated by delimiter characters ("|") .
	 * 
	 * @param str
	 *            a string to be parsed.
	 * @return the string array.
	 */
	private String[] getStringArray(String str) {
		StringTokenizer st = new StringTokenizer(str, "|");
		String[] strs = new String[st.countTokens()];
		for (int i = 0; i < strs.length && st.hasMoreTokens(); i++) {
			strs[i] = st.nextToken();
		}
		return strs;
	}

	/**
	 * Web service operation creating a new card by card template
	 * 
	 * @param userId
	 *            the id to identify the user.
	 * @param password
	 *            the password to identify the user.
	 * @param templete
	 *            the i-card template
	 * @return a new card id
	 */
	public ICardTO createICardByTemplate(String userId, String password, ICardTemplateTO template)
			throws SOAPFaultException {
		// creates an i-card
		// if (log.isDebugEnabled()) log.debug("RPPSServiceImpl: createICardByTemplate");;
		try {
			return RppsService.getInstance(userId, password).createICard(template);
		} catch (Throwable e) {
			// log.error(e, e);
			throw new SOAPFaultException(new QName("urn:RPPSService/wsdlRPPSService", e.getClass().getName()), e
					.getMessage(), "RPPSService", null);
		} finally {
			// if (log.isDebugEnabled()) log.debug("RPPSServiceImpl: createICardByTemplate");
		}
	}

	/**
	 * Web service operation creating a new card for HBX
	 * 
	 * @param userId
	 *            the id to identify the user.
	 * @param password
	 *            the password to identify the user.
	 * @param providerID
	 *            the providerID to identify the card provider.
	 * @param cardType
	 *            the cardType to identify the type of card.
	 * @param description
	 *            the card's description
	 * @param name
	 *            the card's name
	 * @param claimarray
	 *            the card's claims
	 * @return a new card cuid
	 * @deprecated
	 */
	public String createICardFromHBX(String userId, String password, String cardname) throws SOAPFaultException {
		// if (log.isDebugEnabled()) log.debug("RPPSServiceImpl: createICardFromHBX");
		try {
			return RppsService.getInstance(userId, password).createICardFromHBX(cardname);
		} catch (Throwable e) {
			// log.error(e, e);
			throw new SOAPFaultException(new QName("urn:RPPSService/wsdlRPPSService", e.getClass().getName()), e
					.getMessage(), "RPPSService", null);
		} finally {
			// if (log.isDebugEnabled()) log.debug("RPPSServiceImpl: createICardFromHBX");
		}
	}

	public ICardTO createDuplicateICard(String userId, String password, String cardName, String cuid, String pinCode)
			throws SOAPFaultException {
		// if (log.isDebugEnabled()) log.debug("RPPSServiceImpl: createDuplicateICard");
		try {
			return RppsService.getInstance(userId, password).createDuplicateICard(cardName, cuid, pinCode);
		} catch (Throwable e) {
			// log.error(e, e);
			throw new SOAPFaultException(new QName("urn:RPPSService/wsdlRPPSService", e.getClass().getName()), e
					.getMessage(), "RPPSService", null);
		} finally {
			// if (log.isDebugEnabled()) log.debug("RPPSServiceImpl: createDuplicateICard");
		}
	}

	/**
	 * Web service operation. Deletes specified card.
	 * 
	 * 
	 * @param userId
	 *            the id to identify the user.
	 * @param password
	 *            the password to identify the user.
	 * @param cuid
	 *            the icard's CUID
	 * @return Returns ICardUpdateResponseTO which contains delete status for card. If <code>code</code> field equals
	 *         <code>0</code> then card was updated correctly. Otherwise <code>code</code> field contains error code. If
	 *         deleted card had self signed type then <code>cuid</code> field contains cuid of personal card which had
	 *         been used to signed managed card.
	 * @throws SOAPFaultException
	 */
	public ICardUpdateResponseTO deleteICard(String userId, String password, String cuid) throws SOAPFaultException {

		// if (log.isDebugEnabled()) log.debug("RPPSServiceImpl: deleteICard");
		try {
			return RppsService.getInstance(userId, password).deleteICard(cuid);
		} catch (Throwable e) {
			// log.error(e, e);
			throw new SOAPFaultException(new QName("urn:RPPSService/wsdlRPPSService", e.getClass().getName()), e
					.getMessage(), "RPPSService", null);
		} finally {
			// if (log.isDebugEnabled()) log.debug("RPPSServiceImpl: deleteICard");
		}
	}

	/**
	 * Web service operation. Deletes all user's card.
	 * 
	 * @param userId
	 *            the id to identify the user.
	 * @param password
	 *            the password to identify the user.
	 * @param cuid
	 *            the i-card CUID
	 * @throws java.rmi.SOAPFaultException
	 */
	public void deleteAllICard(String userId, String password) throws SOAPFaultException {

		// if (log.isDebugEnabled()) log.debug("RPPSServiceImpl: deleteAllICard");
		try {
			RppsService.getInstance(userId, password).deleteAllICard();
		} catch (Throwable e) {
			// log.error(e, e);
			throw new SOAPFaultException(new QName("urn:RPPSService/wsdlRPPSService", e.getClass().getName()), e
					.getMessage(), "RPPSService", null);
		} finally {
			// if (log.isDebugEnabled()) log.debug("RPPSServiceImpl: deleteAllICard");
		}
	}

	/**
	 * Web service operation. Gets the global i-name registered to the Higgins service (e.g. "=parity").
	 * 
	 * @return the global i-name.
	 * @throws SOAPFaultException
	 */
	public String getHigginsServiceGlobalIName() throws SOAPFaultException {
		// if (log.isDebugEnabled()) log.debug("RPPSServiceImpl: deleteICard");
		try {
			return RppsService.getInstance().getHigginsServiceGlobalIName();
		} catch (Throwable e) {
			// log.error(e, e);
			throw new SOAPFaultException(new QName("urn:RPPSService/wsdlRPPSService", e.getClass().getName()), e
					.getMessage(), "RPPSService", null);
		} finally {
			// if (log.isDebugEnabled()) log.debug("RPPSServiceImpl: deleteICard");
		}
	}

	/**
	 * Web service operation. Imports card from a file.
	 * 
	 * @param userId
	 *            the id to identify the user.
	 * @param password
	 *            the password to identify the user.
	 * @param crd
	 *            the card as a byte array.
	 * @return list of cards
	 * @throws SOAPFaultException
	 */

	public ICardResponseTO importICardAsByteArray(String userId, String password, byte[] crd) throws SOAPFaultException {

		// if (log.isDebugEnabled()) log.debug("RPPSServiceImpl: importICardAsByteArray");
		try {
			return RppsService.getInstance(userId, password).importICards(crd);
		} catch (Throwable e) {
			// log.error(e, e);
			throw new SOAPFaultException(new QName("urn:RPPSService/wsdlRPPSService", e.getClass().getName()), e
					.getMessage(), "RPPSService", null);
		} finally {
			// if (log.isDebugEnabled()) log.debug("RPPSServiceImpl: importICardAsByteArray");
		}
	}

	public ICardResponseTO importICards(String userId, String password, byte[] crd, String formatID,
			UITokenServiceCredentialTO credential) throws SOAPFaultException {
		// if (log.isDebugEnabled()) log.debug("RPPSServiceImpl: importICards");
		try {
			return RppsService.getInstance(userId, password).importICards(crd, formatID, credential);
		} catch (Throwable e) {
			// log.error(e, e);
			throw new SOAPFaultException(new QName("urn:RPPSService/wsdlRPPSService", e.getClass().getName()), e
					.getMessage(), "RPPSService", null);
		} finally {
			// if (log.isDebugEnabled()) log.debug("RPPSServiceImpl: importICards");
		}
	}

	public FormatDescriptorTO checkImportStream(String userId, String password, byte[] crd) throws SOAPFaultException {
		// if (log.isDebugEnabled()) log.debug("RPPSServiceImpl: importICards");
		try {
			return RppsService.getInstance(userId, password).checkImportStream(crd);
		} catch (Throwable e) {
			// log.error(e, e);
			throw new SOAPFaultException(new QName("urn:RPPSService/wsdlRPPSService", e.getClass().getName()), e
					.getMessage(), "RPPSService", null);
		} finally {
			// if (log.isDebugEnabled()) log.debug("RPPSServiceImpl: importICards");
		}
	}

	public byte[] exportICards(String userId, String password, String formatID, String[] cards,
			UITokenServiceCredentialTO credential) throws SOAPFaultException {
		// if (log.isDebugEnabled()) log.debug("RPPSServiceImpl: exportICards");
		try {
			return RppsService.getInstance(userId, password).exportICards(formatID, cards, credential);
		} catch (Throwable e) {
			// log.error(e, e);
			throw new SOAPFaultException(new QName("urn:RPPSService/wsdlRPPSService", e.getClass().getName()), e
					.getMessage(), "RPPSService", null);
		} finally {
			// if (log.isDebugEnabled()) log.debug("RPPSServiceImpl: exportICards");
		}
	}

	public ICardTO[] getICardsByFormat(String userId, String password, String formatID) throws SOAPFaultException {
		// if (log.isDebugEnabled()) log.debug("RPPSServiceImpl: getICardsByFormat");
		try {
			return RppsService.getInstance(userId, password).getICardsByFormat(formatID);
		} catch (Throwable e) {
			// log.error(e, e);
			throw new SOAPFaultException(new QName("urn:RPPSService/wsdlRPPSService", e.getClass().getName()), e
					.getMessage(), "RPPSService", null);
		} finally {
			// if (log.isDebugEnabled()) log.debug("RPPSServiceImpl: getICardsByFormat");
		}
	}

	public FormatDescriptorTO[] getOutputFormats(String userId, String password) throws SOAPFaultException {
		// if (log.isDebugEnabled()) log.debug("RPPSServiceImpl: getOutputFormats");
		try {
			return RppsService.getInstance(userId, password).getOutputFormats();
		} catch (Throwable e) {
			// log.error(e, e);
			throw new SOAPFaultException(new QName("urn:RPPSService/wsdlRPPSService", e.getClass().getName()), e
					.getMessage(), "RPPSService", null);
		} finally {
			// if (log.isDebugEnabled()) log.debug("RPPSServiceImpl: getOutputFormats");
		}
	}

	/**
	 * Web service operation. Imports card from a file.
	 * 
	 * @param userId
	 *            the id to identify the user.
	 * @param password
	 *            the password to identify the user.
	 * @param crd
	 *            the card as a string.
	 * @return list of cards
	 * @throws SOAPFaultException
	 */
	public void importICardsAsString(String userId, String password, String crd) throws SOAPFaultException {
		importICardAsByteArray(userId, password, crd.getBytes());
	}

	/**
	 * Web service operation. The test method that returns inputed string array.
	 * 
	 * @param strArray
	 *            the string array to be returned.
	 * @return the string array inputed as a parameter.
	 * @throws SOAPFaultException
	 */
	public String[] testStringArray(String[] strArray) throws SOAPFaultException {
		return strArray;
	}

	/**
	 * Web service operation. This method takes as input a policy---for example, one that has been retrieved from the
	 * Relying Party Agent (called here the 'Requester')---as well as a description of the Requester. The
	 * UserInterfaceResponse object should contain all information necessary to display both (a) a user interface which
	 * shares the same card metaphor, sequences of experiences and decision points (the ceremony) of CardSpace and (b)
	 * other interface paradigms to be developed in the future
	 * 
	 * @param userId
	 *            the id to identify the user.
	 * @param password
	 *            the password to identify the user.
	 * @param url
	 *            the URL to the html page with <object> tag
	 * @param target
	 * @param sslCert
	 *            the SSL certificate
	 * @param policyType
	 *            the RP Security Policy type
	 * @param policy
	 *            the RP Agent's Relying Party Security Policy
	 * @return the UIResponseTO transfer object.
	 * @throws SOAPFaultException
	 */
	public UIResponseTO getUI(String userId, String password, String url, String target, String sslCert,
			String policytype, String policy) throws SOAPFaultException {

		// if (log.isDebugEnabled()) log.debug("RPPSServiceImpl: getUI");
		try {
			return RppsService.getInstance(userId, password).getUserInterface(url, target, sslCert, policytype, policy);
		} catch (Throwable e) {
			// log.error(e, e);
			throw new SOAPFaultException(new QName("urn:RPPSService/wsdlRPPSService", e.getClass().getName()), e
					.getMessage(), "RPPSService", null);
		} finally {
			// if (log.isDebugEnabled()) log.debug("RPPSServiceImpl: getUI");
		}
	}

	/**
	 * Web service operation return security token by list claims.
	 * 
	 * @param sslCert
	 *            the SSL certificate
	 * @param claimNameArray
	 *            list of claim's names
	 * @param claimValueArray
	 *            list of claim's values
	 * @return the security token as a String
	 */
	public TokenResponseTO getTokenByClaims(String sslCert, String claimNameArray, String claimValueArray)
			throws SOAPFaultException {

		// if (log.isDebugEnabled()) log.debug("RPPSServiceImpl: getTokenByClaims");
		try {
			return RppsService.getInstance().getTokenByClaims(sslCert, getStringArray(claimNameArray),
					getStringArray(claimValueArray));
		} catch (Throwable e) {
			// log.error(e, e);
			throw new SOAPFaultException(new QName("urn:RPPSService/wsdlRPPSService", e.getClass().getName()), e
					.getMessage(), "RPPSService", null);
		} finally {
			// if (log.isDebugEnabled()) log.debug("RPPSServiceImpl: getTokenByClaims");
		}
	}

	/**
	 * Web service operation. Given the Relying Party Agent (aka 'Requester')'s policy, identity information about the
	 * requester, and the set of one or more selections that the user has just made in the ISS Web UI (see
	 * userInterfaceRequest), AND presuming that the protocol involved in interacting with the RPA requires a security
	 * token, request the token that corresponds to the user's selection(s).
	 * 
	 * @param userId
	 *            the id to identify the user.
	 * @param password
	 *            the password to identify the user.
	 * @param policy
	 *            the RP Agent's Relying Party Security Policy
	 * @param policytype
	 *            the RP Security Policy type
	 * @param sslCert
	 *            the SSL certificate
	 * @param cuids
	 *            the user-selected subset of the UIDs (handles) contained in the UserInterfaceResponse object returned
	 *            from an earlier invocation of 'userInterfaceRequest'
	 * @param typeofCredential
	 *            the type of selected credential
	 * @param credentialKey
	 *            the names of fields of selected credential
	 * @param credentialValue
	 *            the value of fields of selected credential
	 * @return the security token as a String
	 * @throws SOAPFaultException
	 */

	public TokenResponseTO getTokenObject(String userId, String password, String policy, String policytype,
			String sslCert, String[] cuids, String typeofCredential, String[] credentialKey, String[] credentialValue)
			throws SOAPFaultException {

		// if (log.isDebugEnabled()) log.debug("RPPSServiceImpl: getTokenByCredential start");

		try {
			return RppsService.getInstance(userId, password).getTokenObject(policy, policytype, sslCert, cuids,
					typeofCredential, credentialKey, credentialValue);
		} catch (Throwable e) {
			// log.error(e, e);
			throw new SOAPFaultException(new QName("urn:RPPSService/wsdlRPPSService", e.getClass().getName()), e
					.getMessage(), "RPPSService", null);
		} finally {
			// if (log.isDebugEnabled()) log.debug("RPPSServiceImpl: getTokenByCredential end");
		}
	}

	/**
	 * Web service operation. Makes security token usage saved users information
	 * 
	 * @param userId
	 *            the id to identify the user.
	 * @param password
	 *            the password to identify the user.
	 * @param policy
	 *            the RP Agent's Relying Party Security Policy
	 * @param policytype
	 *            the RP Security Policy type
	 * @param sslCert
	 *            the SSL certificate
	 * @param url
	 *            the url of site which needs token
	 * @return the security token as a String with extra card's information
	 * @throws SOAPFaultException
	 */
	public ExtraTokenResponseTO getTokenExtraByUrl(String userId, String password, String policy, String policytype,
			String sslCert, String url) throws SOAPFaultException {

		// if (log.isDebugEnabled()) log.debug("RPPSServiceImpl: getTokenByUrl start");
		try {
			return RppsService.getInstance(userId, password).getTokenExtraByUrl(policy, policytype, sslCert, url);
		} catch (Throwable e) {
			// log.error(e, e);
			throw new SOAPFaultException(new QName("urn:RPPSService/wsdlRPPSService", e.getClass().getName()), e
					.getMessage(), "RPPSService", null);
		} finally {
			// if (log.isDebugEnabled()) log.debug("RPPSServiceImpl: getTokenByUrl end");
		}
	}

	/**
	 * Web service operation. Makes security token usage saved users information
	 * 
	 * @param userId
	 *            the id to identify the user.
	 * @param password
	 *            the password to identify the user.
	 * @param policy
	 *            the RP Agent's Relying Party Security Policy
	 * @param policytype
	 *            the RP Security Policy type
	 * @param sslCert
	 *            the SSL certificate
	 * @param url
	 *            the url of site which needs token
	 * @return the security token as a String
	 * @throws SOAPFaultException
	 */
	public TokenResponseTO getTokenByUrl(String userId, String password, String policy, String policytype,
			String sslCert, String url) throws SOAPFaultException {

		// if (log.isDebugEnabled()) log.debug("RPPSServiceImpl: getTokenByUrl start");
		try {
			return RppsService.getInstance(userId, password).getTokenByUrl(policy, policytype, sslCert, url);
		} catch (Throwable e) {
			// log.error(e, e);
			throw new SOAPFaultException(new QName("urn:RPPSService/wsdlRPPSService", e.getClass().getName()), e
					.getMessage(), "RPPSService", null);
		} finally {
			// if (log.isDebugEnabled()) log.debug("RPPSServiceImpl: getTokenByUrl end");
		}
	}

	/**
	 * Web service operation. Update card.
	 * 
	 * @param userId
	 *            the id to identify the user.
	 * @param password
	 *            the password to identify the user.
	 * @param icardTO
	 *            icardTO object with modified claims
	 * @throws SOAPFaultException
	 */
	public ICardTO updateICard(String userId, String password, ICardTO icardTO) throws SOAPFaultException {
		// if (log.isDebugEnabled()) log.debug("RPPSServiceImpl: updateICard");
		try {
			return RppsService.getInstance(userId, password).updateICard(icardTO);
		} catch (Throwable e) {
			// log.error(e, e);
			throw new SOAPFaultException(new QName("urn:RPPSService/wsdlRPPSService", e.getClass().getName()), e
					.getMessage(), "RPPSService", null);
		} finally {
			// if (log.isDebugEnabled()) log.debug("RPPSServiceImpl: updateICard");
		}
	}

	/**
	 * Web service operation. Update list of card.
	 * 
	 * @param userId
	 *            the id to identify the user.
	 * @param password
	 *            the password to identify the user.
	 * @param icardTO
	 *            icardTO object with modified claims
	 * @throws SOAPFaultException
	 */
	public ICardUpdateResponseTO[] updateICards(String userId, String password, ICardTO[] icardTOs)
			throws SOAPFaultException {
		// if (log.isDebugEnabled()) log.debug("RPPSServiceImpl: updateICard");
		try {
			return RppsService.getInstance(userId, password).updateICards(icardTOs);
		} catch (Throwable e) {
			// log.error(e, e);
			throw new SOAPFaultException(new QName("urn:RPPSService/wsdlRPPSService", e.getClass().getName()), e
					.getMessage(), "RPPSService", null);
		} finally {
			// if (log.isDebugEnabled()) log.debug("RPPSServiceImpl: updateICard");
		}
	}

	/**
	 * Gets the card by id
	 * 
	 * @param userId
	 *            the id to identify the user.
	 * @param password
	 *            the password to identify the user.
	 * @param cuid
	 *            the i-card CUID
	 * @return the card
	 * @throws SOAPFaultException
	 */
	public ICardTO getICardById(String userId, String password, String cuid) throws SOAPFaultException {
		// if (log.isDebugEnabled()) log.debug("RPPSServiceImpl: getICardById");
		try {
			return RppsService.getInstance(userId, password).getICardByCUID(cuid);
		} catch (Throwable e) {
			// log.error(e, e);
			throw new SOAPFaultException(new QName("urn:RPPSService/wsdlRPPSService", e.getClass().getName()), e
					.getMessage(), "RPPSService", null);
		} finally {
			// if (log.isDebugEnabled()) log.debug("RPPSServiceImpl: getICardById");
		}
	}

	public ICardsAndCategoryTO getICardsAndCategoriesByUser(String userId, String password) throws SOAPFaultException {
		// if (log.isDebugEnabled()) log.debug("RPPSServiceImpl: getICardsAndCategoriesByUser");
		try {
			return RppsService.getInstance(userId, password).getICardsAndCategoriesByUser();
		} catch (Throwable e) {
			// log.error(e, e);
			throw new SOAPFaultException(new QName("urn:RPPSService/wsdlRPPSService", e.getClass().getName()), e
					.getMessage(), "RPPSService", null);
		} finally {
			// if (log.isDebugEnabled()) log.debug("RPPSServiceImpl: getICardsAndCategoriesByUser");
		}
	}

	/**
	 * Web service operation.
	 * 
	 * @param userId
	 *            the id to identify the user.
	 * @param password
	 *            the password to identify the user.
	 * @param cuid
	 *            the i-card CUID
	 * @param userCredintial
	 *            the authentication information for getting card's claim
	 * @return the card
	 * @throws SOAPFaultException
	 */
	public ICardTO getICardByIdAndTokenServiceCredential(String userId, String password, String cuid,
			UITokenServiceCredentialTO userCredintial) throws StsFaultException, SOAPFaultException {

		// if (log.isDebugEnabled()) log.debug("RPPSServiceImpl: getICardByIdWithResponseTokenService");
		try {
			return RppsService.getInstance(userId, password).getICardByCUIDAndTokenServiceCredential(cuid,
					userCredintial);

		} catch (Throwable e) {
			// log.error(e, e);
			throw new SOAPFaultException(new QName("urn:RPPSService/wsdlRPPSService", e.getClass().getName()), e
					.getMessage(), "RPPSService", null);
		} finally {
			// if (log.isDebugEnabled()) log.debug("RPPSServiceImpl: getICardByIdWithResponseTokenService");
		}
	}

	public UIResponseCardTO getICardClaims(String userId, String password, String cuid, String policy,
			String typeofCredential, String[] credentialKey, String[] credentialValue) throws SOAPFaultException {
		// if (log.isDebugEnabled()) log.debug("RPPSServiceImpl: getICardClaims");
		try {
			return RppsService.getInstance(userId, password).getICardClaims(cuid, policy, typeofCredential,
					credentialKey, credentialValue);
		} catch (Throwable e) {
			// log.error(e, e);
			throw new SOAPFaultException(new QName("urn:RPPSService/wsdlRPPSService", e.getClass().getName()), e
					.getMessage(), "RPPSService", null);
		} finally {
			// if (log.isDebugEnabled()) log.debug("RPPSServiceImpl: getICardClaims");
		}
	}

	/**
	 * Web service operation returns template list for card creating
	 * 
	 * @param userId
	 *            the id to identify the user.
	 * @param password
	 *            the password to identify the user.
	 * @return a template list
	 */
	public ICardTemplateTO[] getICardCreationTemplate(String userId, String password) throws SOAPFaultException {
		// if (log.isDebugEnabled()) log.debug("RPPSServiceImpl: getICardCreationTemplate");
		try {
			return RppsService.getInstance(userId, password).getICardCreationTemplate();
		} catch (Throwable e) {
			// log.error(e, e);
			throw new SOAPFaultException(new QName("urn:RPPSService/wsdlRPPSService", e.getClass().getName()), e
					.getMessage(), "RPPSService", null);
		} finally {
			// if (log.isDebugEnabled()) log.debug("RPPSServiceImpl: getICardCreationTemplate");
		}
	}

	/**
	 * Gets the card list of the specified user.
	 * 
	 * @param userId
	 *            the id to identify the user.
	 * @param password
	 *            the password to identify the user.
	 * @return the card list
	 * @throws SOAPFaultException
	 */
	public ICardResponseTO getICards(String userId, String password) throws SOAPFaultException {
		// if (log.isDebugEnabled()) log.debug("RPPSServiceImpl: getICards");
		try {
			return RppsService.getInstance(userId, password).getICardsByUser();
		} catch (Throwable e) {
			// log.error(e, e);
			throw new SOAPFaultException(new QName("urn:RPPSService/wsdlRPPSService", e.getClass().getName()), e
					.getMessage(), "RPPSService", null);
		} finally {
			// if (log.isDebugEnabled()) log.debug("RPPSServiceImpl: getICards");
		}
	}

	/**
	 * Gets cards by CUIDs array.
	 * 
	 * @param userId
	 *            the id to identify the user.
	 * @param password
	 *            the password to identify the user.
	 * @param cuids
	 *            the card CUIDs.
	 * @return list of cards transfer object.
	 */
	public ICardTO[] getICardsByCUIDs(String userId, String password, String[] cuids) throws SOAPFaultException {
		// if (log.isDebugEnabled()) log.debug("RPPSServiceImpl: getICards");
		try {
			return RppsService.getInstance(userId, password).getICardsByCUIDs(cuids);
		} catch (Throwable e) {
			// log.error(e, e);
			throw new SOAPFaultException(new QName("urn:RPPSService/wsdlRPPSService", e.getClass().getName()), e
					.getMessage(), "RPPSService", null);
		} finally {
			// if (log.isDebugEnabled()) log.debug("RPPSServiceImpl: getICards");
		}
	}

	public UserProfileTO getUserProfile(String userId, String password) throws SOAPFaultException {
		// if (log.isDebugEnabled()) log.debug("RPPSServiceImpl: getUserProfile");
		try {
			return RppsService.getInstance(userId, password).getUserProfile();
		} catch (Throwable e) {
			throw new SOAPFaultException(new QName("urn:RPPSService/wsdlRPPSService", e.getClass().getName()), e
					.getMessage(), "RPPSService", null);
			// log.error(e, e);
			// if (e instanceof AccessException)
			// throw new SOAPFaultException(new QName("urn:RPPSService/wsdlRPPSService", e.getClass().getName()),
			// e.getMessage(), "RPPSService", null);
			// else
			// throw new SOAPFaultException(new QName("urn:RPPSService/wsdlRPPSService", e.getClass().getName()),
			// e.getMessage(), "RPPSService", null);
		} finally {
			// if (log.isDebugEnabled()) log.debug("RPPSServiceImpl: getUserProfile");
		}
	}

	/**
	 * Web service operation. Add user profile
	 * 
	 * @param userId
	 *            the id to identify the user.
	 * @param password
	 *            the password to identify the user.
	 * @throws Exception
	 * 
	 */
	public void addUserProfileFromHBX(String userId, String password) throws SOAPFaultException {
		// if (log.isDebugEnabled()) log.debug("RPPSServiceImpl: addUserProfileFromHBX");
		try {
			UserProfileTO userProfile = new UserProfileTO();
			userProfile.setLoginName(userId);
			RppsService.getInstance().addUserProfile(userProfile, password);
		} catch (Throwable e) {
			// log.error(e, e);
			throw new SOAPFaultException(new QName("urn:RPPSService/wsdlRPPSService", e.getClass().getName()), e
					.getMessage(), "RPPSService", null);
		} finally {
			// if (log.isDebugEnabled()) log.debug("RPPSServiceImpl: addUseaddUserProfileFromHBX");
		}
	}

	/**
	 * Web service operation. Create relation between card and url
	 * 
	 * @param userId
	 *            the id to identify the user.
	 * @param password
	 *            the password to identify the user.
	 * @param cuid
	 *            the icard's CUID
	 * @param url
	 *            the web page url
	 * @return Returns <code>true</code> if relation was added. Otherwise <code>false</code>
	 * @throws SOAPFaultException
	 */
	public boolean addUrlCardRelation(String userId, String password, String cuid, String url)
			throws SOAPFaultException {
		try {
			return RppsService.getInstance(userId, password).addUrlCardRelation(userId, password);
		} catch (Throwable e) {
			throw new SOAPFaultException(new QName("urn:RPPSService/wsdlRPPSService", e.getClass().getName()), e
					.getMessage(), "RPPSService", null);
		} finally {

		}
	}

	/**
	 * Web service operation. Creates a new user profile
	 * 
	 * @param userId
	 *            the id to identify the user.
	 * @param password
	 *            the password to identify the user.
	 * @param userProfile
	 *            the userProfile profile information as UserProfileTO
	 * @return Returns <code>ResponseMessage</code> with <code>code</code> field equals <code>0</code> if profile was
	 *         added. Otherwise returns <code>ResponseMessage</code> with error code or throws an SOAPFaultException
	 * @throws SOAPFaultException
	 */
	public ResponseMessage addUserProfile(String userId, String password, UserProfileTO userProfile)
			throws SOAPFaultException {
		// if (log.isDebugEnabled()) log.debug("RPPSServiceImpl: addUserProfile");
		try {
			return RppsService.getInstance().addUserProfile(userProfile, password);
		} catch (Throwable e) {
			// log.error(e, e);
			return new ResponseMessage(ResponseMessage.unknownError, e.getMessage());
		} finally {
			// if (log.isDebugEnabled()) log.debug("RPPSServiceImpl: addUserProfile");
		}
	}

	public ResponseMessage deleteUserProfile(String userId, String password) throws SOAPFaultException {
		// if (log.isDebugEnabled()) log.debug("RPPSServiceImpl: deleteUserProfile");
		try {
			return RppsService.getInstance(userId, password).deleteUserProfile();
		} catch (Throwable e) {
			// log.error(e, e);
			return new ResponseMessage(ResponseMessage.unknownError, e.getMessage());
		} finally {
			// if (log.isDebugEnabled()) log.debug("RPPSServiceImpl: deleteUserProfile");
		}
	}

	public boolean deleteUrlsCardsRelations(String userId, String password, String cuid) throws SOAPFaultException {
		// if (log.isDebugEnabled()) log.debug("RPPSServiceImpl: deleteUrlCardRelation");
		try {
			return RppsService.getInstance(userId, password).deleteUrlsCardsRelations(cuid);
		} catch (Throwable e) {
			// log.error(e, e);
			throw new SOAPFaultException(new QName("urn:RPPSService/wsdlRPPSService", e.getClass().getName()), e
					.getMessage(), "RPPSService", null);
		} finally {
			// if (log.isDebugEnabled()) log.debug("RPPSServiceImpl: deleteUrlCardRelation");
		}
	}

	public boolean deleteUrlCardRelation(String userId, String password, String cuid, String url)
			throws SOAPFaultException {
		// if (log.isDebugEnabled()) log.debug("RPPSServiceImpl: deleteUrlCardRelation");
		try {
			return RppsService.getInstance(userId, password).deleteUrlCardRelation(cuid, url);
		} catch (Throwable e) {
			// log.error(e, e);
			throw new SOAPFaultException(new QName("urn:RPPSService/wsdlRPPSService", e.getClass().getName()), e
					.getMessage(), "RPPSService", null);
		} finally {
			// if (log.isDebugEnabled()) log.debug("RPPSServiceImpl: deleteUrlCardRelation");
		}
	}

	public boolean userIdentifierExists(String userIdentifier) throws SOAPFaultException {
		// if (log.isDebugEnabled()) log.debug("RPPSServiceImpl: checkUserIdentifier");
		try {
			return RppsService.getInstance().userIdentifierExists(userIdentifier);
		} catch (Throwable e) {
			// log.error(e, e);
			throw new SOAPFaultException(new QName("urn:RPPSService/wsdlRPPSService", e.getClass().getName()), e
					.getMessage(), "RPPSService", null);
		} finally {
			// if (log.isDebugEnabled()) log.debug("RPPSServiceImpl: checkUserIdentifier");
		}
	}

	public UserProfileTO modifyUserProfile(String userId, String password, UserProfileTO userProfile)
			throws SOAPFaultException {
		// if (log.isDebugEnabled()) log.debug("RPPSServiceImpl: modifyUserProfile");
		try {
			return RppsService.getInstance(userId, password).modifyUserProfile(userProfile);
		} catch (Throwable e) {
			// log.error(e, e);
			throw new SOAPFaultException(new QName("urn:RPPSService/wsdlRPPSService", e.getClass().getName()), e
					.getMessage(), "RPPSService", null);
		} finally {
			// if (log.isDebugEnabled()) log.debug("RPPSServiceImpl: modifyUserProfile");
		}
	}

	public ICardTO setPinCode(String userId, String password, String cuid, String pinCode) throws SOAPFaultException {
		// if (log.isDebugEnabled()) log.debug("RPPSServiceImpl: setPinCode");
		try {
			return RppsService.getInstance(userId, password).setPinCode(cuid, pinCode);
		} catch (Throwable e) {
			// log.error(e, e);
			throw new SOAPFaultException(new QName("urn:RPPSService/wsdlRPPSService", e.getClass().getName()), e
					.getMessage(), "RPPSService", null);
		} finally {
			// if (log.isDebugEnabled()) log.debug("RPPSServiceImpl: setPinCode");
		}
	}

	public ICardTO resetPinCode(String userId, String password, String cuid, String oldPinCode)
			throws SOAPFaultException {
		// if (log.isDebugEnabled()) log.debug("RPPSServiceImpl: resetPinCode");
		try {
			return RppsService.getInstance(userId, password).resetPinCode(cuid, oldPinCode);
		} catch (Throwable e) {
			// log.error(e, e);
			throw new SOAPFaultException(new QName("urn:RPPSService/wsdlRPPSService", e.getClass().getName()), e
					.getMessage(), "RPPSService", null);
		} finally {
			// if (log.isDebugEnabled()) log.debug("RPPSServiceImpl: resetPinCode");
		}
	}

	public ICardTO editPinCode(String userId, String password, String cuid, String oldPinCode, String newPinCode)
			throws SOAPFaultException {
		// if (log.isDebugEnabled()) log.debug("RPPSServiceImpl: editPinCode");
		try {
			return RppsService.getInstance(userId, password).editPinCode(cuid, oldPinCode, newPinCode);
		} catch (Throwable e) {
			// log.error(e, e);
			throw new SOAPFaultException(new QName("urn:RPPSService/wsdlRPPSService", e.getClass().getName()), e
					.getMessage(), "RPPSService", null);
		} finally {
			// if (log.isDebugEnabled()) log.debug("RPPSServiceImpl: editPinCode");
		}
	}

	public CategoryTO[] getCategory(String userId, String password) throws SOAPFaultException {
		// if (log.isDebugEnabled()) log.debug("RPPSServiceImpl: getCategory");
		try {
			return RppsService.getInstance(userId, password).getCategory();
		} catch (Throwable e) {
			// log.error(e, e);
			throw new SOAPFaultException(new QName("urn:RPPSService/wsdlRPPSService", e.getClass().getName()), e
					.getMessage(), "RPPSService", null);
		} finally {
			// if (log.isDebugEnabled()) log.debug("RPPSServiceImpl: getCategory");
		}
	}

	public CategoryTO addCategory(String userId, String password, CategoryTO category) throws SOAPFaultException {
		// if (log.isDebugEnabled()) log.debug("RPPSServiceImpl: addCategory");
		try {
			return RppsService.getInstance(userId, password).addCategory(category);
		} catch (Throwable e) {
			// log.error(e, e);
			throw new SOAPFaultException(new QName("urn:RPPSService/wsdlRPPSService", e.getClass().getName()), e
					.getMessage(), "RPPSService", null);
		} finally {
			// if (log.isDebugEnabled()) log.debug("RPPSServiceImpl: addCategory");
		}
	}

	public void deleteCategory(String userId, String password, String categoryId) throws SOAPFaultException {
		// if (log.isDebugEnabled()) log.debug("RPPSServiceImpl: deleteCategory");
		try {
			RppsService.getInstance(userId, password).deleteCategory(categoryId);
		} catch (Throwable e) {
			// log.error(e, e);
			throw new SOAPFaultException(new QName("urn:RPPSService/wsdlRPPSService", e.getClass().getName()), e
					.getMessage(), "RPPSService", null);
		} finally {
			// if (log.isDebugEnabled()) log.debug("RPPSServiceImpl: deleteCategory");
		}
	}

	public CategoryTO modifyCategory(String userId, String password, CategoryTO category) throws SOAPFaultException {
		// if (log.isDebugEnabled()) log.debug("RPPSServiceImpl: modifyCategory");
		try {
			return RppsService.getInstance(userId, password).modifyCategory(category);
		} catch (Throwable e) {
			// log.error(e, e);
			throw new SOAPFaultException(new QName("urn:RPPSService/wsdlRPPSService", e.getClass().getName()), e
					.getMessage(), "RPPSService", null);
		} finally {
			// if (log.isDebugEnabled()) log.debug("RPPSServiceImpl: modifyCategory");
		}
	}

	public ResponseMessage modifyCategories(String userId, String password, CategoryTO[] categories)
			throws SOAPFaultException {
		// if (log.isDebugEnabled()) log.debug("RPPSServiceImpl: modifyCategories");
		try {
			return RppsService.getInstance(userId, password).modifyCategories(categories);
		} catch (Throwable e) {
			// log.error(e, e);
			return new ResponseMessage(ResponseMessage.unknownError, e.getMessage());
		} finally {
			// if (log.isDebugEnabled()) log.debug("RPPSServiceImpl: modifyCategories");
		}
	}

	public CardInformationTO getCardInformation(String userId, String password, String cuid, int size)
			throws SOAPFaultException {
		// if (log.isDebugEnabled()) log.debug("RPPSServiceImpl: getCardInformation");
		try {
			return RppsService.getInstance(userId, password).getCardInformation(cuid, size);
		} catch (Throwable e) {
			// log.error(e, e);
			throw new SOAPFaultException(new QName("urn:RPPSService/wsdlRPPSService", e.getClass().getName()), e
					.getMessage(), "RPPSService", null);
		} finally {
			// if (log.isDebugEnabled()) log.debug("RPPSServiceImpl: getCardInformation");
		}
	}

	public ICardTO clearCardCredential(String userId, String password, String cuid) throws SOAPFaultException {
		// if (log.isDebugEnabled()) log.debug("RPPSServiceImpl: clearCardCredential");
		try {
			return RppsService.getInstance(userId, password).clearCardCredential(cuid);
		} catch (Throwable e) {
			// log.error(e, e);
			throw new SOAPFaultException(new QName("urn:RPPSService/wsdlRPPSService", e.getClass().getName()), e
					.getMessage(), "RPPSService", null);
		} finally {
			// if (log.isDebugEnabled()) log.debug("RPPSServiceImpl: clearCardCredential");
		}
	}

	public boolean setCardCredential(String userId, String password, String cuid,
			UITokenServiceCredentialTO tokenCredential) throws SOAPFaultException {
		// if (log.isDebugEnabled()) log.debug("RPPSServiceImpl: setCardCredential");
		try {
			return RppsService.getInstance(userId, password).setCardCredential(cuid, tokenCredential);
		} catch (Throwable e) {
			// log.error(e, e);
			throw new SOAPFaultException(new QName("urn:RPPSService/wsdlRPPSService", e.getClass().getName()), e
					.getMessage(), "RPPSService", null);
		} finally {
			// if (log.isDebugEnabled()) log.debug("RPPSServiceImpl: setCardCredential");
		}
	}

	public boolean clearCardHistory(String userId, String password, String cuid) throws SOAPFaultException {
		// if (log.isDebugEnabled()) log.debug("RPPSServiceImpl: clearCardHistory");
		try {
			return RppsService.getInstance(userId, password).clearCardHistory(cuid);
		} catch (Throwable e) {
			// log.error(e, e);
			throw new SOAPFaultException(new QName("urn:RPPSService/wsdlRPPSService", e.getClass().getName()), e
					.getMessage(), "RPPSService", null);
		} finally {
			// if (log.isDebugEnabled()) log.debug("RPPSServiceImpl: clearCardHistory");
		}
	}

	public boolean deleteCardHistoryRecord(String userId, String password, String cuid, String url)
			throws SOAPFaultException {
		// if (log.isDebugEnabled()) log.debug("RPPSServiceImpl: clearCardHistory");
		try {
			return RppsService.getInstance(userId, password).deleteCardHistoryRecord(cuid, url);
		} catch (Throwable e) {
			// log.error(e, e);
			throw new SOAPFaultException(new QName("urn:RPPSService/wsdlRPPSService", e.getClass().getName()), e
					.getMessage(), "RPPSService", null);
		} finally {
			// if (log.isDebugEnabled()) log.debug("RPPSServiceImpl: clearCardHistory");
		}
	}

	public CardUsageTO[] getCardHistory(String userId, String password, String cuid, int startFrom, int capacity,
			String orderBy, String orderDirection) throws SOAPFaultException {
		// if (log.isDebugEnabled()) log.debug("RPPSServiceImpl: getCardHistory");
		try {
			return RppsService.getInstance(userId, password).getCardHistory(cuid, startFrom, capacity, orderBy,
					orderDirection);
		} catch (Throwable e) {
			// log.error(e, e);
			throw new SOAPFaultException(new QName("urn:RPPSService/wsdlRPPSService", e.getClass().getName()), e
					.getMessage(), "RPPSService", null);
		} finally {
			// if (log.isDebugEnabled()) log.debug("RPPSServiceImpl: getCardHistory");
		}
	}

	public String[] getHBXUIList(String version, String windowsName) throws SOAPFaultException {
		// if (log.isDebugEnabled()) log.debug("RPPSServiceImpl: getHBXUIList");
		try {
			HttpServlet servlet = (HttpServlet) MessageContext.getCurrentContext().getProperty(
					HTTPConstants.MC_HTTP_SERVLET);

			ServletContext servletContext = (ServletContext) servlet.getServletContext();

			String schemaFile = version + "_" + windowsName + ".txt";

			InputStream stream = servletContext.getResourceAsStream("/WEB-INF/resource/" + schemaFile);

			if (stream == null)
				throw new FileNotFoundException("Schema file " + schemaFile + " does not find in resource!");

			LineNumberReader lnr = new LineNumberReader(new InputStreamReader(stream));
			ArrayList schemalist = new ArrayList();
			try {
				String s = "";

				while ((s = lnr.readLine()) != null) {
					if (s.trim().length() > 3)
						schemalist.add(s.trim());
				}
			} catch (IOException e) {

				// log.error(e);
				throw new RppsException("Schema file " + schemaFile + " can not be read!");
			} finally {
				lnr.close();
			}
			String[] rr = new String[schemalist.size()];
			schemalist.toArray(rr);

			return rr;
		} catch (Throwable e) {
			// log.error(e, e);
			throw new SOAPFaultException(new QName("urn:RPPSService/wsdlRPPSService", e.getClass().getName()), e
					.getMessage(), "RPPSService", null);
		} finally {
			// if (log.isDebugEnabled()) log.debug("RPPSServiceImpl: getHBXUIList");
		}
	}

	public ICardUpdateResponseTO[] updateClaimsAcrossCards(String userId, String password, String[] claimTypes,
			String[] claimValues, ICardUpdateTO[] cardUpdate) throws SOAPFaultException {
		// if (log.isDebugEnabled()) log.debug("RPPSServiceImpl: updateClaimAcrossCards");
		try {
			return RppsService.getInstance(userId, password).updateClaimsAcrossCards(claimTypes, claimValues,
					cardUpdate);
		} catch (Throwable e) {
			// log.error(e, e);
			throw new SOAPFaultException(new QName("urn:RPPSService/wsdlRPPSService", e.getClass().getName()), e
					.getMessage(), "RPPSService", null);
		} finally {
			// if (log.isDebugEnabled()) log.debug("RPPSServiceImpl: updateClaimAcrossCards");
		}
	}

	public MapTO[] getClaimValuesMRU(String userId, String password) throws SOAPFaultException {
		// if (log.isDebugEnabled()) log.debug("RPPSServiceImpl: updateClaimAcrossCards");
		try {
			return RppsService.getInstance(userId, password).getClaimValuesMRU();
		} catch (Throwable e) {
			// log.error(e, e);
			throw new SOAPFaultException(new QName("urn:RPPSService/wsdlRPPSService", e.getClass().getName()), e
					.getMessage(), "RPPSService", null);
		} finally {
			// if (log.isDebugEnabled()) log.debug("RPPSServiceImpl: updateClaimAcrossCards");
		}
	}

	public void addClaimValuesMRU(String userId, String password, String claimType, String claimValue)
			throws SOAPFaultException {
		// if (log.isDebugEnabled()) log.debug("RPPSServiceImpl: addClaimValuesMRU");
		try {
			RppsService.getInstance(userId, password).addClaimValuesMRU(claimType, claimValue);
		} catch (Throwable e) {
			// log.error(e, e);
			throw new SOAPFaultException(new QName("urn:RPPSService/wsdlRPPSService", e.getClass().getName()), e
					.getMessage(), "RPPSService", null);
		} finally {
			// if (log.isDebugEnabled()) log.debug("RPPSServiceImpl: addClaimValuesMRU");
		}
	}

	public void removeClaimValuesMRU(String userId, String password, String claimType, String claimValue)
			throws SOAPFaultException {
		// if (log.isDebugEnabled()) log.debug("RPPSServiceImpl: removeClaimValuesMRU");
		try {
			RppsService.getInstance(userId, password).removeClaimValuesMRU(claimType, claimValue);
		} catch (Throwable e) {
			// log.error(e, e);
			throw new SOAPFaultException(new QName("urn:RPPSService/wsdlRPPSService", e.getClass().getName()), e
					.getMessage(), "RPPSService", null);
		} finally {
			// if (log.isDebugEnabled()) log.debug("RPPSServiceImpl: removeClaimValuesMRU");
		}
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.higgins.rpps.webservices.RPPSServiceSEI#getDefautPCard(java.lang.String, java.lang.String)
	 */
	public ICardTO getDefaultPCard(String userId, String password) throws SOAPFaultException {
		try {
			return RppsService.getInstance(userId, password).getDefaultPCard();

		} catch (Exception e) {
			throw new SOAPFaultException(new QName("urn:RPPSService/wsdlRPPSService", e.getClass().getName()), e
					.getMessage(), "RPPSService", null);
		}
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.higgins.rpps.webservices.RPPSServiceSEI#setDefautPCard(java.lang.String, java.lang.String,
	 * java.lang.String)
	 */
	public void setDefaultPCard(String userId, String password, String pcardId) throws SOAPFaultException {
		try {
			RppsService.getInstance(userId, password).setDefaultPCard(pcardId);
		} catch (Exception e) {
			throw new SOAPFaultException(new QName("urn:RPPSService/wsdlRPPSService", e.getClass().getName()), e
					.getMessage(), "RPPSService", null);
		}
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.higgins.rpps.webservices.RPPSServiceSEI#generatePasswordResetCode(java.lang.String)
	 */
	public void generatePasswordResetCode(final String userIdentifier) throws SOAPFaultException {
		try {
			new RppsService().generatePasswordResetCode(userIdentifier);
		} catch (Exception e) {
			throw new SOAPFaultException(new QName("urn:RPPSService/wsdlRPPSService", e.getClass().getName()), e
					.getMessage(), "RPPSService", null);
		}
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.higgins.rpps.webservices.RPPSServiceSEI#modifyPasswordWithPasswordResetCode(java.lang.String,
	 * java.lang.String, java.lang.String)
	 */
	public void modifyPasswordWithPasswordResetCode(final String userIdentifier, final String passwordResetCode,
			final String newPassword) throws SOAPFaultException {
		try {
			new RppsService().modifyPasswordWithPasswordResetCode(userIdentifier, passwordResetCode, newPassword);
		} catch (Exception e) {
			throw new SOAPFaultException(new QName("urn:RPPSService/wsdlRPPSService", e.getClass().getName()), e
					.getMessage(), "RPPSService", null);
		}
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.higgins.rpps.webservices.RPPSServiceSEI#modifyPasswordWithOldPassword(java.lang.String,
	 * java.lang.String, java.lang.String)
	 */
	public void modifyPasswordWithOldPassword(final String userIdentifier, final String oldPassword,
			final String newPassword) throws SOAPFaultException {
		try {
			RppsService.getInstance(userIdentifier, oldPassword).modifyPasswordWithOldPassword(userIdentifier,
					oldPassword, newPassword);
		} catch (Exception e) {
			throw new SOAPFaultException(new QName("urn:RPPSService/wsdlRPPSService", e.getClass().getName()), e
					.getMessage(), "RPPSService", null);
		}

	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.higgins.rpps.webservices.RPPSServiceSEI#generatePasswordResetCodeWithCaptcha(java.lang.String,
	 * java.lang.String, java.lang.String)
	 */
	public void generatePasswordResetCodeWithCaptcha(final String userIdentifier, final String captchaId,
			final String captchaResponse) throws SOAPFaultException {
		try {
			RppsService rppsService = new RppsService();
			validateCaptcha(captchaId, captchaResponse, rppsService);
			// generate password reset code
			rppsService.generatePasswordResetCode(userIdentifier);
		} catch (Exception e) {
			LOG.error(e,e);
			throw new SOAPFaultException(new QName("urn:RPPSService/wsdlRPPSService", e.getClass().getName()), e
					.getMessage(), "RPPSService", null);
		}

	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.higgins.rpps.webservices.RPPSServiceSEI#getCaptcha()
	 */
	public CaptchaTO getCaptcha() throws SOAPFaultException {
		CaptchaTO captchaTO;
		try {
			captchaTO = new CaptchaTO();
			Captcha captcha = new RppsService().createCaptcha();
			captchaTO.setId(captcha.getId());

			// build captcha image
			BufferedImage challenge = CaptchaServiceSingleton.getInstance().getImageChallengeForID(captcha.getKey(),
					Locale.US);
			ByteArrayOutputStream jpegOutputStream = new ByteArrayOutputStream();
			JPEGImageEncoder jpegEncoder = JPEGCodec.createJPEGEncoder(jpegOutputStream);
			jpegEncoder.encode(challenge);
			captchaTO.setImage(jpegOutputStream.toByteArray());

		} catch (Exception e) {
			throw new SOAPFaultException(new QName("urn:RPPSService/wsdlRPPSService", e.getClass().getName()), e
					.getMessage(), "RPPSService", null);
		}

		return captchaTO;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see
	 * org.eclipse.higgins.rpps.webservices.RPPSServiceSEI#modifyPasswordWithPasswordResetCodeAndCaptcha(java.lang.String
	 * , java.lang.String, java.lang.String, java.lang.String, java.lang.String)
	 */
	public void modifyPasswordWithPasswordResetCodeAndCaptcha(final String userIdentifier,
			final String passwordResetCode, final String newPassword, final String captchaId,
			final String captchaResponse) throws SOAPFaultException {
		try {
			validateCaptcha(captchaId, captchaResponse, new RppsService());
			modifyPasswordWithPasswordResetCode(userIdentifier, passwordResetCode, newPassword);
		} catch (Exception e) {
			throw new SOAPFaultException(new QName("urn:RPPSService/wsdlRPPSService", e.getClass().getName()), e
					.getMessage(), "RPPSService", null);
		}

	}

	/**
	 *Validate captcha response.
	 * 
	 * @param captchaId
	 * @param captchaResponse
	 * @param rppsService
	 */
	private void validateCaptcha(final String captchaId, final String captchaResponse, RppsService rppsService) {
		Captcha captcha = rppsService.getCaptcha(captchaId);
		
		//captcha must be validated just one time
		rppsService.deleteCaptcha(captchaId);
		
		//validate captcha  
		if (!CaptchaServiceSingleton.getInstance().validateResponseForID(captcha.getKey(), captchaResponse)
						.booleanValue()) {
			LOG.warn("Invalid Captcha security code");
			throw new SOAPFaultException(new QName("urn:RPPSService/wsdlRPPSService", this.getClass().getName()),
					"Sorry, you have provided an invalid security code. You might be a bot. "
							+ "This program is only open to real people.", "RPPSService", null);
		}
	}

}
