/**
 * Copyright (c) 2007 Parity Communications, Inc. 
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     Sergei Yakovlev - initial API and implementation
 */

package org.eclipse.higgins.rpps.core.impl;

import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.net.URI;
import java.net.URL;
import java.net.URLConnection;
import java.util.List;
import java.util.Properties;

import org.eclipse.higgins.rsse.RssFeed;
import org.eclipse.higgins.rsse.RssItem;
import org.eclipse.higgins.rsse.parser.FeedParser;
import org.eclipse.higgins.rsse.parser.ParseException;
import org.eclipse.higgins.rsse.util.RssVersion;

/**
 * This class provides helper methods for RSS feed loading. 
 * @author Sergei Yakovlev
 *
 */
public class RssService {
	
	/**
	 * Creates an RssService object.
	 * @return the RssService object.
	 */
	public static RssService getInstance() {
		return new RssService();
	}
	
	/**
	 * Loads RSS feed from the specified URL then merge it with local file ans save the result locally.
	 * @param feedUrl the remote URL from where RSS feed is loaded.
	 * @param rssDir the local path to where the feed is saved.
	 * @return the merged RssFeed object.
	 * @throws IOException
	 * @throws ParseException
	 */
	public RssFeed loadFeed(URL feedUrl, String rssDir) throws IOException, ParseException {
		RssFeed remoteFeed = null;
		InputStream ins = null;
		try {
			URLConnection connection = feedUrl.openConnection();
			connection.setUseCaches(false);
			ins = connection.getInputStream();
			// gets remote feed (subscribed)
			remoteFeed = FeedParser.parse(ins);
		} finally {
			if(ins != null) {
				ins.close();
			}
		}
        // gets the file name of remote feed.
        String feedFileName = feedUrl.getFile();
        if(feedFileName != null && feedFileName.length() > 0) {
        	// gets file name
            feedFileName = feedFileName.substring(feedFileName.lastIndexOf('/'));
            File feedFile = new File(rssDir + feedFileName);
            // gets local feed (published)
            RssFeed localFeed = RssFeed.load(feedFile);
            if(localFeed == null) {
                // create parent dirs if it's needed
                if(feedFile.getParent() != null) {
                    new File(feedFile.getParent()).mkdirs();
                }
                // create a new file
                feedFile.createNewFile();
                localFeed = new RssFeed(remoteFeed.getTitle(), remoteFeed.getLink(), remoteFeed.getDescription());
                localFeed.setVersion(RssVersion.RSS_2_0);
            }
            // merges remote feed with local one
            localFeed.merge(remoteFeed);
            // saves feed
            localFeed.save(feedFile);
            return localFeed;
        } else {
        	return null;
        }
	}
	
	/**
	 * Gets the properties from the loaded RSS feed.
	 * @param feedUrl the remote URL from where RSS feed is loaded.
	 * @param rssDir the local path to where the feed is saved.
	 * @return the property list from the loaded/merged RSS feed.
	 * @throws IOException
	 */
	public FeedProperties loadFeedProperties(URL feedUrl, String rssDir) throws IOException {
		FeedProperties feedProperties = null;
		try {
			RssFeed feed = loadFeed(feedUrl, rssDir);
			if(feed != null) {
				Properties props = new Properties();
				List items = feed.getItems();
				// populates properties
				for(int i = 0; i < items.size(); i++) {
					RssItem item = (RssItem) items.get(i);
					props.setProperty(item.getName(), item.getValue() != null ? item.getValue() : "");
				}
				feedProperties = new FeedProperties(URI.create(feed.getTitle()), props);
			}
		} catch (ParseException e) {
			throw new IOException(e.getMessage());
		}
		return feedProperties;
	}
}
