/**
 * Copyright (c) 2007 Parity Communications, Inc. 
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     Sergei Yakovlev - initial API and implementation
 *     Artem Verkhovets - initial API and implementation
 */

package org.eclipse.higgins.rpps.core.impl;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.InputStream;
import java.io.UnsupportedEncodingException;
import java.net.URI;
import java.rmi.AccessException;
import java.rmi.RemoteException;
import java.security.AccessController;
import java.security.InvalidAlgorithmParameterException;
import java.security.KeyStore;
import java.security.KeyStoreException;
import java.security.PrivilegedAction;
import java.security.Security;
import java.security.cert.CertStoreException;
import java.security.cert.CertificateException;
import java.security.cert.X509Certificate;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashSet;
import java.util.Hashtable;
import java.util.Iterator;
import java.util.List;
import java.util.ListIterator;
import java.util.Map;
import java.util.Properties;
import java.util.Set;

import javax.net.ssl.ManagerFactoryParameters;
import javax.net.ssl.TrustManager;
import javax.net.ssl.TrustManagerFactorySpi;
import javax.net.ssl.X509TrustManager;
import javax.security.auth.callback.CallbackHandler;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.xml.security.utils.Base64;
import org.eclipse.higgins.cache.CacheProviderFactory;
import org.eclipse.higgins.icard.CUID;
import org.eclipse.higgins.icard.CardException;
import org.eclipse.higgins.icard.ICard;
import org.eclipse.higgins.icard.ICardProvider;
import org.eclipse.higgins.icard.ICardTemplate;
import org.eclipse.higgins.icard.IClaim;
import org.eclipse.higgins.icard.IComplexClaim;
import org.eclipse.higgins.icard.IPersonalInformationCard;
import org.eclipse.higgins.icard.IPolicy;
import org.eclipse.higgins.icard.ISimpleClaim;
import org.eclipse.higgins.icard.ISimpleClaimType;
import org.eclipse.higgins.icard.ITemplateContainer;
import org.eclipse.higgins.icard.ITemplateElement;
import org.eclipse.higgins.icard.ITemplateValue;
import org.eclipse.higgins.icard.InvalidTypeException;
import org.eclipse.higgins.icard.TUID;
import org.eclipse.higgins.icard.auth.ICredential;
import org.eclipse.higgins.icard.auth.ICredentialDescriptor;
import org.eclipse.higgins.icard.auth.IPasswordCredential;
import org.eclipse.higgins.icard.auth.IPinCodeCredential;
import org.eclipse.higgins.icard.auth.ITSKerberosV5Credential;
import org.eclipse.higgins.icard.auth.ITSSelfIssuedCredential;
import org.eclipse.higgins.icard.auth.ITSUsernamePasswordCredential;
import org.eclipse.higgins.icard.auth.ITSX509V3Credential;
import org.eclipse.higgins.icard.auth.ITokenServiceCredential;
import org.eclipse.higgins.icard.auth.IUsernamePasswordCredential;
import org.eclipse.higgins.icard.common.auth.PinCodeCredential;
import org.eclipse.higgins.icard.common.auth.callback.PinCodeCallback;
import org.eclipse.higgins.icard.io.IFormatDescriptor;
import org.eclipse.higgins.icard.policy.ICardSpacePolicy;
import org.eclipse.higgins.icard.provider.cardspace.common.CardSpacePolicy;
import org.eclipse.higgins.icard.provider.cardspace.common.PPIDCardSpacePolicy;
import org.eclipse.higgins.icard.registry.ICardRegistry;
import org.eclipse.higgins.icard.userprofile.CardCredential;
import org.eclipse.higgins.icard.userprofile.CardInformation;
import org.eclipse.higgins.icard.userprofile.CardUsage;
import org.eclipse.higgins.icard.userprofile.Category;
import org.eclipse.higgins.icard.userprofile.ICardUsageManager;
import org.eclipse.higgins.icard.userprofile.PolicyVersion;
import org.eclipse.higgins.icard.userprofile.UserProfile;
import org.eclipse.higgins.icard.userprofile.UserProfileAuthenticationException;
import org.eclipse.higgins.icard.userprofile.UserProfileException;
import org.eclipse.higgins.iss.CredentialContainer;
import org.eclipse.higgins.iss.ICardSelectorService;
import org.eclipse.higgins.iss.IICardSelector;
import org.eclipse.higgins.iss.IIdentityToken;
import org.eclipse.higgins.iss.SelectionANDofORs;
import org.eclipse.higgins.iss.SelectionANDofORsElm;
import org.eclipse.higgins.iss.UCTelm;
import org.eclipse.higgins.iss.UserChoiceTree;
import org.eclipse.higgins.iss.UserChoiceTree_ANDofORs;
import org.eclipse.higgins.iss.cardspace.IdentityToken;
import org.eclipse.higgins.iss.cardspace.util.UserProfileTokenHelper;
import org.eclipse.higgins.keystore.IKeyStoreService;
import org.eclipse.higgins.keystore.registry.KeyStoreRegistry;
import org.eclipse.higgins.rpps.core.IRppsService;
import org.eclipse.higgins.rpps.core.RppsException;
import org.eclipse.higgins.rpps.core.tobj.CardInformationTO;
import org.eclipse.higgins.rpps.core.tobj.CardUsageTO;
import org.eclipse.higgins.rpps.core.tobj.CategoryTO;
import org.eclipse.higgins.rpps.core.tobj.ExtraTokenResponseTO;
import org.eclipse.higgins.rpps.core.tobj.FormatDescriptorTO;
import org.eclipse.higgins.rpps.core.tobj.ICardResponseTO;
import org.eclipse.higgins.rpps.core.tobj.ICardTO;
import org.eclipse.higgins.rpps.core.tobj.ICardTemplateTO;
import org.eclipse.higgins.rpps.core.tobj.ICardUpdateResponseTO;
import org.eclipse.higgins.rpps.core.tobj.ICardUpdateTO;
import org.eclipse.higgins.rpps.core.tobj.ICardsAndCategoryTO;
import org.eclipse.higgins.rpps.core.tobj.MapTO;
import org.eclipse.higgins.rpps.core.tobj.PropertyTO;
import org.eclipse.higgins.rpps.core.tobj.RPPolicy;
import org.eclipse.higgins.rpps.core.tobj.ResponseMessage;
import org.eclipse.higgins.rpps.core.tobj.TokenResponseTO;
import org.eclipse.higgins.rpps.core.tobj.UIResponseAndTO;
import org.eclipse.higgins.rpps.core.tobj.UIResponseCardTO;
import org.eclipse.higgins.rpps.core.tobj.UIResponseClauseTO;
import org.eclipse.higgins.rpps.core.tobj.UIResponseOrTO;
import org.eclipse.higgins.rpps.core.tobj.UIResponseRPTO;
import org.eclipse.higgins.rpps.core.tobj.UIResponseTO;
import org.eclipse.higgins.rpps.core.tobj.UIResponseUsableCardsTO;
import org.eclipse.higgins.rpps.core.tobj.UITokenServiceCredentialTO;
import org.eclipse.higgins.rpps.core.tobj.UserProfileTO;
import org.eclipse.higgins.sts.api.ISTSResponse;

/**
 * The object used for executing Higgins API and returning the results it produces.
 * 
 * @author Sergei Yakovlev
 * @author Artem Verkhovets
 */
public class RppsService implements IRppsService {

	final static class DummySecurityProvider extends java.security.Provider {

		private static final long serialVersionUID = 594145003249139851L;

		public DummySecurityProvider() {
			super("Dummy", 1.0D, "");
			AccessController.doPrivileged(new PrivilegedAction() {
				public Object run() {
					put("TrustManagerFactory." + DummyTrustManagerFactory.getAlgorithm(), DummyTrustManagerFactory.class.getName());
					return null;
				}
			});
		}
	}

	public final static class DummyTrustManagerFactory extends TrustManagerFactorySpi {
		public static String getAlgorithm() {
			return "XTrust509";
		}

		public DummyTrustManagerFactory() {
		}

		protected TrustManager[] engineGetTrustManagers() {
			return new TrustManager[] { new X509TrustManager() {
				public void checkClientTrusted(final java.security.cert.X509Certificate[] certs, final String authType) {
				}

				public void checkServerTrusted(final java.security.cert.X509Certificate[] certs, final String authType) {
					for (int i = 0; i < certs.length; i++) {
						System.out.println("  >>> Checking certificate " + certs[i]);
					}
				}

				public java.security.cert.X509Certificate[] getAcceptedIssuers() {
					return null;
				}
			} };
		}

		protected void engineInit(final KeyStore keystore) throws KeyStoreException {
		}

		protected void engineInit(final ManagerFactoryParameters arg0) throws InvalidAlgorithmParameterException {
		}
	}

	private static Log log = LogFactory.getLog(RppsService.class);

	static {
		AllowSelfSignedSSL();
	}

	public static void AllowSelfSignedSSL() {
		if (Security.getProvider("Dummy") == null) {
			Security.insertProviderAt(new DummySecurityProvider(), 2);
			Security.setProperty("ssl.TrustManagerFactory.algorithm", DummyTrustManagerFactory.getAlgorithm());
		}
	}

	public static synchronized void ConfigureCache(final String configurationBase, final String configurationFile) {
		CacheProviderFactory.setConfigurationBase(configurationBase);
		CacheProviderFactory.setFileName(configurationFile);
		CacheProviderFactory.getCacheProvider();

	}

	/**
	 * Creates an RppsService object.
	 * 
	 * @return the RppsService object
	 */
	public static IRppsService getInstance() {
		return new RppsService();
	}

	/**
	 * Creates an RppsService object.
	 * 
	 * @return the RppsService object
	 */
	public static IRppsService getInstance(final String userId, final String password) throws AccessException {
		return new RppsService(userId, password);
	}

	/**
	 * @param privateSelectorINumber
	 * @return userIdentifer
	 */
	public static String resolveUserIdentifier(final String privateSelectorINumber) {
		try {
			return ICardRegistry.getInstance().getUserProfileService().resolveUserIdentifier(privateSelectorINumber);
		} catch (final UserProfileException e) {
			log.error(e, e);
			return "";
		}
	}

	protected UserProfile userProfile = null;

	private CallbackHandler handler = null;

	/**
	 * Gets URI card by context ref and CUID
	 * 
	 * @param handler the callback handler to be used to request user's identity
	 * @param cref the Context Ref as a String
	 * @param cuid the CUID
	 * @return the i-card object
	 */
	/*	private ICard getURIICard(String cref, String cuid) {
			// gets all registered providers
			for (Iterator pit = ICardRegistry.getInstance().getICardProviders(); pit.hasNext();) {
				ICardProvider provider = (ICardProvider) pit.next();
				try {
					for (Iterator icards = provider.getICards(getHandler()); icards.hasNext();) {
						ICard icard = (ICard) icards.next();
						if (icard instanceof IURICard && ((IURICard) icard).getContextRef().equals(cref) && ((IURICard) icard).getCuid().equals(cuid)) {
							return icard;
						}
					}
				} catch (CardException e) {
					e.printStackTrace();
				}
			}
			return null;
		}*/

	/**
	 * Acquires the i-card from RP site.
	 * 
	 * @param userId the id to identify the user.
	 * @param password the password to identify the user.
	 * @param subscribeURL the URL of the site from where i-card will be acquired
	 * @return a new card id
	 */
	/*	public String acquireICard(String subscribeURL) throws MalformedURLException {
			FeedProperties feedProperties = null;
			URL subscribeURL_ = new URL(subscribeURL);
			String rssDir = RssServlet.getRssDir();
			CallbackHandler handler = getHandler();
			try {
				NameCallback name = new NameCallback("name");
				handler.handle(new Callback[] { name });
				feedProperties = RssService.getInstance().loadFeedProperties(subscribeURL_, rssDir + "/" + name.getName());
			} catch (IOException e) {
				throw new RppsException(e);
			} catch (UnsupportedCallbackException e) {
				throw new RppsException(e);
			}
			return feedProperties != null ? createICard(feedProperties.getId(), feedProperties.getProperties()).getCUID().toString() : null;
		}*/

	/**
	 * Locate the URIICard-type I-Card whose URI matches 'cref'. If no such card exists, create it. Find the Node whose unique identifier is 'cuid'. On the uniqueIdentifier attribute value itself: (1) if 'success' is true, add/update to "now" the the lastVerifiedFromSource metadata property's timestamp and (2) add/update the lastVerifiedAttempt metadata property's timestamp to "now".
	 * 
	 * @param userId the id to identify the user.
	 * @param password the password to identify the user.
	 * @param host the host name of the RP site where user is verified
	 * @param username the user name to login to the RP site
	 * @param success the verifying result
	 */
	/*public void connectionNotify(String host, String username, boolean success) throws MalformedURLException {

		URL host_ = new URL(host);
		ICard icard = getURIICard(host_.toString(), username);
		if (icard != null) {
			IClaim lastConnectionAttemptClaim;
			try {
				lastConnectionAttemptClaim = icard.getClaimByLocalName("lastConnectionAttempt");
			} catch (CardException e) {
				throw new RppsException(e);
			}
			List values = new ArrayList();
			values.add(new Date());
			if (lastConnectionAttemptClaim != null) {
				// sets current date
				try {
					((ISimpleClaim) lastConnectionAttemptClaim).setValues(values);
				} catch (InvalidValueException e) {
					throw new RppsException(e.getMessage());
				} catch (InvalidStateException e) {
					throw new RppsException(e.getMessage());
				} catch (ReadOnlyObjectException e) {
					throw new RppsException(e.getMessage());
				} catch (InvalidValueCountException e) {
					throw new RppsException(e.getMessage());
				}
			} else {
				throw new RppsException("There is not found a claim: lastConnectionAttempt");
			}
			IClaim lastConnectionClaim;
			try {
				lastConnectionClaim = icard.getClaimByLocalName("lastConnection");
			} catch (CardException e) {
				throw new RppsException(e);
			}
			if (lastConnectionClaim != null) {
				// sets current date if verification is succeed
				if (success) {
					try {
						((ISimpleClaim) lastConnectionClaim).setValues(values);
					} catch (InvalidValueException e) {
						throw new RppsException(e.getMessage());
					} catch (InvalidStateException e) {
						throw new RppsException(e.getMessage());
					} catch (ReadOnlyObjectException e) {
						throw new RppsException(e.getMessage());
					} catch (InvalidValueCountException e) {
						throw new RppsException(e.getMessage());
					}
					
					 * NameCallback name = new NameCallback("name"); try { handler.handle(new Callback[] {name}); } catch (IOException e) { throw new RppsException(e); } catch (UnsupportedCallbackException e) { throw new RppsException(e); } setValidated(handler, host, username);
					 
				}
			} else {
				throw new RppsException("There is not found a claim: lastConnection");
			}
		} else {
			throw new RppsException("There is not found an i-card for host: " + host_ + ", username: " + username);
		}
	}*/

	/**
	 * The RP site invokes this method to verify that user really has an account on the specified site.
	 * 
	 * @param userId the id to identify the user.
	 * @param password the password to identify the user.
	 * @param host The verified site url.
	 * @param username The username to login onto verified site.
	 * @return The result of verification process.
	 */
	/*	public boolean connectionVerify(String host, String username) throws MalformedURLException {

			URL host_ = new URL(host);
			ICard icard = getURIICard(host_.toString(), username);
			String lastConnectionAttempt = null;
			String lastConnection = null;
			if (icard != null) {
				IClaim lastConnectionAttemptClaim;
				try {
					lastConnectionAttemptClaim = icard.getClaimByLocalName("lastConnectionAttempt");
				} catch (CardException e) {
					throw new RppsException(e);
				}
				if (lastConnectionAttemptClaim != null) {
					lastConnectionAttempt = ((ISimpleClaim) lastConnectionAttemptClaim).getValues().get(0).toString();
				} else {
					throw new RppsException("There is not found a claim: lastConnectionAttempt");
				}
				IClaim lastConnectionClaim;
				try {
					lastConnectionClaim = icard.getClaimByLocalName("lastConnection");
				} catch (CardException e) {
					throw new RppsException(e);
				}
				if (lastConnectionClaim != null) {
					lastConnection = ((ISimpleClaim) lastConnectionClaim).getValues().get(0).toString();
				} else {
					throw new RppsException("There is not found a claim: lastConnection");
				}
			} else {
				throw new RppsException("There is not found an i-card for host: " + host_ + ", username: " + username);
			}
			return lastConnectionAttempt != null && lastConnection != null && lastConnectionAttempt.equals(lastConnection);
		}*/

	/**
	 * Gets a form filling script for the specified site&user
	 * 
	 * @param page the page url where user lands on
	 * @param username the user name
	 * @param form the HTML form name
	 * @return a form filling script
	 */
	/*	public String getFormFillingScript(URL page, String username, String form) {
			return null;
		}*/

	/**
	 * Gets an HTML scraping script for the specified site
	 * 
	 * @param page the page url
	 * @return an HTML scraping script
	 */
	/*	public String getHtmlScrapingScript(String page) throws MalformedURLException, RppsException {
			String script = "";
			URL page_ = new URL(page);
			try {
				BufferedReader br = FileService.getInstance().getBufferedReader(FileService.SCRIPTS_DIR + FileService.convertURLToPath(page_) + FileService.SCRIPTS_EXT);
				String line;
				while ((line = br.readLine()) != null) {
					script += line;
				}
			} catch (IOException e) {
				log.error(e, e);
				throw new RppsException(e);
			}
			return script;
		}*/

	/**
	 * Sets an HTML scraping script for the specified site
	 * 
	 * @param page the page url
	 * @param script the script body
	 */
	/*	public void setHtmlScrapingScript(String page, String script) throws MalformedURLException, RppsException {
			BufferedWriter bw = null;
			URL page_ = new URL(page);
			try {
				bw = FileService.getInstance().getBufferedWriter(FileService.SCRIPTS_DIR + FileService.convertURLToPath(page_) + FileService.SCRIPTS_EXT);
				bw.write(script);
			} catch (IOException e) {
				log.error(e, e);
				throw new RppsException(e);
			} finally {
				try {
					if (bw != null) {
						bw.close();
					}
				} catch (IOException e) {
					log.error(e, e);
					throw new RppsException(e);
				}
			}
		}*/

	/* Method of icard */

	public RppsService() {
	}

	public RppsService(final String userId, final String password) throws AccessException {
		authenticate(userId, password);
	}

	public void addCardHistory(final CUID cuid, final URI site) throws UserProfileException {
		ICardUsageManager cardUsageManager = null;
		try {
			cardUsageManager = ICardRegistry.getInstance().getUserProfileService().getCardUsageManager(getHandler());
			cardUsageManager.addCardHistory(cuid, site);
		} catch (final Exception e) {
			log.error(e, e);
			throw new RppsException(e);
		} finally {
			if (cardUsageManager != null) {
				cardUsageManager.close();
			}
		}
	}

	public CategoryTO addCategory(final CategoryTO categoryTO) throws RppsException {
		try {
			final Category category = ICardRegistry.getInstance().getUserProfileService().addCategory(getHandler(),
					ConvertHelper.convertCategoryTO_to_Category(categoryTO));
			return ConvertHelper.convertCategory_to_CategoryTO(category);

		} catch (final UserProfileException e) {
			log.error(e, e);
			throw new RppsException(e);
		}
	}

	public void addClaimValuesMRU(final String claimType, final String claimValue) throws RppsException {
		ICardUsageManager cardUsageManager = null;
		try {
			cardUsageManager = ICardRegistry.getInstance().getUserProfileService().getCardUsageManager(getHandler());

			addClaimValuesMRU(claimType, claimValue, cardUsageManager);
		} catch (final Exception e) {
			log.error(e, e);
			throw new RppsException(e);
		} finally {
			if (cardUsageManager != null) {
				try {
					cardUsageManager.close();
				} catch (final UserProfileException e) {
					log.error(e, e);
				}
			}
		}
	}

	private void addClaimValuesMRU(final String claimType, final String claimValue, final ICardUsageManager cardUsageManager) throws RppsException {
		try {
			if (claimValue != null && !"".equals(claimValue.trim())) {
				final Set claimValuesMRU = cardUsageManager.getClaimValuesMRU(claimType);
				if (!claimValuesMRU.contains(claimValue)) {
					claimValuesMRU.add(claimValue);
					cardUsageManager.setClaimValuesMRU(claimType, claimValuesMRU);
				}
			}
		} catch (final Exception e) {
			log.error(e, e);
			throw new RppsException(e);
		}
	}

	/**
	 * Methods of storage relation between card and url
	 * 
	 * @throws UserProfileException
	 */
	protected void addUrlCardRelation(final CUID cuid, final URI url, final CardInformationRPPS cardInformation) throws RppsException, UserProfileException {
		ICardUsageManager cardUsageManager = null;
		try {
			final URI url_ = ConvertHelper.convertString_to_URI(url.toString());
			if (cardInformation != null) {
				final URI[] urls = cardInformation.getUris();
				if (urls != null) {
					for (int i = 0; i < urls.length; i++) {
						if (urls[i].equals(url)) {
							return;
						}
					}
				}
			}
			cardUsageManager = ICardRegistry.getInstance().getUserProfileService().getCardUsageManager(getHandler());
			cardUsageManager.setUsedAlways(cuid, url_);
		} catch (final Exception e) {
			log.error(e, e);
			throw new RppsException(e);
		} finally {
			if (cardUsageManager != null) {
				cardUsageManager.close();
			}
		}
	}

	/**
	 * Creates new user profile
	 * 
	 * @param userId the id to identify the user.
	 * @param password the password to identify the user.
	 * @param userProfile user profile trasfer object
	 * @throws RppsException
	 */
	public ResponseMessage addUserProfile(final UserProfileTO userProfile) throws RppsException {
		try {
			final CallbackHandler handler = RppsHelper.getCallbackHandler(userProfile.getLoginName(), userProfile.getPassword());
			ICardRegistry.getInstance().getUserProfileService().addUserProfile(handler,
					ConvertHelper.convertUserProfileTO_to_UserProfile(this.userProfile, userProfile));
			return new ResponseMessage(ResponseMessage.noError);
		} catch (final UserProfileException e) {
			log.error(e, e);
			return new ResponseMessage(ResponseMessage.unknownError, e.getMessage());
		}
	}

	/**
	 * Creates new user profile
	 * 
	 * @param userId the id to identify the user.
	 * @param password the password to identify the user.
	 * @param firstName first name of user
	 * @param lastName last name of user
	 * @param email email of user
	 * @throws RppsException
	 */
	public void addUserProfileFromHBX(final String userId, final String password) throws RppsException {
		final UserProfileTO userProfile = new UserProfileTO();
		userProfile.setLoginName(userId);
		userProfile.setPassword(password);

		final ResponseMessage responseMessage = addUserProfile(userProfile);
		if (responseMessage.getCode() != ResponseMessage.noError) {
			throw new RppsException(responseMessage.getMessage());
		}
	}

	public void authenticate(final String userId, final String password) throws AccessException {
		try {
			this.userProfile = ICardRegistry.getInstance().getUserProfileService().getUserProfile(RppsHelper.getCallbackHandler(userId, password));
			if (this.userProfile != null) {
				this.handler = RppsHelper.getCallbackHandler(this.userProfile.getUserIdentifier());
			} else {
				throw new AccessException("Don't found user profile for userId = <" + userId + ">");
			}
		} catch (final UserProfileAuthenticationException e) {
			throw new AccessException(e.getMessage(), e);
		} catch (final Exception e) {
			throw new AccessException(e.getMessage(), e);
		}
	}

	public FormatDescriptorTO checkImportStream(final byte[] crd) throws RppsException {
		try {
			final IFormatDescriptor formatDescriptor = ICardRegistry.getInstance().checkInput(new ByteArrayInputStream(crd));
			return convertIFormatDescriptor_to_FormatDescriptorTO(formatDescriptor);
		} catch (final Exception e) {
			log.error(e, e);
			throw new RppsException(e);
		}
	}

	protected void clearCardCredential(final CUID cuid) throws RppsException {
		try {
			ICardRegistry.getInstance().getUserProfileService().clearCardCredential(getHandler(), cuid);
		} catch (final Exception e) {
			log.error(e, e);
			throw new RppsException(e);
		}
	}

	public ICardTO clearCardCredential(final String cuid) throws RppsException {
		try {
			clearCardCredential(new CUID(cuid));
			return getICardByCUID(cuid);
		} catch (final Exception e) {
			log.error(e, e);
			throw new RppsException(e);
		}
	}

	protected void clearCardHistory(final CUID cuid) throws UserProfileException {
		ICardUsageManager cardUsageManager = null;
		try {
			cardUsageManager = ICardRegistry.getInstance().getUserProfileService().getCardUsageManager(getHandler());
			cardUsageManager.clearCardHistory(cuid);
		} catch (final Exception e) {
			log.error(e, e);
			throw new RppsException(e);
		} finally {
			if (cardUsageManager != null) {
				cardUsageManager.close();
			}
		}
	}

	public boolean clearCardHistory(final String cuid) throws RppsException {
		try {
			clearCardHistory(new CUID(cuid));
			final List list = getCardHistoryLog(new CUID(cuid));
			if (list == null || list != null && list.size() == 0) {
				return true;
			} else {
				return false;
			}
		} catch (final Exception e) {
			log.error(e, e);
			throw new RppsException(e);
		}
	}

	/**
	 * Converts the card to the card transfer object.
	 * 
	 * @param card the Higgins card.
	 * @param isCatchAuthentication equals "true" if needs to generete AuthenticationRequiredException or AuthenticationException
	 * @return the card transfer object.
	 * @throws UserProfileException
	 * @throws RppsException
	 */
	protected ICardTO convertICard_To_ICardTO(final ICard card, final boolean isCatchAuthentication, final Map icardMap) throws CardException,
			InvalidTypeException, RppsException, UserProfileException {
		final ICardTO icard = new ICardTO();
		if (card != null) {
			icard.setId(card.getID());
			icard.setCuid(card.getCUID().toString());
			icard.setName(card.getName());
			icard.setIssuerName(card.getIssuerName());
			icard.setImage(card.getImage());
			icard.setImageType(card.getImageType());
			icard.setSelfIssued(card.isSelfIssued());

			icard.setType(card.getType());
			if ("IPersonalInformationCard".equals(card.getType())) {
				icard.setDeleteable(false);
			}

			CardCredential cardCredential = null;
			try {
				final CardInformationRPPS cardInformation = getCardInformationByCuid(card.getCUID());
				if (cardInformation != null) {
					cardCredential = cardInformation.getCardCredential();
				}
			} catch (final Exception e) {
				log.error(e, e);
			}

			final UITokenServiceCredentialTO[] tokenServiceCredentials = convertICredentials_to_UITokenServiceCredentialTO(card, cardCredential, icardMap);
			if (tokenServiceCredentials.length > 0) {
				icard.setTokenServiceCredentials(tokenServiceCredentials);
			}

			icard.setProperties(ConvertHelper.convertClaims_to_PropertyTO(card, isCatchAuthentication, card.getSupportedClaimTypes()));
		}
		return icard;
	}

	/**
	 * Converts the card to the transfer object for the CARD element of UserInterfaceResponse
	 * 
	 * @param card the Higgins card.
	 * @param isCatchAuthentication equals "true" if needs to generete AuthenticationRequiredException or AuthenticationException
	 * @return the card transfer object.
	 */
	protected UIResponseCardTO convertICard_To_UIResponseCardTO(final ICard icard, final URI url, final IPolicy policy,
			final CardInformationRPPS cardInformation) {
		final UIResponseCardTO cardTO = new UIResponseCardTO();
		if (icard != null) {

			// sets uid
			cardTO.setUid(icard.getCUID().toString());
			// sets name
			String nm = icard.getName();
			int k;
			nm = (k = nm.lastIndexOf('/')) > 0 ? nm.substring(k + 1) : nm;
			cardTO.setName(nm);
			// sets description
			cardTO.setDescription(icard.getDescription());
			// sets image
			cardTO.setImage(icard.getImage());
			// sets imagetype
			cardTO.setImagetype(icard.getImageType());

			final List supportedClaimTypeList = icard.getSupportedClaimTypesUris();

			if (supportedClaimTypeList != null && supportedClaimTypeList.size() > 0) {
				final String[] supportedClaimType = new String[supportedClaimTypeList.size()];
				supportedClaimTypeList.toArray(supportedClaimType);
				cardTO.setSupportedClaimTypes(supportedClaimType);
			}

			final UITokenServiceCredentialTO[] tokenservices = convertICredentials_to_UITokenServiceCredentialTO(icard,
					cardInformation != null ? cardInformation.getCardCredential() : null, null);

			cardTO.setTokenservices(tokenservices);

			if (cardInformation != null) {

				final URI[] urls = cardInformation.getUris();
				if (urls != null) {
					for (int i = 0; i < urls.length; i++) {
						if (urls[i].equals(url)) {
							cardTO.setSaveCard(true);
						}
					}
				}
			}

			try {
				if (policy.getRequiredClaims() != null && policy.getRequiredClaims().size() > 0) {
					cardTO.setProperties(ConvertHelper.convertClaims_to_PropertyTO(icard, false, policy.getRequiredClaims().iterator()));
				}
			} catch (final Exception e) {
				log.error(e, e);
			}

			if (policy instanceof ICardSpacePolicy) {
				final ICardSpacePolicy policyCS = (ICardSpacePolicy) policy;
				try {
					if (policyCS.getOptionalClaims() != null && policyCS.getOptionalClaims().size() > 0) {
						cardTO.setOptionalClaims(ConvertHelper.convertClaims_to_PropertyTO(icard, false, policyCS.getOptionalClaims().iterator()));
					}
				} catch (final Exception e) {
					log.error(e, e);
				}
			}
		}
		return cardTO;
	}

	private ICardResponseTO convertICards_To_CardResponceTO(final List cards) {
		final Map icardMap = new Hashtable();
		final List iCardList = new ArrayList();
		final ICardResponseTO cardResponceTO = new ICardResponseTO();
		try {
			if (cards != null) {
				for (int i = 0; i < cards.size(); i++) {
					try {
						final ICard card = (ICard) cards.get(i);
						iCardList.add(convertICard_To_ICardTO(card, false, icardMap));
					} catch (final Exception e) {
						log.error(e, e);
					}
				}
			}
		} catch (final Exception e) {
			log.error(e, e);
		} finally {
		}
		final List relationList = new ArrayList();
		if (!icardMap.isEmpty()) {
			for (final Iterator iter = icardMap.keySet().iterator(); iter.hasNext();) {
				final String key = (String) iter.next();
				final List value = (List) icardMap.get(key);
				relationList.add(new MapTO(key, (String[]) value.toArray(new String[value.size()])));
			}
		}
		cardResponceTO.setICardList((ICardTO[]) iCardList.toArray(new ICardTO[iCardList.size()]));
		cardResponceTO.setRelationList((MapTO[]) relationList.toArray(new MapTO[relationList.size()]));
		return cardResponceTO;
	}

	/**
	 * Converts ICredential object to UITokenServiceCredential transfer object
	 * 
	 * @param handler the callback handler to be used to request user's identity
	 * @param card ICard object
	 * @param cardCredential
	 * @return UITokenServiceCredentialTO object
	 */
	protected UITokenServiceCredentialTO[] convertICredentials_to_UITokenServiceCredentialTO(final ICard card, final CardCredential cardCredential,
			final Map icardMap) {
		final List credList = new ArrayList();
		if (card != null) {
			final ICredentialDescriptor[] der = card.getRequiredCredentials();
			if (der != null) {
				for (int i = 0; i < der.length; i++) {
					final ICredential credential = der[i].getCredential();

					final UITokenServiceCredentialTO credentialTO = convertICredentials_to_UITokenServiceCredentialTO(credential, icardMap, card);

					if (credentialTO != null) {
						if (cardCredential != null && credentialTO.getType().equals(cardCredential.getCredentialType())) {
							credentialTO.setSaveCredential(true);
						}

						credList.add(credentialTO);
					}
				}
			}
		}
		final UITokenServiceCredentialTO[] array = new UITokenServiceCredentialTO[credList.size()];
		credList.toArray(array);
		return array;
	}

	/**
	 * Converts ICredential object to UITokenServiceCredential transfer object
	 * 
	 * @param handler the callback handler to be used to request user's identity
	 * @param card ICard object
	 * @param cardCredential
	 * @return UITokenServiceCredentialTO object
	 */
	protected UITokenServiceCredentialTO convertICredentials_to_UITokenServiceCredentialTO(final ICredential credential, final Map icardMap, final ICard icard) {
		UITokenServiceCredentialTO tokenService = null;
		if (credential != null) {
			tokenService = new UITokenServiceCredentialTO();
			if (credential instanceof ITokenServiceCredential) {

				tokenService.setAddress(((ITokenServiceCredential) credential).getAddress().toString());
				tokenService.setMetadataAddress(((ITokenServiceCredential) credential).getMetadataAddress().toString());

				if (credential instanceof ITSUsernamePasswordCredential) {
					tokenService.setPassword(((ITSUsernamePasswordCredential) credential).getPassword());
					tokenService.setUsername(((ITSUsernamePasswordCredential) credential).getUsername());
					tokenService.setType(RppsService.ITSUsernamePasswordCredential);

				} else if (credential instanceof ITSSelfIssuedCredential) {
					tokenService.setType(RppsService.ITSSelfIssuedNonPinCredential);
					try {
						final ICard personalCard = getICardByPPID(((ITSSelfIssuedCredential) credential).getCertificate(),
								((ITSSelfIssuedCredential) credential).getPPID(), ((ITSSelfIssuedCredential) credential).getAddress());
						if (personalCard == null) {
							log.info("Cannot find the Personal card used to authenticate for this managed card.");
							return null;
						}

						if (icardMap != null && icard != null) {
							List relationCUIDs = null;

							if (icardMap.containsKey(personalCard.getCUID().toString())) {
								relationCUIDs = (List) icardMap.get(personalCard.getCUID().toString());
							}

							if (relationCUIDs == null) {
								relationCUIDs = new ArrayList();
							}

							if (!relationCUIDs.contains(icard.getCUID().toString())) {
								relationCUIDs.add(icard.getCUID().toString());
							}

							icardMap.put(personalCard.getCUID().toString(), relationCUIDs);
						}

						tokenService.setAssociatedCardName(personalCard.getName());
						tokenService.setAssociatedCardImage(personalCard.getImage());
						tokenService.setAssociatedCardImageType(personalCard.getDescription());
						tokenService.setAssociatedCardDescription(personalCard.getDescription());

						ICredential subCredential = null;
						final ICredentialDescriptor[] derPersonal = personalCard.getRequiredCredentials();
						if (derPersonal != null && derPersonal.length > 0) {
							subCredential = derPersonal[0].getCredential();
						}

						if (subCredential != null) {
							for (int index = 0; index < subCredential.getCallbacks().length; index++) {
								if (subCredential.getCallbacks()[index] instanceof PinCodeCallback) {

									tokenService.setType(RppsService.ITSSelfIssuedCredential);
									break;
								}
							}
						}
					} catch (final Exception e) {
						log.error(e, e);
					}
				} else if (credential instanceof ITSKerberosV5Credential) {
					tokenService.setType(RppsService.ITSKerberosV5Credential);

				} else if (credential instanceof ITSX509V3Credential) {
					tokenService.setType(RppsService.ITSX509V3Credential);
				} else {
					tokenService = null;
				}
			} else {
				if (credential instanceof IPinCodeCredential) {
					tokenService.setType(RppsService.IPinCodeCredential);
				} else if (credential instanceof IUsernamePasswordCredential) {
					tokenService.setType(RppsService.IUsernamePasswordCredential);
					tokenService.setUsername(((IUsernamePasswordCredential) credential).getUsername());
				} else if (credential instanceof IPasswordCredential) {
					tokenService.setType(RppsService.IPasswordCredential);
				} else {
					tokenService = null;
				}
			}
		}
		return tokenService;
	}

	protected FormatDescriptorTO convertIFormatDescriptor_to_FormatDescriptorTO(final IFormatDescriptor descriptor) {
		FormatDescriptorTO descriptorTO = null;
		if (descriptor != null) {
			descriptorTO = new FormatDescriptorTO();
			descriptorTO.setFormatID(descriptor.getID());
			descriptorTO.setName(descriptor.getDescription());
			descriptorTO.setFileExtension(descriptor.getFileExtension());
			if (descriptor.getCredentialDescriptor() != null && descriptor.getCredentialDescriptor().getCredential() != null) {
				final UITokenServiceCredentialTO credential = convertICredentials_to_UITokenServiceCredentialTO(descriptor.getCredentialDescriptor()
						.getCredential(), null, null);
				descriptorTO.setCredential(credential);
			}
		}

		return descriptorTO;
	}

	protected ICard createDuplicateICard(final String cardName, final CUID cuid, final String pinCode) throws RppsException, UserProfileException {
		try {
			ICard card = ICardRegistry.getInstance().getICardByCUID(getHandler(), cuid);

			if (!(card instanceof IPersonalInformationCard)) {
				throw new RppsException("You can not create dublicate this card.");
			}
			if (card.getRequiredCredentials() != null && card.getRequiredCredentials().length > 0) {
				ICredential credential = null;
				if (pinCode != null && pinCode.trim().length() > 0) {
					credential = new PinCodeCredential();
					try {
						((PinCodeCredential) credential).setPinCode(pinCode.getBytes("UTF-8"));
					} catch (final UnsupportedEncodingException e) {
						((PinCodeCredential) credential).setPinCode(pinCode.getBytes());
					}
				} else {
					final CardInformationRPPS cardInformation = getCardInformationByCuid(cuid);
					if (cardInformation != null && cardInformation.getCardCredential() != null) {
						credential = ConvertHelper.convertCardCredential_to_ICredential(card, cardInformation.getCardCredential());
					} else {
						throw new RppsException("Rpps can not retrieve card. You have to send pin code.");
					}
				}
				card = card.getProvider().getICardByCUID(getHandler(), cuid, credential);
			}

			final ICardTemplate[] cardTemplates = card.getProvider().getCardCreationTemplates(getHandler());
			ICardTemplate cardTemplate = null;
			for (int i = 0; i < cardTemplates.length; i++) {

				if (!cardTemplates[i].getCardType().isInstance(card)) {
					continue;
				}

				cardTemplate = cardTemplates[i];

				if (cardTemplate.getTemplateElementByID("cardName") != null) {
					((ITemplateValue) cardTemplate.getTemplateElementByID("cardName")).setValue(cardName);
				}
				if (cardTemplate.getTemplateElementByID("cardPicture") != null && card.getImage() != null) {
					((ITemplateValue) cardTemplate.getTemplateElementByID("cardPicture")).setValue(Base64.encode(card.getImage()));
				}
				if (cardTemplate.getTemplateElementByID("claimList") != null) {
					for (final Iterator iter = ((ITemplateContainer) cardTemplate.getTemplateElementByID("claimList")).getTemplateElements(); iter.hasNext();) {
						final ITemplateElement element = (ITemplateElement) iter.next();
						final IClaim claim = card.getClaim(element.getID());
						if (claim != null) {
							if (((ISimpleClaim) claim).getValues().size() > 0) {
								((ITemplateValue) element).setValue((String) ((ISimpleClaim) claim).getValues().get(0));
							}
						}
					}
				}
				break;
			}
			if (cardTemplate != null) {
				return createICard(cardTemplate);
			}
		} catch (final CardException e) {
			log.error(e, e);
			throw new RppsException(e);
		}

		return null;
	}

	/**
	 * Creates the duplicate card with new card's name
	 * 
	 * @param userId the id to identify the user.
	 * @param password the password to identify the user.
	 * @param cardName new card's name
	 * @param cuid cuid of old card
	 * @param pinCode pincode if it needs
	 * @return card trasfer object
	 * @throws RppsException
	 */
	public ICardTO createDuplicateICard(final String cardName, final String cuid, final String pinCode) throws RppsException {
		ICard card = null;
		try {
			card = createDuplicateICard(cardName, new CUID(cuid), pinCode);
			return convertICard_To_ICardTO(card, false, null);
		} catch (final Exception e) {
			log.error(e, e);
			throw new RppsException(e);
		} finally {

		}
	}

	protected ICard createICard(final ICardTemplate template) throws RppsException {
		try {
			if (template != null) {
				// gets all registered providers
				final String extID = template.getProviderID();
				final ICardProvider provider = ICardRegistry.getInstance().getICardProvider(extID);
				if (provider != null) {
					// check if current provider supports needed protocol
					if (provider.getCardCreationTemplates(getHandler()) != null) {
						try {
							return provider.createCard(getHandler(), template);
						} catch (final Exception ex) {
							throw new RppsException(ex);
						}
					}
				}
				throw new RppsException("There is not a registered provider for this id.");
			}
			throw new RppsException("Paramenter <<template>> is null");
		} catch (final RppsException e) {
			log.error(e, e);
			throw e;
		}
	}

	/**
	 * Creates a new card by template
	 * 
	 * @param userId the id to identify the user.
	 * @param password the password to identify the user.
	 * @param templete template
	 * @return a new card id
	 */
	public ICardTO createICard(final ICardTemplateTO template) throws RppsException {
		ICard card = null;
		try {
			final ICardTemplate[] templates = getICardCreationTemplateByProvider(template.getProviderID());
			card = createICard(RppsHelper.fillICardTemplate_from_ICardTemplateTO(templates, template));
			return convertICard_To_ICardTO(card, false, null);
		} catch (final Exception e) {
			log.error(e, e);
			throw new RppsException(e);
		} finally {

		}
	}

	/**
	 * Creates a new card
	 * 
	 * @param userId the id to identify the user.
	 * @param password the password to identify the user.
	 * @param id the i-card id
	 * @param properties the i-card properties
	 * @return a new card id
	 */
	public ICardTO createICard(final String id, final String[] properties) throws RppsException {
		ICard card = null;
		try {
			card = createICard(URI.create(id), RppsHelper.loadProperties(properties));
			return convertICard_To_ICardTO(card, false, null);

		} catch (final Exception e) {
			log.error(e, e);
			throw new RppsException(e);
		}
	}

	protected ICard createICard(final URI id, final Properties properties) throws RppsException {
		// gets all registered providers
		for (final Iterator it = ICardRegistry.getInstance().getICardProviders(); it.hasNext();) {
			final ICardProvider provider = (ICardProvider) it.next();
			// check if current provider supports needed protocol
			if (provider.canCreateCard(getHandler(), id.toString(), properties)) {
				try {
					return provider.createCard(getHandler(), id.toString(), properties);
				} catch (final Exception ex) {
					throw new RppsException(ex);
				}
			}
		}
		throw new RppsException("There is not a registered provider for this type of i-card.");
	}

	/**
	 * Creates new card
	 * 
	 * @param userId the id to identify the user.
	 * @param password the password to identify the user.
	 * @param cardname name of card
	 * @param surname user's last name
	 * @param givenname user's first name
	 * @param emailaddress user's email
	 * @return cuid of new card
	 * @throws RppsException
	 */
	public String createICardFromHBX(final String cardname) throws RppsException {
		String ret = null;
		try {
			ICardTemplate template = getICardCreationTemplateByTUID(new TUID("org.eclipse.higgins.icard.provider.cardspace.personal.db", "PersonalCardTemplate"));
			template = RppsHelper.fillPersonalCardTemplate(template, cardname, "", "", "", "", "", "", "", "", "", "", "", "", "", "");
			ret = createICard(template).getCUID().toString();
		} catch (final Exception e) {
			throw new RppsException(e);
		} finally {
		}
		return ret;
	}

	/**
	 * Deletes all user's card.
	 * 
	 * @param userId the id to identify the user.
	 * @param password the password to identify the user.
	 */
	public void deleteAllICard() throws RppsException {
		final List icards = getICardsByUser(getHandler());

		for (final Iterator iter = icards.iterator(); iter.hasNext();) {
			deleteICardOnly(((ICard) iter.next()).getCUID().toString());
		}
	}

	protected boolean deleteCardHistoryRecord(final CUID cuid, final URI url, final ICardUsageManager cardUsageManager) throws RppsException {
		try {
			// cardUsageManager = ICardRegistry.getInstance().getUserProfileService().getCardUsageManager(getHandler());
			cardUsageManager.clearCardHistory(cuid, url);
			return true;
		} catch (final Exception e) {
			log.error(e, e);
			throw new RppsException(e);
		}
	}

	public boolean deleteCardHistoryRecord(final String cuid, final String url) throws RppsException {
		ICardUsageManager cardUsageManager = null;
		try {
			cardUsageManager = ICardRegistry.getInstance().getUserProfileService().getCardUsageManager(getHandler());
			return deleteCardHistoryRecord(new CUID(cuid), new URI(url), cardUsageManager);
		} catch (final Exception e) {
			log.error(e, e);
			throw new RppsException(e);
		} finally {
			if (cardUsageManager != null) {
				try {
					cardUsageManager.close();
				} catch (final UserProfileException e) {
					log.error(e, e);
				}
			}
		}
	}

	public void deleteCategory(final String categoryId) throws RppsException {
		try {
			ICardRegistry.getInstance().getUserProfileService().deleteCategory(getHandler(), categoryId);
		} catch (final Exception e) {
			log.error(e, e);
			throw new RppsException(e);
		}
	}

	/**
	 * Deletes specified card.
	 * 
	 * @param cuid the i-card CUID
	 */
	public ICardUpdateResponseTO deleteICard(final String cuid) throws RppsException {
		ICard personalCard = null;
		try {
			final ICard icard = getICardByCUID(new CUID(cuid));
			final ICredentialDescriptor[] credDescriptor = icard.getRequiredCredentials();
			if (credDescriptor != null) {
				for (int i = 0; i < credDescriptor.length; i++) {
					if (credDescriptor[i].getCredential() != null && credDescriptor[i].getCredential() instanceof ITSSelfIssuedCredential) {
						final ITSSelfIssuedCredential credential = (ITSSelfIssuedCredential) credDescriptor[i].getCredential();
						personalCard = getICardByPPID(credential.getCertificate(), credential.getPPID(), credential.getAddress());
					}
				}
			}
			deleteICardOnly(cuid);
		} catch (final RppsException e) {
			return new ICardUpdateResponseTO(ICardUpdateResponseTO.unknownError, e.getMessage());
		} catch (final Exception e) {
			log.error(e, e);
			return new ICardUpdateResponseTO(ICardUpdateResponseTO.unknownError, e.getMessage());
		}
		if (personalCard != null) {
			return new ICardUpdateResponseTO(personalCard.getCUID().toString());
		}
		return new ICardUpdateResponseTO(ICardUpdateResponseTO.noError, "");
	}

	/**
	 * Deletes specified card.
	 * 
	 * @param cuid the i-card CUID
	 */
	private void deleteICardOnly(final String cuid) throws RppsException {
		try {
			final CallbackHandler handler = getHandler();
			final CUID cuid_ = new CUID(cuid);
			ICardRegistry.getInstance().getUserProfileService().clearCardCredential(getHandler(), cuid_);
			ICardUsageManager cardUsageManager = null;
			try {
				cardUsageManager = ICardRegistry.getInstance().getUserProfileService().getCardUsageManager(handler);
				cardUsageManager.clearUsedAlwaysURLs(cuid_);
				cardUsageManager.clearCardHistory(cuid_);
			} finally {
				if (cardUsageManager != null) {
					cardUsageManager.close();
				}
			}
			ICardRegistry.getInstance().deleteICard(getHandler(), new CUID(cuid));
		} catch (final CardException e) {
			log.error(e, e);
			throw new RppsException(e);
		} catch (final UserProfileException e) {
			log.error(e, e);
			throw new RppsException(e);
		}
	}

	/**
	 * @throws UserProfileException
	 */
	public void deleteOldPassword() throws UserProfileException {
		try {
			ICardRegistry.getInstance().getUserProfileService().deleteOldPassword(getHandler());
		} catch (final Exception e) {
			log.error(e, e);
			throw new RppsException(e);
		}
	}

	protected CardUsage deleteUrlCardRelation(final CUID cuid, final URI url) throws RppsException, UserProfileException {

		ICardUsageManager cardUsageManager = null;
		try {
			cardUsageManager = ICardRegistry.getInstance().getUserProfileService().getCardUsageManager(getHandler());
			return cardUsageManager.deleteUsedAlways(cuid, url);
		} catch (final Exception e) {
			log.error(e, e);
			throw new RppsException(e);
		} finally {
			if (cardUsageManager != null) {
				cardUsageManager.close();
			}
		}
	}

	public boolean deleteUrlCardRelation(final String cuid, final String url) throws RppsException {
		try {
			if (url != null) {
				deleteUrlCardRelation(new CUID(cuid), new URI(url));
			}
			return true;
		} catch (final Exception e) {
			log.error(e, e);
			throw new RppsException(e);
		}
	}

	/**
	 * Deletes user profile
	 * 
	 * @param userId the id to identify the user.
	 * @param password the password to identify the user.
	 * @throws RppsException
	 */
	public ResponseMessage deleteUserProfile() throws RppsException {
		try {
			deleteUserProfile(getHandler());
			return new ResponseMessage(ResponseMessage.noError);
		} catch (final RppsException e) {
			log.error(e, e);
			return new ResponseMessage(ResponseMessage.unknownError, e.getMessage());
		}
	}

	/* Methods of HBX */

	/**
	 * Deletes user profile
	 * 
	 * @param userId the id to identify the user.
	 * @param password the password to identify the user.
	 * @throws RppsException
	 */
	protected void deleteUserProfile(final CallbackHandler handler) throws RppsException {
		try {
			deleteAllICard();
			ICardRegistry.getInstance().getUserProfileService().deleteUserProfile(handler);
		} catch (final UserProfileException e) {
			log.error(e, e);
			throw new RppsException(e);
		}
	}

	/* Methods of getting token */

	public ICardTO editPinCode(final String cuid, final String oldPinCode, final String newPinCode) throws RppsException {
		ICard card = null;
		byte[] oldPinCode_ = null;
		try {
			card = getICardByCUID(new CUID(cuid));

			if (card == null) {
				throw new RppsException("Can't find card with CUID " + cuid);
			}

			if (!(card instanceof IPersonalInformationCard)) {
				new RppsException("This card is not personal");
			}
			CardInformationRPPS cardInformation = null;
			if (oldPinCode == null) {
				cardInformation = getCardInformationByCuid(new CUID(cuid));
				if (cardInformation.getCardCredential() != null
						&& IRppsService.IPinCodeCredential.equals(cardInformation.getCardCredential().getCredentialType())) {
					oldPinCode_ = cardInformation.getCardCredential().getPinCode();
				}
			} else {
				oldPinCode_ = oldPinCode.getBytes("UTF-8");
			}

			card = resetPinCode((IPersonalInformationCard) card, oldPinCode_);
			card = setPinCode((IPersonalInformationCard) card, newPinCode.getBytes("UTF-8"));
			if (cardInformation != null) {
				try {
					if (cardInformation.getCardCredential() != null) {
						final CardCredential cardCredential = cardInformation.getCardCredential();
						cardCredential.setPinCode(newPinCode.getBytes("UTF-8"));
						setCardCredential(new CUID(cuid), cardCredential);
					}
				} catch (final Exception e) {
					log.error(e, e);
				}
			}
			return convertICard_To_ICardTO(card, false, null);
		} catch (final Exception e) {
			log.error(e, e);
			throw new RppsException(e);
		}
	}

	public byte[] exportICards(final String formatID, final String[] cards, final UITokenServiceCredentialTO credential) throws RppsException {
		try {
			final ByteArrayOutputStream out = new ByteArrayOutputStream();
			if (cards != null && cards.length > 0) {
				final CUID[] cuids = new CUID[cards.length];
				for (int i = 0; i < cards.length; i++) {
					cuids[i] = new CUID(cards[i]);
				}
				final ICredential credential_ = ConvertHelper.convertUITokenServiceCredentialTO_to_ICredentials(credential);
				ICardRegistry.getInstance().exportICards(getHandler(), out, formatID, cuids, credential_);
				return out.toByteArray();
			}
		} catch (final Exception e) {
			log.error(e, e);
			throw new RppsException(e);
		}
		return null;
	}

	/**
	 * Creates AND element as well as subelements of the UIResponse transfer object.
	 * 
	 * @param uct the UserChoiceTree object.
	 * @return the array of UIResponseAndTO objects.
	 * @throws UserProfileException
	 * @throws RppsException
	 */
	protected UIResponseAndTO[] getAnds(final UserChoiceTree uct, final URI url) throws RppsException, UserProfileException {
		final List andList = new ArrayList();
		if (uct != null) {
			for (final ListIterator li = uct.getListIterator(); li.hasNext();) {
				final Object child = li.next();
				final UIResponseAndTO and = new UIResponseAndTO();
				if (child instanceof UserChoiceTree) {
					final List orList = new ArrayList();
					for (final ListIterator li2 = ((UserChoiceTree) child).getListIterator(); li2.hasNext();) {
						final Object child2 = li2.next();
						final UIResponseOrTO or = new UIResponseOrTO();
						if (child2 instanceof UserChoiceTree) {
							final List andList2 = new ArrayList();
							for (final ListIterator li3 = ((UserChoiceTree) child2).getListIterator(); li3.hasNext();) {
								final Object child3 = li3.next();
								final UIResponseAndTO and2 = new UIResponseAndTO();
								if (child3 instanceof UCTelm) {
									// ???
								}
								andList2.add(and2);
							}
						} else if (child2 instanceof UCTelm) {
							final IPolicy atom = ((UCTelm) child2).getAtom();
							final UIResponseClauseTO clause = new UIResponseClauseTO();

							final RPPolicy policy = ConvertHelper.convertIPolicyToRPPolicy(atom);

							clause.setLabelType(policy.getLabelType());
							clause.setLabel(policy.getLabel());
							clause.setLonglabel(policy.getLonglabel());
							clause.setOptional(policy.getOptional());
							clause.setLongOptional(policy.getLongOptional());
							clause.setOptionalType(policy.getOptionalType());

							final PolicyVersion policyVersion = ICardRegistry.getInstance().getUserProfileService().getRPPolicyVersion(getHandler(), url);

							if (atom instanceof ICardSpacePolicy) {
								final ICardSpacePolicy icp = (ICardSpacePolicy) atom;
								if (policyVersion != null && policyVersion.getVersion() != null && !policyVersion.getVersion().equals(icp.getPrivacyVersion())) {
									try {
										final IPolicy oldPolicy = ICardSelectorService.getInstance().parsePolicy(policyVersion.getPolicy());
										final RPPolicy oldRpPolicy = ConvertHelper.convertIPolicyToRPPolicy(oldPolicy);
										clause.setOldPolicy(oldRpPolicy);
									} catch (final Exception e) {
										log.error(e, e);
									}
								}
							}

							final UIResponseUsableCardsTO uc = new UIResponseUsableCardsTO();
							final List cardList = new ArrayList();
							for (final Iterator it = ((UCTelm) child2).getCredsIterator(); it != null && it.hasNext();) {
								UIResponseCardTO card = new UIResponseCardTO();
								final CredentialContainer cc = (CredentialContainer) it.next();
								final ICard cred = cc.getCredential();

								final CardInformationRPPS cardInformation = getCardInformationByCuid(cred.getCUID());
								// sets uid
								card = convertICard_To_UIResponseCardTO(cred, url, atom, cardInformation);

								card.setUid(cc.key);

								if (cardInformation != null) {
									final URI[] urls = cardInformation.getUris();
									if (urls != null) {
										for (int i = 0; i < urls.length; i++) {
											if (urls[i].equals(url)) {
												card.setSaveCard(true);
											}
										}
									}
								}

								cardList.add(card);
							}

							final UIResponseCardTO[] cards = new UIResponseCardTO[cardList.size()];
							cardList.toArray(cards);
							uc.setCard(cards);
							clause.setUsableCards(uc);

							final List usedCard = getUsedCards(url);
							final String[] usedCardCuids = new String[usedCard.size()];
							for (int index = 0; index < usedCard.size(); index++) {
								usedCardCuids[index] = ((CardUsage) usedCard.get(index)).getCuid().toString();
							}
							clause.setUsedCardCuids(usedCardCuids);

							or.setClause(clause);
						}
						orList.add(or);
					}
					final UIResponseOrTO[] ors = new UIResponseOrTO[orList.size()];
					orList.toArray(ors);
					and.setOr(ors);
				} else if (child instanceof UCTelm) {
					// ???
				}
				andList.add(and);
			}
		}
		final UIResponseAndTO[] ands = new UIResponseAndTO[andList.size()];
		andList.toArray(ands);
		return ands;
	}

	protected List getCardHistory(final CUID cuid) throws RppsException, UserProfileException {

		ICardUsageManager cardUsageManager = null;
		try {
			cardUsageManager = ICardRegistry.getInstance().getUserProfileService().getCardUsageManager(getHandler());
			return cardUsageManager.getCardUsageSummary(cuid);
		} catch (final Exception e) {
			log.error(e, e);
			throw new RppsException(e);
		} finally {
			if (cardUsageManager != null) {
				cardUsageManager.close();
			}
		}
	}

	protected List getCardHistory(final CUID cuid, final int startFrom, final int size, final String orderBy, final boolean desc) throws RppsException,
			UserProfileException {

		ICardUsageManager cardUsageManager = null;
		try {
			cardUsageManager = ICardRegistry.getInstance().getUserProfileService().getCardUsageManager(getHandler());
			return cardUsageManager.getCardUsageSummary(cuid, startFrom, size, orderBy, desc);
		} catch (final Exception e) {
			log.error(e, e);
			throw new RppsException(e);
		} finally {
			if (cardUsageManager != null) {
				cardUsageManager.close();
			}
		}
	}

	public CardUsageTO[] getCardHistory(final String cuid, final int startFrom, final int size, final String orderBy, final String orderDirection) {
		final boolean desc = orderDirection.equalsIgnoreCase("desc") ? true : false;
		String order = CardUsage.BY_SITE;
		if ("date".equalsIgnoreCase(orderBy)) {
			order = CardUsage.BY_DATE;
		} else if ("site".equalsIgnoreCase(orderBy)) {
			order = CardUsage.BY_USE;
		}

		try {
			final List list = getCardHistory(new CUID(cuid), startFrom, size, order, desc);
			final CardUsageTO[] arrayUsage = new CardUsageTO[list.size()];
			for (int i = 0; i < list.size(); i++) {
				final CardUsage cardUsage = (CardUsage) list.get(i);
				arrayUsage[i] = ConvertHelper.convertCardUsage_to_CardUsageTO(cardUsage);
			}
			return arrayUsage;
		} catch (final Exception e) {
			log.error(e, e);
			throw new RppsException(e);
		}
	}

	private List getCardHistoryLog(final CUID cuid) throws UserProfileException {
		ICardUsageManager cardUsageManager = null;
		try {
			cardUsageManager = ICardRegistry.getInstance().getUserProfileService().getCardUsageManager(getHandler());
			return cardUsageManager.getCardHistory(cuid);
		} catch (final Exception e) {
			log.error(e, e);
			throw new RppsException(e);
		} finally {
			if (cardUsageManager != null) {
				cardUsageManager.close();
			}
		}
	}

	private List getCardHistoryLog(final CUID cuid, final int startFrom, final int size, final String orderBy, final boolean desc) throws RppsException,
			UserProfileException {

		ICardUsageManager cardUsageManager = null;
		try {
			cardUsageManager = ICardRegistry.getInstance().getUserProfileService().getCardUsageManager(getHandler());
			return cardUsageManager.getCardHistory(cuid);

		} catch (final Exception e) {
			log.error(e, e);
			throw new RppsException(e);
		} finally {
			if (cardUsageManager != null) {
				cardUsageManager.close();
			}
		}
	}

	public CardUsageTO[] getCardHistoryLog(final String cuid, final int startFrom, final int size, final String orderBy, final String orderDirection) {
		final boolean desc = orderDirection.equalsIgnoreCase("desc") ? true : false;
		String order = CardUsage.BY_SITE;
		if ("date".equalsIgnoreCase(orderBy)) {
			order = CardUsage.BY_DATE;
		} else if ("site".equalsIgnoreCase(orderBy)) {
			order = CardUsage.BY_USE;
		}

		try {
			final List list = getCardHistoryLog(new CUID(cuid), startFrom, size, order, desc);
			final CardUsageTO[] arrayUsage = new CardUsageTO[list.size()];
			for (int i = 0; i < list.size(); i++) {
				final CardUsage cardUsage = (CardUsage) list.get(i);
				arrayUsage[i] = ConvertHelper.convertCardUsage_to_CardUsageTO(cardUsage);
			}
			return arrayUsage;
		} catch (final Exception e) {
			log.error(e, e);
			throw new RppsException(e);
		}
	}

	public CardInformationTO getCardInformation(final String cuid, final int size) throws RppsException {
		try {
			List cardHistory = getCardHistory(new CUID(cuid));
			int sizeAll = 0;
			if (cardHistory != null) {
				sizeAll = cardHistory.size();
			}
			cardHistory = getCardHistory(new CUID(cuid), 0, size, CardUsage.BY_DATE, false);
			return ConvertHelper.createCardInformationTO(cardHistory, sizeAll);
		} catch (final Exception e) {
			log.error(e, e);
			throw new RppsException(e);
		}
	}

	protected CardInformationRPPS getCardInformationByCuid(final CUID cuid) throws RppsException, UserProfileException {
		final CardInformationRPPS cardInformation = null;
		ICardUsageManager cardUsageManager = null;
		try {
			cardUsageManager = ICardRegistry.getInstance().getUserProfileService().getCardUsageManager(getHandler());
			return getCardInformationByCuid(cuid, cardUsageManager);

		} catch (final Exception e) {
			log.error(e, e);
			throw new RppsException(e);
		} finally {
			if (cardUsageManager != null) {
				cardUsageManager.close();
			}
		}
	}

	protected CardInformationRPPS getCardInformationByCuid(final CUID cuid, final ICardUsageManager cardUsageManager) throws RppsException,
			UserProfileException {
		CardInformationRPPS cardInformation = null;
		try {
			final CardInformation cardInformationUP = ICardRegistry.getInstance().getUserProfileService().getCardInformation(getHandler(), cuid);

			cardInformation = new CardInformationRPPS(cardInformationUP, cuid);
			final List cardUsages = cardUsageManager.getUsedAlwaysURLs(cuid);
			final URI[] uris = new URI[cardUsages.size()];
			for (int i = 0; i < cardUsages.size(); i++) {
				uris[i] = ((CardUsage) cardUsages.get(i)).getSite();
			}
			cardInformation.setUris(uris);

		} catch (final Exception e) {
			log.error(e, e);
			throw new RppsException(e);
		} finally {
		}
		return cardInformation;
	}

	protected List getCategories(final CallbackHandler handler) throws RppsException {
		try {
			final List list = ICardRegistry.getInstance().getUserProfileService().getCategories(handler);

			Collections.sort(list, new CategoryComparator());

			return list;
		} catch (final UserProfileException e) {
			log.error(e, e);
			throw new RppsException(e);
		}
	}

	/**
	 * @param userId
	 * @param password
	 * @return
	 * @throws RppsException
	 */
	public CategoryTO[] getCategory() throws RppsException {
		try {
			final List category = getCategories(getHandler());
			CategoryTO[] categorysTO = new CategoryTO[0];
			if (category != null) {
				categorysTO = new CategoryTO[category.size()];
				for (int i = 0; i < category.size(); i++) {
					categorysTO[i] = ConvertHelper.convertCategory_to_CategoryTO((Category) category.get(i));
				}
			}
			return categorysTO;
		} catch (final RppsException e) {
			log.error(e, e);
			throw e;
		}
	}

	public MapTO[] getClaimValuesMRU() throws RppsException {
		ICardUsageManager cardUsageManager = null;
		try {
			cardUsageManager = ICardRegistry.getInstance().getUserProfileService().getCardUsageManager(getHandler());

			final Map claimValuesMRU = cardUsageManager.getClaimValuesMRU();
			if (claimValuesMRU != null && claimValuesMRU.keySet() != null) {
				final List result = new ArrayList();
				for (final Iterator iter = claimValuesMRU.keySet().iterator(); iter.hasNext();) {
					final MapTO mruTO = new MapTO();
					final String key = (String) iter.next();
					final Set valuesSet = (Set) claimValuesMRU.get(key);
					if (valuesSet != null) {
						final String[] values = new String[valuesSet.size()];
						if (values != null && values.length > 0) {
							valuesSet.toArray(values);
							if (values != null) {
								Arrays.sort(values, new StringComparator());
							}

							mruTO.setKey(key);
							mruTO.setValues(values);
							result.add(mruTO);
						}
					}
				}
				final MapTO[] resultMRU = new MapTO[result.size()];
				result.toArray(resultMRU);
				return resultMRU;
			}
			return new MapTO[0];
		} catch (final Exception e) {
			log.error(e, e);
			throw new RppsException(e);
		} finally {
			if (cardUsageManager != null) {
				try {
					cardUsageManager.close();
				} catch (final UserProfileException e) {
					log.error(e, e);
				}
			}
		}
	}

	protected CUID getCuidByUrl(final URI url) throws RppsException, UserProfileException {
		ICardUsageManager cardUsageManager = null;
		try {
			cardUsageManager = ICardRegistry.getInstance().getUserProfileService().getCardUsageManager(getHandler());
			return cardUsageManager.getUsedAlwaysCard(url);
		} catch (final Exception e) {
			log.error(e, e);
			throw new RppsException(e);
		} finally {
			if (cardUsageManager != null) {
				cardUsageManager.close();
			}
		}
	}

	public CallbackHandler getHandler() {
		return this.handler;
	}

	/**
	 * Gets the global i-name registered to the Higgins service (e.g. "=parity").
	 * 
	 * @return the global i-name.
	 */
	public String getHigginsServiceGlobalIName() {
		return System.getProperty(WebKeys.HIGGINS_SERVICE_GLOBAL_NAME_PROPERTY, "=parity");
	}

	/* Methods of card history */

	protected ICard getICardByCUID(final CUID cuid) throws RppsException {
		try {
			return ICardRegistry.getInstance().getICardByCUID(getHandler(), cuid);
		} catch (final CardException e) {
			log.error(e, e);
			throw new RppsException(e);
		}
	}

	/**
	 * Gets a card by CUID.
	 * 
	 * @param userId the id to identify the user.
	 * @param password the password to identify the user.
	 * @param cuid the card CUID.
	 * @return the card transfer object.
	 */
	public ICardTO getICardByCUID(final String cuid) throws RppsException {
		try {
			final ICard card = getICardByCUID(new CUID(cuid));
			return convertICard_To_ICardTO(card, false, null);
		} catch (final Exception e) {
			log.error(e, e);
			throw new RppsException(e);
		} finally {
		}
	}

	/**
	 * Gets a card by CUID and Credential.
	 * 
	 * @param userId the id to identify the user.
	 * @param password the password to identify the user.
	 * @param cuid the card CUID.
	 * @param tokenCredential the tokenCredential transfer object
	 * @return the card transfer object.
	 */
	public ICardTO getICardByCUIDAndTokenServiceCredential(final String cuid, final UITokenServiceCredentialTO tokenCredential) throws RppsException {
		try {
			ICard card = getICardByCUID(new CUID(cuid));

			if (card == null) {
				throw new RppsException("Can't find card with CUID " + cuid);
			}

			final CardInformationRPPS cardInformation = getCardInformationByCuid(new CUID(cuid));
			ICredential credential = null;
			if (cardInformation.getCardCredential() != null) {
				credential = ConvertHelper.convertCardCredential_to_ICredential(card, cardInformation.getCardCredential());
			} else {
				if (tokenCredential != null) {
					credential = ConvertHelper.convertUITokenServiceCredentialTO_to_ICredentials(card, tokenCredential);
				}
			}
			if (credential != null) {
				card = card.getProvider().getICardByCUID(getHandler(), card.getCUID(), credential);
			}

			return convertICard_To_ICardTO(card, true, null);
		} catch (final Exception e) {
			log.error(e, e);
			throw new RppsException(e);
		} finally {
		}
	}

	protected ICard getICardByPPID(final X509Certificate x509cert, final String ppid, final URI address) throws RppsException {

		try {
			final byte[] bppid = Base64.decode(ppid);
			final IKeyStoreService securityService = KeyStoreRegistry.getInstance().getSecurityService();
			final X509Certificate[] chainOriginal = new X509Certificate[] { x509cert };
			X509Certificate[] chain = null;
			if (chainOriginal != null) {
				try {
					chain = securityService.getValidateCertificateChain(getHandler(), chainOriginal, address);
				} catch (final CertificateException e) {
					log.error(e, e);
				} catch (final CertStoreException e) {
					log.error(e, e);
				}
			}
			if (chain == null) {
				chain = chainOriginal;
			}
			final PPIDCardSpacePolicy policy = new PPIDCardSpacePolicy(chain, securityService.getCertStore(getHandler()), bppid);
			final IICardSelector selector = ICardSelectorService.getInstance().getICardSelector(getHandler(), policy);

			final UserChoiceTree_ANDofORs uct = (UserChoiceTree_ANDofORs) selector.getUserChoice();
			final SelectionANDofORs subselection = uct.getDefaultSelection();

			if (subselection.isEmpty()) {
				log.debug("Cannot find the Personal card used to authenticate for this managed card.");
				return null;
			}
			final String cuid = ((SelectionANDofORsElm) subselection.getElements().get(0)).getUUID();

			return getICardByCUID(new CUID(cuid));
		} catch (final Exception e) {
			log.error(e, e);
			throw new RppsException(e);
		}
	}

	public UIResponseCardTO getICardClaims(final String cuid, final String policy, final String typeofCredential, final String[] credentialKey,
			final String[] credentialValue) throws RppsException {
		try {
			AllowSelfSignedSSL();
			final Hashtable table = new Hashtable();
			for (int i = 0; i < credentialKey.length; i++) {
				if (credentialKey[i] != null && credentialValue[i] != null) {
					table.put(credentialKey[i], credentialValue[i]);
				}
			}
			URI url = null;
			String urlStr;
			if (table.get("url") != null) {
				urlStr = (String) table.get("url");
				url = ConvertHelper.convertString_to_URI(urlStr);
			}

			ICredential credential;

			ICard card = getICardByCUID(new CUID(cuid));
			if (card != null) {

				final CardInformationRPPS cardCredentialInfo = getCardInformationByCuid(new CUID(cuid));

				if (cardCredentialInfo != null && cardCredentialInfo.getCardCredential() != null) {
					credential = ConvertHelper.convertCardCredential_to_ICredential(card, cardCredentialInfo.getCardCredential());
				} else {
					credential = ConvertHelper.convertHashtable_to_Credential(card, typeofCredential, table);
				}

				if (credential != null) {
					card = card.getProvider().getICardByCUID(getHandler(), card.getCUID(), credential);
				}
				final ICardSpacePolicy policy_ = (ICardSpacePolicy) ICardSelectorService.getInstance().parsePolicy(policy);
				return convertICard_To_UIResponseCardTO(card, url, policy_, cardCredentialInfo);
			} else {
				throw new RppsException("Didn't find card with cuid=" + cuid);
			}

		} catch (final Exception e) {
			log.error(e, e);
			throw new RppsException(e);
		} finally {
		}
	}

	/**
	 * Return templates for creating cards
	 * 
	 * @param userId the id to identify the user.
	 * @param password the password to identify the user.
	 * @return a template list
	 */
	public ICardTemplateTO[] getICardCreationTemplate() {
		List templates = null;
		try {
			templates = getICardCreationTemplate(getHandler());
		} finally {
		}
		ICardTemplateTO[] telmplateList = new ICardTemplateTO[0];
		if (templates != null) {
			telmplateList = new ICardTemplateTO[templates.size()];
			for (int index = 0; index < templates.size(); index++) {
				telmplateList[index] = ConvertHelper.convertICardTemplate_to_ICardTemplateTO((ICardTemplate) templates.get(index));
			}
		}
		return telmplateList;
	}

	/**
	 * Return templates for creating cards
	 * 
	 * @param userId the id to identify the user.
	 * @param password the password to identify the user.
	 * @return a template list
	 */
	protected List getICardCreationTemplate(final CallbackHandler handler) throws RppsException {
		return ICardRegistry.getInstance().getCardCreationTemplates(handler);
	}

	public ICardTemplate[] getICardCreationTemplateByProvider(final String extID) throws RppsException {
		final ICardProvider provider = ICardRegistry.getInstance().getICardProvider(extID);
		return provider.getCardCreationTemplates(getHandler());

	}

	/**
	 * Return templates for creating cards
	 * 
	 * @return a template list
	 */
	public ICardTemplateTO getICardCreationTemplateByTUID(final String tuid) throws RppsException {

		ICardTemplate template = null;
		try {
			template = getICardCreationTemplateByTUID(new TUID(tuid));
		} catch (final CardException e) {
			log.error(e, e);
			throw new RppsException(e);
		} finally {
		}
		if (template != null) {
			return ConvertHelper.convertICardTemplate_to_ICardTemplateTO(template);
		}
		return null;
	}

	/*
		*//**
				 * Get a icard template for creating personal card
				 * 
				 * @param userId the id to identify the user.
				 * @param password the password to identify the user.
				 * @return a template
				 */
	/*
		public ICardTemplateTO getPersonalICardCreationTemplate() {
			ICardTemplate template = null;
			try {
				template = getPersonalICardCreationTemplate(getHandler());
			} finally {
			}
			if (template != null) {
				return ConvertHelper.convertICardTemplate_to_ICardTemplateTO(template);
			}
			return null;
		}*/

	/**
	 * Return templates for creating cards
	 * 
	 * @return a template list
	 */
	public ICardTemplate getICardCreationTemplateByTUID(final TUID tuid) throws CardException {
		try {
			return ICardRegistry.getInstance().getCardCreationTemplatesByTUID(getHandler(), tuid);
		} finally {
		}
	}

	/**
	 * Gets the card and category lists of the specified user.
	 * 
	 * @param userId the id to identify the user.
	 * @param password the password to identify the user.
	 * @return the card list
	 */
	public ICardsAndCategoryTO getICardsAndCategoriesByUser() throws RppsException {
		try {
			final ICardsAndCategoryTO cardsAndCategoryTO = new ICardsAndCategoryTO();
			try {
				final ICardResponseTO cardResponceTO = getICardsByUser();
				cardsAndCategoryTO.setRelationList(cardResponceTO.getRelationList());
				cardsAndCategoryTO.setICardList(cardResponceTO.getICardList());
			} catch (final Exception e) {
				log.error(e, e);
			}
			try {
				cardsAndCategoryTO.setCategoryList(getCategory());
			} catch (final Exception e) {
				log.error(e, e);
			}
			return cardsAndCategoryTO;

		} finally {
		}
	}

	/**
	 * Gets cards by CUIDs array.
	 * 
	 * @param userId the id to identify the user.
	 * @param password the password to identify the user.
	 * @param cuids the card CUIDs.
	 * @return list of cards transfer object.
	 */
	public ICardTO[] getICardsByCUIDs(final String[] cuids) {
		final ArrayList result = new ArrayList();
		for (int i = 0; i < cuids.length; i++) {
			try {
				result.add(getICardByCUID(cuids[i]));
			} catch (final Exception e) {
				log.error(e, e);
			}
		}
		return (ICardTO[]) result.toArray(new ICardTO[result.size()]);
	}

	public ICardTO[] getICardsByFormat(final String formatID) throws RppsException {
		ICardTO[] icards = new ICardTO[0];
		try {
			final List cards = new ArrayList();
			try {
				for (final Iterator iter = ICardRegistry.getInstance().getICardsByFormat(getHandler(), formatID); iter.hasNext();) {
					try {
						cards.add(iter.next());
					} catch (final Exception e) {
						log.error(e, e);
					}
				}
			} catch (final Exception e) {
				log.error(e, e);
			}
			if (cards != null) {
				icards = new ICardTO[cards.size()];
				for (int i = 0; i < cards.size(); i++) {
					try {
						icards[i] = convertICard_To_ICardTO((ICard) cards.get(i), false, null);
					} catch (final Exception e) {
						log.error(e, e);
					}
				}
			}
		} catch (final Exception e) {
			log.error(e, e);
		} finally {
		}
		return icards;
	}

	/**
	 * Gets the card list of the specified user.
	 * 
	 * @param userId the id to identify the user.
	 * @param password the password to identify the user.
	 * @return the card list
	 */
	public ICardResponseTO getICardsByUser() {
		ICardResponseTO cardResponceTO = new ICardResponseTO();
		try {
			List cards = null;
			cards = getICardsByUser(getHandler());
			if (cards != null) {
				cardResponceTO = convertICards_To_CardResponceTO(cards);
			}
		} catch (final Exception e) {
			log.error(e, e);
			// throw new RppsException(e);
		} finally {
		}
		return cardResponceTO;
	}

	/* Methods of card category */

	protected List getICardsByUser(final CallbackHandler handler) {
		final List cards = new ArrayList();
		// gets all registered providers
		for (final Iterator pit = ICardRegistry.getInstance().getICardProviders(); pit.hasNext();) {
			final ICardProvider provider = (ICardProvider) pit.next();
			try {
				for (final Iterator icards = provider.getICards(handler); icards.hasNext();) {
					cards.add(icards.next());
				}
			} catch (final Exception e) {
				log.error(e, e);
			}
		}
		return cards;
	}

	protected ICredential getICredentialByPPID(final X509Certificate x509cert, final String ppid, final URI address) throws RppsException {

		try {
			final ICard card = getICardByPPID(x509cert, ppid, address);
			if (card == null) {
				return null;
			}
			final ICredentialDescriptor[] der = card.getRequiredCredentials();
			if (der == null) {
				return null;
			}
			if (der.length == 0) {
				return null;
			}
			return der[0].getCredential();
		} catch (final Exception e) {
			log.error(e, e);
			throw new RppsException(e);
		}
	}

	public FormatDescriptorTO[] getOutputFormats() throws RppsException {
		try {
			final IFormatDescriptor[] formatDescriptor = ICardRegistry.getInstance().getOutputFormats();
			if (formatDescriptor != null && formatDescriptor.length > 0) {
				final FormatDescriptorTO[] formatDescriptorTO = new FormatDescriptorTO[formatDescriptor.length];
				for (int i = 0; i < formatDescriptor.length; i++) {
					formatDescriptorTO[i] = convertIFormatDescriptor_to_FormatDescriptorTO(formatDescriptor[i]);
				}
				return formatDescriptorTO;
			}
			return null;
		} catch (final Exception e) {
			log.error(e, e);
			throw new RppsException(e);
		}
	}

	/**
	 * Web service operation. Makes xmlToken usage claim list.
	 * 
	 * @param userId the id to identify the user.
	 * @param password the password to identify the user.
	 * @param policy the RP Agent's Relying Party Security Policy
	 * @param policytype the RP Security Policy type
	 * @param sslCert the SSL certificate
	 * @param user the username to login to the STS
	 * @param pass the password to login to the STS
	 * @param sels the user-selected subset of the UIDs (handles) contained in the UserInterfaceResponse object returned from an earlier invocation of 'userInterfaceRequest'
	 * @return the security token transfer object
	 * @throws RppsException
	 */
	public TokenResponseTO getTokenByClaims(final String sslCert, final String[] claimName, final String[] claimValue) throws RppsException {
		try {
			final Hashtable claimHashtable = new Hashtable();
			for (int i = 0; i < claimName.length; i++) {
				try {
					claimHashtable.put(claimName[i], claimValue[i]);
				} catch (final RuntimeException e) {
					e.printStackTrace();
				}
			}
			X509Certificate cert = null;
			final X509Certificate[] certChain = RppsHelper.parseSSLCert(sslCert);
			if (certChain != null && certChain.length > 0) {
				cert = KeyStoreRegistry.getInstance().getSecurityService().findLeafFromCertificates(certChain);
			}
			final ISTSResponse resp = UserProfileTokenHelper.getToken(cert, claimHashtable);

			final IdentityToken itoken = new IdentityToken(resp, Base64.encode(cert.getEncoded()));

			final String tokenAsString = (String) itoken.getAs(String.class);

			return ConvertHelper.createTokenResponse(tokenAsString);
		} catch (final Exception e) {
			log.error(e, e);
			throw new RppsException(e);
		}
	}

	/**
	 * Web service operation. Makes security token usage saved users information
	 * 
	 * @param userId the id to identify the user.
	 * @param password the password to identify the user.
	 * @param policy the RP Agent's Relying Party Security Policy
	 * @param policytype the RP Security Policy type
	 * @param sslCert the SSL certificate
	 * @param url the url of site which needs token
	 * @return the security token transfer object
	 * @throws RemoteException
	 */
	public TokenResponseTO getTokenByUrl(final String policy, final String policytype, final String sslCert, final String url) throws RppsException {
		TokenResponseTO token = null;
		try {
			final URI url_ = ConvertHelper.convertString_to_URI(url);
			final CUID cuidUrl = getCuidByUrl(url_);
			if (cuidUrl == null) {
				return null;
			}

			final CardInformationRPPS cardInformation = getCardInformationByCuid(cuidUrl);
			if (cardInformation != null) {
				final String[] cuids = new String[] { cuidUrl.toString() };

				final ICard card = getICardByCUID(cuidUrl);
				if (card == null) {
					new RppsException("Does not find card with cuid " + cuidUrl.toString());
				}

				ICredential credential = null;
				credential = ConvertHelper.convertCardCredential_to_ICredential(card, cardInformation.getCardCredential());

				List optionalClaims = null;
				try {
					optionalClaims = getUsersChoiceOfOptionalClaims(card.getCUID(), url_);
				} catch (final Exception exc) {
					log.error(exc, exc);
				}
				final IPolicy iPolicy = ICardSelectorService.getInstance().parsePolicy(policy);
				if (iPolicy instanceof CardSpacePolicy && optionalClaims != null && optionalClaims.size() > 0) {
					((CardSpacePolicy) iPolicy).setOptionalClaims(optionalClaims);
				}

				token = getTokenObject(policy, policytype, sslCert, cuids, credential, url_);

				if (token != null && token.getTkn() != null && token.getTkn().length() > 0) {
					if (url_ != null) {
						addCardHistory(card.getCUID(), url_);
					}
				}
			}
		} catch (final Exception e) {
			log.error(e, e);
		}
		return token;
	}

	/**
	 * Web service operation. Makes security token usage saved users information
	 * 
	 * @param userId the id to identify the user.
	 * @param password the password to identify the user.
	 * @param policy the RP Agent's Relying Party Security Policy
	 * @param policytype the RP Security Policy type
	 * @param sslCert the SSL certificate
	 * @param url the url of site which needs token
	 * @return the security token transfer object
	 * @throws RemoteException
	 */
	public ExtraTokenResponseTO getTokenExtraByUrl(final String policy, final String policytype, final String sslCert, final String url) throws RppsException {
		ExtraTokenResponseTO token = null;
		try {
			final URI url_ = ConvertHelper.convertString_to_URI(url);
			final CUID cuidUrl = getCuidByUrl(url_);
			if (cuidUrl == null) {
				return null;
			}

			final CardInformationRPPS cardCredentialInfo = getCardInformationByCuid(cuidUrl);
			if (cardCredentialInfo != null) {
				final String[] cuids = new String[] { cardCredentialInfo.getCuid().toString() };

				final ICard card = getICardByCUID(cardCredentialInfo.getCuid());
				if (card == null) {
					new RppsException("Does not find card with cuid " + cardCredentialInfo.getCuid().toString());
				}

				ICredential credential = null;
				credential = ConvertHelper.convertCardCredential_to_ICredential(card, cardCredentialInfo.getCardCredential());

				List optionalClaims = null;
				try {
					optionalClaims = getUsersChoiceOfOptionalClaims(card.getCUID(), url_);
				} catch (final Exception exc) {
					log.error(exc, exc);
				}
				final IPolicy iPolicy = ICardSelectorService.getInstance().parsePolicy(policy);
				if (iPolicy instanceof CardSpacePolicy && optionalClaims != null && optionalClaims.size() > 0) {
					((CardSpacePolicy) iPolicy).setOptionalClaims(optionalClaims);
				}

				final TokenResponseTO tokenTO = getTokenObject(policy, policytype, sslCert, cuids, credential, url_);

				if (tokenTO != null && tokenTO.getTkn() != null && tokenTO.getTkn().length() > 0) {
					if (url_ != null) {
						addCardHistory(card.getCUID(), url_);
					}
				}
				token = new ExtraTokenResponseTO();
				token.setTkn(tokenTO.getTkn());
				token.setCardName(card.getName());
				token.setCardDescription(card.getDescription());
				token.setCardImage(card.getImage());
				token.setCardImageType(card.getImageType());
			}
		} catch (final Exception e) {
			log.error(e, e);
		}
		return token;
	}

	/* Methods of user profile */

	/**
	 * Web service operation. Given the Relying Party Agent (aka 'Requester')'s policy, identity information about the requester, and the set of one or more selections that the user has just made in the ISS Web UI (see userInterfaceRequest), AND presuming that the protocol involved in interacting with the RPA requires a security token, request the token that corresponds to the user's selection(s).
	 * 
	 * @param userId the id to identify the user.
	 * @param password the password to identify the user.
	 * @param policy the RP Agent's Relying Party Security Policy
	 * @param policytype the RP Security Policy type
	 * @param sslCert the SSL certificate
	 * @param cuids the user-selected subset of the UIDs (handles) contained in the UserInterfaceResponse object returned from an earlier invocation of 'userInterfaceRequest'
	 * @param typeofCredential the type of selected credential
	 * @param credentialKey the names of fields of selected credential
	 * @param credentialValue the value of fields of selected credential
	 * @return the security token transfer object
	 * @throws RemoteException
	 */
	protected TokenResponseTO getTokenObject(final String policy, final String policytype, final String sslCert, final String[] cuids,
			final ICredential credential, final CardInformationRPPS cardCredentialInfo, final URI url, final boolean saveCard, final boolean saveCredential)
			throws RppsException {
		TokenResponseTO token = null;
		try {

			final CUID cuid = new CUID(cuids[0]);

			if (!saveCredential) {
				if (cardCredentialInfo != null && cardCredentialInfo.getCardCredential() != null) {
					clearCardCredential(cuid);
				}
			}

			token = getTokenObject(policy, policytype, sslCert, cuids, credential, url);

			if (token != null && token.getTkn() != null && token.getTkn().length() > 0) {
				if (url != null) {
					addCardHistory(cuid, url);
					final IPolicy iPolicy = ICardSelectorService.getInstance().parsePolicy(policy);
					if (iPolicy instanceof CardSpacePolicy) {
						final List claimTypeList = ((CardSpacePolicy) iPolicy).getOptionalClaims();
						if (claimTypeList != null && claimTypeList.size() > 0) {
							saveUsersChoiceOfOptionalClaims(cuid, url, claimTypeList);
						}
						final ICardSpacePolicy iCardSpacePolicy = (ICardSpacePolicy) iPolicy;
						if (iCardSpacePolicy != null && iCardSpacePolicy.getPrivacyVersion() != null && iCardSpacePolicy.getPrivacyVersion().length() > 0) {
							final PolicyVersion policyVersion = new PolicyVersion();
							policyVersion.setUrl(url);
							policyVersion.setPolicy(policy);
							policyVersion.setVersion(iCardSpacePolicy.getPrivacyVersion());
							ICardRegistry.getInstance().getUserProfileService().setRPPolicyVersion(getHandler(), policyVersion);
						}
					}
				}
				if (saveCard && url != null) {
					addUrlCardRelation(cuid, url, cardCredentialInfo);
				}

				if (saveCredential) {
					final CardCredential cardCredential = ConvertHelper.convertICredential_to_CardCredential(credential);
					setCardCredential(cuid, cardCredential);
				}
			}
		} catch (final Exception e) {
			log.error(e, e);
			throw new RppsException(e);
		}
		return token;
	}

	/**
	 * Given the Relying Party Agent (aka 'Requester')'s policy, identity information about the requester, and the set of one or more selections that the user has just made in the ISS Web UI (see userInterfaceRequest), AND presuming that the protocol involved in interacting with the RPA requires a security token, request the token that corresponds to the user's selection(s).
	 * 
	 * @param handler
	 * @param policy the RP Agent's Relying Party Security Policy
	 * @param policytype the RP Security Policy type
	 * @param sslCert the SSL certificate
	 * @param cuids the user-selected subset of the UIDs (handles) contained in the UserInterfaceResponse object returned from an earlier invocation of 'userInterfaceRequest'
	 * @param credential
	 * @param cardCredentialInfo
	 * @param url
	 * @return the security token transfer object
	 * @throws RppsException
	 */
	protected TokenResponseTO getTokenObject(final String policy, final String policytype, final String sslCert, final String[] cuids,
			final ICredential credential, final URI uri) throws RppsException {
		String token = null;

		final IKeyStoreService securityService = KeyStoreRegistry.getInstance().getSecurityService();

		// get the selections from user's selection
		final SelectionANDofORs selection = new SelectionANDofORs();
		if (uri != null) {
			selection.action = uri.toString();
		}
		selection.idemix_credential_db_uri = "credentials.db";
		selection.cardspace_credential_db_uri = "infocards.properties";
		selection.publickey_db_uri = "pk.db";
		final X509Certificate[] chainOriginal = RppsHelper.parseSSLCert(sslCert);
		X509Certificate[] chain = null;
		if (chainOriginal != null) {
			try {
				chain = securityService.getValidateCertificateChain(getHandler(), chainOriginal, uri);
				securityService.putCertChain(chain);
			} catch (final CertificateException e) {
				log.error(e, e);
			} catch (final CertStoreException e) {
				log.error(e, e);
			}
		}
		if (chain == null) {
			chain = chainOriginal;
		}
		selection.sslCertChain = chain;
		selection.setCredential(credential);
		for (int i = 0; i < cuids.length; i++) {
			final String uid = cuids[i];
			// n = (Node)i.next();
			int ui = 0;
			try {
				if (i < cuids.length - 1) {
					ui = Integer.parseInt(cuids[++i]);
				}
			} catch (final Exception ew) {
				ew.printStackTrace();
			}
			final SelectionANDofORsElm se = new SelectionANDofORsElm(uid, ui, 0);
			selection.add(se);
		}
		try {

			final IICardSelector selector = ICardSelectorService.getInstance().getICardSelector(getHandler(), policy);
			final IIdentityToken iToken = selector.getIdentityToken(selection);
			token = (String) iToken.getAs(String.class);

		} catch (final Exception e) {
			log.error(e, e);
			throw new RppsException(e);
		}
		return ConvertHelper.createTokenResponse(token);
	}

	/**
	 * Web service operation. Given the Relying Party Agent (aka 'Requester')'s policy, identity information about the requester, and the set of one or more selections that the user has just made in the ISS Web UI (see userInterfaceRequest), AND presuming that the protocol involved in interacting with the RPA requires a security token, request the token that corresponds to the user's selection(s).
	 * 
	 * @param userId the id to identify the user.
	 * @param password the password to identify the user.
	 * @param policy the RP Agent's Relying Party Security Policy
	 * @param policytype the RP Security Policy type
	 * @param sslCert the SSL certificate
	 * @param cuids the user-selected subset of the UIDs (handles) contained in the UserInterfaceResponse object returned from an earlier invocation of 'userInterfaceRequest'
	 * @param typeofCredential the type of selected credential
	 * @param credentialKey the names of fields of selected credential
	 * @param credentialValue the value of fields of selected credential
	 * @return the security token transfer object
	 * @throws RemoteException
	 */
	public TokenResponseTO getTokenObject(final String policy, final String policytype, final String sslCert, final String[] cuids,
			final String typeofCredential, final String[] credentialKey, final String[] credentialValue) throws RppsException {
		TokenResponseTO token = null;
		try {
			AllowSelfSignedSSL();
			final Hashtable table = new Hashtable();
			boolean saveCard = false;
			boolean saveCredential = false;
			String urlStr = null;
			for (int i = 0; i < credentialKey.length; i++) {
				if (credentialKey[i] != null && credentialValue[i] != null) {
					table.put(credentialKey[i], credentialValue[i]);
					log.debug("Put in table key='" + credentialKey[i] + "' with value='" + credentialValue[i] + "'");
				}
			}
			if (table.get("saveCard") != null && "true".equalsIgnoreCase((String) table.get("saveCard"))) {
				saveCard = true;
			}
			if (table.get("saveCredential") != null && "true".equalsIgnoreCase((String) table.get("saveCredential"))) {
				saveCredential = true;
			}
			if (table.get("url") != null) {
				urlStr = (String) table.get("url");
			}
			if (urlStr != null && urlStr.indexOf("?") > -1) {
				urlStr = urlStr.substring(0, urlStr.indexOf("?"));
			}

			log.debug("CUID parameter is " + cuids[0]);
			log.debug("SaveCard parameter is " + Boolean.toString(saveCard));
			log.debug("SaveCredential parameter is " + Boolean.toString(saveCredential));
			log.debug("Url parameter is " + urlStr);

			URI url = null;
			if (urlStr != null && urlStr.length() > 0) {
				url = ConvertHelper.convertString_to_URI(urlStr);
			}

			ICredential credential;

			final ICard card = getICardByCUID(new CUID(cuids[0]));
			if (card != null) {

				CardInformationRPPS cardCredentialInfo = null;
				final CUID cuidUrl = getCuidByUrl(url);

				if (cuidUrl != null) {
					if (cuidUrl.equals(card.getCUID()) && !saveCard) {
						deleteUrlCardRelation(card.getCUID(), url);
					}
				}

				cardCredentialInfo = getCardInformationByCuid(card.getCUID());

				if (cardCredentialInfo != null && cardCredentialInfo.getCardCredential() != null) {
					credential = ConvertHelper.convertCardCredential_to_ICredential(card, cardCredentialInfo.getCardCredential());
				} else {
					credential = ConvertHelper.convertHashtable_to_Credential(card, typeofCredential, table);
				}

				token = getTokenObject(policy, policytype, sslCert, cuids, credential, cardCredentialInfo, url, saveCard, saveCredential);
			} else {
				throw new RppsException("Didn't find card with cuid=" + cuids[0]);
			}
		} catch (final Exception e) {
			log.error(e, e);
			throw new RppsException(e);
		} finally {

		}

		return token;
	}

	private List getUsedCards(final URI url) throws RppsException, UserProfileException {
		ICardUsageManager cardUsageManager = null;
		try {
			cardUsageManager = ICardRegistry.getInstance().getUserProfileService().getCardUsageManager(getHandler());
			return cardUsageManager.getUsedCards(url);
		} catch (final Exception e) {
			log.error(e, e);
			throw new RppsException(e);
		} finally {
			if (cardUsageManager != null) {
				cardUsageManager.close();
			}
		}
	}

	/**
	 * This method takes as input a policy---for example, one that has been retrieved from the Relying Party Agent (called here the 'Requester')---as well as a description of the Requester. The UserInterfaceResponse object should contain all information necessary to display both (a) a user interface which shares the same card metaphor, sequences of experiences and decision points (the ceremony) of CardSpace and (b) other interface paradigms to be developed in the future
	 * 
	 * @param userId the id to identify the user.
	 * @param password the password to identify the user.
	 * @param url the URL to the html page with <object> tag
	 * @param target
	 * @param sslCert the SSL certificate
	 * @param policyType the RP Security Policy type
	 * @param policy the RP Agent's Relying Party Security Policy
	 * @return the UIResponseTO transfer object.
	 * @throws RemoteException
	 */

	public UIResponseTO getUserInterface(final String url, final String target, final String sslCert, final String policytype, final String policy)
			throws RppsException {

		final UserInterfaceResponse uir = new UserInterfaceResponse();
		try {
			final IICardSelector selector = ICardSelectorService.getInstance().getICardSelector(getHandler(), policy);
			uir.setUCL(selector.getUserChoice());
			X509Certificate[] sslCerts = RppsHelper.parseSSLCert(sslCert);
			if (sslCerts != null) {
				X509Certificate cert = null;
				if (sslCerts != null && sslCerts.length > 0) {
					cert = KeyStoreRegistry.getInstance().getSecurityService().findLeafFromCertificates(sslCerts);
				}

				uir.setRequestorInformation(cert);

				final IKeyStoreService securityService = KeyStoreRegistry.getInstance().getSecurityService();
				boolean isValid = false;
				boolean isExtendedValid = false;
				try {
					sslCerts = securityService.getValidateCertificateChain(getHandler(), sslCerts, new URI(url));
					if (sslCerts != null) {
						isValid = true;
					}
				} catch (final CertificateException e) {
					log.error(e, e);
				} catch (final CertStoreException e) {
					log.error(e, e);
				}
				if (isValid) {
					isExtendedValid = securityService.extendendValidationCertificateChain(getHandler(), sslCerts, new URI(url));
				}
				uir.setValid(isValid);
				uir.setExtendedValid(isExtendedValid);
			}
		} catch (final Exception e) {
			log.error(e, e);
			throw new RppsException(e);
		}
		final UIResponseTO uirto = new UIResponseTO();
		try {
			final URI url_ = ConvertHelper.convertString_to_URI(url);
			if (uir != null) {
				uirto.setStatus("ok");
				final UIResponseRPTO rp = new UIResponseRPTO();
				rp.setName(uir.getName());
				rp.setAddr1(uir.getAddr1());
				rp.setAddr2(uir.getAddr2());
				rp.setPhone(uir.getPhone());
				rp.setVerifier(uir.getVerifier());
				rp.setValid(uir.isValid());
				rp.setExtendedValid(uir.isExtendedValid());
				uirto.setRP(rp);
				uirto.setAnd(getAnds(uir.getUCL(), url_));
			}
		} catch (final Exception e) {
			log.error(e, e);
			throw new RppsException(e);
		}
		return uirto;
	}

	public UserProfileTO getUserProfile() throws RppsException, AccessException {
		final UserProfile userProfile = getUserProfile(getHandler());
		return ConvertHelper.convertUserProfile_to_UserProfileTO(userProfile);
	}

	private UserProfile getUserProfile(final CallbackHandler handler) throws RppsException, AccessException {
		try {
			if (this.userProfile == null) {
				this.userProfile = ICardRegistry.getInstance().getUserProfileService().getUserProfile(handler);
			}

			return this.userProfile;
		} catch (final UserProfileAuthenticationException e) {
			log.error(e, e);
			throw new AccessException(e.getMessage(), e);
		} catch (final UserProfileException e) {
			log.error(e, e);
			throw new RppsException(e.getMessage(), e);
		}
	}

	private List getUsersChoiceOfOptionalClaims(final CUID cuid, final URI site) throws UserProfileException {
		ICardUsageManager cardUsageManager = null;
		try {
			cardUsageManager = ICardRegistry.getInstance().getUserProfileService().getCardUsageManager(getHandler());
			final Set optionalClaims = cardUsageManager.getOptionalClaims(cuid, site);
			if (optionalClaims != null) {
				return new ArrayList(optionalClaims);
			} else {
				return new ArrayList();
			}
		} catch (final Exception e) {
			log.error(e, e);
			throw new RppsException(e);
		} finally {
			if (cardUsageManager != null) {
				cardUsageManager.close();
			}
		}
	}

	/* Methods of card information */

	/**
	 * Imports card from a file.
	 * 
	 * @param userId the id to identify the user.
	 * @param password the password to identify the user.
	 * @param crd the card as a byte array.
	 */
	public ICardResponseTO importICards(final byte[] crd) throws RppsException {
		return importICards(crd, null, null);
	}

	/**
	 * Imports card from a file.
	 * 
	 * @param userId the id to identify the user.
	 * @param password the password to identify the user.
	 * @param crd the card as a byte array.
	 * @param formatID the data format of the input stream if available. If <code>null</code> is used all available data formats will be checked in order to import.
	 * @param credential the authentication data to proccess the input stream if required
	 */
	public ICardResponseTO importICards(final byte[] crd, final String formatID, final UITokenServiceCredentialTO credential) throws RppsException {
		try {
			List cards = null;
			cards = importICards(new ByteArrayInputStream(crd), formatID, ConvertHelper.convertUITokenServiceCredentialTO_to_ICredentials(credential));

			return convertICards_To_CardResponceTO(cards);
		} catch (final Exception e) {
			log.error(e, e);
			throw new RppsException(e);
		}
	}

	/**
	 * Imports card from a stream.
	 * 
	 * @param in the input stream to import from
	 * @param formatID the data format of the input stream if available. If <code>null</code> is used all available data formats will be checked in order to import.
	 * @param credential the authentication data to proccess the input stream if required
	 */
	protected List importICards(final InputStream in, final String formatID, final ICredential credential) throws RppsException {
		try {
			final List result = new ArrayList();
			for (final Iterator iter = ICardRegistry.getInstance().importICards(getHandler(), in, formatID, credential); iter.hasNext();) {
				result.add(iter.next());
			}
			return result;
		} catch (final CardException e) {
			log.error(e, e);
			throw new RppsException(e);
		}
	}

	public ResponseMessage modifyCategories(final CategoryTO[] categoriesTO) throws RppsException {
		try {
			for (int i = 0; i < categoriesTO.length; i++) {
				ICardRegistry.getInstance().getUserProfileService().modifyCategory(getHandler(), ConvertHelper.convertCategoryTO_to_Category(categoriesTO[i]));
			}
			return new ResponseMessage(ResponseMessage.noError, "");
		} catch (final UserProfileException e) {
			log.error(e, e);
			return new ResponseMessage(ResponseMessage.unknownError, e.getMessage());
		}
	}

	/* Methods of storage card credential */

	public CategoryTO modifyCategory(final CategoryTO categoryTO) throws RppsException {
		try {
			final Category category = ICardRegistry.getInstance().getUserProfileService().modifyCategory(getHandler(),
					ConvertHelper.convertCategoryTO_to_Category(categoryTO));
			return ConvertHelper.convertCategory_to_CategoryTO(category);
		} catch (final UserProfileException e) {
			log.error(e, e);
			throw new RppsException(e);
		}
	}

	/**
	 * Update user profile
	 * 
	 * @param userId the id to identify the user.
	 * @param password the password to identify the user.
	 * @param userProfile user profile transfer object
	 * @return user profile transfer object
	 * @throws RppsException
	 */
	public UserProfileTO modifyUserProfile(final UserProfileTO userProfile) throws RppsException {
		try {
			this.userProfile = ICardRegistry.getInstance().getUserProfileService().modifyUserProfile(getHandler(),
					ConvertHelper.convertUserProfileTO_to_UserProfile(this.userProfile, userProfile));
			return getUserProfile();
		} catch (final Exception e) {
			log.error(e, e);
			throw new RppsException(e);
		}
	}

	public void removeClaimValuesMRU(final String claimType, final String claimValue) throws RppsException {
		ICardUsageManager cardUsageManager = null;
		try {
			cardUsageManager = ICardRegistry.getInstance().getUserProfileService().getCardUsageManager(getHandler());

			final Set claimValuesMRU = cardUsageManager.getClaimValuesMRU(claimType);
			if (claimValuesMRU.contains(claimValue)) {
				claimValuesMRU.remove(claimValue);
				cardUsageManager.setClaimValuesMRU(claimType, claimValuesMRU);
			}
		} catch (final Exception e) {
			log.error(e, e);
			throw new RppsException(e);
		} finally {
			if (cardUsageManager != null) {
				try {
					cardUsageManager.close();
				} catch (final UserProfileException e) {
					log.error(e, e);
				}
			}
		}
	}

	protected ICard resetPinCode(IPersonalInformationCard card, final byte[] oldPinCode) throws RppsException {
		try {

			final PinCodeCredential pinCodeCredential = new PinCodeCredential();
			pinCodeCredential.setPinCode(oldPinCode);
			card = (IPersonalInformationCard) card.getProvider().getICardByCUID(getHandler(), card.getCUID(), pinCodeCredential);

			try {
				final PinCodeCredential pinCodeCredentialNew = new PinCodeCredential();
				pinCodeCredentialNew.setPinCode(null);

				card.beginUpdates();
				card.setPinCode(pinCodeCredentialNew);
				card.applyUpdates();
			} catch (final CardException e) {
				if (card.isEditMode()) {
					card.cancelUpdates();
				}
				throw new RppsException(e);
			}
			return ICardRegistry.getInstance().getICardByCUID(getHandler(), card.getCUID());
		} catch (final Exception e) {
			log.error(e, e);
			throw new RppsException(e);
		}
	}

	public ICardTO resetPinCode(final String cuid, final String oldPinCode) throws RppsException {
		ICard card = null;
		byte[] oldPinCode_ = null;
		try {
			card = getICardByCUID(new CUID(cuid));

			if (card == null) {
				throw new RppsException("Can't find card with CUID " + cuid);
			}

			if (!(card instanceof IPersonalInformationCard)) {
				new RppsException("This card is not personal");
			}

			CardInformationRPPS cardInformation = null;

			if (oldPinCode == null) {
				cardInformation = getCardInformationByCuid(new CUID(cuid));
				if (cardInformation.getCardCredential() != null
						&& IRppsService.IPinCodeCredential.equals(cardInformation.getCardCredential().getCredentialType())) {
					oldPinCode_ = cardInformation.getCardCredential().getPinCode();
				}
			} else {
				oldPinCode_ = oldPinCode.getBytes("UTF-8");
			}

			card = resetPinCode((IPersonalInformationCard) card, oldPinCode_);

			if (cardInformation != null) {
				clearCardCredential(card.getCUID());
			}

			return convertICard_To_ICardTO(card, false, null);
		} catch (final Exception e) {
			log.error(e, e);
			throw new RppsException(e);
		}
	}

	private void saveUsersChoiceOfOptionalClaims(final CUID cuid, final URI site, final List claims) throws UserProfileException {
		ICardUsageManager cardUsageManager = null;
		try {
			final Set optionalClaims = new HashSet(claims);
			cardUsageManager = ICardRegistry.getInstance().getUserProfileService().getCardUsageManager(getHandler());
			cardUsageManager.setOptionalClaims(cuid, site, optionalClaims);
		} catch (final Exception e) {
			log.error(e, e);
			throw new RppsException(e);
		} finally {
			if (cardUsageManager != null) {
				cardUsageManager.close();
			}
		}
	}

	protected boolean setCardCredential(final CUID cuid, final CardCredential cardCredential) throws RppsException {
		try {
			final CardInformationRPPS cardInformation = new CardInformationRPPS(ICardRegistry.getInstance().getUserProfileService().setCardCredential(
					getHandler(), cuid, cardCredential), cuid);
			if (cardInformation.getCardCredential() != null && cardCredential != null || cardInformation.getCardCredential() == null && cardCredential == null) {
				return true;
			}
		} catch (final Exception e) {
			log.error(e, e);
			throw new RppsException(e);
		}
		return false;
	}

	public boolean setCardCredential(final String cuid, final UITokenServiceCredentialTO tokenCredential) throws RppsException {
		try {
			ICard card = getICardByCUID(new CUID(cuid));

			if (card == null) {
				throw new RppsException("Can't find card with CUID " + cuid);
			}

			final ICredential credential = ConvertHelper.convertUITokenServiceCredentialTO_to_ICredentials(card, tokenCredential);

			final ICardProvider provider = card.getProvider();

			card = null;

			if (credential != null) {
				card = provider.getICardByCUID(getHandler(), new CUID(cuid), credential);
			}

			if (card != null) {
				final CardCredential cardCredential = ConvertHelper.convertUITokenServiceCredentialTO_to_CardCredential(tokenCredential);
				final boolean result = setCardCredential(new CUID(cuid), cardCredential);
				if (result) {
					return true;
				} else {
					return false;
				}
			}
			throw new RppsException("Didn't set card credential.");

		} catch (final Exception e) {
			log.error(e, e);
			throw new RppsException(e.getMessage(), e);
		}
	}

	/**
	 * @param claim
	 * @param value
	 * @throws RppsException
	 */
	private void setComplexValues(final IClaim claim, final PropertyTO value, final ICardUsageManager cardUsageManager) throws Exception {
		if (claim != null && claim instanceof IComplexClaim) {
			final int size = value.getComplexValue().length;
			for (int index = 0; index < size; index++) {
				final PropertyTO propTO = value.getComplexValue()[index];
				IClaim claimValue = ((IComplexClaim) claim).getClaim(propTO.getClaimType());
				if (claimValue == null) {
					claimValue = ((IComplexClaim) claim).createClaim(propTO.getClaimType());
				}
				if (propTO.getValueType() != null) {
					setSimpleValues(claimValue, propTO, cardUsageManager);
				} else {
					setComplexValues(claimValue, propTO, cardUsageManager);
				}
			}
		}
	}

	/**
	 * Gets list of random word
	 * 
	 * @param count
	 * @return list of word
	 */
	/*
	 * public String[] getRandomWordList(int count) throws RppsException { try { if (listword == null) { listword = new ArrayList(); String dictionaryFile = System.getProperty(WebKeys.HIGGINS_RPPS_WEB_DICTIONARY_PROPERTY) != null ? System.getProperty(WebKeys.HIGGINS_RPPS_WEB_DICTIONARY_PROPERTY) : WebKeys.HIGGINS_RPPS_WEB_DICTIONARY_FILEPATH; File file = new File(dictionaryFile); if (!file.exists()) throw new FileNotFoundException("File " + dictionaryFile + " does not find!"); if (!file.canRead()) throw new IOException("File " + dictionaryFile + " can not be read!"); try { FileReader fr = new FileReader(file); LineNumberReader lnr = new LineNumberReader(fr); String s = ""; while ((s = lnr.readLine()) != null) { if (s.trim().length() > 3) listword.add(s.trim()); } lnr.close(); fr.close(); } catch (IOException e) { e.printStackTrace(); throw new IOException("File " + dictionaryFile + " can
	 * not be read!"); } } if (listword.size() < count) throw new IOException("Dictionary file does not contain enough words"); ArrayList result = new ArrayList(); Random random = new Random(new Date().getTime()); for (int i = 0; i < count; i++) { int index = 0; do { index = random.nextInt(listword.size()); } while (result.contains(listword.get(index)) || (((String) listword.get(index)).length() < 4)); result.add(listword.get(index)); } String[] rr = new String[count]; result.toArray(rr); return rr; } catch (Exception e) { log.error(e,e); throw new RppsException(e); } }
	 */

	public void setHandler(final CallbackHandler handler) {
		this.handler = handler;
	}

	// protected static ArrayList listword = null;

	/**
	 * @param newPassword
	 * @throws UserProfileException
	 */
	public void setNewPrivateUserPassword(final String newPassword) throws UserProfileException {
		try {
			ICardRegistry.getInstance().getUserProfileService().setNewPassword(getHandler(), newPassword);
		} catch (final Exception e) {
			log.error(e, e);
			throw new RppsException(e);
		}
	}

	protected ICard setPinCode(final IPersonalInformationCard card, final byte[] pinCode) throws RppsException {
		try {
			PinCodeCredential pinCodeCredential = new PinCodeCredential();
			try {
				card.beginUpdates();
				pinCodeCredential.setPinCode(pinCode);
				card.setPinCode(pinCodeCredential);
				card.applyUpdates();
			} catch (final CardException e) {
				if (card.isEditMode()) {
					card.cancelUpdates();
				}
				throw new RppsException(e);
			}
			pinCodeCredential = new PinCodeCredential();
			pinCodeCredential.setPinCode(pinCode);

			return card.getProvider().getICardByCUID(getHandler(), card.getCUID(), pinCodeCredential);
		} catch (final Exception e) {
			log.error(e, e);
			throw new RppsException(e);
		}
	}

	/**
	 * Get a icard template for creating personal card
	 * 
	 * @param handler the callback handler to be used to request user's identity
	 * @return a template
	 */
	/*protected ICardTemplate getPersonalICardCreationTemplate(CallbackHandler handler) throws RppsException {
		for (Iterator it = ICardRegistry.getInstance().getICardProviders(); it.hasNext();) {
			ICardProvider provider = (ICardProvider) it.next();
			try {
				// check if current provider supports creating cards
				if (provider.getCardCreationTemplates(handler) != null) {
					ICardTemplate[] templates = provider.getCardCreationTemplates(handler);
					for (int i = 0; i < templates.length; i++) {
						if (templates[i].getCardType().isAssignableFrom(IPersonalInformationCard.class)) {
							return templates[i];
						}
					}
				}
			} catch (Exception e) {
				log.error(e, e);
				throw new RppsException(e);
			}
		}
		return null;
	}*/

	public ICardTO setPinCode(final String cuid, final String pinCode) throws RppsException {
		ICard card = null;
		try {
			card = ICardRegistry.getInstance().getICardByCUID(getHandler(), cuid);
			if (!(card instanceof IPersonalInformationCard)) {
				new RppsException("This card is not personal");
			}

			card = setPinCode((IPersonalInformationCard) card, pinCode.getBytes("UTF-8"));
			return convertICard_To_ICardTO(card, false, null);
		} catch (final Exception e) {
			log.error(e, e);
			throw new RppsException(e);
		}
	}

	/**
	 * @param claim
	 * @param value
	 * @throws Exception
	 */
	private void setSimpleValues(final IClaim claim, final PropertyTO value, final ICardUsageManager cardUsageManager) throws Exception {
		if (claim != null && claim instanceof ISimpleClaim) {
			if (((ISimpleClaimType) claim.getType()).getDataType().getName().equals(value.getValueType())) {
				final List newValues = new ArrayList();
				for (int index = 0; index < value.getValue().length; index++) {
					newValues.add(value.getValue()[index]);
					try {
						addClaimValuesMRU(claim.getType().getType(), value.getValue()[index], cardUsageManager);
					} catch (final Exception e) {
						log.error(e, e);
					}

				}
				((ISimpleClaim) claim).setValues(newValues);
			}
		}
	}

	/**
	 * Update claim across cards.
	 */
	public ICardUpdateResponseTO[] updateClaimsAcrossCards(final String[] claimTypes, final String[] claimValues, final ICardUpdateTO[] cardUpdate)
			throws RppsException {

		ICard card = null;
		if (cardUpdate == null) {
			return new ICardUpdateResponseTO[0];
		}

		final ICardUpdateResponseTO[] result = new ICardUpdateResponseTO[cardUpdate.length];
		for (int i = 0; i < cardUpdate.length; i++) {
			try {

				card = getICardByCUID(new CUID(cardUpdate[i].getCuid()));

				if (card instanceof IPersonalInformationCard) {

					final CardInformationRPPS cardInformation = getCardInformationByCuid(card.getCUID());
					ICredential credential = null;
					if (cardInformation.getCardCredential() != null) {
						credential = ConvertHelper.convertCardCredential_to_ICredential(card, cardInformation.getCardCredential());
					} else {
						if (cardUpdate[i].getTokenServiceCredential() != null) {
							credential = ConvertHelper.convertUITokenServiceCredentialTO_to_ICredentials(card, cardUpdate[i].getTokenServiceCredential());
						}
					}

					if (credential != null) {
						card = card.getProvider().getICardByCUID(getHandler(), card.getCUID(), credential);
					}

					card.beginUpdates();

					for (int index = 0; index < claimTypes.length; index++) {
						final String claimType = claimTypes[index];
						final String claimValue = claimValues[index];
						IClaim claim = card.getClaim(claimType);

						if (claim == null) {
							claim = card.createClaim(claimType);
						}

						if (claim != null && claim instanceof ISimpleClaim) {
							final List newValues = new ArrayList();
							newValues.add(claimValue);
							((ISimpleClaim) claim).setValues(newValues);
							try {
								addClaimValuesMRU(claimType, claimValue);
							} catch (final Exception e) {
								log.error(e, e);
							}
						}
					}
					card.applyUpdates();
					result[i] = new ICardUpdateResponseTO(card.getCUID().toString());
				}
			} catch (final Exception e) {
				log.error(e, e);
				result[i] = new ICardUpdateResponseTO(ICardUpdateResponseTO.unknownError, e.getMessage());
				try {
					if (card != null && card.isEditMode()) {
						card.cancelUpdates();
					}
				} catch (final Throwable tr) {
					log.error(tr, tr);
				}
			} finally {
			}
		}
		return result;
	}

	/**
	 * Update card.
	 * 
	 * @param userId the id to identify the user.
	 * @param password the password to identify the user.
	 * @param icardTO the card transfer object.
	 * @return the card transfer object.
	 */
	public ICardTO updateICard(final ICardTO icardTO) throws RppsException {
		ICardUsageManager cardUsageManager = null;
		try {
			cardUsageManager = ICardRegistry.getInstance().getUserProfileService().getCardUsageManager(getHandler());

			return convertICard_To_ICardTO(updateICard(icardTO, cardUsageManager), false, null);
		} catch (final Exception e) {
			log.error(e, e);
			throw new RppsException(e);
		} finally {
			if (cardUsageManager != null) {
				try {
					cardUsageManager.close();
				} catch (final UserProfileException e) {
					log.error(e, e);
				}
			}
		}
	}

	private ICard updateICard(final ICardTO icardTO, final ICardUsageManager cardUsageManager) throws RppsException {

		ICard card = null;
		try {
			card = getICardByCUID(new CUID(icardTO.getCuid()));

			final CardInformationRPPS cardInformation = getCardInformationByCuid(card.getCUID(), cardUsageManager);
			ICredential credential = null;
			if (cardInformation.getCardCredential() != null) {
				credential = ConvertHelper.convertCardCredential_to_ICredential(card, cardInformation.getCardCredential());
			} else {
				if (icardTO.getTokenServiceCredentials() != null && icardTO.getTokenServiceCredentials().length != 0) {
					credential = ConvertHelper.convertUITokenServiceCredentialTO_to_ICredentials(card, icardTO.getTokenServiceCredentials()[0]);
				}
			}

			if (credential != null) {
				card = card.getProvider().getICardByCUID(getHandler(), card.getCUID(), credential);
			}

			card.beginUpdates();
			if (!RppsHelper.equalsObject(card.getName(), icardTO.getName())) {
				card.setName(icardTO.getName());
			}
			try {
				if (!RppsHelper.equalsObject(card.getIssuerName(), icardTO.getIssuerName())) {
					card.setIssuerName(icardTO.getIssuerName());
				}
			} catch (final Exception exc) {
				log.error(exc, exc);
			}

			if (!RppsHelper.equalsObject(card.getImageType(), icardTO.getImageType()) || !RppsHelper.equalsObject(card.getImage(), icardTO.getImage())) {
				card.setImage(icardTO.getImage(), icardTO.getImageType());
			}

			if (card instanceof IPersonalInformationCard) {
				try {
					if (!RppsHelper.equalsObject(card.getTimeExpires(), icardTO.getExpiredTime())) {
						card.setTimeExpires(icardTO.getExpiredTime());
					}
				} catch (final Exception exc) {
					log.error(exc, exc);
				}

				if (icardTO.getProperties() != null) {
					final int size = icardTO.getProperties().length;
					for (int index = 0; index < size; index++) {
						final PropertyTO propTO = icardTO.getProperties()[index];
						IClaim claim = card.getClaim(propTO.getClaimType());
						if (claim == null) {
							claim = card.createClaim(propTO.getClaimType());
						}
						if (propTO.getComplexValue() == null) {
							setSimpleValues(claim, propTO, cardUsageManager);
						} else {
							setComplexValues(claim, propTO, cardUsageManager);
						}
					}
				}
			}

			card.applyUpdates();

			card = getICardByCUID(new CUID(icardTO.getCuid()));
			return card;

		} catch (final Exception e) {
			log.error(e, e);
			try {
				if (card != null && card.isEditMode()) {
					card.cancelUpdates();
				}
			} catch (final Throwable tr) {
			}
			throw new RppsException(e);
		} finally {
		}
	}

	/**
	 * Update card.
	 * 
	 * @param userId the id to identify the user.
	 * @param password the password to identify the user.
	 * @param icardTO the card transfer object.
	 * @return the card transfer object.
	 */
	public ICardUpdateResponseTO[] updateICards(final ICardTO[] icardTOList) throws RppsException {
		ICardUsageManager cardUsageManager = null;
		try {
			final ICardUpdateResponseTO[] result = new ICardUpdateResponseTO[icardTOList.length];
			cardUsageManager = ICardRegistry.getInstance().getUserProfileService().getCardUsageManager(getHandler());
			for (int i = 0; i < icardTOList.length; i++) {
				try {
					final ICard card = updateICard(icardTOList[i], cardUsageManager);
					result[i] = new ICardUpdateResponseTO(card.getCUID().toString());
				} catch (final Exception e) {
					log.error(e, e);
					result[i] = new ICardUpdateResponseTO(ICardUpdateResponseTO.unknownError, e.getMessage());
				}
			}
			return result;
		} catch (final Exception e) {
			log.error(e, e);
			throw new RppsException(e);
		} finally {
			if (cardUsageManager != null) {
				try {
					cardUsageManager.close();
				} catch (final UserProfileException e) {
					log.error(e, e);
				}
			}
		}
	}

	/**
	 * Checks availability of user profile by userIdentifier
	 * 
	 * @param userIdentifier
	 * @return true if user profile exists otherwise false
	 * @throws RppsException
	 */
	public boolean userIdentifierExists(final String userIdentifier) {
		try {
			return ICardRegistry.getInstance().getUserProfileService().userIdentifierExists(userIdentifier);
		} catch (final UserProfileException e) {
		}
		return false;
	}
}