/**
 * Copyright (c) 2007 Parity Communications, Inc. 
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     Sergei Yakovlev - initial API and implementation
 *     Artem Verkhovets - initial API and implementation
 */

package org.eclipse.higgins.rpps.core;

import java.rmi.AccessException;
import java.rmi.RemoteException;

import org.eclipse.higgins.icard.ICardTemplate;
import org.eclipse.higgins.icard.provider.cardspace.common.STSFaultException;
import org.eclipse.higgins.rpps.core.tobj.CardInformationTO;
import org.eclipse.higgins.rpps.core.tobj.CardUsageTO;
import org.eclipse.higgins.rpps.core.tobj.CategoryTO;
import org.eclipse.higgins.rpps.core.tobj.ExtraTokenResponseTO;
import org.eclipse.higgins.rpps.core.tobj.FormatDescriptorTO;
import org.eclipse.higgins.rpps.core.tobj.ICardResponseTO;
import org.eclipse.higgins.rpps.core.tobj.ICardTO;
import org.eclipse.higgins.rpps.core.tobj.ICardTemplateTO;
import org.eclipse.higgins.rpps.core.tobj.ICardUpdateResponseTO;
import org.eclipse.higgins.rpps.core.tobj.ICardUpdateTO;
import org.eclipse.higgins.rpps.core.tobj.ICardsAndCategoryTO;
import org.eclipse.higgins.rpps.core.tobj.MapTO;
import org.eclipse.higgins.rpps.core.tobj.RPEndPointTO;
import org.eclipse.higgins.rpps.core.tobj.ResponseMessage;
import org.eclipse.higgins.rpps.core.tobj.TokenResponseTO;
import org.eclipse.higgins.rpps.core.tobj.UIResponseCardTO;
import org.eclipse.higgins.rpps.core.tobj.UIResponseTO;
import org.eclipse.higgins.rpps.core.tobj.UITokenServiceCredentialTO;
import org.eclipse.higgins.rpps.core.tobj.UserProfileTO;
import org.eclipse.higgins.userprofile.UserProfileException;

/**
 * The object used for executing Higgins API and returning the results it produces.
 * 
 * @author Sergei Yakovlev
 * @author Artem Verkhovets
 */
public interface IRppsService {

	public final static String ITSUsernamePasswordCredential = "ITSUsernamePasswordCredential";

	public final static String ITSSelfIssuedCredential = "ITSSelfIssuedCredential";

	public final static String ITSKerberosV5Credential = "ITSKerberosV5Credential";

	public final static String ITSX509V3Credential = "ITSX509V3Credential";

	public final static String IPinCodeCredential = "IPinCodeCredential";

	public final static String ITSSelfIssuedNonPinCredential = "ITSSelfIssuedNonPinCredential";

	public final static String IUsernamePasswordCredential = "IUsernamePasswordCredential";

	public final static String IPasswordCredential = "IPasswordCredential";

	/**
	 * Creates a new card's category
	 * 
	 * @param categoryTO
	 *            the card category information
	 * @return Returns saved card category object
	 * @throws RppsException
	 */
	public CategoryTO addCategory(final CategoryTO categoryTO) throws RppsException;

	/**
	 * Adds most recently used record
	 * 
	 * @param claimType
	 *            claim's type
	 * @param claimValue
	 *            claim's value
	 * @throws RppsException
	 */
	public void addClaimValuesMRU(final String claimType, final String claimValue) throws RppsException;

	/**
	 * Creates a new user profile
	 * 
	 * @param userProfile
	 *            the userProfile profile information as UserProfileTO
	 * @return Returns <code>ResponseMessage</code> with <code>code</code> field equals <code>0</code> if profile was
	 *         added. Otherwise returns <code>ResponseMessage</code> with error code or throws an RemoteException
	 * @throws RemoteException
	 */
	public ResponseMessage addUserProfile(UserProfileTO userProfile, String password) throws RppsException;

	/**
	 * Create relation between card and url
	 * 
	 * @param cuid
	 *            the icard's CUID
	 * @param webForm
	 *            the object which represents logging form on html page
	 * @return
	 * @return Returns <code>true</code> if relation was added. Otherwise <code>false</code>
	 * @throws RppsException
	 * @throws UserProfileException
	 */
	public boolean addRPEndPointCardRelation(String cuid, RPEndPointTO rpEndPointTO) throws RppsException,
			UserProfileException;

	/**
	 * Create relation between card and url
	 * 
	 * @param cuid
	 *            the icard's CUID
	 * @param webForm
	 *            the object which represents logging form on html page
	 * @return
	 * @return Returns <code>true</code> if relation was added. Otherwise <code>false</code>
	 * @throws RppsException
	 * @throws UserProfileException
	 * @deprecated
	 */
	public boolean addUrlCardRelation(String cuid, String url) throws RppsException, UserProfileException;

	/**
	 * Checks whether <code>RppsService</code> can recognize data format of the provided input stream. Some formats
	 * require authentication information in order to process input data. This method should be used prior to actual
	 * attempt to import data from the input stream in order to retrieve an information about data format and required
	 * authentication data to process the data successfully.
	 * 
	 * @param crd
	 *            context of file as byte array
	 * @return the information about input stream data format if it was recognized successfully and <code>null</code>
	 *         otherwise.
	 * @throws RppsException
	 */
	public FormatDescriptorTO checkImportStream(byte[] crd) throws RppsException;

	/**
	 * Deletes saved authentication card information of the specified card
	 * 
	 * @param cuid
	 *            the icard's CUID
	 * @return Returns updated card object
	 * @throws RppsException
	 */
	public ICardTO clearCardCredential(String cuid) throws RppsException;

	/**
	 * Deletes card history of the specified card on all URLs unless it is checked as "Use Always".
	 * 
	 * @param cuid
	 *            the icard's CUID
	 * @return Returns <code>true</code> if card history was cleared. Otherwise <code>false</code>
	 * @throws RppsException
	 */
	public boolean clearCardHistory(String cuid) throws RppsException;

	/**
	 * Creates duplicate of personal card only with new card's name
	 * 
	 * @param cardName
	 *            new card's name
	 * @param cuid
	 *            the old icard's cuid
	 * @param pinCode
	 *            pinCode for retrieve claims of personal card if that is pin protected. Otherwise <code>null</code>
	 * @return Returns updated card object
	 * @throws RppsException
	 */
	public ICardTO createDuplicateICard(String cardName, String cuid, String pinCode) throws RppsException;

	/**
	 * Creates a new card by template
	 * 
	 * @param template
	 *            the icard's template
	 * @return a new card object
	 * @throws RppsException
	 */
	public ICardTO createICard(ICardTemplateTO template) throws RppsException;

	/**
	 * Creates a new card for HBX
	 * 
	 * @param cardname
	 *            the card's name
	 * @return a new card's cuid
	 * @throws RppsException
	 * @deprecated
	 */
	public String createICardFromHBX(String cardname) throws RppsException;

	/**
	 * Deletes all user's card.
	 * 
	 * @throws RppsException
	 */
	public void deleteAllICard() throws RppsException;

	/**
	 * Delete specified record of card history.
	 * 
	 * @param cuid
	 *            the icard's CUID
	 * @param webForm
	 *            the object which represents logging form on html page
	 * @return Returns <code>true</code> if card history record was deleted. Otherwise <code>false</code>
	 * @throws RppsException
	 */
	public boolean deleteCardHistoryRecord(String cuid, RPEndPointTO rpEndPointTO) throws RppsException;

	/**
	 * Delete specified record of card history.
	 * 
	 * @param cuid
	 *            the icard's CUID
	 * @param webForm
	 *            the object which represents logging form on html page
	 * @return Returns <code>true</code> if card history record was deleted. Otherwise <code>false</code>
	 * @throws RppsException
	 * @deprecated
	 */
	public boolean deleteCardHistoryRecord(String cuid, String url) throws RppsException;

	/**
	 * Delete the card's category
	 * 
	 * @param categoryId
	 *            category identifier
	 * @throws RppsException
	 */
	public void deleteCategory(String categoryId) throws RppsException;

	/**
	 * Deletes specified card.
	 * 
	 * @param cuid
	 *            the icard's CUID
	 * @return
	 * @throws RppsException
	 */
	public ICardUpdateResponseTO deleteICard(String cuid) throws RppsException;

	public void deleteOldPassword() throws UserProfileException;

	/**
	 * 
	 * @param cuid
	 * @param url
	 * @return
	 * @throws RppsException
	 * @deprecated
	 */
	public boolean deleteUrlCardRelation(String cuid, String url) throws RppsException;

	public boolean deleteRPEndPointCardRelation(String cuid, RPEndPointTO rpEndPointTO) throws RppsException;

	/**
	 * 
	 * @param cuid
	 * @return
	 * @throws RppsException
	 * @deprecated
	 */
	public boolean deleteUrlsCardsRelations(String cuid) throws RppsException;

	public boolean deleteRPEndPointsCardsRelations(String cuid) throws RppsException;

	/**
	 * Deletes user profile
	 * 
	 * @param userId
	 *            the id to identify the user.
	 * @param password
	 *            the password to identify the user.
	 * @throws RppsException
	 */
	public ResponseMessage deleteUserProfile() throws RppsException;

	public ICardTO editPinCode(String cuid, String oldPinCode, String newPinCode) throws RppsException;

	public byte[] exportICards(String formatID, String[] cards, UITokenServiceCredentialTO credential)
			throws RppsException;

	public CardUsageTO[] getCardHistory(String cuid, int startFrom, int capacity, String orderBy, String orderDirection);

	/**
	 * @param userId
	 * @param password
	 * @param cuid
	 * @return
	 * @throws RppsException
	 */
	public CardInformationTO getCardInformation(String cuid, int size) throws RppsException;

	public CategoryTO[] getCategory() throws RppsException;

	public MapTO[] getClaimValuesMRU() throws RppsException;

	/**
	 * Gets the global i-name registered to the Higgins service (e.g. "=parity").
	 * 
	 * @return the global i-name.
	 */
	public String getHigginsServiceGlobalIName();

	/**
	 * Gets a card by CUID.
	 * 
	 * @param userId
	 *            the id to identify the user.
	 * @param password
	 *            the password to identify the user.
	 * @param cuid
	 *            the card CUID.
	 * @return the card transfer object.
	 */
	public ICardTO getICardByCUID(String cuid) throws RppsException;

	/**
	 * Gets a card by CUID and Credential.
	 * 
	 * @param userId
	 *            the id to identify the user.
	 * @param password
	 *            the password to identify the user.
	 * @param cuid
	 *            the card CUID.
	 * @param tokenCredential
	 *            the tokenCredential transfer object
	 * @return the card transfer object.
	 */
	public ICardTO getICardByCUIDAndTokenServiceCredential(String cuid, UITokenServiceCredentialTO tokenCredential)
			throws RppsException, STSFaultException;

	/**
	 * 
	 * @param cuid
	 * @param policy
	 * @param typeofCredential
	 * @param credentialKey
	 * @param credentialValue
	 * @return
	 * @throws RppsException
	 * @deprecated
	 */
	public UIResponseCardTO getICardClaims(String cuid, String policy, String typeofCredential, String[] credentialKey,
			String[] credentialValue) throws RppsException;

	public UIResponseCardTO getICardClaims(String cuid, String policy, String typeofCredential, String[] credentialKey,
			String[] credentialValue, RPEndPointTO rpEndPointTO) throws RppsException;

	/**
	 * Return templates for creating cards
	 * 
	 * @param userId
	 *            the id to identify the user.
	 * @param password
	 *            the password to identify the user.
	 * @return a template list
	 */
	public ICardTemplateTO[] getICardCreationTemplate() throws RppsException;

	public ICardTemplate[] getICardCreationTemplateByProvider(String extID) throws RppsException;

	/**
	 * Gets the card and category lists of the specified user.
	 * 
	 * @param userId
	 *            the id to identify the user.
	 * @param password
	 *            the password to identify the user.
	 * @return the card list
	 */
	public ICardsAndCategoryTO getICardsAndCategoriesByUser() throws RppsException;

	/**
	 * Gets cards by CUIDs array.
	 * 
	 * @param userId
	 *            the id to identify the user.
	 * @param password
	 *            the password to identify the user.
	 * @param cuids
	 *            the card CUIDs.
	 * @return list of cards transfer object.
	 */
	public ICardTO[] getICardsByCUIDs(String[] cuids);

	public ICardTO[] getICardsByFormat(String formatID) throws RppsException;

	/**
	 * Gets the card list of the specified user.
	 * 
	 * @param userId
	 *            the id to identify the user.
	 * @param password
	 *            the password to identify the user.
	 * @return the card list
	 */
	public ICardResponseTO getICardsByUser();

	public FormatDescriptorTO[] getOutputFormats() throws RppsException;

	/**
	 * Web service operation. Makes xmlToken usage claim list.
	 * 
	 * @param userId
	 *            the id to identify the user.
	 * @param password
	 *            the password to identify the user.
	 * @param policy
	 *            the RP Agent's Relying Party Security Policy
	 * @param policytype
	 *            the RP Security Policy type
	 * @param sslCert
	 *            the SSL certificate
	 * @param user
	 *            the username to login to the STS
	 * @param pass
	 *            the password to login to the STS
	 * @param sels
	 *            the user-selected subset of the UIDs (handles) contained in the UserInterfaceResponse object returned
	 *            from an earlier invocation of 'userInterfaceRequest'
	 * @return the security token as a String
	 * @throws RppsException
	 */
	public TokenResponseTO getTokenByClaims(String sslCert, String[] claimName, String[] claimValue)
			throws RppsException;

	/**
	 * Web service operation. Makes security token usage saved users information
	 * 
	 * @param userId
	 *            the id to identify the user.
	 * @param password
	 *            the password to identify the user.
	 * @param policy
	 *            the RP Agent's Relying Party Security Policy
	 * @param policytype
	 *            the RP Security Policy type
	 * @param sslCert
	 *            the SSL certificate
	 * @param webForm
	 *            the object which represents logging form on html page
	 * @return the security token as a String
	 * @throws RemoteException
	 */
	public TokenResponseTO getTokenByRPEndPoint(String policy, String policytype, String sslCert,
			RPEndPointTO rpEndPointTO) throws RppsException;

	/**
	 * Web service operation. Makes security token usage saved users information
	 * 
	 * @param userId
	 *            the id to identify the user.
	 * @param password
	 *            the password to identify the user.
	 * @param policy
	 *            the RP Agent's Relying Party Security Policy
	 * @param policytype
	 *            the RP Security Policy type
	 * @param sslCert
	 *            the SSL certificate
	 * @param webForm
	 *            the object which represents logging form on html page
	 * @return the security token as a String
	 * @throws RemoteException
	 * @deprecated
	 */
	public TokenResponseTO getTokenByUrl(String policy, String policytype, String sslCert, String url)
			throws RppsException;

	/**
	 * Web service operation. Makes security token usage saved users information
	 * 
	 * @param userId
	 *            the id to identify the user.
	 * @param password
	 *            the password to identify the user.
	 * @param policy
	 *            the RP Agent's Relying Party Security Policy
	 * @param policytype
	 *            the RP Security Policy type
	 * @param sslCert
	 *            the SSL certificate
	 * @param webForm
	 *            the object which represents logging form on html page
	 * @return the security token as a String with extra card's information
	 * @throws RemoteException
	 * @deprecated
	 */
	public ExtraTokenResponseTO getTokenExtraByUrl(String policy, String policytype, String sslCert, String url)
			throws RppsException;

	/**
	 * Web service operation. Makes security token usage saved users information
	 * 
	 * @param userId
	 *            the id to identify the user.
	 * @param password
	 *            the password to identify the user.
	 * @param policy
	 *            the RP Agent's Relying Party Security Policy
	 * @param policytype
	 *            the RP Security Policy type
	 * @param sslCert
	 *            the SSL certificate
	 * @param webForm
	 *            the object which represents logging form on html page
	 * @return the security token as a String with extra card's information
	 * @throws RemoteException
	 */
	public ExtraTokenResponseTO getTokenExtraByRPEndPoint(String policy, String policytype, String sslCert,
			RPEndPointTO rpEndPointTO) throws RppsException;

	/**
	 * Web service operation. Given the Relying Party Agent (aka 'Requester')'s policy, identity information about the
	 * requester, and the set of one or more selections that the user has just made in the ISS Web UI (see
	 * userInterfaceRequest), AND presuming that the protocol involved in interacting with the RPA requires a security
	 * token, request the token that corresponds to the user's selection(s).
	 * 
	 * @param userId
	 *            the id to identify the user.
	 * @param password
	 *            the password to identify the user.
	 * @param policy
	 *            the RP Agent's Relying Party Security Policy
	 * @param policytype
	 *            the RP Security Policy type
	 * @param sslCert
	 *            the SSL certificate
	 * @param cuids
	 *            the user-selected subset of the UIDs (handles) contained in the UserInterfaceResponse object returned
	 *            from an earlier invocation of 'userInterfaceRequest'
	 * @param typeofCredential
	 *            the type of selected credential
	 * @param credentialKey
	 *            the names of fields of selected credential
	 * @param credentialValue
	 *            the value of fields of selected credential
	 * @return the security token as a String
	 * @throws RemoteException
	 * @deprecated
	 */
	public TokenResponseTO getTokenObject(String policy, String policytype, String sslCert, String[] cuids,
			String typeofCredential, String[] credentialKey, String[] credentialValue) throws RppsException;

	/**
	 * Web service operation. Given the Relying Party Agent (aka 'Requester')'s policy, identity information about the
	 * requester, and the set of one or more selections that the user has just made in the ISS Web UI (see
	 * userInterfaceRequest), AND presuming that the protocol involved in interacting with the RPA requires a security
	 * token, request the token that corresponds to the user's selection(s).
	 * 
	 * @param userId
	 *            the id to identify the user.
	 * @param password
	 *            the password to identify the user.
	 * @param policy
	 *            the RP Agent's Relying Party Security Policy
	 * @param policytype
	 *            the RP Security Policy type
	 * @param sslCert
	 *            the SSL certificate
	 * @param cuids
	 *            the user-selected subset of the UIDs (handles) contained in the UserInterfaceResponse object returned
	 *            from an earlier invocation of 'userInterfaceRequest'
	 * @param typeofCredential
	 *            the type of selected credential
	 * @param credentialKey
	 *            the names of fields of selected credential
	 * @param credentialValue
	 *            the value of fields of selected credential
	 * @return the security token as a String
	 * @throws RemoteException
	 * 
	 */
	public TokenResponseTO getTokenObject(String policy, String policytype, String sslCert, String[] cuids,
			String typeofCredential, String[] credentialKey, String[] credentialValue, RPEndPointTO rpEndPointTO)
			throws RppsException;

	/**
	 * This method takes as input a policy---for example, one that has been retrieved from the Relying Party Agent
	 * (called here the 'Requester')---as well as a description of the Requester. The UserInterfaceResponse object
	 * should contain all information necessary to display both (a) a user interface which shares the same card
	 * metaphor, sequences of experiences and decision points (the ceremony) of CardSpace and (b) other interface
	 * paradigms to be developed in the future
	 * 
	 * @param userId
	 *            the id to identify the user.
	 * @param password
	 *            the password to identify the user.
	 * @param webForm
	 *            the object which represents logging form on html page
	 * @param target
	 * @param sslCert
	 *            the SSL certificate
	 * @param policyType
	 *            the RP Security Policy type
	 * @param policy
	 *            the RP Agent's Relying Party Security Policy
	 * @return the UIResponseTO transfer object.
	 * @throws RemoteException
	 * @deprecated
	 */
	public UIResponseTO getUserInterface(String url, String target, String sslCert, String policytype, String policy)
			throws RppsException;

	/**
	 * This method takes as input a policy---for example, one that has been retrieved from the Relying Party Agent
	 * (called here the 'Requester')---as well as a description of the Requester. The UserInterfaceResponse object
	 * should contain all information necessary to display both (a) a user interface which shares the same card
	 * metaphor, sequences of experiences and decision points (the ceremony) of CardSpace and (b) other interface
	 * paradigms to be developed in the future
	 * 
	 * @param userId
	 *            the id to identify the user.
	 * @param password
	 *            the password to identify the user.
	 * @param webForm
	 *            the object which represents logging form on html page
	 * @param target
	 * @param sslCert
	 *            the SSL certificate
	 * @param policyType
	 *            the RP Security Policy type
	 * @param policy
	 *            the RP Agent's Relying Party Security Policy
	 * @return the UIResponseTO transfer object.
	 * @throws RemoteException
	 */
	public UIResponseTO getUserInterface(RPEndPointTO rpEndPointTO, String target, String sslCert, String policytype,
			String policy) throws RppsException;

	public UserProfileTO getUserProfile() throws RppsException, AccessException;

	/**
	 * Imports card from a file.
	 * 
	 * @param userId
	 *            the id to identify the user.
	 * @param password
	 *            the password to identify the user.
	 * @param crd
	 *            the card as a byte array.
	 */
	public ICardResponseTO importICards(byte[] crd) throws RppsException;

	public ICardResponseTO importICards(byte[] crd, String formatID, UITokenServiceCredentialTO credential)
			throws RppsException;

	public ResponseMessage modifyCategories(CategoryTO[] categoriesTO) throws RppsException;

	public CategoryTO modifyCategory(CategoryTO categoryTO) throws RppsException;

	/**
	 * Update user profile
	 * 
	 * @param userId
	 *            the id to identify the user.
	 * @param password
	 *            the password to identify the user.
	 * @param userProfile
	 *            user profile transfer object
	 * @return user profile transfer object
	 * @throws RppsException
	 */
	public UserProfileTO modifyUserProfile(UserProfileTO userProfile) throws RppsException;

	public void removeClaimValuesMRU(String claimType, String claimValue) throws RppsException;

	public ICardTO resetPinCode(String cuid, String oldPinCode) throws RppsException;

	public boolean setCardCredential(String cuid, UITokenServiceCredentialTO tokenCredential) throws RppsException;

	public void setNewPrivateUserPassword(String newPassword) throws UserProfileException;

	public ICardTO setPinCode(String cuid, String pinCode) throws RppsException;

	public ICardUpdateResponseTO[] updateClaimsAcrossCards(String[] claimTypes, String[] claimValues,
			ICardUpdateTO[] cardUpdate) throws RppsException;

	/**
	 * Update card.
	 * 
	 * @param userId
	 *            the id to identify the user.
	 * @param password
	 *            the password to identify the user.
	 * @param icardTO
	 *            the card transfer object.
	 * @return the card transfer object.
	 */
	public ICardTO updateICard(ICardTO icardTO) throws RppsException;

	public ICardUpdateResponseTO[] updateICards(ICardTO[] icardTOList) throws RppsException;

	/**
	 * Checks availability of user profile by userIdentifier
	 * 
	 * @param userIdentifier
	 * @return true if user profile exists otherwise false
	 * @throws RppsException
	 */
	public boolean userIdentifierExists(String userIdentifier);
}
