/**
 * Copyright (c) 2007 Parity Communications, Inc. 
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     Sergei Yakovlev - initial API and implementation
 *     Artem Verkhovets - initial API and implementation
 */

package org.eclipse.higgins.rpps.core;

import java.rmi.AccessException;

import org.eclipse.higgins.icard.ICardTemplate;
import org.eclipse.higgins.icard.provider.cardspace.common.STSFaultException;
import org.eclipse.higgins.rpps.core.tobj.CardInformationTO;
import org.eclipse.higgins.rpps.core.tobj.CardUsageTO;
import org.eclipse.higgins.rpps.core.tobj.CategoryTO;
import org.eclipse.higgins.rpps.core.tobj.ExtraTokenResponseTO;
import org.eclipse.higgins.rpps.core.tobj.FormatDescriptorTO;
import org.eclipse.higgins.rpps.core.tobj.ICardResponseTO;
import org.eclipse.higgins.rpps.core.tobj.ICardTO;
import org.eclipse.higgins.rpps.core.tobj.ICardTemplateTO;
import org.eclipse.higgins.rpps.core.tobj.ICardUpdateResponseTO;
import org.eclipse.higgins.rpps.core.tobj.ICardUpdateTO;
import org.eclipse.higgins.rpps.core.tobj.ICardsAndCategoryTO;
import org.eclipse.higgins.rpps.core.tobj.MapTO;
import org.eclipse.higgins.rpps.core.tobj.RPEndPointTO;
import org.eclipse.higgins.rpps.core.tobj.ResponseMessage;
import org.eclipse.higgins.rpps.core.tobj.TokenResponseTO;
import org.eclipse.higgins.rpps.core.tobj.UIResponseCardTO;
import org.eclipse.higgins.rpps.core.tobj.UIResponseTO;
import org.eclipse.higgins.rpps.core.tobj.UITokenServiceCredentialTO;
import org.eclipse.higgins.rpps.core.tobj.UserProfileTO;
import org.eclipse.higgins.userprofile.UserProfileException;
import org.eclipse.higgins.userprofile.entity.Captcha;

/**
 * The object used for executing Higgins API and returning the results it produces.
 * 
 * @author Sergei Yakovlev
 * @author Artem Verkhovets
 */
public interface IRppsService {

	String ITSUsernamePasswordCredential = "ITSUsernamePasswordCredential";

	String ITSSelfIssuedCredential = "ITSSelfIssuedCredential";

	String ITSKerberosV5Credential = "ITSKerberosV5Credential";

	String ITSX509V3Credential = "ITSX509V3Credential";

	String IPinCodeCredential = "IPinCodeCredential";

	String ITSSelfIssuedNonPinCredential = "ITSSelfIssuedNonPinCredential";

	String IUsernamePasswordCredential = "IUsernamePasswordCredential";

	String IPasswordCredential = "IPasswordCredential";

	/**
	 * Creates a new card's category.
	 * 
	 * @param categoryTO
	 *            the card category information
	 * @return Returns saved card category object
	 * @throws RppsException
	 */
	CategoryTO addCategory(CategoryTO categoryTO) throws RppsException;

	/**
	 * Adds most recently used record.
	 * 
	 * @param claimType
	 *            claim's type
	 * @param claimValue
	 *            claim's value
	 * @throws RppsException
	 */
	void addClaimValuesMRU(String claimType, String claimValue) throws RppsException;

	/**
	 * Creates a new user profile.
	 * 
	 * @param userProfile
	 *            the userProfile profile information as UserProfileTO
	 * @param password
	 *            the password to identify the user.
	 * @return Returns <code>ResponseMessage</code> with <code>code</code> field equals <code>0</code> if profile was
	 *         added. Otherwise returns <code>ResponseMessage</code> with error code or throws an RppsException
	 * @throws RppsException
	 */
	ResponseMessage addUserProfile(UserProfileTO userProfile, String password) throws RppsException;

	/**
	 * Create relation between card and relying party.
	 * 
	 * @param cuid
	 *            the icard's CUID
	 * @param rpEndPointTO
	 *            the object which represents logging form on relying party
	 * @return Returns <code>true</code> if relation was added. Otherwise <code>false</code>
	 * @throws RppsException
	 * @throws UserProfileException
	 */
	boolean addRPEndPointCardRelation(String cuid, RPEndPointTO rpEndPointTO) throws RppsException,
			UserProfileException;

	/**
	 * Create relation between card and url.
	 * 
	 * @param cuid
	 *            the icard's CUID
	 * @param url
	 *            the url of site
	 * @return Returns <code>true</code> if relation was added. Otherwise <code>false</code>
	 * @throws RppsException
	 * @throws UserProfileException
	 * @deprecated
	 */
	boolean addUrlCardRelation(String cuid, String url) throws RppsException, UserProfileException;

	/**
	 * Checks whether <code>RppsService</code> can recognize data format of the provided input stream. Some formats
	 * require authentication information in order to process input data. This method should be used prior to actual
	 * attempt to import data from the input stream in order to retrieve an information about data format and required
	 * authentication data to process the data successfully.
	 * 
	 * @param crd
	 *            context of file as byte array
	 * @return the information about input stream data format if it was recognized successfully and <code>null</code>
	 *         otherwise.
	 * @throws RppsException
	 */
	FormatDescriptorTO checkImportStream(byte[] crd) throws RppsException;

	/**
	 * Deletes saved authentication card information of the specified card.
	 * 
	 * @param cuid
	 *            the icard's CUID
	 * @return Returns updated card object
	 * @throws RppsException
	 */
	ICardTO clearCardCredential(String cuid) throws RppsException;

	/**
	 * Deletes card history of the specified card on all URLs unless it is checked as "Use Always".
	 * 
	 * @param cuid
	 *            the icard's CUID
	 * @return Returns <code>true</code> if card history was cleared. Otherwise <code>false</code>
	 * @throws RppsException
	 */
	boolean clearCardHistory(String cuid) throws RppsException;

	/**
	 * Creates duplicate of personal card only with new card's name.
	 * 
	 * @param cardName
	 *            new card's name
	 * @param cuid
	 *            the old icard's cuid
	 * @param pinCode
	 *            pinCode for retrieve claims of personal card if that is pin protected. Otherwise <code>null</code>
	 * @return Returns updated card object
	 * @throws RppsException
	 */
	ICardTO createDuplicateICard(String cardName, String cuid, String pinCode) throws RppsException;

	/**
	 * Creates a new card by template.
	 * 
	 * @param template
	 *            the icard's template
	 * @return a new card object
	 * @throws RppsException
	 */
	ICardTO createICard(ICardTemplateTO template) throws RppsException;

	/**
	 * Creates a new card for HBX.
	 * 
	 * @param cardname
	 *            the card's name
	 * @return a new card's cuid
	 * @throws RppsException
	 * @deprecated
	 */
	String createICardFromHBX(String cardname) throws RppsException;

	/**
	 * Deletes all user's card.
	 * 
	 * @throws RppsException
	 */
	void deleteAllICard() throws RppsException;

	/**
	 * Delete specified record of card history.
	 * 
	 * @param cuid
	 *            the icard's CUID
	 * @param rpEndPointTO
	 *            the object which represents logging form on relying party
	 * @return Returns <code>true</code> if card history record was deleted. Otherwise <code>false</code>
	 * @throws RppsException
	 */
	boolean deleteCardHistoryRecord(String cuid, RPEndPointTO rpEndPointTO) throws RppsException;

	/**
	 * Delete specified record of card history.
	 * 
	 * @param cuid
	 *            the icard's CUID
	 * @param url
	 *            the url of site
	 * @return Returns <code>true</code> if card history record was deleted. Otherwise <code>false</code>
	 * @throws RppsException
	 * @deprecated
	 */
	boolean deleteCardHistoryRecord(String cuid, String url) throws RppsException;

	/**
	 * Delete the card's category.
	 * 
	 * @param categoryId
	 *            category identifier
	 * @throws RppsException
	 */
	void deleteCategory(String categoryId) throws RppsException;

	/**
	 * Deletes specified card.
	 * 
	 * @param cuid
	 *            the icard's CUID
	 * @return Returns ICardUpdateResponseTO which contains delete status for card. If <code>code</code> field equals
	 *         <code>0</code> then card was updated correctly. Otherwise <code>code</code> field contains error code. If
	 *         deleted card had self signed type then <code>cuid</code> field contains cuid of personal card which had
	 *         been used to signed managed card.
	 * @throws RppsException
	 */
	ICardUpdateResponseTO deleteICard(String cuid) throws RppsException;

	void deleteOldPassword() throws UserProfileException;

	/**
	 * Delete relation between card and site's url.
	 * 
	 * @param cuid
	 *            the icard's CUID
	 * @param url
	 *            the url of site
	 * @return Returns <code>true</code> if relation was deleted. Otherwise <code>false</code>
	 * @throws RppsException
	 * @deprecated
	 */
	boolean deleteUrlCardRelation(String cuid, String url) throws RppsException;

	/**
	 * Delete relation between card and relying party.
	 * 
	 * @param cuid
	 *            the icard's CUID
	 * @param rpEndPointTO
	 *            the object which represents logging form on relying party
	 * @return Returns <code>true</code> if relation was deleted. Otherwise <code>false</code>
	 * @throws RppsException
	 */
	boolean deleteRPEndPointCardRelation(String cuid, RPEndPointTO rpEndPointTO) throws RppsException;

	/**
	 * Delete all relations between card and site's urls.
	 * 
	 * @param cuid
	 *            the icard's CUID
	 * @return Returns <code>true</code> if relations were deleted. Otherwise <code>false</code>
	 * @throws RppsException
	 * @deprecated
	 */
	boolean deleteUrlsCardsRelations(String cuid) throws RppsException;

	/**
	 * Delete all relations between card and relying party.
	 * 
	 * @param cuid
	 *            the icard's CUID
	 * @return Returns <code>true</code> if relations were deleted. Otherwise <code>false</code>
	 * @throws RppsException
	 */
	boolean deleteRPEndPointsCardsRelations(String cuid) throws RppsException;

	/**
	 * Deletes user profile.
	 * 
	 * @return ResponseMessage
	 * @throws RppsException
	 */
	ResponseMessage deleteUserProfile() throws RppsException;

	/**
	 * Changes pin code for personal card.
	 * 
	 * @param cuid
	 *            the icard's CUID
	 * @param oldPinCode
	 *            Old pin code
	 * @param newPinCode
	 *            New pin code
	 * @return Returns updated card object
	 * @throws RppsException
	 */
	ICardTO editPinCode(String cuid, String oldPinCode, String newPinCode) throws RppsException;

	/**
	 * Returns file as byte array with special which contains select cards.
	 * 
	 * @param formatID
	 *            file's format identifier
	 * @param cards
	 *            the icard's CUIDs
	 * @param credential
	 *            authentication information
	 * @return array of byte
	 * @throws RppsException
	 */
	byte[] exportICards(String formatID, String[] cards, UITokenServiceCredentialTO credential) throws RppsException;

	/**
	 * Return list of card history records, beginning at the specified position in selection.
	 * 
	 * @param cuid
	 *            the icard's CUID
	 * @param startFrom
	 *            starting position in the selection
	 * @param capacity
	 *            count of record which will be returned
	 * @param orderBy
	 *            name of field which be used for sorting. Can take one of these values: "<code>date</code>","
	 *            <code>site</code>"," <code>useAlways</code>"
	 * @param orderDirection
	 *            If value of this parameter is "<code>desc</code>" then result will be sorted in descending order.
	 *            Otherwise in ascending order.
	 * @return array of CardUsageTO
	 * @throws RppsException
	 */
	CardUsageTO[] getCardHistory(String cuid, int startFrom, int capacity, String orderBy, String orderDirection);

	/**
	 * Gets card information object.
	 * 
	 * @param cuid
	 *            the icard's CUID
	 * @param size
	 *            count of card history element which will be returned
	 * @return Return card information object
	 * @throws RppsException
	 */
	CardInformationTO getCardInformation(String cuid, int size) throws RppsException;

	/**
	 * Gets list of card categories.
	 * 
	 * @return Returns array of category transfer object
	 * @throws RppsException
	 */
	CategoryTO[] getCategory() throws RppsException;

	/**
	 * Gets most recently used records.
	 * 
	 * @return Returns array of MapTO
	 * @throws RppsException
	 */
	MapTO[] getClaimValuesMRU() throws RppsException;

	/**
	 * Gets the global i-name registered to the Higgins service (e.g. "=parity").
	 * 
	 * @return the global i-name.
	 */
	String getHigginsServiceGlobalIName();

	/**
	 * Gets a card by CUID.
	 * 
	 * @param cuid
	 *            the card CUID.
	 * @return the card transfer object.
	 */
	ICardTO getICardByCUID(String cuid) throws RppsException;

	/**
	 * This method gets the card with retrieved claims.
	 * 
	 * @param cuid
	 *            the card's cuid
	 * @param tokenCredential
	 *            filled card credential for retrieve claims
	 * @return the card as a ICardTO
	 * @throws RppsException
	 */
	ICardTO getICardByCUIDAndTokenServiceCredential(String cuid, UITokenServiceCredentialTO tokenCredential)
			throws RppsException, STSFaultException;

	/**
	 * Created for some demo.
	 * 
	 * @param cuid
	 *            the card's cuid.
	 * @param policy
	 * @param typeofCredential
	 * @param credentialKey
	 * @param credentialValue
	 * @return UIResponseCardTO
	 * @throws RppsException
	 * @deprecated
	 */
	UIResponseCardTO getICardClaims(String cuid, String policy, String typeofCredential, String[] credentialKey,
			String[] credentialValue) throws RppsException;

	/**
	 * Created for some demo.
	 * 
	 * @param cuid
	 *            the card's cuid.
	 * @param policy
	 * @param typeofCredential
	 * @param credentialKey
	 * @param credentialValue
	 * @param rpEndPointTO
	 *            the object which represents logging form on relying party
	 * @return UIResponseCardTO
	 * @throws RppsException
	 */
	UIResponseCardTO getICardClaims(String cuid, String policy, String typeofCredential, String[] credentialKey,
			String[] credentialValue, RPEndPointTO rpEndPointTO) throws RppsException;

	/**
	 * This method return list of card creation template as array of ICardTemplateTO.
	 * 
	 * @return a template list as array of ICardTemplateTO
	 * @throws RppsException
	 */
	ICardTemplateTO[] getICardCreationTemplate() throws RppsException;

	/**
	 * This method return list of card creation template by provider id.
	 * 
	 * @param extID
	 *            the id to identify the card provider.
	 * @return a template list as array of ICardTemplateTO
	 * @throws RppsException
	 */
	ICardTemplate[] getICardCreationTemplateByProvider(String extID) throws RppsException;

	/**
	 * This method gets the card list and category list of the specified user.
	 * 
	 * @return ICardsAndCategoryTO which contain the cards and the categorys lists
	 * @throws RppsException
	 */
	ICardsAndCategoryTO getICardsAndCategoriesByUser() throws RppsException;

	/**
	 * This method gets cards as array of ICardTO by CUIDs array.
	 * 
	 * @param cuids
	 *            the card CUIDs.
	 * @return array of ICardTO.
	 */
	ICardTO[] getICardsByCUIDs(String[] cuids);

	/**
	 * Gets cards which may be export to special file format.
	 * 
	 * @param formatID
	 *            the id of format file
	 * @return list of cards as a array of CardTO
	 * @throws RppsException
	 */
	ICardTO[] getICardsByFormat(String formatID) throws RppsException;

	/**
	 * This method gets ICardResponseTO which contains the card list and card relation list of the specified user.
	 * 
	 * @return ICardResponseTO
	 */
	ICardResponseTO getICardsByUser();

	/**
	 * Gets available file format for cards export.
	 * 
	 * @return array of FormatDescriptorTO
	 * @throws RppsException
	 */
	FormatDescriptorTO[] getOutputFormats() throws RppsException;

	/**
	 * Makes xmlToken usage claim list.
	 * 
	 * @param sslCert
	 *            the SSL certificate
	 * @param claimName
	 *            the array which contains claim names
	 * @param claimValue
	 *            the array which contains claim values
	 * @return the security token as a String
	 * @throws RppsException
	 */
	TokenResponseTO getTokenByClaims(String sslCert, String[] claimName, String[] claimValue) throws RppsException;

	/**
	 * Makes security token usage saved users information.
	 * 
	 * @param policy
	 *            the RP Agent's Relying Party Security Policy
	 * @param policytype
	 *            the RP Security Policy type
	 * @param sslCert
	 *            the SSL certificate
	 * @param rpEndPointTO
	 *            the object which represents logging form on relying party
	 * @return the security token as a String
	 * @throws RppsException
	 */
	TokenResponseTO getTokenByRPEndPoint(String policy, String policytype, String sslCert, RPEndPointTO rpEndPointTO)
			throws RppsException;

	/**
	 * Makes security token usage saved users information.
	 * 
	 * @param policy
	 *            the RP Agent's Relying Party Security Policy
	 * @param policytype
	 *            the RP Security Policy type
	 * @param sslCert
	 *            the SSL certificate
	 * @param url
	 *            the url of site which needs token
	 * @return the security token as a String
	 * @throws RppsException
	 * @deprecated
	 */
	TokenResponseTO getTokenByUrl(String policy, String policytype, String sslCert, String url) throws RppsException;

	/**
	 * Makes security token with extra card's information usage saved users information.
	 * 
	 * @param policy
	 *            the RP Agent's Relying Party Security Policy
	 * @param policytype
	 *            the RP Security Policy type
	 * @param sslCert
	 *            the SSL certificate
	 * @param url
	 *            the url of site which needs token
	 * @return the security token as a String with extra card's information
	 * @throws RppsException
	 * @deprecated
	 */
	ExtraTokenResponseTO getTokenExtraByUrl(String policy, String policytype, String sslCert, String url)
			throws RppsException;

	/**
	 * Makes security token with extra card's information usage saved users information.
	 * 
	 * @param policy
	 *            the RP Agent's Relying Party Security Policy
	 * @param policytype
	 *            the RP Security Policy type
	 * @param sslCert
	 *            the SSL certificate
	 * @param rpEndPointTO
	 *            the object which represents logging form on relying party
	 * @return the security token as a String with extra card's information
	 * @throws RppsException
	 */
	ExtraTokenResponseTO getTokenExtraByRPEndPoint(String policy, String policytype, String sslCert,
			RPEndPointTO rpEndPointTO) throws RppsException;

	/**
	 * Given the Relying Party Agent (aka 'Requester')'s policy, identity information about the requester, and the set
	 * of one or more selections that the user has just made in the ISS Web UI (see userInterfaceRequest), AND presuming
	 * that the protocol involved in interacting with the RPA requires a security token, request the token that
	 * corresponds to the user's selection(s).
	 * 
	 * @param policy
	 *            the RP Agent's Relying Party Security Policy
	 * @param policytype
	 *            the RP Security Policy type
	 * @param sslCert
	 *            the SSL certificate
	 * @param cuids
	 *            the user-selected subset of the UIDs (handles) contained in the UserInterfaceResponse object returned
	 *            from an earlier invocation of 'userInterfaceRequest'
	 * @param typeofCredential
	 *            the type of selected credential
	 * @param credentialKey
	 *            the names of fields of selected credential
	 * @param credentialValue
	 *            the value of fields of selected credential
	 * @return the security token as a String
	 * @throws RppsException
	 * @deprecated
	 */
	TokenResponseTO getTokenObject(String policy, String policytype, String sslCert, String[] cuids,
			String typeofCredential, String[] credentialKey, String[] credentialValue) throws RppsException;

	/**
	 * Given the Relying Party Agent (aka 'Requester')'s policy, identity information about the requester, and the set
	 * of one or more selections that the user has just made in the ISS Web UI (see userInterfaceRequest), AND presuming
	 * that the protocol involved in interacting with the RPA requires a security token, request the token that
	 * corresponds to the user's selection(s).
	 * 
	 * @param policy
	 *            the RP Agent's Relying Party Security Policy
	 * @param policytype
	 *            the RP Security Policy type
	 * @param sslCert
	 *            the SSL certificate
	 * @param cuids
	 *            the user-selected subset of the UIDs (handles) contained in the UserInterfaceResponse object returned
	 *            from an earlier invocation of 'userInterfaceRequest'
	 * @param typeofCredential
	 *            the type of selected credential
	 * @param credentialKey
	 *            the names of fields of selected credential
	 * @param credentialValue
	 *            the value of fields of selected credential
	 * @param rpEndPointTO
	 *            the object which represents logging form on relying party
	 * @return the security token as a String
	 * @throws RppsException
	 */
	TokenResponseTO getTokenObject(String policy, String policytype, String sslCert, String[] cuids,
			String typeofCredential, String[] credentialKey, String[] credentialValue, RPEndPointTO rpEndPointTO)
			throws RppsException;

	/**
	 * This method takes as input a policy---for example, one that has been retrieved from the Relying Party Agent
	 * (called here the 'Requester')---as well as a description of the Requester. The UserInterfaceResponse object
	 * should contain all information necessary to display both (a) a user interface which shares the same card
	 * metaphor, sequences of experiences and decision points (the ceremony) of CardSpace and (b) other interface
	 * paradigms to be developed in the future.
	 * 
	 * @param url
	 *            the URL to the html page with special tag
	 * @param target
	 *            some information
	 * @param sslCert
	 *            the SSL certificate
	 * @param policytype
	 *            the RP Security Policy type
	 * @param policy
	 *            the RP Agent's Relying Party Security Policy
	 * @return the UIResponseTO transfer object.
	 * @throws RppsException
	 * @deprecated
	 */
	UIResponseTO getUserInterface(String url, String target, String sslCert, String policytype, String policy)
			throws RppsException;

	/**
	 * This method takes as input a policy---for example, one that has been retrieved from the Relying Party Agent
	 * (called here the 'Requester')---as well as a description of the Requester. The UserInterfaceResponse object
	 * should contain all information necessary to display both (a) a user interface which shares the same card
	 * metaphor, sequences of experiences and decision points (the ceremony) of CardSpace and (b) other interface
	 * paradigms to be developed in the future.
	 * 
	 * @param rpEndPointTO
	 *            the object which represents logging form on relying party
	 * @param target
	 *            some information
	 * @param sslCert
	 *            the SSL certificate
	 * @param policytype
	 *            the RP Security Policy type
	 * @param policy
	 *            the RP Agent's Relying Party Security Policy
	 * @return the UIResponseTO transfer object.
	 * @throws RppsException
	 */
	UIResponseTO getUserInterface(RPEndPointTO rpEndPointTO, String target, String sslCert, String policytype,
			String policy) throws RppsException;

	/**
	 * Gets user profile as a UserProfileTO.
	 * 
	 * @return user profile information
	 * @throws RppsException
	 * @throws AccessException
	 */
	UserProfileTO getUserProfile() throws RppsException, AccessException;

	/**
	 * Imports card from a file.
	 * 
	 * @param crd
	 *            the file's context as a byte array.
	 * @return ICardResponseTO
	 * @throws RppsException
	 */
	ICardResponseTO importICards(byte[] crd) throws RppsException;

	/**
	 * Import cards from special format file.
	 * 
	 * @param crd
	 *            file's context
	 * @param formatID
	 *            file's format identifier
	 * @param credential
	 *            authentication information
	 * @return ICardResponseTO
	 * @throws RppsException
	 */
	ICardResponseTO importICards(byte[] crd, String formatID, UITokenServiceCredentialTO credential)
			throws RppsException;

	/**
	 * Modifies list of card's category.
	 * 
	 * @param categoriesTO
	 *            list of card's category
	 * @return Returns <code>ResponseMessage</code> with <code>code</code> field equals <code>0</code> if categories
	 *         were updated. Otherwise returns <code>ResponseMessage</code> with error code or throws an RppsException
	 * @throws RppsException
	 */
	ResponseMessage modifyCategories(CategoryTO[] categoriesTO) throws RppsException;

	/**
	 * Modifies the card's category.
	 * 
	 * @param categoryTO
	 *            category transfer object
	 * @return Returns modified category
	 * @throws RppsException
	 */
	CategoryTO modifyCategory(CategoryTO categoryTO) throws RppsException;

	/**
	 * Update user profile.
	 * 
	 * @param userProfile
	 *            user profile transfer object
	 * @return user profile transfer object
	 * @throws RppsException
	 */
	UserProfileTO modifyUserProfile(UserProfileTO userProfile) throws RppsException;

	/**
	 * Deletes most recently used record.
	 * 
	 * @param claimType
	 *            claim's type
	 * @param claimValue
	 *            claim's value
	 * @throws RppsException
	 */
	void removeClaimValuesMRU(String claimType, String claimValue) throws RppsException;

	/**
	 * Remove pin code protection from personal card.
	 * 
	 * @param cuid
	 *            the icard's CUID
	 * @param oldPinCode
	 *            pin code
	 * @return Returns updated card object
	 * @throws RppsException
	 */
	ICardTO resetPinCode(String cuid, String oldPinCode) throws RppsException;

	/**
	 * Saves authentication card information of the specified card.
	 * 
	 * @param cuid
	 *            the icard's CUID
	 * @param tokenCredential
	 *            authentication card information
	 * @return Returns <code>true</code> if card credential was set. Otherwise <code>false</code>
	 * @throws RppsException
	 */
	boolean setCardCredential(String cuid, UITokenServiceCredentialTO tokenCredential) throws RppsException;

	void setNewPrivateUserPassword(String newPassword) throws UserProfileException;

	/**
	 * Sets pin code protection on personal card.
	 * 
	 * @param cuid
	 *            the icard's CUID
	 * @param pinCode
	 *            pin code
	 * @return Returns updated card object
	 * @throws RppsException
	 */
	ICardTO setPinCode(String cuid, String pinCode) throws RppsException;

	/**
	 * This method updates some claim values in list of cards.
	 * 
	 * @param claimTypes
	 *            array of claim's types
	 * @param claimValues
	 *            array of claim's types. Must be in the same order as claimTypes
	 * @param cardUpdate
	 *            array of ICardUpdateTO
	 * @return Return array of ICardUpdateResponseTO which contains update status for every card. If <code>code</code>
	 *         field equals <code>0</code> then card was updated correctly. Otherwise <code>code</code> field contains
	 *         error code
	 * @throws RppsException
	 */
	ICardUpdateResponseTO[] updateClaimsAcrossCards(String[] claimTypes, String[] claimValues,
			ICardUpdateTO[] cardUpdate) throws RppsException;

	/**
	 * This method update card.
	 * 
	 * @param icardTO
	 *            the card transfer object.
	 * @return the card transfer object.
	 * @throws RppsException
	 */
	ICardTO updateICard(ICardTO icardTO) throws RppsException;

	/**
	 * This method update list of cards.
	 * 
	 * @param icardTOList
	 *            list of cards which may be updated
	 * @return list of ICardUpdateResponseTO
	 * @throws RppsException
	 */
	ICardUpdateResponseTO[] updateICards(ICardTO[] icardTOList) throws RppsException;

	/**
	 * Checks availability of user profile by userIdentifier.
	 * 
	 * @param userIdentifier
	 * @return Returns <code>true</code> if user identifier exists. Otherwise <code>false</code>
	 * @throws RppsException
	 */
	boolean userIdentifierExists(String userIdentifier);

	/**
	 * Set default personal card.
	 * 
	 * This method is part of workflow ' Instant credential m-Cards' http://graceland.parityinc.net:8090/browse/WWM-1150
	 * if pcardId is null or empty string it just removes old default pcard.
	 * 
	 * @param pcardId
	 *            personal card id (cuid)
	 * @throws RppsException
	 *             if coudn't find pcardId throw RppsException
	 */
	void setDefaultPCard(String pcardId) throws RppsException;

	/**
	 * Return default personal card.
	 * 
	 * This method is part of workflow ' Instant credential m-Cards' http://graceland.parityinc.net:8090/browse/WWM-1150
	 * 
	 * @return default pcard or null if user doesn't have default pcard
	 * @throws RppsException
	 */
	ICardTO getDefaultPCard() throws RppsException;

	/**
	 * Generate and send user email with new password reset code. This method is part of reset user password work-flow.
	 * 
	 * @see WWM-1264 RPPS Changes for password reset http://graceland.parityinc.net:8090/browse/WWM-1264
	 * @param userIdentifier
	 *            user login name
	 * @return UserProfile with new generated password reset code
	 * @throws RppsException
	 */
	void generatePasswordResetCode(String userIdentifier) throws RppsException;

	/**
	 * Update user password, requires password reset code for identify.
	 * 
	 * This method is part of reset user password work-flow.
	 * 
	 * @see WWM-1264 RPPS Changes for password reset http://graceland.parityinc.net:8090/browse/WWM-1264
	 * 
	 * @param userIdentifier
	 * @param newPassword
	 * @param passwordResetCode
	 * @throws RppsException
	 */
	void modifyPasswordWithPasswordResetCode(final String userIdentifier, final String passwordResetCode,
			final String newPassword) throws RppsException;

	/**
	 * Update user password, requires oldPassword for identify.
	 * 
	 * This method is part of reset user password work-flow.
	 * 
	 * @see WWM-1264 RPPS Changes for password reset http://graceland.parityinc.net:8090/browse/WWM-1264
	 * 
	 * @param userIdentifier
	 * @param oldPassword
	 * @param newPassword
	 * @throws RppsException
	 */
	void modifyPasswordWithOldPassword(final String userIdentifier, final String oldPassword, final String newPassword)
			throws RppsException;

	/**
	 * Create new Captcha entity and store it via UserProfileService.
	 * 
	 * This method is part of reset user password work-flow.
	 * 
	 * @see WWM-1264 RPPS Changes for password reset http://graceland.parityinc.net:8090/browse/WWM-1264
	 * 
	 * @return Captcha
	 * @throws RppsException
	 */
	Captcha createCaptcha() throws RppsException;

	/**
	 * Retrieve Captcha entity via UserProfileService.
	 * 
	 * This method is part of reset user password work-flow.
	 * 
	 * @see WWM-1264 RPPS Changes for password reset http://graceland.parityinc.net:8090/browse/WWM-1264
	 * @param id
	 *            entity Id
	 * @return Captcha entity
	 * @throws RppsException
	 */
	Captcha getCaptcha(final String id) throws RppsException;

	/**
	 * Delete Captcha entity via UserProfileService.
	 * 
	 * This method is part of reset user password work-flow.
	 * 
	 * @see WWM-1264 RPPS Changes for password reset http://graceland.parityinc.net:8090/browse/WWM-1264
	 * @param id
	 * @throws RppsException
	 */
	void deleteCaptcha(final String id) throws RppsException;

}
