/** 
 * Copyright (c) 2008 Parity Communications, Inc.  
 * All rights reserved. This program and the accompanying materials  
 * are made available under the terms of the Eclipse Public License v1.0  
 * which accompanies this distribution, and is available at 
 * http://www.eclipse.org/legal/epl-v10.html 
 *  
 * Contributors: 
 * Alexander Yuhimenko -  initial API and implementation
 * 
 */
package org.eclipse.higgins.rpps.core.utils;

import java.io.UnsupportedEncodingException;
import java.text.MessageFormat;
import java.util.ArrayList;
import java.util.MissingResourceException;
import java.util.Properties;
import java.util.ResourceBundle;
import java.util.StringTokenizer;

import javax.mail.Message;
import javax.mail.MessagingException;
import javax.mail.PasswordAuthentication;
import javax.mail.Session;
import javax.mail.Transport;
import javax.mail.internet.AddressException;
import javax.mail.internet.InternetAddress;
import javax.mail.internet.MimeMessage;

/**
 * @author Alexander Yuhimenko
 * 
 */
public class EmailUtils {

	private static final String BUNDLE_NAME = "org.eclipse.higgins.rpps.core.utils.email"; //$NON-NLS-1$

	private static final ResourceBundle RESOURCE_BUNDLE = ResourceBundle.getBundle(BUNDLE_NAME);

	public static String getString(String key) {
		try {
			return RESOURCE_BUNDLE.getString(key);
		} catch (MissingResourceException e) {
			return '!' + key + '!';
		}
	}

	/**
	 * Send password reset code.
	 * 
	 * @param emailAddres
	 * @param code
	 * @throws UnsupportedEncodingException
	 * @throws MessagingException
	 */
	public void sendPasswordResetCode(final String emailAddres, final String code) throws UnsupportedEncodingException,
			MessagingException {
		Properties props = getSMTPPropertie(getString("email.smtp.host"), getString("email.smtp.port"));
		Session session = getEmailSession(getString("email.smtp.user"), getString("email.smtp.password"), props);
		sendEmailMessage(getString("email.header.from"), getString("email.header.personal.from"), emailAddres,
				getString("email.header.subj.reset_code"), MessageFormat.format(getString("email.body.reset_code"),
						new String[] { code }), getString("email.encoding"), session);
	}

	/**
	 * Send confirmation for password update.
	 * 
	 * @param emailAddres
	 * @param code
	 * @throws UnsupportedEncodingException
	 * @throws MessagingException
	 */
	public void sendPasswordUpdatedEmail(final String emailAddres) throws UnsupportedEncodingException,
			MessagingException {
		if (null != emailAddres) {
			Properties props = getSMTPPropertie(getString("email.smtp.host"), getString("email.smtp.port"));
			Session session = getEmailSession(getString("email.smtp.user"), getString("email.smtp.password"), props);
			sendEmailMessage(getString("email.header.from"), getString("email.header.personal.from"), emailAddres,
					getString("email.header.subj.password_updated"), getString("email.body.password_updated"),
					getString("email.encoding"), session);
		}
	}

	private Properties getSMTPPropertie(final String host, final String port) {
		Properties props = new Properties();
		props.put("mail.smtp.host", host);
		// props.put("mail.smtp.starttls.enable", "true");
		// props.put("mail.transport.protocol", "smtps");
		props.put("mail.transport.protocol", "smtp");
		// props.put("mail.smtps.quitwait", "false");
		// props.put("mail.smtp.quitwait", "false");
		props.put("mail.smtp.auth", "true");

		// props.put("mail.smtps.auth", "true");
		props.put("mail.debug", "false");
		props.put("mail.smtp.port", port);
		// props.put("mail.smtp.socketFactory.port", port);
		// props.put("mail.smtp.socketFactory.class", SSL_FACTORY);
		// props.put("mail.smtp.socketFactory.fallback", "false");
		return props;
	}

	private Session getEmailSession(final String smtpUserName, final String smtpPassword, Properties props) {
		Session session = Session.getInstance(props, new javax.mail.Authenticator() {
			protected PasswordAuthentication getPasswordAuthentication() {
				return new PasswordAuthentication(smtpUserName, smtpPassword);
			}
		});
		return session;
	}

	/**
	 * @param from
	 * @param personalFrom
	 * @param to
	 * @param subject
	 * @param body
	 * @param encoding
	 * @param session
	 * @throws UnsupportedEncodingException
	 * @throws MessagingException
	 */
	private void sendEmailMessage(final String from, final String personalFrom, final String to, final String subject,
			final String body, final String encoding, Session session) throws UnsupportedEncodingException,
			MessagingException {
		Message msg = new MimeMessage(session);
		msg.setFrom(new InternetAddress(from, personalFrom, encoding));
		msg.setRecipients(Message.RecipientType.TO, convertStringToInternetAddress(to));
		msg.setSentDate(new java.util.Date());
		((MimeMessage) msg).setSubject(subject, encoding);

		msg.setContent(body, "text/html; charset=" + encoding);

		session.getTransport("smtp");
		// tr.connect(emailClient.getSmtpUser(), emailClient.getSmtpPassword());
		Transport.send(msg);

	}

	/**
	 * @param emailAddress
	 * @return
	 */
	private static InternetAddress[] convertStringToInternetAddress(String emailAddress) {
		ArrayList addressList = new ArrayList();
		StringTokenizer st = new StringTokenizer(emailAddress, " ,;");
		while (st.hasMoreTokens()) {
			try {
				addressList.add(new InternetAddress(st.nextToken()));
			} catch (AddressException ae) {
				ae.printStackTrace();
			}
		}
		InternetAddress[] address = new InternetAddress[addressList.size()];
		addressList.toArray(address);
		return address;
	}

}
