/**
 * Copyright (c) 2007 Parity Communications, Inc. 
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     Sergei Yakovlev - initial API and implementation
 *     Artem Verkhovets - initial API and implementation
 */

package org.eclipse.higgins.rpps.core.impl;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.InputStream;
import java.io.UnsupportedEncodingException;
import java.net.URI;
import java.rmi.AccessException;
import java.rmi.RemoteException;
import java.security.AccessController;
import java.security.InvalidAlgorithmParameterException;
import java.security.KeyStore;
import java.security.KeyStoreException;
import java.security.PrivilegedAction;
import java.security.SecureRandom;
import java.security.Security;
import java.security.cert.CertStoreException;
import java.security.cert.CertificateException;
import java.security.cert.X509Certificate;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.Date;
import java.util.HashSet;
import java.util.Hashtable;
import java.util.Iterator;
import java.util.List;
import java.util.ListIterator;
import java.util.Map;
import java.util.Set;

import javax.net.ssl.ManagerFactoryParameters;
import javax.net.ssl.TrustManager;
import javax.net.ssl.TrustManagerFactorySpi;
import javax.net.ssl.X509TrustManager;
import javax.security.auth.callback.CallbackHandler;

import org.apache.axiom.om.util.UUIDGenerator;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.xml.security.utils.Base64;
import org.eclipse.higgins.cache.CacheProviderFactory;
import org.eclipse.higgins.icard.CUID;
import org.eclipse.higgins.icard.CardException;
import org.eclipse.higgins.icard.ICard;
import org.eclipse.higgins.icard.ICardProvider;
import org.eclipse.higgins.icard.ICardTemplate;
import org.eclipse.higgins.icard.IClaim;
import org.eclipse.higgins.icard.IClaimType;
import org.eclipse.higgins.icard.IComplexClaim;
import org.eclipse.higgins.icard.IInformationCard;
import org.eclipse.higgins.icard.IInformationCardExtension;
import org.eclipse.higgins.icard.IManagedInformationCard;
import org.eclipse.higgins.icard.IPersonalInformationCard;
import org.eclipse.higgins.icard.IPolicy;
import org.eclipse.higgins.icard.ISimpleClaim;
import org.eclipse.higgins.icard.ISimpleClaimType;
import org.eclipse.higgins.icard.ITemplateContainer;
import org.eclipse.higgins.icard.ITemplateElement;
import org.eclipse.higgins.icard.ITemplateValue;
import org.eclipse.higgins.icard.TUID;
import org.eclipse.higgins.icard.auth.ICredential;
import org.eclipse.higgins.icard.auth.ICredentialDescriptor;
import org.eclipse.higgins.icard.auth.IPasswordCredential;
import org.eclipse.higgins.icard.auth.IPinCodeCredential;
import org.eclipse.higgins.icard.auth.ITSKerberosV5Credential;
import org.eclipse.higgins.icard.auth.ITSSelfIssuedCredential;
import org.eclipse.higgins.icard.auth.ITSUsernamePasswordCredential;
import org.eclipse.higgins.icard.auth.ITSX509V3Credential;
import org.eclipse.higgins.icard.auth.ITokenServiceCredential;
import org.eclipse.higgins.icard.auth.IUsernamePasswordCredential;
import org.eclipse.higgins.icard.common.auth.PinCodeCredential;
import org.eclipse.higgins.icard.common.auth.callback.PinCodeCallback;
import org.eclipse.higgins.icard.io.IFormatDescriptor;
import org.eclipse.higgins.icard.policy.ICardSpacePolicy;
import org.eclipse.higgins.icard.policy.ISTSPrivacyPolicy;
import org.eclipse.higgins.icard.provider.cardspace.common.CardSpacePolicy;
import org.eclipse.higgins.icard.provider.cardspace.common.InformationCard;
import org.eclipse.higgins.icard.provider.cardspace.common.PPIDCardSpacePolicy;
import org.eclipse.higgins.icard.provider.cardspace.common.STSFaultException;
import org.eclipse.higgins.icard.registry.ICardRegistry;
import org.eclipse.higgins.iss.CredentialContainer;
import org.eclipse.higgins.iss.ICardSelectorService;
import org.eclipse.higgins.iss.IICardSelector;
import org.eclipse.higgins.iss.IIdentityToken;
import org.eclipse.higgins.iss.SelectionANDofORs;
import org.eclipse.higgins.iss.SelectionANDofORsElm;
import org.eclipse.higgins.iss.UCTelm;
import org.eclipse.higgins.iss.UserChoiceTree;
import org.eclipse.higgins.iss.UserChoiceTree_ANDofORs;
import org.eclipse.higgins.iss.cardspace.IdentityToken;
import org.eclipse.higgins.iss.cardspace.util.UserProfileTokenHelper;
import org.eclipse.higgins.keystore.IKeyStoreService;
import org.eclipse.higgins.keystore.registry.KeyStoreRegistry;
import org.eclipse.higgins.rpps.core.IRppsService;
import org.eclipse.higgins.rpps.core.RppsException;
import org.eclipse.higgins.rpps.core.tobj.CardExtensionTO;
import org.eclipse.higgins.rpps.core.tobj.CardInformationTO;
import org.eclipse.higgins.rpps.core.tobj.CardUsageTO;
import org.eclipse.higgins.rpps.core.tobj.CategoryTO;
import org.eclipse.higgins.rpps.core.tobj.ExtraTokenResponseTO;
import org.eclipse.higgins.rpps.core.tobj.FormatDescriptorTO;
import org.eclipse.higgins.rpps.core.tobj.ICardResponseTO;
import org.eclipse.higgins.rpps.core.tobj.ICardTO;
import org.eclipse.higgins.rpps.core.tobj.ICardTemplateTO;
import org.eclipse.higgins.rpps.core.tobj.ICardUpdateResponseTO;
import org.eclipse.higgins.rpps.core.tobj.ICardUpdateTO;
import org.eclipse.higgins.rpps.core.tobj.ICardsAndCategoryTO;
import org.eclipse.higgins.rpps.core.tobj.MapTO;
import org.eclipse.higgins.rpps.core.tobj.PropertyTO;
import org.eclipse.higgins.rpps.core.tobj.RPEndPointTO;
import org.eclipse.higgins.rpps.core.tobj.RPPolicy;
import org.eclipse.higgins.rpps.core.tobj.ResponseMessage;
import org.eclipse.higgins.rpps.core.tobj.STSPolicyTO;
import org.eclipse.higgins.rpps.core.tobj.TokenResponseTO;
import org.eclipse.higgins.rpps.core.tobj.UIResponseAndTO;
import org.eclipse.higgins.rpps.core.tobj.UIResponseCardTO;
import org.eclipse.higgins.rpps.core.tobj.UIResponseClauseTO;
import org.eclipse.higgins.rpps.core.tobj.UIResponseOrTO;
import org.eclipse.higgins.rpps.core.tobj.UIResponseRPTO;
import org.eclipse.higgins.rpps.core.tobj.UIResponseTO;
import org.eclipse.higgins.rpps.core.tobj.UIResponseUsableCardsTO;
import org.eclipse.higgins.rpps.core.tobj.UITokenServiceCredentialTO;
import org.eclipse.higgins.rpps.core.tobj.UserProfileTO;
import org.eclipse.higgins.rpps.core.utils.EmailUtils;
import org.eclipse.higgins.rpps.core.utils.Utils;
import org.eclipse.higgins.sts.api.ISTSResponse;
import org.eclipse.higgins.sts.client.PPIDHelper;
import org.eclipse.higgins.userprofile.ICardUsageManager;
import org.eclipse.higgins.userprofile.IUserProfileService;
import org.eclipse.higgins.userprofile.UserProfileAuthenticationException;
import org.eclipse.higgins.userprofile.UserProfileException;
import org.eclipse.higgins.userprofile.UserProfileRegistry;
import org.eclipse.higgins.userprofile.entity.Captcha;
import org.eclipse.higgins.userprofile.entity.CardCredential;
import org.eclipse.higgins.userprofile.entity.CardInformation;
import org.eclipse.higgins.userprofile.entity.CardUsage;
import org.eclipse.higgins.userprofile.entity.Category;
import org.eclipse.higgins.userprofile.entity.PolicyVersion;
import org.eclipse.higgins.userprofile.entity.UserProfile;
import org.eclipse.higgins.userprofile.entity.WebForm;

/**
 * The object used for executing Higgins API and returning the results it produces.
 * 
 * @author Sergei Yakovlev
 * @author Artem Verkhovets
 */
public class RppsService implements IRppsService {

	final static class DummySecurityProvider extends java.security.Provider {

		private static final long serialVersionUID = 594145003249139851L;

		public DummySecurityProvider() {
			super("Dummy", 1.0D, "");
			AccessController.doPrivileged(new PrivilegedAction() {
				public Object run() {
					put("TrustManagerFactory." + DummyTrustManagerFactory.getAlgorithm(),
							DummyTrustManagerFactory.class.getName());
					return null;

				}
			});
		}
	}

	public final static class DummyTrustManagerFactory extends TrustManagerFactorySpi {
		public static String getAlgorithm() {
			return "XTrust509";
		}

		public DummyTrustManagerFactory() {
		}

		protected TrustManager[] engineGetTrustManagers() {
			return new TrustManager[] { new X509TrustManager() {
				public void checkClientTrusted(final java.security.cert.X509Certificate[] certs, final String authType) {
				}

				public void checkServerTrusted(final java.security.cert.X509Certificate[] certs, final String authType) {
					for (int i = 0; i < certs.length; i++) {
						LOG.info("  >>> Checking certificate " + certs[i]);
					}
				}

				public java.security.cert.X509Certificate[] getAcceptedIssuers() {
					return null;
				}
			} };
		}

		protected void engineInit(final KeyStore keystore) throws KeyStoreException {
		}

		protected void engineInit(final ManagerFactoryParameters arg0) throws InvalidAlgorithmParameterException {
		}
	}

	private static final Log LOG = LogFactory.getLog(RppsService.class);

	static {
		AllowSelfSignedSSL();
	}

	public static void AllowSelfSignedSSL() {
		if (Security.getProvider("Dummy") == null) {
			Security.insertProviderAt(new DummySecurityProvider(), 2);
			Security.setProperty("ssl.TrustManagerFactory.algorithm", DummyTrustManagerFactory.getAlgorithm());
		}
	}

	public static synchronized void ConfigureCache(final String configurationBase, final String configurationFile) {
		CacheProviderFactory.setConfigurationBase(configurationBase);
		CacheProviderFactory.setFileName(configurationFile);
		CacheProviderFactory.getCacheProvider();

	}

	/**
	 * Creates an RppsService object.
	 * 
	 * @return the RppsService object
	 */
	public static IRppsService getInstance() {
		return new RppsService();
	}

	/**
	 * Creates an RppsService object.
	 * 
	 * @param userId
	 *            the id to identify the user.
	 * @param password
	 *            the password to identify the user.
	 * @param password
	 * @return the RppsService object
	 * @throws AccessException
	 *             some situation
	 */
	public static IRppsService getInstance(final String userId, final String password) throws AccessException {
		return new RppsService(userId, password);
	}

	private static IUserProfileService getUserProfileService() throws UserProfileException {
		return UserProfileRegistry.getInstance().getUserProfileService();
	}

	/**
	 * @param privateSelectorINumber
	 *            privateSelectorINumber
	 * @return userIdentifer
	 */
	public static String resolveUserIdentifier(final String privateSelectorINumber) {
		try {
			return getUserProfileService().resolveUserIdentifier(privateSelectorINumber);
		} catch (final UserProfileException e) {
			LOG.error(e, e);
			return "";
		}
	}

	/**
	 * Instance of userProfile.
	 */
	protected UserProfile userProfileInstance = null;

	private CallbackHandler handler = null;

	/**
	 * Creates an RppsService object.
	 */
	public RppsService() {
	}

	/**
	 * Creates an RppsService object and authenticated user.
	 * 
	 * @param userId
	 *            the id to identify the user.
	 * @param password
	 *            the password to identify the user.
	 * @throws AccessException
	 *             some situation
	 */
	public RppsService(final String userId, final String password) throws AccessException {
		authenticate(userId, password);
	}

	/**
	 * Creates new card history record.
	 * 
	 * @param cuid
	 *            the icard's cuid.
	 * @param webForm
	 *            the object which represents logging form on relying party in UserProfileService
	 * @throws RppsException
	 *             some situation
	 * @throws UserProfileException
	 *             some situation
	 */
	protected void addCardHistory(final CUID cuid, final WebForm webForm) throws RppsException, UserProfileException {
		ICardUsageManager cardUsageManager = null;
		try {
			Utils.cleanURLQuery(webForm);
			cardUsageManager = getUserProfileService().getCardUsageManager(getHandler());
			cardUsageManager.addCardHistory(cuid, webForm);
		} catch (final Exception e) {
			LOG.error(e, e);
			throw new RppsException(e);
		} finally {
			if (cardUsageManager != null) {
				cardUsageManager.close();
			}
		}
	}

	/**
	 * Creates a new card's category.
	 * 
	 * @param categoryTO
	 *            the card category information
	 * @return Returns saved card category object
	 * @throws RppsException
	 *             some situation
	 */
	public CategoryTO addCategory(final CategoryTO categoryTO) throws RppsException {
		try {
			final Category category = getUserProfileService().addCategory(getHandler(),
					ConvertHelper.convertCategoryTO_to_Category(categoryTO));
			return ConvertHelper.convertCategory_to_CategoryTO(category);

		} catch (final UserProfileException e) {
			LOG.error(e, e);
			throw new RppsException(e);
		}
	}

	/**
	 * Adds most recently used record.
	 * 
	 * @param claimType
	 *            claim's type
	 * @param claimValue
	 *            claim's value
	 * @throws RppsException
	 *             some situation
	 */
	public void addClaimValuesMRU(final String claimType, final String claimValue) throws RppsException {
		ICardUsageManager cardUsageManager = null;
		try {
			cardUsageManager = getUserProfileService().getCardUsageManager(getHandler());
			addClaimValuesMRU(claimType, claimValue, cardUsageManager);
		} catch (final Exception e) {
			LOG.error(e, e);
			throw new RppsException(e);
		} finally {
			if (cardUsageManager != null) {
				try {
					cardUsageManager.close();
				} catch (final UserProfileException e) {
					LOG.error(e, e);
				}
			}
		}
	}

	/**
	 * Adds most recently used record.
	 * 
	 * @param claimType
	 *            claim's type
	 * @param claimValue
	 *            claim's value
	 * @param cardUsageManager
	 *            instance of ICardUsageManager
	 * @throws RppsException
	 *             some situation
	 */
	private void addClaimValuesMRU(final String claimType, final String claimValue,
			final ICardUsageManager cardUsageManager) throws RppsException {
		try {
			if (claimValue != null && !"".equals(claimValue.trim())) {
				final Set claimValuesMRU = cardUsageManager.getClaimValuesMRU(claimType);
				if (!claimValuesMRU.contains(claimValue)) {
					claimValuesMRU.add(claimValue);
					cardUsageManager.setClaimValuesMRU(claimType, claimValuesMRU);
				}
			}
		} catch (final Exception e) {
			LOG.error(e, e);
			throw new RppsException(e);
		}
	}

	/**
	 * Create relation between card and url.
	 * 
	 * @param cuid
	 *            the icard's CUID
	 * @param url
	 *            the url of site
	 * @return Returns <code>true</code> if relation was added. Otherwise <code>false</code>
	 * @throws RppsException
	 *             some situation
	 * @throws UserProfileException
	 * @deprecated
	 */
	public boolean addUrlCardRelation(final String cuid, final String url) throws RppsException, UserProfileException {
		ICardUsageManager cardUsageManager = null;
		try {
			CUID cuidTemp = new CUID(cuid);
			cardUsageManager = getUserProfileService().getCardUsageManager(getHandler());
			CardInformationRPPS cardCredentialInfo = getCardInformationByCuid(cuidTemp, cardUsageManager);
			return addWebFormCardRelation(cuidTemp, new WebForm(new URI(Utils.cleanURLQuery(url)), null, null, null),
					cardCredentialInfo, cardUsageManager);
		} catch (final Exception e) {
			LOG.error(e, e);
			throw new RppsException(e);
		} finally {
			if (cardUsageManager != null) {
				cardUsageManager.close();
			}
		}
	}

	/**
	 * Create relation between card and relying party.
	 * 
	 * @param cuid
	 *            the icard's CUID
	 * @param rpEndPointTO
	 *            the object which represents logging form on relying party
	 * @return Returns <code>true</code> if relation was added. Otherwise <code>false</code>
	 * @throws RppsException
	 *             some situation
	 * @throws UserProfileException
	 *             some situation
	 */
	public boolean addRPEndPointCardRelation(final String cuid, final RPEndPointTO rpEndPointTO) throws RppsException,
			UserProfileException {
		ICardUsageManager cardUsageManager = null;
		try {
			CUID cuidTemp = new CUID(cuid);
			cardUsageManager = getUserProfileService().getCardUsageManager(getHandler());
			CardInformationRPPS cardCredentialInfo = getCardInformationByCuid(cuidTemp, cardUsageManager);
			WebForm webForm = rpEndPointTO.toWebForm();
			Utils.cleanURLQuery(webForm);
			return addWebFormCardRelation(cuidTemp, webForm, cardCredentialInfo, cardUsageManager);

		} catch (final Exception e) {
			LOG.error(e, e);
			throw new RppsException(e);
		} finally {
			if (cardUsageManager != null) {
				cardUsageManager.close();
			}
		}
	}

	/**
	 * Create relation between card and relying party.
	 * 
	 * @param cuid
	 *            the icard's CUID
	 * @param webForm
	 *            the object which represents logging form on relying party in UserProfileService
	 * @param cardCredentialInfo
	 * @return Returns <code>true</code> if relation was added. Otherwise <code>false</code>
	 * @throws RppsException
	 *             some situation
	 * @throws UserProfileException
	 *             some situation
	 */
	protected boolean addWebFormCardRelation(final CUID cuid, final WebForm webForm,
			final CardInformationRPPS cardCredentialInfo) throws RppsException, UserProfileException {
		ICardUsageManager cardUsageManager = null;
		try {
			Utils.cleanURLQuery(webForm);
			cardUsageManager = getUserProfileService().getCardUsageManager(getHandler());
			return addWebFormCardRelation(cuid, webForm, cardCredentialInfo, cardUsageManager);

		} catch (final Exception e) {
			LOG.error(e, e);
			throw new RppsException(e);
		} finally {
			if (cardUsageManager != null) {
				cardUsageManager.close();
			}
		}
	}

	/**
	 * Create relation between card and relying party.
	 * 
	 * @param cuid
	 *            the icard's CUID
	 * @param webForm
	 *            the object which represents logging form on relying party in UserProfileService
	 * @param cardCredentialInfo
	 * @param cardUsageManager
	 *            instance of ICardUsageManager
	 * @return Returns <code>true</code> if relation was added. Otherwise <code>false</code>
	 * @throws RppsException
	 *             some situation
	 * @throws UserProfileException
	 *             some situation
	 */
	protected boolean addWebFormCardRelation(final CUID cuid, final WebForm webForm,
			final CardInformationRPPS cardInformation, final ICardUsageManager cardUsageManager) throws RppsException,
			UserProfileException {
		try {

			Utils.cleanURLQuery(webForm);

			if (cardInformation != null) {
				final WebForm[] webForms = cardInformation.getWebForms();
				if (webForms != null) {
					for (int i = 0; i < webForms.length; i++) {
						if (webForms[i].equals(webForm)) {
							return true;
						}
					}
				}
			}

			cardUsageManager.setUsedAlways(cuid, webForm);
			return true;
		} catch (final Exception e) {
			LOG.error(e, e);
			throw new RppsException(e);
		}
	}

	/**
	 * Creates a new user profile.
	 * 
	 * @param userProfile
	 *            the user profile information.
	 * @param password
	 *            the password to identify the user.
	 * @return Returns <code>ResponseMessage</code> with <code>code</code> field equals <code>0</code> if profile was
	 *         added. Otherwise returns <code>ResponseMessage</code> with error code or throws an RemoteException
	 * @throws RppsException
	 *             some situation
	 */
	public ResponseMessage addUserProfile(final UserProfileTO userProfile, final String password) throws RppsException {
		try {
			final CallbackHandler handlerTemp = RppsHelper.getCallbackHandler(userProfile.getLoginName(), password);
			getUserProfileService().addUserProfile(handlerTemp,
					ConvertHelper.convertUserProfileTO_to_UserProfile(this.userProfileInstance, userProfile));
			return new ResponseMessage(ResponseMessage.noError);
		} catch (final UserProfileException e) {
			LOG.error(e, e);
			return new ResponseMessage(ResponseMessage.unknownError, e.getMessage());
		}
	}

	/**
	 * Method gets user profile information and prepare authenticated callback handler.
	 * 
	 * @param userId
	 *            the id to identify the user.
	 * @param password
	 *            the password to identify the user.
	 * @throws AccessException
	 *             some situation
	 */
	public void authenticate(final String userId, final String password) throws AccessException {
		try {
			this.userProfileInstance = getUserProfileService().getUserProfile(
					RppsHelper.getCallbackHandler(userId, password));
			if (this.userProfileInstance != null) {
				this.handler = RppsHelper.getCallbackHandler(this.userProfileInstance.getUserIdentifier());
			} else {
				throw new AccessException("Don't found user profile for userId = <" + userId + ">");
			}
		} catch (final UserProfileAuthenticationException e) {
			throw new AccessException(e.getMessage(), e);
		} catch (final Exception e) {
			throw new AccessException(e.getMessage(), e);
		}
	}

	/**
	 * Checks whether <code>RppsService</code> can recognize data format of the provided input stream. Some formats
	 * require authentication information in order to process input data. This method should be used prior to actual
	 * attempt to import data from the input stream in order to retrieve an information about data format and required
	 * authentication data to process the data successfully.
	 * 
	 * @param crd
	 *            context of file as byte array
	 * @return the information about input stream data format if it was recognized successfully and <code>null</code>
	 *         otherwise.
	 * @throws RppsException
	 *             some situation
	 */
	public FormatDescriptorTO checkImportStream(final byte[] crd) throws RppsException {
		try {
			final IFormatDescriptor formatDescriptor = ICardRegistry.getInstance().checkInput(
					new ByteArrayInputStream(crd));
			return convertIFormatDescriptor_to_FormatDescriptorTO(formatDescriptor);
		} catch (final Exception e) {
			LOG.error(e, e);
			throw new RppsException(e);
		}
	}

	/**
	 * Deletes saved authentication card information for specified card.
	 * 
	 * @param cuid
	 *            the icard's CUID
	 * @return Returns updated card object
	 * @throws RppsException
	 *             some situation
	 */
	protected void clearCardCredential(final CUID cuid) throws RppsException {
		try {
			getUserProfileService().clearCardCredential(getHandler(), cuid);
		} catch (final Exception e) {
			LOG.error(e, e);
			throw new RppsException(e);
		}
	}

	/**
	 * Deletes saved authentication card information for specified card.
	 * 
	 * @param cuid
	 *            the icard's CUID
	 * @return Returns updated card object
	 * @throws RppsException
	 *             some situation
	 */
	public ICardTO clearCardCredential(final String cuid) throws RppsException {
		try {
			clearCardCredential(new CUID(cuid));
			return getICardByCUID(cuid);
		} catch (final Exception e) {
			LOG.error(e, e);
			throw new RppsException(e);
		}
	}

	/**
	 * Deletes card history of the specified card on all URLs unless it is checked as "Use Always".
	 * 
	 * @param cuid
	 *            the icard's CUID
	 * @return Returns <code>true</code> if card history was cleared. Otherwise <code>false</code>
	 * @throws RppsException
	 *             some situation
	 */
	protected void clearCardHistory(final CUID cuid) throws RppsException, UserProfileException {
		ICardUsageManager cardUsageManager = null;
		try {
			cardUsageManager = getUserProfileService().getCardUsageManager(getHandler());
			cardUsageManager.clearCardHistory(cuid);
		} catch (final Exception e) {
			LOG.error(e, e);
			throw new RppsException(e);
		} finally {
			if (cardUsageManager != null) {
				cardUsageManager.close();
			}
		}
	}

	/**
	 * Deletes card history of the specified card on all URLs unless it is checked as "Use Always".
	 * 
	 * @param cuid
	 *            the icard's CUID
	 * @return Returns <code>true</code> if card history was cleared. Otherwise <code>false</code>
	 * @throws RppsException
	 *             some situation
	 */
	public boolean clearCardHistory(final String cuid) throws RppsException {
		try {
			clearCardHistory(new CUID(cuid));
			final List list = getCardHistoryLog(new CUID(cuid));
			if (list == null || (list != null && list.size() == 0)) {
				return true;
			} else {
				return false;
			}
		} catch (final Exception e) {
			LOG.error(e, e);
			throw new RppsException(e);
		}
	}

	/**
	 * Converts the card to the card transfer object.
	 * 
	 * @param card
	 *            the Higgins card.
	 * @param isCatchAuthentication
	 *            equals "true" if needs to generate AuthenticationRequiredException or AuthenticationException
	 * @return the card transfer object.
	 * @throws CardException
	 *             some situation
	 * @throws UserProfileException
	 *             some situation
	 * @throws RppsException
	 *             some situation
	 */
	protected ICardTO convertICard_To_ICardTO(final ICard card, final boolean isCatchAuthentication, final Map icardMap)
			throws CardException, RppsException, UserProfileException {
		final ICardTO icard = new ICardTO();
		if (card != null) {
			icard.setId(card.getID());
			icard.setCuid(card.getCUID().toString());
			icard.setName(card.getName());
			icard.setIssuerName(card.getIssuerName());
			icard.setImage(card.getImage());
			icard.setImageType(card.getImageType());
			icard.setSelfIssued(card.isSelfIssued());

			icard.setType(card.getType());

			if (card instanceof IInformationCard) {
				if (((IInformationCard) card).hasExtensions()) {
					IInformationCardExtension[] extension = ((IInformationCard) card).getExtensions();
					if (extension != null) {
						List cardExtensionTOs = new ArrayList();
						for (int i = 0; i < extension.length; i++) {
							try {
								boolean enabled = extension[i].isEnabled();
								String element = (String) extension[i].getElement().getAs(String.class);
								cardExtensionTOs.add(new CardExtensionTO(enabled, element));
							} catch (Exception e) {
								LOG.error(e, e);
							}
						}
						icard.setCardExtensions((CardExtensionTO[]) cardExtensionTOs
								.toArray(new CardExtensionTO[cardExtensionTOs.size()]));
					}
				}
			}
			if (card instanceof IManagedInformationCard) {
				ISTSPrivacyPolicy policy = ((IManagedInformationCard) card).getPrivacyNotice();
				if (policy != null) {
					icard.setStsPolicy(new STSPolicyTO(policy.getPrivacyVersion(), policy.getPrivacyUrl()));
				}
			}

			CardCredential cardCredential = null;
			try {
				final CardInformationRPPS cardInformation = getCardInformationByCuid(card.getCUID());
				if (cardInformation != null) {
					cardCredential = cardInformation.getCardCredential();
				}
			} catch (final Exception e) {
				LOG.error(e, e);
			}

			final UITokenServiceCredentialTO[] tokenServiceCredentials = convertICredentials_to_UITokenServiceCredentialTO(
					card, cardCredential, icardMap);
			if (tokenServiceCredentials.length > 0) {
				icard.setTokenServiceCredentials(tokenServiceCredentials);
			}
			Map propMap = ConvertHelper.convertClaims_to_PropertyTO(card, isCatchAuthentication, card
					.getSupportedClaimTypes());
			if (propMap.size() > 0) {
				icard.setProperties((PropertyTO[]) propMap.values().toArray(new PropertyTO[propMap.values().size()]));
			}
		}
		return icard;
	}

	/**
	 * Converts the card to the transfer object for the CARD element of UserInterfaceResponse.
	 * 
	 * @param card
	 *            the Higgins card.
	 * @param isCatchAuthentication
	 *            equals "true" if needs to generete AuthenticationRequiredException or AuthenticationException
	 * @return the card transfer object.
	 * 
	 * 
	 */
	protected UIResponseCardTO convertICard_To_UIResponseCardTO(final ICard icard, final WebForm webForm,
			final IPolicy policy, final CardInformationRPPS cardInformation, final X509Certificate[] sslCerts) {
		final UIResponseCardTO cardTO = new UIResponseCardTO();
		if (icard != null) {
			// sets uid
			cardTO.setCardId(icard.getCUID().toString());
			cardTO.setUid(icard.getCUID().toString());
			// sets name
			String nm = icard.getName();
			int k;
			nm = (k = nm.lastIndexOf('/')) > 0 ? nm.substring(k + 1) : nm;
			cardTO.setName(nm);
			// sets description
			cardTO.setDescription(icard.getDescription());
			// sets image
			cardTO.setImage(icard.getImage());
			// sets imagetype
			cardTO.setImagetype(icard.getImageType());

			cardTO.setIssuerName(icard.getIssuerName());

			cardTO.setType(icard.getType());

			final List supportedClaimTypeList = icard.getSupportedClaimTypesUris();

			if (supportedClaimTypeList != null && supportedClaimTypeList.size() > 0) {
				final String[] supportedClaimType = new String[supportedClaimTypeList.size()];
				supportedClaimTypeList.toArray(supportedClaimType);
				cardTO.setSupportedClaimTypes(supportedClaimType);
			}

			final UITokenServiceCredentialTO[] tokenservices = convertICredentials_to_UITokenServiceCredentialTO(icard,
					cardInformation != null ? cardInformation.getCardCredential() : null, null);

			cardTO.setTokenservices(tokenservices);

			if (cardInformation != null) {

				final WebForm[] webForms = cardInformation.getWebForms();
				if (webForms != null) {
					for (int i = 0; i < webForms.length; i++) {
						if (webForms[i].equals(webForm)) {
							cardTO.setSaveCard(true);
						}
					}
				}
			}
			String PPID_CLAIM = "http://schemas.xmlsoap.org/ws/2005/05/identity/claims/privatepersonalidentifier";
			Map requiredMap = null;
			try {
				if (policy.getRequiredClaims() != null && policy.getRequiredClaims().size() > 0) {
					requiredMap = ConvertHelper.convertClaims_to_PropertyTO(icard, false, policy.getRequiredClaims()
							.iterator());
				}
			} catch (final Exception e) {
				LOG.error(e, e);
			}
			// if (requiredMap == null) {
			// requiredMap = new LinkedHashMap();
			// }
			if (policy instanceof ICardSpacePolicy) {
				final ICardSpacePolicy policyCS = (ICardSpacePolicy) policy;
				try {
					// calc user friendly ppid
					if (icard instanceof IPersonalInformationCard && requiredMap != null
							&& requiredMap.containsKey(PPID_CLAIM)) {
						String ppid = null;
						byte[] pinDigets = ((IPersonalInformationCard) icard).getPinDigest();
						if (pinDigets == null && sslCerts != null && sslCerts.length > 0 && webForm != null
								&& webForm.getUrl() != null) {
							String uriSite = webForm.getUrl().toString();
							byte[] ppidArray = PPIDHelper.computeClaimValuePPID((InformationCard) icard, sslCerts,
									KeyStoreRegistry.getInstance().getSecurityService().getCertStore(getHandler()),
									uriSite);
							if (ppidArray != null) {
								ppid = PPIDHelper.getUserFriendlyPPID(ppidArray);
							}
						}
						if (ppid != null) {
							requiredMap.put(PPID_CLAIM, new PropertyTO("Site Specific Card ID", PPID_CLAIM, ""
									.getClass().getName(), new String[] { ppid }, true,
									0, new String[] { "" }, "", ""));
						}
					}

					if (policyCS.getOptionalClaims() != null && policyCS.getOptionalClaims().size() > 0) {
						Map propMap = ConvertHelper.convertClaims_to_PropertyTO(icard, false, policyCS
								.getOptionalClaims().iterator());
						if (propMap.size() > 0) {
							cardTO.setOptionalClaims(((PropertyTO[]) propMap.values().toArray(
									new PropertyTO[propMap.values().size()])));
						}
					}
				} catch (final Exception e) {
					LOG.error(e, e);
				}
			}

			try {
				if (requiredMap != null && requiredMap.size() > 0) {
					cardTO.setProperties((PropertyTO[]) requiredMap.values().toArray(
							new PropertyTO[requiredMap.values().size()]));
				}
			} catch (final Exception e) {
			}
		}
		return cardTO;
	}

	/**
	 * Converts the list of card to the ICardResponseTO.
	 * 
	 * @param cards
	 *            the Higgins cards.
	 * @return ICardResponseTO
	 */
	private ICardResponseTO convertICards_To_CardResponceTO(final List cards) {
		final Map icardMap = new Hashtable();
		final List iCardList = new ArrayList();
		final ICardResponseTO cardResponceTO = new ICardResponseTO();
		try {
			if (cards != null) {
				for (int i = 0; i < cards.size(); i++) {
					try {
						final ICard card = (ICard) cards.get(i);
						iCardList.add(convertICard_To_ICardTO(card, false, icardMap));
					} catch (final Exception e) {
						LOG.error(e, e);
					}
				}
			}
		} catch (final Exception e) {
			LOG.error(e, e);
		}
		final List relationList = new ArrayList();
		if (!icardMap.isEmpty()) {
			for (final Iterator iter = icardMap.keySet().iterator(); iter.hasNext();) {
				final String key = (String) iter.next();
				final List value = (List) icardMap.get(key);
				relationList.add(new MapTO(key, (String[]) value.toArray(new String[value.size()])));
			}
		}
		cardResponceTO.setICardList((ICardTO[]) iCardList.toArray(new ICardTO[iCardList.size()]));
		cardResponceTO.setRelationList((MapTO[]) relationList.toArray(new MapTO[relationList.size()]));
		return cardResponceTO;
	}

	/**
	 * Converts ICredential object to UITokenServiceCredential transfer object.
	 * 
	 * @param handler
	 *            the callback handler to be used to request user's identity
	 * @param card
	 *            ICard object
	 * @param cardCredential
	 * @return UITokenServiceCredentialTO object
	 */
	protected UITokenServiceCredentialTO[] convertICredentials_to_UITokenServiceCredentialTO(final ICard card,
			final CardCredential cardCredential, final Map icardMap) {
		final List credList = new ArrayList();
		if (card != null) {
			final ICredentialDescriptor[] der = card.getRequiredCredentials();
			if (der != null) {
				for (int i = 0; i < der.length; i++) {
					final ICredential credential = der[i].getCredential();

					final UITokenServiceCredentialTO credentialTO = convertICredentials_to_UITokenServiceCredentialTO(
							credential, icardMap, card);

					if (credentialTO != null) {
						if (cardCredential != null && credentialTO.getType().equals(cardCredential.getCredentialType())) {
							credentialTO.setSaveCredential(true);
						}

						credList.add(credentialTO);
					}
				}
			}
		}
		final UITokenServiceCredentialTO[] array = new UITokenServiceCredentialTO[credList.size()];
		credList.toArray(array);
		return array;
	}

	/**
	 * Converts ICredential object to UITokenServiceCredential transfer object.
	 * 
	 * @param handler
	 *            the callback handler to be used to request user's identity
	 * @param card
	 *            ICard object
	 * @param cardCredential
	 * @return UITokenServiceCredentialTO object
	 */
	protected UITokenServiceCredentialTO convertICredentials_to_UITokenServiceCredentialTO(
			final ICredential credential, final Map icardMap, final ICard icard) {
		UITokenServiceCredentialTO tokenService = null;
		if (credential != null) {
			tokenService = new UITokenServiceCredentialTO();
			if (credential instanceof ITokenServiceCredential) {

				tokenService.setAddress(((ITokenServiceCredential) credential).getAddress().toString());
				tokenService.setMetadataAddress(((ITokenServiceCredential) credential).getMetadataAddress().toString());

				if (credential instanceof ITSUsernamePasswordCredential) {
					tokenService.setPassword(((ITSUsernamePasswordCredential) credential).getPassword());
					tokenService.setUsername(((ITSUsernamePasswordCredential) credential).getUsername());
					tokenService.setType(RppsService.ITSUsernamePasswordCredential);

				} else
					if (credential instanceof ITSSelfIssuedCredential) {
						tokenService.setType(RppsService.ITSSelfIssuedNonPinCredential);
						try {
							final ICard personalCard = getICardByPPID(((ITSSelfIssuedCredential) credential)
									.getCertificate(), ((ITSSelfIssuedCredential) credential).getPPID(),
									((ITSSelfIssuedCredential) credential).getAddress());
							if (personalCard == null) {
								LOG.info("Cannot find the Personal card used to authenticate for this managed card.");
								return null;
							}

							if (icardMap != null && icard != null) {
								List relationCUIDs = null;

								if (icardMap.containsKey(personalCard.getCUID().toString())) {
									relationCUIDs = (List) icardMap.get(personalCard.getCUID().toString());
								}

								if (relationCUIDs == null) {
									relationCUIDs = new ArrayList();
								}

								if (!relationCUIDs.contains(icard.getCUID().toString())) {
									relationCUIDs.add(icard.getCUID().toString());
								}

								icardMap.put(personalCard.getCUID().toString(), relationCUIDs);
							}

							tokenService.setAssociatedCardName(personalCard.getName());
							tokenService.setAssociatedCardImage(personalCard.getImage());
							tokenService.setAssociatedCardImageType(personalCard.getDescription());
							tokenService.setAssociatedCardDescription(personalCard.getDescription());

							ICredential subCredential = null;
							final ICredentialDescriptor[] derPersonal = personalCard.getRequiredCredentials();
							if (derPersonal != null && derPersonal.length > 0) {
								subCredential = derPersonal[0].getCredential();
							}

							if (subCredential != null) {
								for (int index = 0; index < subCredential.getCallbacks().length; index++) {
									if (subCredential.getCallbacks()[index] instanceof PinCodeCallback) {

										tokenService.setType(RppsService.ITSSelfIssuedCredential);
										break;
									}
								}
							}
						} catch (final Exception e) {
							LOG.error(e, e);
						}
					} else
						if (credential instanceof ITSKerberosV5Credential) {
							tokenService.setType(RppsService.ITSKerberosV5Credential);

						} else
							if (credential instanceof ITSX509V3Credential) {
								tokenService.setType(RppsService.ITSX509V3Credential);
							} else {
								tokenService = null;
							}
			} else {
				if (credential instanceof IPinCodeCredential) {
					tokenService.setType(RppsService.IPinCodeCredential);
				} else
					if (credential instanceof IUsernamePasswordCredential) {
						tokenService.setType(RppsService.IUsernamePasswordCredential);
						tokenService.setUsername(((IUsernamePasswordCredential) credential).getUsername());
					} else
						if (credential instanceof IPasswordCredential) {
							tokenService.setType(RppsService.IPasswordCredential);
						} else {
							tokenService = null;
						}
			}
		}
		return tokenService;
	}

	/**
	 * Converts IFormatDescriptor object to FormatDescriptorTO transfer object.
	 * 
	 * @param descriptor
	 * @return FormatDescriptorTO
	 */
	protected FormatDescriptorTO convertIFormatDescriptor_to_FormatDescriptorTO(final IFormatDescriptor descriptor) {
		FormatDescriptorTO descriptorTO = null;
		if (descriptor != null) {
			descriptorTO = new FormatDescriptorTO();
			descriptorTO.setFormatID(descriptor.getID());
			descriptorTO.setName(descriptor.getDescription());
			descriptorTO.setFileExtension(descriptor.getFileExtension());
			if (descriptor.getCredentialDescriptor() != null
					&& descriptor.getCredentialDescriptor().getCredential() != null) {
				final UITokenServiceCredentialTO credential = convertICredentials_to_UITokenServiceCredentialTO(
						descriptor.getCredentialDescriptor().getCredential(), null, null);
				descriptorTO.setCredential(credential);
			}
		}

		return descriptorTO;
	}

	/**
	 * Creates duplicate of personal card only with new card's name.
	 * 
	 * @param cardName
	 *            new card's name
	 * @param cuid
	 *            the old icard's cuid
	 * @param pinCode
	 *            pinCode for retrieve claims of personal card if that is pin protected. Otherwise <code>null</code>
	 * @return Returns updated card object
	 * @throws RppsException
	 *             some situation
	 */
	protected ICard createDuplicateICard(final String cardName, final CUID cuid, final String pinCode)
			throws RppsException, UserProfileException {
		try {
			ICard card = ICardRegistry.getInstance().getICardByCUID(getHandler(), cuid);

			if (!(card instanceof IPersonalInformationCard)) {
				throw new RppsException("You can not create dublicate this card.");
			}
			if (card.getRequiredCredentials() != null && card.getRequiredCredentials().length > 0) {
				ICredential credential = null;
				if (pinCode != null && pinCode.trim().length() > 0) {
					credential = new PinCodeCredential();
					try {
						((PinCodeCredential) credential).setPinCode(pinCode.getBytes("UTF-8"));
					} catch (final UnsupportedEncodingException e) {
						((PinCodeCredential) credential).setPinCode(pinCode.getBytes());
					}
				} else {
					final CardInformationRPPS cardInformation = getCardInformationByCuid(cuid);
					if (cardInformation != null && cardInformation.getCardCredential() != null) {
						credential = ConvertHelper.convertCardCredential_to_ICredential(card, cardInformation
								.getCardCredential());
					} else {
						throw new RppsException("Rpps can not retrieve card. You have to send pin code.");
					}
				}
				card = card.getProvider().getICardByCUID(getHandler(), cuid, credential);
			}

			final ICardTemplate[] cardTemplates = card.getProvider().getCardCreationTemplates(getHandler());
			ICardTemplate cardTemplate = null;
			for (int i = 0; i < cardTemplates.length; i++) {

				if (!cardTemplates[i].getCardType().isInstance(card)) {
					continue;
				}

				cardTemplate = cardTemplates[i];

				if (cardTemplate.getTemplateElementByID("cardName") != null) {
					((ITemplateValue) cardTemplate.getTemplateElementByID("cardName")).setValue(cardName);
				}
				if (cardTemplate.getTemplateElementByID("cardPicture") != null && card.getImage() != null) {
					((ITemplateValue) cardTemplate.getTemplateElementByID("cardPicture")).setValue(Base64.encode(card
							.getImage()));
				}
				if (cardTemplate.getTemplateElementByID("claimList") != null) {
					for (final Iterator iter = ((ITemplateContainer) cardTemplate.getTemplateElementByID("claimList"))
							.getTemplateElements(); iter.hasNext();) {
						final ITemplateElement element = (ITemplateElement) iter.next();
						final IClaim claim = card.getClaim(element.getID());
						if (claim != null) {
							if (((ISimpleClaim) claim).getValues().size() > 0) {
								((ITemplateValue) element).setValue((String) ((ISimpleClaim) claim).getValues().get(0));
							}
						}
					}
				}
				break;
			}
			if (cardTemplate != null) {
				return createICard(cardTemplate);
			}
		} catch (final CardException e) {
			LOG.error(e, e);
			throw new RppsException(e);
		}

		return null;
	}

	/**
	 * Creates duplicate of personal card only with new card's name.
	 * 
	 * @param cardName
	 *            new card's name
	 * @param cuid
	 *            the old icard's cuid
	 * @param pinCode
	 *            pinCode for retrieve claims of personal card if that is pin protected. Otherwise <code>null</code>
	 * @return Returns updated card object
	 * @throws RppsException
	 *             some situation
	 */
	public ICardTO createDuplicateICard(final String cardName, final String cuid, final String pinCode)
			throws RppsException {
		ICard card = null;
		try {
			card = createDuplicateICard(cardName, new CUID(cuid), pinCode);
			return convertICard_To_ICardTO(card, false, null);
		} catch (final Exception e) {
			LOG.error(e, e);
			throw new RppsException(e);
		} finally {

		}
	}

	/**
	 * Creates a new card by template.
	 * 
	 * @param template
	 *            the icard's template
	 * @return a new card object
	 * @throws RemoteException
	 */
	protected ICard createICard(final ICardTemplate template) throws RppsException {
		try {
			if (template != null) {
				// gets all registered providers
				final String extID = template.getProviderID();
				final ICardProvider provider = ICardRegistry.getInstance().getICardProvider(extID);
				if (provider != null) {
					// check if current provider supports needed protocol
					if (provider.getCardCreationTemplates(getHandler()) != null) {
						try {
							return provider.createCard(getHandler(), template);
						} catch (final Exception ex) {
							throw new RppsException(ex);
						}
					}
				}
				throw new RppsException("There is not a registered provider for this id.");
			}
			throw new RppsException("Paramenter <<template>> is null");
		} catch (final RppsException e) {
			LOG.error(e, e);
			throw e;
		}
	}

	/**
	 * Creates a new card by template.
	 * 
	 * @param template
	 *            the icard's template
	 * @return a new card object
	 * @throws RemoteException
	 */
	public ICardTO createICard(final ICardTemplateTO template) throws RppsException {
		ICard card = null;
		try {
			final ICardTemplate[] templates = getICardCreationTemplateByProvider(template.getProviderID());
			card = createICard(RppsHelper.fillICardTemplate_from_ICardTemplateTO(templates, template));
			return convertICard_To_ICardTO(card, false, null);
		} catch (final Exception e) {
			LOG.error(e, e);
			throw new RppsException(e);
		} finally {

		}
	}

	/**
	 * Creates a new card for HBX.
	 * 
	 * @param cardname
	 *            the card's name
	 * @return a new card's cuid
	 * @throws RppsException
	 *             some situation
	 * @deprecated
	 */
	public String createICardFromHBX(final String cardname) throws RppsException {
		String ret = null;
		try {
			ICardTemplate template = getICardCreationTemplateByTUID(new TUID(
					"org.eclipse.higgins.icard.provider.cardspace.personal.db", "PersonalCardTemplate"));
			template = RppsHelper.fillPersonalCardTemplate(template, cardname, "", "", "", "", "", "", "", "", "", "",
					"", "", "", "");
			ret = createICard(template).getCUID().toString();
		} catch (final Exception e) {
			throw new RppsException(e);
		} finally {
		}
		return ret;
	}

	/**
	 * Deletes all user's card.
	 * 
	 * @throws RppsException
	 *             some situation
	 */
	public void deleteAllICard() throws RppsException {
		final List icards = getICardsByUser(getHandler());

		for (final Iterator iter = icards.iterator(); iter.hasNext();) {
			deleteICardOnly(((ICard) iter.next()).getCUID().toString());
		}
	}

	/**
	 * Delete specified record of card history.
	 * 
	 * @param cuid
	 *            the icard's CUID
	 * @param webForm
	 *            webForm on web page which will be deleted from card history.
	 * @param cardUsageManager
	 *            instance of ICardUsageManager
	 * @return Returns <code>true</code> if card history record was deleted. Otherwise <code>false</code>
	 * @throws RppsException
	 *             some situation
	 */
	protected boolean deleteCardHistoryRecord(final CUID cuid, final WebForm webForm,
			final ICardUsageManager cardUsageManager) throws RppsException {
		try {
			Utils.cleanURLQuery(webForm);
			cardUsageManager.clearCardHistory(cuid, webForm);
			return true;
		} catch (final Exception e) {
			LOG.error(e, e);
			throw new RppsException(e);
		}
	}

	/**
	 * Delete specified record of card history.
	 * 
	 * @param cuid
	 *            the icard's CUID
	 * @param url
	 *            url of site which will be deleted from card history
	 * @return Returns <code>true</code> if card history record was deleted. Otherwise <code>false</code>
	 * @throws RppsException
	 *             some situation
	 * @deprecated
	 */
	public boolean deleteCardHistoryRecord(final String cuid, final String url) throws RppsException {
		ICardUsageManager cardUsageManager = null;
		try {
			cardUsageManager = getUserProfileService().getCardUsageManager(getHandler());
			return deleteCardHistoryRecord(new CUID(cuid), new WebForm(new URI(Utils.cleanURLQuery(url)), null, null,
					null), cardUsageManager);
		} catch (final Exception e) {
			LOG.error(e, e);
			throw new RppsException(e);
		} finally {
			if (cardUsageManager != null) {
				try {
					cardUsageManager.close();
				} catch (final UserProfileException e) {
					LOG.error(e, e);
				}
			}
		}
	}

	/**
	 * Delete specified record of card history.
	 * 
	 * @param cuid
	 *            the icard's CUID
	 * @param rpEndPointTO
	 *            the object which represents logging form on relying party
	 * @return Returns <code>true</code> if card history record was deleted. Otherwise <code>false</code>
	 * @throws RppsException
	 *             some situation
	 */
	public boolean deleteCardHistoryRecord(final String cuid, final RPEndPointTO rpEndPointTO) throws RppsException {
		ICardUsageManager cardUsageManager = null;
		try {
			cardUsageManager = getUserProfileService().getCardUsageManager(getHandler());
			WebForm webForm = rpEndPointTO.toWebForm();
			Utils.cleanURLQuery(webForm);
			return deleteCardHistoryRecord(new CUID(cuid), webForm, cardUsageManager);
		} catch (final Exception e) {
			LOG.error(e, e);
			throw new RppsException(e);
		} finally {
			if (cardUsageManager != null) {
				try {
					cardUsageManager.close();
				} catch (final UserProfileException e) {
					LOG.error(e, e);
				}
			}
		}
	}

	/**
	 * Delete the card's category.
	 * 
	 * @param categoryId
	 *            category identifier
	 * @throws RppsException
	 *             some situation
	 */
	public void deleteCategory(final String categoryId) throws RppsException {
		try {
			getUserProfileService().deleteCategory(getHandler(), categoryId);
		} catch (final Exception e) {
			LOG.error(e, e);
			throw new RppsException(e);
		}
	}

	/**
	 * Deletes specified card.
	 * 
	 * @param cuid
	 *            the icard's CUID
	 * @return Returns ICardUpdateResponseTO which contains delete status for card. If <code>code</code> field equals
	 *         <code>0</code> then card was updated correctly. Otherwise <code>code</code> field contains error code. If
	 *         deleted card had self signed type then <code>cuid</code> field contains cuid of personal card which had
	 *         been used to signed managed card.
	 * @throws RppsException
	 *             some situation
	 */
	public ICardUpdateResponseTO deleteICard(final String cuid) throws RppsException {
		ICard personalCard = null;
		try {
			final ICard icard = getICardByCUID(new CUID(cuid));
			final ICredentialDescriptor[] credDescriptor = icard.getRequiredCredentials();
			if (credDescriptor != null) {
				for (int i = 0; i < credDescriptor.length; i++) {
					if (credDescriptor[i].getCredential() != null
							&& credDescriptor[i].getCredential() instanceof ITSSelfIssuedCredential) {
						final ITSSelfIssuedCredential credential = (ITSSelfIssuedCredential) credDescriptor[i]
								.getCredential();
						personalCard = getICardByPPID(credential.getCertificate(), credential.getPPID(), credential
								.getAddress());
					}
				}
			}
			deleteICardOnly(cuid);
		} catch (final RppsException e) {
			LOG.error(e, e);
			return new ICardUpdateResponseTO(ICardUpdateResponseTO.unknownError, e.getMessage());
		} catch (final Exception e) {
			LOG.error(e, e);
			return new ICardUpdateResponseTO(ICardUpdateResponseTO.unknownError, e.getMessage());
		}
		if (personalCard != null) {
			return new ICardUpdateResponseTO(personalCard.getCUID().toString());
		}
		return new ICardUpdateResponseTO(ICardUpdateResponseTO.noError, "");
	}

	/**
	 * Deletes specified card and all information which associate with card.
	 * 
	 * @param cuid
	 *            the icard's CUID
	 * @throws RppsException
	 *             some situation
	 */
	private void deleteICardOnly(final String cuid) throws RppsException {
		try {
			final CUID cuidTemp = new CUID(cuid);
			getUserProfileService().clearCardCredential(getHandler(), cuidTemp);
			ICardUsageManager cardUsageManager = null;
			try {
				cardUsageManager = getUserProfileService().getCardUsageManager(getHandler());
				cardUsageManager.clearUsedAlwaysURLs(cuidTemp);
				cardUsageManager.clearCardHistory(cuidTemp);
			} finally {
				if (cardUsageManager != null) {
					cardUsageManager.close();
				}
			}
			ICardRegistry.getInstance().deleteICard(getHandler(), new CUID(cuid));

			// check if card is default p-card
			if (cuid.equals(getUserProfileInstance().getDefaultPCardCUID())) {
				// clear default personal card setting
				setDefaultPCard(null);
			}
		} catch (final CardException e) {
			LOG.error(e, e);
			throw new RppsException(e);
		} catch (final UserProfileException e) {
			LOG.error(e, e);
			throw new RppsException(e);
		}
	}

	/**
	 * @throws UserProfileException
	 */
	public void deleteOldPassword() throws UserProfileException {
		try {
			getUserProfileService().deleteOldPassword(getHandler());
		} catch (final Exception e) {
			LOG.error(e, e);
			throw new RppsException(e);
		}
	}

	/**
	 * Delete relation between card and relying party.
	 * 
	 * @param cuid
	 *            the icard's CUID
	 * @param webForm
	 *            the object which represents logging form on relying party in UserProfileService
	 * @return Returns <code>true</code> if relation was deleted. Otherwise <code>false</code>
	 * @throws RppsException
	 *             some situation
	 */
	protected CardUsage deleteWebFormCardRelation(final CUID cuid, final WebForm webForm) throws RppsException,
			UserProfileException {

		ICardUsageManager cardUsageManager = null;
		try {
			Utils.cleanURLQuery(webForm);
			cardUsageManager = getUserProfileService().getCardUsageManager(getHandler());
			return cardUsageManager.deleteUsedAlways(cuid, webForm);
		} catch (final Exception e) {
			LOG.error(e, e);
			throw new RppsException(e);
		} finally {
			if (cardUsageManager != null) {
				cardUsageManager.close();
			}
		}
	}

	/**
	 * Delete relation between card and relying party.
	 * 
	 * @param cuid
	 *            the icard's CUID
	 * @param rpEndPointTO
	 *            the object which represents logging form on relying party
	 * @return Returns <code>true</code> if relation was deleted. Otherwise <code>false</code>
	 * @throws RppsException
	 *             some situation
	 */
	public boolean deleteRPEndPointCardRelation(final String cuid, final RPEndPointTO rpEndPointTO)
			throws RppsException {
		try {
			if (rpEndPointTO != null) {
				WebForm webForm = rpEndPointTO.toWebForm();
				Utils.cleanURLQuery(webForm);
				deleteWebFormCardRelation(new CUID(cuid), webForm);
			}
			return true;
		} catch (final Exception e) {
			LOG.error(e, e);
			throw new RppsException(e);
		}
	}

	/**
	 * Delete relation between card and site's url.
	 * 
	 * @param cuid
	 *            the icard's CUID
	 * @param url
	 *            the url of site
	 * @return Returns <code>true</code> if relation was deleted. Otherwise <code>false</code>
	 * @throws RppsException
	 *             some situation
	 * @deprecated
	 */
	public boolean deleteUrlCardRelation(final String cuid, final String url) throws RppsException {
		try {
			if (url != null) {
				deleteWebFormCardRelation(new CUID(cuid), new WebForm(new URI(Utils.cleanURLQuery(url)), null, null,
						null));
			}
			return true;
		} catch (final Exception e) {
			LOG.error(e, e);
			throw new RppsException(e);
		}
	}

	/**
	 * Delete all relations between card and site's urls.
	 * 
	 * @param cuid
	 *            the icard's CUID
	 * @return Returns <code>true</code> if relations were deleted. Otherwise <code>false</code>
	 * @throws RppsException
	 *             some situation
	 * @deprecated
	 */
	public boolean deleteUrlsCardsRelations(final String cuid) throws RppsException {
		return deleteRPEndPointsCardsRelations(cuid);
	}

	/**
	 * Delete relation between card and relying party.
	 * 
	 * @param cuid
	 *            the icard's CUID
	 * @return Returns <code>true</code> if relation was deleted. Otherwise <code>false</code>
	 * @throws RppsException
	 *             some situation
	 */
	public boolean deleteRPEndPointsCardsRelations(final String cuid) throws RppsException {
		try {
			ICardUsageManager cardUsageManager = null;
			try {
				cardUsageManager = getUserProfileService().getCardUsageManager(getHandler());
				cardUsageManager.clearUsedAlwaysURLs(new CUID(cuid));
				return true;
			} finally {
				if (cardUsageManager != null) {
					cardUsageManager.close();
				}
			}
		} catch (final CardException e) {
			LOG.error(e, e);
			throw new RppsException(e);
		} catch (final UserProfileException e) {
			LOG.error(e, e);
			throw new RppsException(e);
		}
	}

	/**
	 * Deletes user profile.
	 * 
	 * @throws RppsException
	 *             some situation
	 */
	public ResponseMessage deleteUserProfile() throws RppsException {
		try {
			deleteUserProfile(getHandler());
			return new ResponseMessage(ResponseMessage.noError);
		} catch (final RppsException e) {
			LOG.error(e, e);
			return new ResponseMessage(ResponseMessage.unknownError, e.getMessage());
		}
	}

	/**
	 * Deletes user profile.
	 * 
	 * @param handler
	 *            authenticated callback handler
	 * @throws RppsException
	 *             some situation
	 */
	protected void deleteUserProfile(final CallbackHandler handlerTemp) throws RppsException {
		try {
			deleteAllICard();
			getUserProfileService().deleteUserProfile(handlerTemp);
		} catch (final UserProfileException e) {
			LOG.error(e, e);
			throw new RppsException(e);
		}
	}

	/**
	 * Changes pin code for personal card.
	 * 
	 * @param cuid
	 *            the icard's CUID
	 * @param oldPinCode
	 *            Old pin code
	 * @param newPinCode
	 *            New pin code
	 * @return Returns updated card object
	 * @throws RppsException
	 *             some situation
	 */
	public ICardTO editPinCode(final String cuid, final String oldPinCode, final String newPinCode)
			throws RppsException {
		ICard card = null;
		byte[] oldPinCodeTemp = null;
		try {
			card = getICardByCUID(new CUID(cuid));

			if (card == null) {
				throw new RppsException("Can't find card with CUID " + cuid);
			}

			if (!(card instanceof IPersonalInformationCard)) {
				new RppsException("This card is not personal");
			}
			CardInformationRPPS cardInformation = null;
			if (oldPinCode == null) {
				cardInformation = getCardInformationByCuid(new CUID(cuid));
				if (cardInformation.getCardCredential() != null
						&& IRppsService.IPinCodeCredential.equals(cardInformation.getCardCredential()
								.getCredentialType())) {
					oldPinCodeTemp = cardInformation.getCardCredential().getPinCode();
				}
			} else {
				oldPinCodeTemp = oldPinCode.getBytes("UTF-8");
			}

			card = resetPinCode((IPersonalInformationCard) card, oldPinCodeTemp);
			card = setPinCode((IPersonalInformationCard) card, newPinCode.getBytes("UTF-8"));
			if (cardInformation != null) {
				try {
					if (cardInformation.getCardCredential() != null) {
						final CardCredential cardCredential = cardInformation.getCardCredential();
						cardCredential.setPinCode(newPinCode.getBytes("UTF-8"));
						setCardCredential(new CUID(cuid), cardCredential);
					}
				} catch (final Exception e) {
					LOG.error(e, e);
				}
			}
			return convertICard_To_ICardTO(card, false, null);
		} catch (final Exception e) {
			LOG.error(e, e);
			throw new RppsException(e);
		}
	}

	/**
	 * Returns file as byte array with special which contains select cards.
	 * 
	 * @param formatID
	 *            file's format identifier
	 * @param cards
	 *            the icard's CUIDs
	 * @param credential
	 *            authentication information
	 * @return array of byte
	 * @throws RppsException
	 *             some situation
	 */
	public byte[] exportICards(final String formatID, final String[] cards, final UITokenServiceCredentialTO credential)
			throws RppsException {
		try {
			final ByteArrayOutputStream out = new ByteArrayOutputStream();
			if (cards != null && cards.length > 0) {
				final CUID[] cuids = new CUID[cards.length];
				for (int i = 0; i < cards.length; i++) {
					cuids[i] = new CUID(cards[i]);
				}
				final ICredential credentialTemp = ConvertHelper
						.convertUITokenServiceCredentialTO_to_ICredentials(credential);
				ICardRegistry.getInstance().exportICards(getHandler(), out, formatID, cuids, credentialTemp);
				return out.toByteArray();
			}
		} catch (final Exception e) {
			LOG.error(e, e);
			throw new RppsException(e);
		}
		return null;
	}

	/**
	 * Creates AND element as well as sub-elements of the UIResponse transfer object.
	 * 
	 * @param uct
	 *            the UserChoiceTree object.
	 * @return the array of UIResponseAndTO objects.
	 * @throws UserProfileException
	 * @throws RppsException
	 *             some situation
	 */
	protected UIResponseAndTO[] getAnds(final UserChoiceTree uct, final WebForm webForm,
			final X509Certificate[] sslCerts) throws RppsException, UserProfileException {
		Utils.cleanURLQuery(webForm);
		final List andList = new ArrayList();
		if (uct != null) {
			for (final ListIterator li = uct.getListIterator(); li.hasNext();) {
				final Object child = li.next();
				final UIResponseAndTO and = new UIResponseAndTO();
				if (child instanceof UserChoiceTree) {
					final List orList = new ArrayList();
					for (final ListIterator li2 = ((UserChoiceTree) child).getListIterator(); li2.hasNext();) {
						final Object child2 = li2.next();
						final UIResponseOrTO or = new UIResponseOrTO();
						if (child2 instanceof UserChoiceTree) {
							final List andList2 = new ArrayList();
							for (final ListIterator li3 = ((UserChoiceTree) child2).getListIterator(); li3.hasNext();) {
								final Object child3 = li3.next();
								final UIResponseAndTO and2 = new UIResponseAndTO();
								if (child3 instanceof UCTelm) {
									// ???
								}
								andList2.add(and2);
							}
						} else
							if (child2 instanceof UCTelm) {

								final IPolicy atom = ((UCTelm) child2).getAtom();
								final UIResponseClauseTO clause = new UIResponseClauseTO();

								final RPPolicy policy = ConvertHelper.convertIPolicyToRPPolicy(atom);

								clause.setLabelType(policy.getLabelType());
								// TODO It must be re-factored
								// Quick fix http://graceland.parityinc.net:8090/browse/WWM-1414
								// clause.setLabel(policy.getLabel());
								clause.setLabel(policy.getLabel());

								clause.setLonglabel(policy.getLonglabel());
								clause.setOptional(policy.getOptional());
								clause.setLongOptional(policy.getLongOptional());
								clause.setOptionalType(policy.getOptionalType());

								final PolicyVersion policyVersion = getUserProfileService().getRPPolicyVersion(
										getHandler(), webForm);

								if (atom instanceof ICardSpacePolicy) {
									final ICardSpacePolicy icp = (ICardSpacePolicy) atom;
									if (policyVersion != null && policyVersion.getVersion() != null
											&& !policyVersion.getVersion().equals(icp.getPrivacyVersion())) {
										try {
											final IPolicy oldPolicy = ICardSelectorService.getInstance().parsePolicy(
													policyVersion.getPolicy());
											final RPPolicy oldRpPolicy = ConvertHelper
													.convertIPolicyToRPPolicy(oldPolicy);
											clause.setOldPolicy(oldRpPolicy);
										} catch (final Exception e) {
											LOG.error(e, e);
										}
									}
								}

								final UIResponseUsableCardsTO uc = new UIResponseUsableCardsTO();
								final List cardList = new ArrayList();
								for (final Iterator it = ((UCTelm) child2).getCredsIterator(); it != null
										&& it.hasNext();) {
									UIResponseCardTO card = new UIResponseCardTO();
									final CredentialContainer cc = (CredentialContainer) it.next();
									final ICard cred = cc.getCredential();

									// @see http://graceland.parityinc.net:8090/browse/WWM-1414
									// set UIResponseClauseTO.label from Display tag
									try {
										for (Iterator iterator = cred.getSupportedClaimTypes(); iterator.hasNext();) {
											IClaimType claimTypeItem = (IClaimType) iterator.next();

											// get displayName from ICard if cardspace unknown that type
											// @see implementation of
											// org.eclipse.higgins.icard.provider.cardspace.common.CardSpacePolicy#getSummary()
											for (int i = 0; i < clause.getLabel().length; i++) {
												if (claimTypeItem != null
														&& claimTypeItem.getType().equals(clause.getLabel()[i])) {
													clause.getLabel()[i] = claimTypeItem.getDisplayName();
												}
												// claimTypeItem.getType().equals(clause.getLabelType())
											}
										}
									} catch (Exception e) {
										LOG.warn(e, e);
									}

									final CardInformationRPPS cardInformation = getCardInformationByCuid(cred.getCUID());
									// sets uid
									card = convertICard_To_UIResponseCardTO(cred, webForm, atom, cardInformation,
											sslCerts);

									card.setCardId(cc.key);

									if (cardInformation != null) {
										final WebForm[] webForms = cardInformation.getWebForms();
										if (webForms != null) {
											for (int i = 0; i < webForms.length; i++) {
												if (webForms[i].equals(webForm)) {
													card.setSaveCard(true);
												}
											}
										}
									}

									cardList.add(card);
								}

								final UIResponseCardTO[] cards = new UIResponseCardTO[cardList.size()];
								cardList.toArray(cards);
								uc.setCard(cards);
								clause.setUsableCards(uc);

								final List usedCard = getUsedCards(webForm);
								final String[] usedCardCuids = new String[usedCard.size()];
								for (int index = 0; index < usedCard.size(); index++) {
									usedCardCuids[index] = ((CardUsage) usedCard.get(index)).getCuid().toString();
								}
								clause.setUsedCardCuids(usedCardCuids);

								or.setClause(clause);
							}
						orList.add(or);
					}
					final UIResponseOrTO[] ors = new UIResponseOrTO[orList.size()];
					orList.toArray(ors);
					and.setOr(ors);
				} else
					if (child instanceof UCTelm) {
						// ???
					}
				andList.add(and);
			}
		}
		final UIResponseAndTO[] ands = new UIResponseAndTO[andList.size()];
		andList.toArray(ands);
		return ands;
	}

	/**
	 * Return list of card history records.
	 * 
	 * @param cuid
	 *            the icard's CUID
	 * @return list of CardUsage object
	 * @throws RppsException
	 *             some situation
	 * @throws UserProfileException
	 *             some situation
	 */
	protected List getCardHistory(final CUID cuid) throws RppsException, UserProfileException {

		ICardUsageManager cardUsageManager = null;
		try {
			cardUsageManager = getUserProfileService().getCardUsageManager(getHandler());
			return cardUsageManager.getCardUsageSummary(cuid);
		} catch (final Exception e) {
			LOG.error(e, e);
			throw new RppsException(e);
		} finally {
			if (cardUsageManager != null) {
				cardUsageManager.close();
			}
		}
	}

	/**
	 * Return list of card history records, beginning at the specified position in selection.
	 * 
	 * @param cuid
	 *            the icard's CUID
	 * @param startFrom
	 *            starting position in the selection
	 * @param size
	 *            count of record which will be returned
	 * @param orderBy
	 *            name of field which be used for sorting. Can take one of these values: "<code>date</code>","
	 *            <code>site</code>","<code>useAlways</code>"
	 * @param desc
	 *            If value of this parameter is "<code>desc</code>" then result will be sorted in descending order.
	 *            Otherwise in ascending order.
	 * @return list of CardUsage object
	 * @throws RppsException
	 *             some situation
	 * @throws UserProfileException
	 *             some situation
	 */
	protected List getCardHistory(final CUID cuid, final int startFrom, final int size, final String orderBy,
			final boolean desc) throws RppsException, UserProfileException {

		ICardUsageManager cardUsageManager = null;
		try {
			cardUsageManager = getUserProfileService().getCardUsageManager(getHandler());
			return cardUsageManager.getCardUsageSummary(cuid, startFrom, size, orderBy, desc);
		} catch (final Exception e) {
			LOG.error(e, e);
			throw new RppsException(e);
		} finally {
			if (cardUsageManager != null) {
				cardUsageManager.close();
			}
		}
	}

	/**
	 * Return list of card history records, beginning at the specified position in selection.
	 * 
	 * @param cuid
	 *            the icard's CUID
	 * @param startFrom
	 *            starting position in the selection
	 * @param size
	 *            count of record which will be returned
	 * @param orderBy
	 *            name of field which be used for sorting. Can take one of these values: "<code>date</code>","
	 *            <code>site</code>","<code>useAlways</code>"
	 * @param orderDirection
	 *            If value of this parameter is "<code>desc</code>" then result will be sorted in descending order.
	 *            Otherwise in ascending order.
	 * @return array of CardUsageTO
	 * @throws RppsException
	 *             some situation
	 */
	public CardUsageTO[] getCardHistory(final String cuid, final int startFrom, final int size, final String orderBy,
			final String orderDirection) {
		final boolean desc = orderDirection.equalsIgnoreCase("desc") ? true : false;
		String order = CardUsage.BY_SITE;
		if ("date".equalsIgnoreCase(orderBy)) {
			order = CardUsage.BY_DATE;
		} else
			if ("site".equalsIgnoreCase(orderBy)) {
				order = CardUsage.BY_USE;
			}

		try {
			final List list = getCardHistory(new CUID(cuid), startFrom, size, order, desc);
			final CardUsageTO[] arrayUsage = new CardUsageTO[list.size()];
			for (int i = 0; i < list.size(); i++) {
				final CardUsage cardUsage = (CardUsage) list.get(i);
				arrayUsage[i] = ConvertHelper.convertCardUsage_to_CardUsageTO(cardUsage);
			}
			return arrayUsage;
		} catch (final Exception e) {
			LOG.error(e, e);
			throw new RppsException(e);
		}
	}

	private List getCardHistoryLog(final CUID cuid) throws UserProfileException {
		ICardUsageManager cardUsageManager = null;
		try {
			cardUsageManager = getUserProfileService().getCardUsageManager(getHandler());
			return cardUsageManager.getCardHistory(cuid);
		} catch (final Exception e) {
			LOG.error(e, e);
			throw new RppsException(e);
		} finally {
			if (cardUsageManager != null) {
				cardUsageManager.close();
			}
		}
	}

	private List getCardHistoryLog(final CUID cuid, final int startFrom, final int size, final String orderBy,
			final boolean desc) throws RppsException, UserProfileException {

		ICardUsageManager cardUsageManager = null;
		try {
			cardUsageManager = getUserProfileService().getCardUsageManager(getHandler());
			return cardUsageManager.getCardHistory(cuid);

		} catch (final Exception e) {
			LOG.error(e, e);
			throw new RppsException(e);
		} finally {
			if (cardUsageManager != null) {
				cardUsageManager.close();
			}
		}
	}

	public CardUsageTO[] getCardHistoryLog(final String cuid, final int startFrom, final int size,
			final String orderBy, final String orderDirection) {
		final boolean desc = orderDirection.equalsIgnoreCase("desc") ? true : false;
		String order = CardUsage.BY_SITE;
		if ("date".equalsIgnoreCase(orderBy)) {
			order = CardUsage.BY_DATE;
		} else
			if ("site".equalsIgnoreCase(orderBy)) {
				order = CardUsage.BY_USE;
			}

		try {
			final List list = getCardHistoryLog(new CUID(cuid), startFrom, size, order, desc);
			final CardUsageTO[] arrayUsage = new CardUsageTO[list.size()];
			for (int i = 0; i < list.size(); i++) {
				final CardUsage cardUsage = (CardUsage) list.get(i);
				arrayUsage[i] = ConvertHelper.convertCardUsage_to_CardUsageTO(cardUsage);
			}
			return arrayUsage;
		} catch (final Exception e) {
			LOG.error(e, e);
			throw new RppsException(e);
		}
	}

	/**
	 * Gets card information object.
	 * 
	 * @param cuid
	 *            the icard's CUID
	 * @param size
	 *            count of card history element which will be returned
	 * @return Return card information object
	 * @throws RppsException
	 *             some situation
	 */
	public CardInformationTO getCardInformation(final String cuid, final int size) throws RppsException {
		try {
			List cardHistory = getCardHistory(new CUID(cuid));
			int sizeAll = 0;
			if (cardHistory != null) {
				sizeAll = cardHistory.size();
			}
			cardHistory = getCardHistory(new CUID(cuid), 0, size, CardUsage.BY_DATE, false);
			return ConvertHelper.createCardInformationTO(cardHistory, sizeAll);
		} catch (final Exception e) {
			LOG.error(e, e);
			throw new RppsException(e);
		}
	}

	/**
	 * Gets CardInformationRPPS object.
	 * 
	 * @param cuid
	 *            the icard's CUID
	 * @return Return CardInformationRPPS object
	 * @throws RppsException
	 *             some situation
	 */
	protected CardInformationRPPS getCardInformationByCuid(final CUID cuid) throws RppsException, UserProfileException {
		// final CardInformationRPPS cardInformation = null;
		ICardUsageManager cardUsageManager = null;
		try {
			cardUsageManager = getUserProfileService().getCardUsageManager(getHandler());
			return getCardInformationByCuid(cuid, cardUsageManager);

		} catch (final Exception e) {
			LOG.error(e, e);
			throw new RppsException(e);
		} finally {
			if (cardUsageManager != null) {
				cardUsageManager.close();
			}
		}
	}

	/**
	 * Gets CardInformationRPPS object.
	 * 
	 * @param cuid
	 *            the icard's CUID
	 * @param cardUsageManager
	 *            instance of ICardUsageManager
	 * @return Return CardInformationRPPS object
	 * @throws RppsException
	 *             some situation
	 */
	protected CardInformationRPPS getCardInformationByCuid(final CUID cuid, final ICardUsageManager cardUsageManager)
			throws RppsException, UserProfileException {
		CardInformationRPPS cardInformation = null;
		try {
			final CardInformation cardInformationUP = getUserProfileService().getCardInformation(getHandler(), cuid);

			cardInformation = new CardInformationRPPS(cardInformationUP, cuid);
			List cardUsages = null;
			try {
				cardUsages = cardUsageManager.getUsedAlwaysURLs(cuid);

			} catch (Exception ex) {
				LOG.error(ex, ex);
			}
			if (null != cardUsages) {
				final WebForm[] webForms = new WebForm[cardUsages.size()];
				for (int i = 0; i < cardUsages.size(); i++) {
					webForms[i] = ((CardUsage) cardUsages.get(i)).getForm();
				}
				cardInformation.setWebForms(webForms);
			}

		} catch (Exception e) {
			LOG.error(e, e);
			throw new RppsException(e);
		}
		return cardInformation;
	}

	protected List getCategories(final CallbackHandler handler) throws RppsException {
		LOG.trace("RppsService.getCategories(CallbackHandler) started");
		try {
			final List list = getUserProfileService().getCategories(handler);

			Collections.sort(list, new CategoryComparator());

			return list;
		} catch (final UserProfileException e) {
			LOG.error(e, e);
			throw new RppsException(e);
		}
	}

	/**
	 * @param userId
	 * @param password
	 * @return
	 * @throws RppsException
	 *             some situation
	 */
	public CategoryTO[] getCategory() throws RppsException {
		LOG.trace("RppsService.getCategory() started");
		try {
			final List category = getCategories(getHandler());
			CategoryTO[] categorysTO = new CategoryTO[0];
			if (category != null) {
				categorysTO = new CategoryTO[category.size()];
				for (int i = 0; i < category.size(); i++) {
					categorysTO[i] = ConvertHelper.convertCategory_to_CategoryTO((Category) category.get(i));
				}
			}
			return categorysTO;
		} catch (final RppsException e) {
			LOG.error(e, e);
			throw e;
		}
	}

	public MapTO[] getClaimValuesMRU() throws RppsException {
		ICardUsageManager cardUsageManager = null;
		try {
			cardUsageManager = getUserProfileService().getCardUsageManager(getHandler());

			final Map claimValuesMRU = cardUsageManager.getClaimValuesMRU();
			if (claimValuesMRU != null && claimValuesMRU.keySet() != null) {
				final List result = new ArrayList();
				for (final Iterator iter = claimValuesMRU.keySet().iterator(); iter.hasNext();) {
					final MapTO mruTO = new MapTO();
					final String key = (String) iter.next();
					final Set valuesSet = (Set) claimValuesMRU.get(key);
					if (valuesSet != null) {
						final String[] values = new String[valuesSet.size()];
						if (values != null && values.length > 0) {
							valuesSet.toArray(values);
							if (values != null) {
								Arrays.sort(values, new StringComparator());
							}

							mruTO.setKey(key);
							mruTO.setValues(values);
							result.add(mruTO);
						}
					}
				}
				final MapTO[] resultMRU = new MapTO[result.size()];
				result.toArray(resultMRU);
				return resultMRU;
			}
			return new MapTO[0];
		} catch (final Exception e) {
			LOG.error(e, e);
			throw new RppsException(e);
		} finally {
			if (cardUsageManager != null) {
				try {
					cardUsageManager.close();
				} catch (final UserProfileException e) {
					LOG.error(e, e);
				}
			}
		}
	}

	protected CUID getCuidByWebForm(final WebForm webForm) throws RppsException, UserProfileException {
		ICardUsageManager cardUsageManager = null;
		try {
			Utils.cleanURLQuery(webForm);
			cardUsageManager = getUserProfileService().getCardUsageManager(getHandler());
			return cardUsageManager.getUsedAlwaysCard(webForm);
		} catch (final Exception e) {
			LOG.error(e, e);
			throw new RppsException(e);
		} finally {
			if (cardUsageManager != null) {
				cardUsageManager.close();
			}
		}
	}

	public CallbackHandler getHandler() {
		return this.handler;
	}

	/**
	 * Gets the global i-name registered to the Higgins service (e.g. "=parity").
	 * 
	 * @return the global i-name.
	 */
	public String getHigginsServiceGlobalIName() {
		return System.getProperty(WebKeys.HIGGINS_SERVICE_GLOBAL_NAME_PROPERTY, "=parity");
	}

	/* Methods of card history */

	protected ICard getICardByCUID(final CUID cuid) throws RppsException {
		try {
			return ICardRegistry.getInstance().getICardByCUID(getHandler(), cuid);
		} catch (final CardException e) {
			LOG.error(e, e);
			throw new RppsException(e);
		}
	}

	/**
	 * Gets a card by CUID.
	 * 
	 * @param cuid
	 *            the card CUID.
	 * @return the card transfer object.
	 */
	public ICardTO getICardByCUID(final String cuid) throws RppsException {
		try {
			final ICard card = getICardByCUID(new CUID(cuid));
			return convertICard_To_ICardTO(card, false, null);
		} catch (final Exception e) {
			LOG.error(e, e);
			throw new RppsException(e);
		}
	}

	/**
	 * Gets a card by CUID and Credential.
	 * 
	 * @param cuid
	 *            the card CUID.
	 * @param tokenCredential
	 *            the tokenCredential transfer object
	 * @return the card transfer object.
	 */
	public ICardTO getICardByCUIDAndTokenServiceCredential(final String cuid,
			final UITokenServiceCredentialTO tokenCredential) throws RppsException, STSFaultException {
		try {
			ICard card = getICardByCUID(new CUID(cuid));

			if (card == null) {
				throw new RppsException("Can't find card with CUID " + cuid);
			}

			final CardInformationRPPS cardInformation = getCardInformationByCuid(new CUID(cuid));
			ICredential credential = null;
			if (cardInformation.getCardCredential() != null) {
				credential = ConvertHelper.convertCardCredential_to_ICredential(card, cardInformation
						.getCardCredential());
			} else {
				if (tokenCredential != null) {
					credential = ConvertHelper.convertUITokenServiceCredentialTO_to_ICredentials(card, tokenCredential);
				}
			}
			if (credential != null) {
				card = card.getProvider().getICardByCUID(getHandler(), card.getCUID(), credential);
			}

			return convertICard_To_ICardTO(card, true, null);
//		} catch (STSFaultException faultException) {
//			// return STS fault exception
//			throw faultException;
//
		} catch (final Exception e) {
			LOG.error(e, e);
			throw new RppsException(e);
		}
	}

	protected ICard getICardByPPID(final X509Certificate x509cert, final String ppid, final URI address)
			throws RppsException {

		try {
			final byte[] bppid = Base64.decode(ppid);
			final IKeyStoreService securityService = KeyStoreRegistry.getInstance().getSecurityService();
			final X509Certificate[] chainOriginal = new X509Certificate[] { x509cert };
			X509Certificate[] chain = null;
			if (chainOriginal != null) {
				try {
					chain = securityService.getNormalizedCertificateChain(getHandler(), chainOriginal, null);
				} catch (final CertificateException e) {
					LOG.error(e, e);
				} catch (final CertStoreException e) {
					LOG.error(e, e);
				}
			}
			if (chain == null) {
				chain = chainOriginal;
			}
			final PPIDCardSpacePolicy policy = new PPIDCardSpacePolicy(chain, securityService
					.getCertStore(getHandler()), bppid, address.toString());
			final IICardSelector selector = ICardSelectorService.getInstance().getICardSelector(getHandler(), policy);

			final UserChoiceTree_ANDofORs uct = (UserChoiceTree_ANDofORs) selector.getUserChoice();
			final SelectionANDofORs subselection = uct.getDefaultSelection();

			if (subselection.isEmpty()) {
				LOG.debug("Cannot find the Personal card used to authenticate for this managed card.");
				return null;
			}
			final String cuid = ((SelectionANDofORsElm) subselection.getElements().get(0)).getUUID();

			return getICardByCUID(new CUID(cuid));
		} catch (final Exception e) {
			LOG.error(e, e);
			throw new RppsException(e);
		}
	}

	/**
	 * Created for some demo.
	 * 
	 * @param cuid
	 *            the card's cuid.
	 * @param policy
	 * @param typeofCredential
	 * @param credentialKey
	 * @param credentialValue
	 * @return UIResponseCardTO
	 * @throws RppsException
	 * @deprecated
	 */
	public UIResponseCardTO getICardClaims(final String cuid, final String policy, final String typeofCredential,
			final String[] credentialKey, final String[] credentialValue) throws RppsException {
		try {
			AllowSelfSignedSSL();
			final Hashtable table = new Hashtable();
			for (int i = 0; i < credentialKey.length; i++) {
				if (credentialKey[i] != null && credentialValue[i] != null) {
					table.put(credentialKey[i], credentialValue[i]);
				}
			}
			URI url = null;
			String urlStr;
			if (table.get("url") != null) {
				// String cleanUrl = Utils.cleanURLQuery(((String) table.get("url")));
				// if (null != cleanUrl) {
				url = new URI((String) table.get("url"));
				// }
			}

			// build certificate chain, needs for PPID
			X509Certificate[] sslCertsOriginal = null;
			if (table.get("cert") != null) {
				sslCertsOriginal = RppsHelper.parseSSLCert((String) table.get("cert"));
				if (sslCertsOriginal != null) {
					X509Certificate[] sslCerts = null;
					try {
						sslCerts = KeyStoreRegistry.getInstance().getSecurityService().getNormalizedCertificateChain(
								getHandler(), sslCertsOriginal, null);
						if (sslCerts != null) {
							sslCertsOriginal = sslCerts;
						}
					} catch (final CertificateException e) {
						LOG.error(e, e);
					} catch (final CertStoreException e) {
						LOG.error(e, e);
					}
				}
			}

			ICredential credential;
			ICard card = getICardByCUID(new CUID(cuid));
			if (card != null) {

				final CardInformationRPPS cardCredentialInfo = getCardInformationByCuid(new CUID(cuid));

				if (cardCredentialInfo != null && cardCredentialInfo.getCardCredential() != null) {
					credential = ConvertHelper.convertCardCredential_to_ICredential(card, cardCredentialInfo
							.getCardCredential());
				} else {
					credential = ConvertHelper.convertHashtable_to_Credential(card, typeofCredential, table);
				}

				if (credential != null) {
					card = card.getProvider().getICardByCUID(getHandler(), card.getCUID(), credential);
				}
				final ICardSpacePolicy policyTemp = (ICardSpacePolicy) ICardSelectorService.getInstance().parsePolicy(
						policy);

				return convertICard_To_UIResponseCardTO(card, new WebForm(url, null, null, null), policyTemp,
						cardCredentialInfo, sslCertsOriginal);
			} else {
				throw new RppsException("Didn't find card with cuid=" + cuid);
			}

		} catch (final Exception e) {
			LOG.error(e, e);
			throw new RppsException(e);
		}
	}

	/**
	 * Created for some demo.
	 * 
	 * @param cuid
	 *            the card's cuid.
	 * @param policy
	 * @param typeofCredential
	 * @param credentialKey
	 * @param credentialValue
	 * @param rpEndPointTO
	 *            the object which represents logging form on relying party
	 * @return UIResponseCardTO
	 * @throws RppsException
	 */
	public UIResponseCardTO getICardClaims(final String cuid, final String policy, final String typeofCredential,
			final String[] credentialKey, final String[] credentialValue, final RPEndPointTO rpEndPointTO)
			throws RppsException {
		try {
			AllowSelfSignedSSL();
			final Hashtable table = new Hashtable();
			for (int i = 0; i < credentialKey.length; i++) {
				if (credentialKey[i] != null && credentialValue[i] != null) {
					table.put(credentialKey[i], credentialValue[i]);
				}
			}
			WebForm webForm = rpEndPointTO.toWebForm();
			Utils.cleanURLQuery(webForm);

			// build certificate chain, needs for PPID
			X509Certificate[] sslCertsOriginal = null;
			if (table.get("cert") != null) {
				sslCertsOriginal = RppsHelper.parseSSLCert((String) table.get("cert"));
				if (sslCertsOriginal != null) {
					X509Certificate[] sslCerts = null;
					try {
						sslCerts = KeyStoreRegistry.getInstance().getSecurityService().getNormalizedCertificateChain(
								getHandler(), sslCertsOriginal, null);
						if (sslCerts != null) {
							sslCertsOriginal = sslCerts;
						}
					} catch (final CertificateException e) {
						LOG.error(e, e);
					} catch (final CertStoreException e) {
						LOG.error(e, e);
					}
				}
			}

			ICredential credential;

			ICard card = getICardByCUID(new CUID(cuid));
			if (card != null) {

				final CardInformationRPPS cardCredentialInfo = getCardInformationByCuid(new CUID(cuid));

				if (cardCredentialInfo != null && cardCredentialInfo.getCardCredential() != null) {
					credential = ConvertHelper.convertCardCredential_to_ICredential(card, cardCredentialInfo
							.getCardCredential());
				} else {
					credential = ConvertHelper.convertHashtable_to_Credential(card, typeofCredential, table);
				}

				if (credential != null) {
					card = card.getProvider().getICardByCUID(getHandler(), card.getCUID(), credential);
				}
				final ICardSpacePolicy policyTemp = (ICardSpacePolicy) ICardSelectorService.getInstance().parsePolicy(
						policy);
				return convertICard_To_UIResponseCardTO(card, webForm, policyTemp, cardCredentialInfo, sslCertsOriginal);
			} else {
				throw new RppsException("Didn't find card with cuid=" + cuid);
			}

		} catch (final Exception e) {
			LOG.error(e, e);
			throw new RppsException(e);
		}
	}

	/**
	 * Return templates for creating cards
	 * 
	 * @return a template list
	 */
	public ICardTemplateTO[] getICardCreationTemplate() {
		List templates = null;
		templates = getICardCreationTemplate(getHandler());
		ICardTemplateTO[] telmplateList = new ICardTemplateTO[0];
		if (templates != null) {
			telmplateList = new ICardTemplateTO[templates.size()];
			for (int index = 0; index < templates.size(); index++) {
				telmplateList[index] = ConvertHelper.convertICardTemplate_to_ICardTemplateTO((ICardTemplate) templates
						.get(index));
			}
		}
		return telmplateList;
	}

	/**
	 * Return templates for creating cards.
	 * 
	 * @param userId
	 *            the id to identify the user.
	 * @param password
	 *            the password to identify the user.
	 * @return a template list
	 */
	protected List getICardCreationTemplate(final CallbackHandler handler) throws RppsException {
		return ICardRegistry.getInstance().getCardCreationTemplates(handler);
	}

	/**
	 * This method return list of card creation template by provider id.
	 * 
	 * @param extID
	 *            the id to identify the card provider.
	 * @return a template list as array of ICardTemplateTO
	 * @throws RppsException
	 */
	public ICardTemplate[] getICardCreationTemplateByProvider(final String extID) throws RppsException {
		final ICardProvider provider = ICardRegistry.getInstance().getICardProvider(extID);
		return provider.getCardCreationTemplates(getHandler());

	}

	/**
	 * Return templates for creating cards.
	 * 
	 * @return a template list
	 */
	public ICardTemplateTO getICardCreationTemplateByTUID(final String tuid) throws RppsException {

		ICardTemplate template = null;
		try {
			template = getICardCreationTemplateByTUID(new TUID(tuid));
		} catch (final CardException e) {
			LOG.error(e, e);
			throw new RppsException(e);
		} finally {
		}
		if (template != null) {
			return ConvertHelper.convertICardTemplate_to_ICardTemplateTO(template);
		}
		return null;
	}

	/**
	 * Return templates for creating cards.
	 * 
	 * @return a template list
	 */
	public ICardTemplate getICardCreationTemplateByTUID(final TUID tuid) throws CardException {
		try {
			return ICardRegistry.getInstance().getCardCreationTemplatesByTUID(getHandler(), tuid);
		} finally {
		}
	}

	/**
	 * Gets the card and category lists of the specified user.
	 * 
	 * @return the card list
	 */
	public ICardsAndCategoryTO getICardsAndCategoriesByUser() throws RppsException {
		LOG.trace("RppsService.getICardsAndCategoriesByUser() started");
		final ICardsAndCategoryTO cardsAndCategoryTO = new ICardsAndCategoryTO();
		try {
			final ICardResponseTO cardResponceTO = getICardsByUser();
			cardsAndCategoryTO.setRelationList(cardResponceTO.getRelationList());
			cardsAndCategoryTO.setICardList(cardResponceTO.getICardList());
		} catch (final Exception e) {
			LOG.error(e, e);
		}
		try {
			cardsAndCategoryTO.setCategoryList(getCategory());
		} catch (final Exception e) {
			LOG.error(e, e);
		}
		return cardsAndCategoryTO;
	}

	/**
	 * Gets cards by CUIDs array.
	 * 
	 * @param cuids
	 *            the card CUIDs.
	 * @return list of cards transfer object.
	 */
	public ICardTO[] getICardsByCUIDs(final String[] cuids) {
		final ArrayList result = new ArrayList();
		for (int i = 0; i < cuids.length; i++) {
			try {
				result.add(getICardByCUID(cuids[i]));
			} catch (final Exception e) {
				LOG.error(e, e);
			}
		}
		return (ICardTO[]) result.toArray(new ICardTO[result.size()]);
	}

	/**
	 * Gets cards which may be export to special file format.
	 * 
	 * @param formatID
	 *            the id of format file
	 * @return list of cards as a array of CardTO
	 * @throws RppsException
	 */
	public ICardTO[] getICardsByFormat(final String formatID) throws RppsException {
		ICardTO[] icards = new ICardTO[0];
		try {
			final List cards = new ArrayList();
			try {
				for (final Iterator iter = ICardRegistry.getInstance().getICardsByFormat(getHandler(), formatID); iter
						.hasNext();) {
					try {
						cards.add(iter.next());
					} catch (final Exception e) {
						LOG.error(e, e);
					}
				}
			} catch (final Exception e) {
				LOG.error(e, e);
			}
			if (cards != null) {
				icards = new ICardTO[cards.size()];
				for (int i = 0; i < cards.size(); i++) {
					try {
						icards[i] = convertICard_To_ICardTO((ICard) cards.get(i), false, null);
					} catch (final Exception e) {
						LOG.error(e, e);
					}
				}
			}
		} catch (final Exception e) {
			LOG.error(e, e);
		}
		return icards;
	}

	/**
	 * Gets the card list of the specified user.
	 * 
	 * @return the card list
	 */
	public ICardResponseTO getICardsByUser() {
		LOG.trace("RppsService.getICardsByUser() started");
		ICardResponseTO cardResponceTO = new ICardResponseTO();
		try {
			List cards = null;
			cards = getICardsByUser(getHandler());
			if (cards != null) {
				cardResponceTO = convertICards_To_CardResponceTO(cards);
			}
		} catch (final Exception e) {
			LOG.error(e, e);
			// throw new RppsException(e);
		}
		return cardResponceTO;
	}

	/* Methods of card category */

	protected List getICardsByUser(final CallbackHandler handlerLocal) {
		LOG.trace("RppsService.getICardsByUser() List started");
		final List cards = new ArrayList();
		// gets all registered providers
		for (final Iterator pit = ICardRegistry.getInstance().getICardProviders(); pit.hasNext();) {
			final ICardProvider provider = (ICardProvider) pit.next();
			LOG.trace("RppsService.getICardsByUser() processes icard probvider : " + provider.getID());
			try {
				for (final Iterator icards = provider.getICards(handlerLocal); icards.hasNext();) {
					cards.add(icards.next());
				}
			} catch (final Exception e) {
				LOG.error(e, e);
			}
		}
		LOG.trace("RppsService.getICardsByUser() List finished");
		return cards;
	}

	protected ICredential getICredentialByPPID(final X509Certificate x509cert, final String ppid, final URI address)
			throws RppsException {

		try {
			final ICard card = getICardByPPID(x509cert, ppid, address);
			if (card == null) {
				return null;
			}
			final ICredentialDescriptor[] der = card.getRequiredCredentials();
			if (der == null) {
				return null;
			}
			if (der.length == 0) {
				return null;
			}
			return der[0].getCredential();
		} catch (final Exception e) {
			LOG.error(e, e);
			throw new RppsException(e);
		}
	}

	/**
	 * Gets available file format for cards export.
	 * 
	 * @return array of FormatDescriptorTO
	 * @throws RppsException
	 */
	public FormatDescriptorTO[] getOutputFormats() throws RppsException {
		try {
			final IFormatDescriptor[] formatDescriptor = ICardRegistry.getInstance().getOutputFormats();
			if (formatDescriptor != null && formatDescriptor.length > 0) {
				final FormatDescriptorTO[] formatDescriptorTO = new FormatDescriptorTO[formatDescriptor.length];
				for (int i = 0; i < formatDescriptor.length; i++) {
					formatDescriptorTO[i] = convertIFormatDescriptor_to_FormatDescriptorTO(formatDescriptor[i]);
				}
				return formatDescriptorTO;
			}
			return null;
		} catch (final Exception e) {
			LOG.error(e, e);
			throw new RppsException(e);
		}
	}

	/**
	 * Makes xmlToken usage claim list.
	 * 
	 * @param sslCert
	 *            the SSL certificate
	 * @param claimName
	 *            the array which contains claim names
	 * @param claimValue
	 *            the array which contains claim values
	 * @return the security token transfer object
	 * @throws RppsException
	 *             some situation
	 */
	public TokenResponseTO getTokenByClaims(final String sslCert, final String[] claimName, final String[] claimValue)
			throws RppsException {
		try {
			final Hashtable claimHashtable = new Hashtable();
			for (int i = 0; i < claimName.length; i++) {
				try {
					claimHashtable.put(claimName[i], claimValue[i]);
				} catch (final RuntimeException e) {
					LOG.error(e, e);
				}
			}
			X509Certificate cert = null;
			final X509Certificate[] certChain = RppsHelper.parseSSLCert(sslCert);
			if (certChain != null && certChain.length > 0) {
				cert = KeyStoreRegistry.getInstance().getSecurityService().findLeafFromCertificates(certChain);
			}
			final ISTSResponse resp = UserProfileTokenHelper.getToken(cert, claimHashtable);

			final IdentityToken itoken = new IdentityToken(resp, Base64.encode(cert.getEncoded()));

			final String tokenAsString = (String) itoken.getAs(String.class);

			return ConvertHelper.createTokenResponse(tokenAsString);
		} catch (final Exception e) {
			LOG.error(e, e);
			throw new RppsException(e);
		}
	}

	/**
	 * Web service operation. Makes security token usage saved users information
	 * 
	 * @param policy
	 *            the RP Agent's Relying Party Security Policy
	 * @param policytype
	 *            the RP Security Policy type
	 * @param sslCert
	 *            the SSL certificate
	 * @param url
	 *            the url of site which needs token
	 * @return the security token transfer object
	 * @throws RemoteException
	 * @deprecated
	 */
	public TokenResponseTO getTokenByUrl(final String policy, final String policytype, final String sslCert,
			final String url) throws RppsException {
		TokenResponseTO token = null;
		try {
			token = getTokenByRPEndPoint(policy, policytype, sslCert, new RPEndPointTO(Utils.cleanURLQuery(url), null,
					null, null));
		} catch (Exception e) {
			LOG.error(e, e);
		}
		return token;
	}

	/**
	 * Web service operation. Makes security token usage saved users information
	 * 
	 * @param policy
	 *            the RP Agent's Relying Party Security Policy
	 * @param policytype
	 *            the RP Security Policy type
	 * @param sslCert
	 *            the SSL certificate
	 * @param rpEndPointTO
	 *            the object which represents logging form on relying party
	 * @return the security token transfer object
	 * @throws RemoteException
	 */
	public TokenResponseTO getTokenByRPEndPoint(final String policy, final String policytype, final String sslCert,
			final RPEndPointTO rpEndPointTO) throws RppsException {
		TokenResponseTO token = null;
		try {

			WebForm webForm = rpEndPointTO.toWebForm();
			Utils.cleanURLQuery(webForm);

			PolicyVersion policyVersion = getUserProfileService().getRPPolicyVersion(handler, webForm);
			final IPolicy iPolicy = ICardSelectorService.getInstance().parsePolicy(policy);
			if (policyVersion != null && iPolicy instanceof CardSpacePolicy) {
				boolean result = policyVersion.getVersion().equals(((CardSpacePolicy) iPolicy).getPrivacyVersion());
				if (!result) {
					return null;
				}
			}

			final CUID cuidUrl = getCuidByWebForm(webForm);
			if (cuidUrl == null) {
				return null;
			}

			final CardInformationRPPS cardInformation = getCardInformationByCuid(cuidUrl);
			if (cardInformation != null) {
				final String[] cuids = new String[] { cuidUrl.toString() };

				final ICard card = getICardByCUID(cuidUrl);
				if (card == null) {
					new RppsException("Does not find card with cuid " + cuidUrl.toString());
				}

				ICredential credential = null;
				credential = ConvertHelper.convertCardCredential_to_ICredential(card, cardInformation
						.getCardCredential());

				List optionalClaims = null;
				try {
					optionalClaims = getUsersChoiceOfOptionalClaims(card.getCUID(), webForm);
				} catch (final Exception exc) {
					LOG.error(exc, exc);
				}

				if (iPolicy instanceof CardSpacePolicy && optionalClaims != null && optionalClaims.size() > 0) {
					((CardSpacePolicy) iPolicy).setOptionalClaims(optionalClaims);
				}

				token = getTokenObject(iPolicy, policytype, sslCert, cuids, credential, webForm);

				if (token != null && token.getTkn() != null && token.getTkn().length() > 0) {
					if (webForm != null) {
						addCardHistory(card.getCUID(), webForm);
					}
				}
			}
		} catch (final Exception e) {
			LOG.error(e, e);
		}
		return token;
	}

	/**
	 * Web service operation. Makes security token usage saved users information
	 * 
	 * @param policy
	 *            the RP Agent's Relying Party Security Policy
	 * @param policytype
	 *            the RP Security Policy type
	 * @param sslCert
	 *            the SSL certificate
	 * @param rpEndPointTO
	 *            the object which represents logging form on relying party
	 * @return the security token transfer object
	 * @throws RemoteException
	 */
	public ExtraTokenResponseTO getTokenExtraByRPEndPoint(final String policy, final String policytype,
			final String sslCert, final RPEndPointTO rpEndPointTO) throws RppsException {
		ExtraTokenResponseTO token = null;
		try {
			WebForm webForm = rpEndPointTO.toWebForm();
			Utils.cleanURLQuery(webForm);

			PolicyVersion policyVersion = getUserProfileService().getRPPolicyVersion(handler, webForm);
			IPolicy iPolicy = ICardSelectorService.getInstance().parsePolicy(policy);
			if (policyVersion != null && iPolicy instanceof CardSpacePolicy) {
				String privacyVersion = ((CardSpacePolicy) iPolicy).getPrivacyVersion();
				// compare privacyVersion
				if (null != privacyVersion && !privacyVersion.trim().equals("")
						&& !policyVersion.getVersion().equals(privacyVersion)) {
					return null;
				}
			}

			final CUID cuidUrl = getCuidByWebForm(webForm);
			if (cuidUrl == null) {
				return null;
			}

			final CardInformationRPPS cardCredentialInfo = getCardInformationByCuid(cuidUrl);
			if (cardCredentialInfo != null) {
				final String[] cuids = new String[] { cardCredentialInfo.getCuid().toString() };

				final ICard card = getICardByCUID(cardCredentialInfo.getCuid());
				if (card == null) {
					new RppsException("Does not find card with cuid " + cardCredentialInfo.getCuid().toString());
				}

				ICredential credential = null;
				credential = ConvertHelper.convertCardCredential_to_ICredential(card, cardCredentialInfo
						.getCardCredential());

				// Restore user selected optional claims
				List optionalClaims = null;
				if (iPolicy instanceof CardSpacePolicy) {
					try {
						optionalClaims = getUsersChoiceOfOptionalClaims(card.getCUID(), webForm);
						((CardSpacePolicy) iPolicy).setOptionalClaims(optionalClaims);
					} catch (final Exception exc) {
						LOG.error(exc, exc);
					}
				}

				final TokenResponseTO tokenTO = getTokenObject(iPolicy, policytype, sslCert, cuids, credential, webForm);

				if (tokenTO != null && tokenTO.getTkn() != null && tokenTO.getTkn().length() > 0) {
					if (webForm != null) {
						addCardHistory(card.getCUID(), webForm);
					}
				}
				token = new ExtraTokenResponseTO();
				token.setTkn(tokenTO.getTkn());
				token.setCardName(card.getName());
				token.setCardDescription(card.getDescription());
				token.setCardImage(card.getImage());
				token.setCardImageType(card.getImageType());
			}
		} catch (final Exception e) {
			LOG.error(e, e);
		}
		return token;
	}

	/**
	 * Web service operation. Makes security token usage saved users information
	 * 
	 * @param policy
	 *            the RP Agent's Relying Party Security Policy
	 * @param policytype
	 *            the RP Security Policy type
	 * @param sslCert
	 *            the SSL certificate
	 * @param url
	 *            the url of site which needs token
	 * @return the security token transfer object
	 * @throws RemoteException
	 * @deprecated
	 */
	public ExtraTokenResponseTO getTokenExtraByUrl(final String policy, final String policytype, final String sslCert,
			final String url) throws RppsException {
		ExtraTokenResponseTO token = null;
		try {
			token = getTokenExtraByRPEndPoint(policy, policytype, sslCert, new RPEndPointTO(Utils.cleanURLQuery(url),
					null, null, null));
		} catch (final Exception e) {
			LOG.error(e, e);
		}
		return token;
	}

	/* Methods of user profile */

	/**
	 * Web service operation. Given the Relying Party Agent (aka 'Requester')'s policy, identity information about the
	 * requester, and the set of one or more selections that the user has just made in the ISS Web UI (see
	 * userInterfaceRequest), AND presuming that the protocol involved in interacting with the RPA requires a security
	 * token, request the token that corresponds to the user's selection(s).
	 * 
	 * @param userId
	 *            the id to identify the user.
	 * @param password
	 *            the password to identify the user.
	 * @param policy
	 *            the RP Agent's Relying Party Security Policy
	 * @param policytype
	 *            the RP Security Policy type
	 * @param sslCert
	 *            the SSL certificate
	 * @param cuids
	 *            the user-selected subset of the UIDs (handles) contained in the UserInterfaceResponse object returned
	 *            from an earlier invocation of 'userInterfaceRequest'
	 * @param typeofCredential
	 *            the type of selected credential
	 * @param credentialKey
	 *            the names of fields of selected credential
	 * @param credentialValue
	 *            the value of fields of selected credential
	 * @return the security token transfer object
	 * @throws RemoteException
	 */
	protected TokenResponseTO getTokenObject(final String policy, final String policytype, final String sslCert,
			final String[] cuids, final ICredential credential, final CardInformationRPPS cardCredentialInfo,
			final WebForm webForm, final boolean saveCard, final boolean saveCredential) throws RppsException {
		TokenResponseTO token = null;
		Utils.cleanURLQuery(webForm);
		try {
			CUID cuid = null;
			if (cuids != null && cuids.length > 0) {
				cuid = new CUID(cuids[0]);

				if (!saveCredential) {
					if (cardCredentialInfo != null && cardCredentialInfo.getCardCredential() != null) {
						clearCardCredential(cuid);
					}
				}
			}

			final IPolicy iPolicy = ICardSelectorService.getInstance().parsePolicy(policy);

			token = getTokenObject(iPolicy, policytype, sslCert, cuids, credential, webForm);

			if (cuid != null) {
				if (token != null && token.getTkn() != null && token.getTkn().length() > 0) {
					if (webForm != null) {
						addCardHistory(cuid, webForm);
						if (iPolicy instanceof CardSpacePolicy) {
							final List claimTypeList = ((CardSpacePolicy) iPolicy).getOptionalClaims();
							if (claimTypeList != null && claimTypeList.size() > 0) {
								saveUsersChoiceOfOptionalClaims(cuid, webForm, claimTypeList);
							}
							final ICardSpacePolicy iCardSpacePolicy = (ICardSpacePolicy) iPolicy;
							if (iCardSpacePolicy != null && iCardSpacePolicy.getPrivacyVersion() != null
									&& iCardSpacePolicy.getPrivacyVersion().length() > 0) {
								final PolicyVersion policyVersion = new PolicyVersion();
								policyVersion.setForm(webForm);
								policyVersion.setPolicy(policy);
								policyVersion.setVersion(iCardSpacePolicy.getPrivacyVersion());
								getUserProfileService().setRPPolicyVersion(getHandler(), policyVersion);
							}
						}
					}
					if (saveCard && webForm != null) {
						addWebFormCardRelation(cuid, webForm, cardCredentialInfo);
					}

					if (saveCredential) {
						final CardCredential cardCredential = ConvertHelper
								.convertICredential_to_CardCredential(credential);
						setCardCredential(cuid, cardCredential);
					}
				}
			}
		} catch (final Exception e) {
			LOG.error(e, e);
			throw new RppsException(e);
		}
		return token;
	}

	/**
	 * Given the Relying Party Agent (aka 'Requester')'s policy, identity information about the requester, and the set
	 * of one or more selections that the user has just made in the ISS Web UI (see userInterfaceRequest), AND presuming
	 * that the protocol involved in interacting with the RPA requires a security token, request the token that
	 * corresponds to the user's selection(s).
	 * 
	 * @param handler
	 * @param policy
	 *            the RP Agent's Relying Party Security Policy
	 * @param policytype
	 *            the RP Security Policy type
	 * @param sslCert
	 *            the SSL certificate
	 * @param cuids
	 *            the user-selected subset of the UIDs (handles) contained in the UserInterfaceResponse object returned
	 *            from an earlier invocation of 'userInterfaceRequest'
	 * @param credential
	 * @param cardCredentialInfo
	 * @param url
	 * @return the security token transfer object
	 * @throws RppsException
	 *             some situation
	 */
	protected TokenResponseTO getTokenObject(final IPolicy policy, final String policytype, final String sslCert,
			final String[] cuids, final ICredential credential, final WebForm webForm) throws RppsException {
		String token = null;

		Utils.cleanURLQuery(webForm);

		final IKeyStoreService securityService = KeyStoreRegistry.getInstance().getSecurityService();

		// get the selections from user's selection
		final SelectionANDofORs selection = new SelectionANDofORs();

		if (null != webForm && null != webForm.getUrl()) {
			selection.action = webForm.getUrl().toString();
		}
		selection.idemix_credential_db_uri = "credentials.db";
		selection.cardspace_credential_db_uri = "infocards.properties";
		selection.publickey_db_uri = "pk.db";
		final X509Certificate[] chainOriginal = RppsHelper.parseSSLCert(sslCert);
		X509Certificate[] chain = null;
		if (chainOriginal != null) {
			try {
				chain = securityService.getNormalizedCertificateChain(getHandler(), chainOriginal, null);
				securityService.putCertChain(chain);
			} catch (final CertificateException e) {
				LOG.error(e, e);
			} catch (final CertStoreException e) {
				LOG.error(e, e);
			}
		}
		if (chain == null) {
			chain = chainOriginal;
		}
		selection.sslCertChain = chain;
		selection.setCredential(credential);
		for (int i = 0; i < cuids.length; i++) {
			final String uid = cuids[i];
			// n = (Node)i.next();
			int ui = 0;
			try {
				if (i < cuids.length - 1) {
					ui = Integer.parseInt(cuids[++i]);
				}
			} catch (final Exception ew) {
				ew.printStackTrace();
			}
			final SelectionANDofORsElm se = new SelectionANDofORsElm(uid, ui, 0);
			selection.add(se);
		}
		try {

			final IICardSelector selector = ICardSelectorService.getInstance().getICardSelector(getHandler(), policy);
			final IIdentityToken iToken = selector.getIdentityToken(selection);
			if (null != iToken) {
				token = (String) iToken.getAs(String.class);
			} else {
				token = "";
			}
		} catch (final Exception e) {
			LOG.error(e, e);
			throw new RppsException(e);
		}
		return ConvertHelper.createTokenResponse(token);
	}

	/**
	 * Web service operation. Given the Relying Party Agent (aka 'Requester')'s policy, identity information about the
	 * requester, and the set of one or more selections that the user has just made in the ISS Web UI (see
	 * userInterfaceRequest), AND presuming that the protocol involved in interacting with the RPA requires a security
	 * token, request the token that corresponds to the user's selection(s).
	 * 
	 * @param policy
	 *            the RP Agent's Relying Party Security Policy
	 * @param policytype
	 *            the RP Security Policy type
	 * @param sslCert
	 *            the SSL certificate
	 * @param cuids
	 *            the user-selected subset of the UIDs (handles) contained in the UserInterfaceResponse object returned
	 *            from an earlier invocation of 'userInterfaceRequest'
	 * @param typeofCredential
	 *            the type of selected credential
	 * @param credentialKey
	 *            the names of fields of selected credential
	 * @param credentialValue
	 *            the value of fields of selected credential
	 * @return the security token transfer object
	 * @throws RemoteException
	 */
	public TokenResponseTO getTokenObject(final String policy, final String policytype, final String sslCert,
			final String[] cuids, final String typeofCredential, final String[] credentialKey,
			final String[] credentialValue, final RPEndPointTO rpEndPointTO) throws RppsException {
		TokenResponseTO token = null;
		try {
			AllowSelfSignedSSL();
			final Hashtable table = new Hashtable();
			boolean saveCard = false;
			boolean saveCredential = false;
			String urlStr = null;
			for (int i = 0; i < credentialKey.length; i++) {
				if (credentialKey[i] != null && credentialValue[i] != null) {
					table.put(credentialKey[i], credentialValue[i]);
					LOG.debug("Put in table key='" + credentialKey[i] + "' with value='" + credentialValue[i] + "'");
				}
			}
			if (table.get("saveCard") != null && "true".equalsIgnoreCase((String) table.get("saveCard"))) {
				saveCard = true;
			}
			if (table.get("saveCredential") != null && "true".equalsIgnoreCase((String) table.get("saveCredential"))) {
				saveCredential = true;
			}

			// if (urlStr != null && urlStr.indexOf("?") > -1) {
			// urlStr = urlStr.substring(0, urlStr.indexOf("?"));
			// }

			// log.debug("CUID parameter is " + cuids[0]);
			// log.debug("SaveCard parameter is " + Boolean.toString(saveCard));
			// log.debug("SaveCredential parameter is " + Boolean.toString(saveCredential));
			// log.debug("Url parameter is " + urlStr);

			WebForm webForm = rpEndPointTO.toWebForm();
			Utils.cleanURLQuery(webForm);

			ICredential credential = null;
			CardInformationRPPS cardCredentialInfo = null;
			if (cuids != null && cuids.length > 0) {
				final ICard card = getICardByCUID(new CUID(cuids[0]));
				if (card != null) {

					cardCredentialInfo = null;
					CUID cuidUrl = null;
					if (webForm != null) {
						cuidUrl = getCuidByWebForm(webForm);
					}

					if (cuidUrl != null) {
						if (cuidUrl.equals(card.getCUID()) && !saveCard) {
							deleteWebFormCardRelation(card.getCUID(), webForm);
						}
					}

					cardCredentialInfo = getCardInformationByCuid(card.getCUID());

					if (cardCredentialInfo != null && cardCredentialInfo.getCardCredential() != null) {
						credential = ConvertHelper.convertCardCredential_to_ICredential(card, cardCredentialInfo
								.getCardCredential());
					} else {
						credential = ConvertHelper.convertHashtable_to_Credential(card, typeofCredential, table);
					}
				}
			}
			token = getTokenObject(policy, policytype, sslCert, cuids, credential, cardCredentialInfo, webForm,
					saveCard, saveCredential);
		} catch (final Exception e) {
			LOG.error(e, e);
			throw new RppsException(e);
		}

		return token;
	}

	/**
	 * Web service operation. Given the Relying Party Agent (aka 'Requester')'s policy, identity information about the
	 * requester, and the set of one or more selections that the user has just made in the ISS Web UI (see
	 * userInterfaceRequest), AND presuming that the protocol involved in interacting with the RPA requires a security
	 * token, request the token that corresponds to the user's selection(s).
	 * 
	 * @param policy
	 *            the RP Agent's Relying Party Security Policy
	 * @param policytype
	 *            the RP Security Policy type
	 * @param sslCert
	 *            the SSL certificate
	 * @param cuids
	 *            the user-selected subset of the UIDs (handles) contained in the UserInterfaceResponse object returned
	 *            from an earlier invocation of 'userInterfaceRequest'
	 * @param typeofCredential
	 *            the type of selected credential
	 * @param credentialKey
	 *            the names of fields of selected credential
	 * @param credentialValue
	 *            the value of fields of selected credential
	 * @return the security token transfer object
	 * @throws RemoteException
	 * @deprecated
	 */
	public TokenResponseTO getTokenObject(final String policy, final String policytype, final String sslCert,
			final String[] cuids, final String typeofCredential, final String[] credentialKey,
			final String[] credentialValue) throws RppsException {
		TokenResponseTO token = null;
		try {
			AllowSelfSignedSSL();
			final Hashtable table = new Hashtable();
			boolean saveCard = false;
			boolean saveCredential = false;
			String urlStr = null;
			for (int i = 0; i < credentialKey.length; i++) {
				if (credentialKey[i] != null && credentialValue[i] != null) {
					table.put(credentialKey[i], credentialValue[i]);
					LOG.debug("Put in table key='" + credentialKey[i] + "' with value='" + credentialValue[i] + "'");
				}
			}
			if (table.get("saveCard") != null && "true".equalsIgnoreCase((String) table.get("saveCard"))) {
				saveCard = true;
			}
			if (table.get("saveCredential") != null && "true".equalsIgnoreCase((String) table.get("saveCredential"))) {
				saveCredential = true;
			}
			if (table.get("url") != null) {
				urlStr = Utils.cleanURLQuery((String) table.get("url"));
			}
			// if (urlStr != null && urlStr.indexOf("?") > -1) {
			//
			// urlStr = urlStr.substring(0, urlStr.indexOf("?"));
			// }

			// log.debug("CUID parameter is " + cuids[0]);
			// log.debug("SaveCard parameter is " + Boolean.toString(saveCard));
			// log.debug("SaveCredential parameter is " + Boolean.toString(saveCredential));
			// log.debug("Url parameter is " + urlStr);

			WebForm webForm = null;
			if (urlStr != null && urlStr.length() > 0) {
				webForm = new WebForm(new URI(urlStr), null, null, null);
			}

			ICredential credential = null;
			CardInformationRPPS cardCredentialInfo = null;
			if (cuids != null && cuids.length > 0) {
				final ICard card = getICardByCUID(new CUID(cuids[0]));
				if (card != null) {
					CUID cuidUrl = null;
					if (webForm != null) {
						cuidUrl = getCuidByWebForm(webForm);
					}

					if (cuidUrl != null) {
						if (cuidUrl.equals(card.getCUID()) && !saveCard) {
							deleteWebFormCardRelation(card.getCUID(), webForm);
						}
					}

					cardCredentialInfo = getCardInformationByCuid(card.getCUID());

					if (cardCredentialInfo != null && cardCredentialInfo.getCardCredential() != null) {
						credential = ConvertHelper.convertCardCredential_to_ICredential(card, cardCredentialInfo
								.getCardCredential());
					} else {
						credential = ConvertHelper.convertHashtable_to_Credential(card, typeofCredential, table);
					}
				}
			}
			token = getTokenObject(policy, policytype, sslCert, cuids, credential, cardCredentialInfo, webForm,
					saveCard, saveCredential);
		} catch (final Exception e) {
			LOG.error(e, e);
			throw new RppsException(e);
		}

		return token;
	}

	private List getUsedCards(final WebForm webForm) throws RppsException, UserProfileException {
		ICardUsageManager cardUsageManager = null;
		try {
			Utils.cleanURLQuery(webForm);
			cardUsageManager = getUserProfileService().getCardUsageManager(getHandler());
			return cardUsageManager.getUsedCards(webForm);
		} catch (final Exception e) {
			LOG.error(e, e);
			throw new RppsException(e);
		} finally {
			if (cardUsageManager != null) {
				cardUsageManager.close();
			}
		}
	}

	/**
	 * This method takes as input a policy---for example, one that has been retrieved from the Relying Party Agent
	 * (called here the 'Requester')---as well as a description of the Requester. The UserInterfaceResponse object
	 * should contain all information necessary to display both (a) a user interface which shares the same card
	 * metaphor, sequences of experiences and decision points (the ceremony) of CardSpace and (b) other interface
	 * paradigms to be developed in the future
	 * 
	 * @param userId
	 *            the id to identify the user.
	 * @param password
	 *            the password to identify the user.
	 * @param url
	 *            the URL to the html page with <object> tag
	 * @param target
	 * @param sslCert
	 *            the SSL certificate
	 * @param policyType
	 *            the RP Security Policy type
	 * @param policy
	 *            the RP Agent's Relying Party Security Policy
	 * @return the UIResponseTO transfer object.
	 * @throws RemoteException
	 */

	public UIResponseTO getUserInterface(final RPEndPointTO rpEndPointTO, final String target, final String sslCert,
			final String policytype, final String policy) throws RppsException {

		final UserInterfaceResponse uir = new UserInterfaceResponse();
		X509Certificate[] sslCertsOriginal = null;
		try {
			WebForm webForm = rpEndPointTO.toWebForm();
			final IICardSelector selector = ICardSelectorService.getInstance().getICardSelector(getHandler(), policy);
			uir.setUCL(selector.getUserChoice());

			sslCertsOriginal = RppsHelper.parseSSLCert(sslCert);
			if (sslCertsOriginal != null) {
				X509Certificate cert = null;
				if (sslCertsOriginal != null && sslCertsOriginal.length > 0) {
					cert = KeyStoreRegistry.getInstance().getSecurityService().findLeafFromCertificates(
							sslCertsOriginal);
				}
				uir.setRequestorInformation(cert);
				final IKeyStoreService securityService = KeyStoreRegistry.getInstance().getSecurityService();
				boolean isValid = false;
				boolean isExtendedValid = false;
				X509Certificate[] sslCerts = null;
				try {
					sslCerts = securityService.getNormalizedCertificateChain(getHandler(), sslCertsOriginal, null);
					if (sslCerts != null) {
						isValid = true;
						sslCertsOriginal = sslCerts;
					}
				} catch (final CertificateException e) {
					LOG.error(e, e);
				} catch (final CertStoreException e) {
					LOG.error(e, e);
				}
				if (isValid) {
					isExtendedValid = securityService.extendendValidationCertificateChain(getHandler(),
							sslCertsOriginal, webForm.getUrl());
				}
				uir.setValid(isValid);
				uir.setExtendedValid(isExtendedValid);
			}
		} catch (final Exception e) {
			LOG.error(e, e);
			throw new RppsException(e);
		}
		final UIResponseTO uirto = new UIResponseTO();
		try {
			WebForm webForm = rpEndPointTO.toWebForm();
			Utils.cleanURLQuery(webForm);
			if (uir != null) {
				uirto.setStatus("ok");
				final UIResponseRPTO rp = new UIResponseRPTO();
				rp.setName(uir.getName());
				rp.setAddr1(uir.getAddr1());
				rp.setAddr2(uir.getAddr2());
				rp.setPhone(uir.getPhone());
				rp.setVerifier(uir.getVerifier());
				rp.setValid(uir.isValid());
				rp.setExtendedValid(uir.isExtendedValid());
				uirto.setRP(rp);
				uirto.setAnd(getAnds(uir.getUCL(), webForm, sslCertsOriginal));
			}
		} catch (final Exception e) {
			LOG.error(e, e);
			throw new RppsException(e);
		}
		return uirto;
	}

	/**
	 * This method takes as input a policy---for example, one that has been retrieved from the Relying Party Agent
	 * (called here the 'Requester')---as well as a description of the Requester. The UserInterfaceResponse object
	 * should contain all information necessary to display both (a) a user interface which shares the same card
	 * metaphor, sequences of experiences and decision points (the ceremony) of CardSpace and (b) other interface
	 * paradigms to be developed in the future
	 * 
	 * @param userId
	 *            the id to identify the user.
	 * @param password
	 *            the password to identify the user.
	 * @param url
	 *            the URL to the html page with <object> tag
	 * @param target
	 * @param sslCert
	 *            the SSL certificate
	 * @param policyType
	 *            the RP Security Policy type
	 * @param policy
	 *            the RP Agent's Relying Party Security Policy
	 * @return the UIResponseTO transfer object.
	 * @throws RemoteException
	 * @deprecated
	 */
	public UIResponseTO getUserInterface(final String url, final String target, final String sslCert,
			final String policytype, final String policy) throws RppsException {

		try {
			return getUserInterface(new RPEndPointTO(url, null, null, null), target, sslCert, policytype, policy);
		} catch (final Exception e) {
			LOG.error(e, e);
			throw new RppsException(e);
		}
	}

	public UserProfileTO getUserProfile() throws RppsException, AccessException {
		final UserProfile userProfile = getUserProfile(getHandler());
		return ConvertHelper.convertUserProfile_to_UserProfileTO(userProfile);
	}

	private UserProfile getUserProfile(final CallbackHandler handler) throws RppsException, AccessException {
		try {
			if (this.userProfileInstance == null) {
				this.userProfileInstance = getUserProfileService().getUserProfile(handler);
			}

			return this.userProfileInstance;
		} catch (final UserProfileAuthenticationException e) {
			LOG.error(e, e);
			throw new AccessException(e.getMessage(), e);
		} catch (final UserProfileException e) {
			LOG.error(e, e);
			throw new RppsException(e.getMessage(), e);
		}
	}

	private List getUsersChoiceOfOptionalClaims(final CUID cuid, final WebForm webForm) throws UserProfileException {
		ICardUsageManager cardUsageManager = null;
		try {
			Utils.cleanURLQuery(webForm);
			cardUsageManager = getUserProfileService().getCardUsageManager(getHandler());
			final Set optionalClaims = cardUsageManager.getOptionalClaims(cuid, webForm);
			if (optionalClaims != null) {
				return new ArrayList(optionalClaims);
			} else {
				return new ArrayList();
			}
		} catch (final Exception e) {
			LOG.error(e, e);
			throw new RppsException(e);
		} finally {
			if (cardUsageManager != null) {
				cardUsageManager.close();
			}
		}
	}

	/* Methods of card information */

	/**
	 * Imports card from a file.
	 * 
	 * @param crd
	 *            the card as a byte array.
	 */
	public ICardResponseTO importICards(final byte[] crd) throws RppsException {
		return importICards(crd, null, null);
	}

	/**
	 * Imports card from a file.
	 * 
	 * @param crd
	 *            the card as a byte array.
	 * @param formatID
	 *            the data format of the input stream if available. If <code>null</code> is used all available data
	 *            formats will be checked in order to import.
	 * @param credential
	 *            the authentication data to proccess the input stream if required
	 */
	public ICardResponseTO importICards(final byte[] crd, final String formatID,
			final UITokenServiceCredentialTO credential) throws RppsException {
		try {
			List cards = null;
			cards = importICards(new ByteArrayInputStream(crd), formatID, ConvertHelper
					.convertUITokenServiceCredentialTO_to_ICredentials(credential));

			return convertICards_To_CardResponceTO(cards);
		} catch (final Exception e) {
			LOG.error(e, e);
			throw new RppsException(e);
		}
	}

	/**
	 * Imports card from a stream.
	 * 
	 * @param in
	 *            the input stream to import from
	 * @param formatID
	 *            the data format of the input stream if available. If <code>null</code> is used all available data
	 *            formats will be checked in order to import.
	 * @param credential
	 *            the authentication data to proccess the input stream if required
	 */
	protected List importICards(final InputStream in, final String formatID, final ICredential credential)
			throws RppsException {
		try {
			final List result = new ArrayList();
			for (final Iterator iter = ICardRegistry.getInstance().importICards(getHandler(), in, formatID, credential); iter
					.hasNext();) {
				result.add(iter.next());
			}
			return result;
		} catch (final CardException e) {
			LOG.error(e, e);
			throw new RppsException(e);
		}
	}

	public ResponseMessage modifyCategories(final CategoryTO[] categoriesTO) throws RppsException {
		try {
			for (int i = 0; i < categoriesTO.length; i++) {
				getUserProfileService().modifyCategory(getHandler(),
						ConvertHelper.convertCategoryTO_to_Category(categoriesTO[i]));
			}
			return new ResponseMessage(ResponseMessage.noError, "");
		} catch (final UserProfileException e) {
			LOG.error(e, e);
			return new ResponseMessage(ResponseMessage.unknownError, e.getMessage());
		}
	}

	/* Methods of storage card credential */

	public CategoryTO modifyCategory(final CategoryTO categoryTO) throws RppsException {
		try {
			final Category category = getUserProfileService().modifyCategory(getHandler(),
					ConvertHelper.convertCategoryTO_to_Category(categoryTO));
			return ConvertHelper.convertCategory_to_CategoryTO(category);
		} catch (final UserProfileException e) {
			LOG.error(e, e);
			throw new RppsException(e);
		}
	}

	/**
	 * Update user profile.
	 * 
	 * @param userProfile
	 *            user profile transfer object
	 * @return user profile transfer object
	 * @throws RppsException
	 *             some situation
	 */
	public UserProfileTO modifyUserProfile(final UserProfileTO userProfile) throws RppsException {
		try {
			this.userProfileInstance = getUserProfileService().modifyUserProfile(getHandler(),
					ConvertHelper.convertUserProfileTO_to_UserProfile(this.userProfileInstance, userProfile));
			return getUserProfile();
		} catch (final Exception e) {
			LOG.error(e, e);
			throw new RppsException(e);
		}
	}

	public void removeClaimValuesMRU(final String claimType, final String claimValue) throws RppsException {
		ICardUsageManager cardUsageManager = null;
		try {
			cardUsageManager = getUserProfileService().getCardUsageManager(getHandler());

			final Set claimValuesMRU = cardUsageManager.getClaimValuesMRU(claimType);
			if (claimValuesMRU.contains(claimValue)) {
				claimValuesMRU.remove(claimValue);
				cardUsageManager.setClaimValuesMRU(claimType, claimValuesMRU);
			}
		} catch (final Exception e) {
			LOG.error(e, e);
			throw new RppsException(e);
		} finally {
			if (cardUsageManager != null) {
				try {
					cardUsageManager.close();
				} catch (final UserProfileException e) {
					LOG.error(e, e);
				}
			}
		}
	}

	protected ICard resetPinCode(IPersonalInformationCard card, final byte[] oldPinCode) throws RppsException {
		try {

			final PinCodeCredential pinCodeCredential = new PinCodeCredential();
			pinCodeCredential.setPinCode(oldPinCode);
			card = (IPersonalInformationCard) card.getProvider().getICardByCUID(getHandler(), card.getCUID(),
					pinCodeCredential);

			try {
				final PinCodeCredential pinCodeCredentialNew = new PinCodeCredential();
				pinCodeCredentialNew.setPinCode(null);

				card.beginUpdates();
				card.setPinCode(pinCodeCredentialNew);
				card.applyUpdates();
			} catch (final CardException e) {
				if (card.isEditMode()) {
					card.cancelUpdates();
				}
				throw new RppsException(e);
			}
			return ICardRegistry.getInstance().getICardByCUID(getHandler(), card.getCUID());
		} catch (final Exception e) {
			LOG.error(e, e);
			throw new RppsException(e);
		}
	}

	public ICardTO resetPinCode(final String cuid, final String oldPinCode) throws RppsException {
		ICard card = null;
		byte[] oldPinCode_ = null;
		try {
			card = getICardByCUID(new CUID(cuid));

			if (card == null) {
				throw new RppsException("Can't find card with CUID " + cuid);
			}

			if (!(card instanceof IPersonalInformationCard)) {
				new RppsException("This card is not personal");
			}

			CardInformationRPPS cardInformation = null;

			if (oldPinCode == null) {
				cardInformation = getCardInformationByCuid(new CUID(cuid));
				if (cardInformation.getCardCredential() != null
						&& IRppsService.IPinCodeCredential.equals(cardInformation.getCardCredential()
								.getCredentialType())) {
					oldPinCode_ = cardInformation.getCardCredential().getPinCode();
				}
			} else {
				oldPinCode_ = oldPinCode.getBytes("UTF-8");
			}

			card = resetPinCode((IPersonalInformationCard) card, oldPinCode_);

			if (cardInformation != null) {
				clearCardCredential(card.getCUID());
			}

			return convertICard_To_ICardTO(card, false, null);
		} catch (final Exception e) {
			LOG.error(e, e);
			throw new RppsException(e);
		}
	}

	private void saveUsersChoiceOfOptionalClaims(final CUID cuid, final WebForm webForm, final List claims)
			throws UserProfileException {
		ICardUsageManager cardUsageManager = null;
		try {
			Utils.cleanURLQuery(webForm);
			final Set optionalClaims = new HashSet(claims);
			cardUsageManager = getUserProfileService().getCardUsageManager(getHandler());
			cardUsageManager.setOptionalClaims(cuid, webForm, optionalClaims);
		} catch (final Exception e) {
			LOG.error(e, e);
			throw new RppsException(e);
		} finally {
			if (cardUsageManager != null) {
				cardUsageManager.close();
			}
		}
	}

	protected boolean setCardCredential(final CUID cuid, final CardCredential cardCredential) throws RppsException {
		try {
			final CardInformationRPPS cardInformation = new CardInformationRPPS(getUserProfileService()
					.setCardCredential(getHandler(), cuid, cardCredential), cuid);
			if (cardInformation.getCardCredential() != null && cardCredential != null
					|| cardInformation.getCardCredential() == null && cardCredential == null) {
				return true;
			}
		} catch (final Exception e) {
			LOG.error(e, e);
			throw new RppsException(e);
		}
		return false;
	}

	public boolean setCardCredential(final String cuid, final UITokenServiceCredentialTO tokenCredential)
			throws RppsException {
		try {
			ICard card = getICardByCUID(new CUID(cuid));

			if (card == null) {
				throw new RppsException("Can't find card with CUID " + cuid);
			}

			final ICredential credential = ConvertHelper.convertUITokenServiceCredentialTO_to_ICredentials(card,
					tokenCredential);

			final ICardProvider provider = card.getProvider();

			card = null;

			if (credential != null) {
				card = provider.getICardByCUID(getHandler(), new CUID(cuid), credential);
			}

			if (card != null) {
				final CardCredential cardCredential = ConvertHelper
						.convertUITokenServiceCredentialTO_to_CardCredential(tokenCredential);
				final boolean result = setCardCredential(new CUID(cuid), cardCredential);
				if (result) {
					return true;
				} else {
					return false;
				}
			}
			throw new RppsException("Didn't set card credential.");

		} catch (final Exception e) {
			LOG.error(e, e);
			throw new RppsException(e.getMessage(), e);
		}
	}

	private void setComplexValues(final IClaim claim, final PropertyTO value, final ICardUsageManager cardUsageManager)
			throws Exception {
		if (claim != null && claim instanceof IComplexClaim) {
			final int size = value.getComplexValue().length;
			for (int index = 0; index < size; index++) {
				final PropertyTO propTO = value.getComplexValue()[index];
				IClaim claimValue = ((IComplexClaim) claim).getClaim(propTO.getClaimType());
				if (claimValue == null) {
					claimValue = ((IComplexClaim) claim).createClaim(propTO.getClaimType());
				}
				if (propTO.getValueType() != null) {
					setSimpleValues(claimValue, propTO, cardUsageManager);
				} else {
					setComplexValues(claimValue, propTO, cardUsageManager);
				}
			}
		}
	}

	public void setHandler(final CallbackHandler handler) {
		this.handler = handler;
	}

	public void setNewPrivateUserPassword(final String newPassword) throws UserProfileException {
		try {
			getUserProfileService().setNewPassword(getHandler(), newPassword);
		} catch (final Exception e) {
			LOG.error(e, e);
			throw new RppsException(e);
		}
	}

	protected ICard setPinCode(final IPersonalInformationCard card, final byte[] pinCode) throws RppsException {
		try {
			PinCodeCredential pinCodeCredential = new PinCodeCredential();
			try {
				card.beginUpdates();
				pinCodeCredential.setPinCode(pinCode);
				card.setPinCode(pinCodeCredential);
				card.applyUpdates();
			} catch (final CardException e) {
				if (card.isEditMode()) {
					card.cancelUpdates();
				}
				throw new RppsException(e);
			}
			pinCodeCredential = new PinCodeCredential();
			pinCodeCredential.setPinCode(pinCode);

			return card.getProvider().getICardByCUID(getHandler(), card.getCUID(), pinCodeCredential);
		} catch (final Exception e) {
			LOG.error(e, e);
			throw new RppsException(e);
		}
	}

	public ICardTO setPinCode(final String cuid, final String pinCode) throws RppsException {
		ICard card = null;
		try {
			card = ICardRegistry.getInstance().getICardByCUID(getHandler(), cuid);
			if (!(card instanceof IPersonalInformationCard)) {
				new RppsException("This card is not personal");
			}

			card = setPinCode((IPersonalInformationCard) card, pinCode.getBytes("UTF-8"));
			return convertICard_To_ICardTO(card, false, null);
		} catch (final Exception e) {
			LOG.error(e, e);
			throw new RppsException(e);
		}
	}

	private void setSimpleValues(final IClaim claim, final PropertyTO value, final ICardUsageManager cardUsageManager)
			throws Exception {
		if (claim != null && claim instanceof ISimpleClaim) {
			if (((ISimpleClaimType) claim.getType()).getDataType().getName().equals(value.getValueType())) {
				final List newValues = new ArrayList();
				for (int index = 0; index < value.getValue().length; index++) {
					newValues.add(value.getValue()[index]);
					try {
						addClaimValuesMRU(claim.getType().getType(), value.getValue()[index], cardUsageManager);
					} catch (final Exception e) {
						LOG.error(e, e);
					}

				}
				((ISimpleClaim) claim).setValues(newValues);
			}
		}
	}

	/**
	 * Update claim across cards.
	 */
	public ICardUpdateResponseTO[] updateClaimsAcrossCards(final String[] claimTypes, final String[] claimValues,
			final ICardUpdateTO[] cardUpdate) throws RppsException {

		ICard card = null;
		if (cardUpdate == null) {
			return new ICardUpdateResponseTO[0];
		}

		final ICardUpdateResponseTO[] result = new ICardUpdateResponseTO[cardUpdate.length];
		for (int i = 0; i < cardUpdate.length; i++) {
			try {

				card = getICardByCUID(new CUID(cardUpdate[i].getCuid()));

				if (card instanceof IPersonalInformationCard) {

					final CardInformationRPPS cardInformation = getCardInformationByCuid(card.getCUID());
					ICredential credential = null;
					if (cardInformation.getCardCredential() != null) {
						credential = ConvertHelper.convertCardCredential_to_ICredential(card, cardInformation
								.getCardCredential());
					} else {
						if (cardUpdate[i].getTokenServiceCredential() != null) {
							credential = ConvertHelper.convertUITokenServiceCredentialTO_to_ICredentials(card,
									cardUpdate[i].getTokenServiceCredential());
						}
					}

					if (credential != null) {
						card = card.getProvider().getICardByCUID(getHandler(), card.getCUID(), credential);
					}

					card.beginUpdates();

					for (int index = 0; index < claimTypes.length; index++) {
						final String claimType = claimTypes[index];
						final String claimValue = claimValues[index];
						IClaim claim = card.getClaim(claimType);

						if (claim == null) {
							claim = card.createClaim(claimType);
						}

						if (claim != null && claim instanceof ISimpleClaim) {
							final List newValues = new ArrayList();
							newValues.add(claimValue);
							((ISimpleClaim) claim).setValues(newValues);
							try {
								addClaimValuesMRU(claimType, claimValue);
							} catch (final Exception e) {
								LOG.error(e, e);
							}
						}
					}
					card.applyUpdates();
					result[i] = new ICardUpdateResponseTO(card.getCUID().toString());
				}
			} catch (final Exception e) {
				LOG.error(e, e);
				result[i] = new ICardUpdateResponseTO(ICardUpdateResponseTO.unknownError, e.getMessage());
				try {
					if (card != null && card.isEditMode()) {
						card.cancelUpdates();
					}
				} catch (final Throwable tr) {
					LOG.error(tr, tr);
				}
			} finally {
			}
		}
		return result;
	}

	/**
	 * Update card.
	 * 
	 * @param userId
	 *            the id to identify the user.
	 * @param password
	 *            the password to identify the user.
	 * @param icardTO
	 *            the card transfer object.
	 * @return the card transfer object.
	 */
	public ICardTO updateICard(final ICardTO icardTO) throws RppsException {
		ICardUsageManager cardUsageManager = null;
		try {
			cardUsageManager = getUserProfileService().getCardUsageManager(getHandler());

			return convertICard_To_ICardTO(updateICard(icardTO, cardUsageManager), false, null);
		} catch (final Exception e) {
			LOG.error(e, e);
			throw new RppsException(e);
		} finally {
			if (cardUsageManager != null) {
				try {
					cardUsageManager.close();
				} catch (final UserProfileException e) {
					LOG.error(e, e);
				}
			}
		}
	}

	private ICard updateICard(final ICardTO icardTO, final ICardUsageManager cardUsageManager) throws RppsException {

		ICard card = null;
		try {
			card = getICardByCUID(new CUID(icardTO.getCuid()));

			final CardInformationRPPS cardInformation = getCardInformationByCuid(card.getCUID(), cardUsageManager);
			ICredential credential = null;
			if (cardInformation.getCardCredential() != null) {
				credential = ConvertHelper.convertCardCredential_to_ICredential(card, cardInformation
						.getCardCredential());
			} else {
				if (icardTO.getTokenServiceCredentials() != null && icardTO.getTokenServiceCredentials().length != 0) {
					credential = ConvertHelper.convertUITokenServiceCredentialTO_to_ICredentials(card, icardTO
							.getTokenServiceCredentials()[0]);
				}
			}

			if (credential != null) {
				card = card.getProvider().getICardByCUID(getHandler(), card.getCUID(), credential);
			}

			card.beginUpdates();
			if (!RppsHelper.equalsObject(card.getName(), icardTO.getName())) {
				card.setName(icardTO.getName());
			}
			try {
				if (!RppsHelper.equalsObject(card.getIssuerName(), icardTO.getIssuerName())) {
					card.setIssuerName(icardTO.getIssuerName());
				}
			} catch (final Exception exc) {
				LOG.error(exc, exc);
			}

			if (!RppsHelper.equalsObject(card.getImageType(), icardTO.getImageType())
					|| !RppsHelper.equalsObject(card.getImage(), icardTO.getImage())) {
				card.setImage(icardTO.getImage(), icardTO.getImageType());
			}

			if (card instanceof IPersonalInformationCard) {
				try {
					if (!RppsHelper.equalsObject(card.getTimeExpires(), icardTO.getExpiredTime())) {
						card.setTimeExpires(icardTO.getExpiredTime());
					}
				} catch (final Exception exc) {
					LOG.error(exc, exc);
				}

				if (icardTO.getProperties() != null) {
					final int size = icardTO.getProperties().length;
					for (int index = 0; index < size; index++) {
						final PropertyTO propTO = icardTO.getProperties()[index];
						IClaim claim = card.getClaim(propTO.getClaimType());
						if (claim == null) {
							claim = card.createClaim(propTO.getClaimType());
						}
						if (propTO.getComplexValue() == null) {
							setSimpleValues(claim, propTO, cardUsageManager);
						} else {
							setComplexValues(claim, propTO, cardUsageManager);
						}
					}
				}
			}

			card.applyUpdates();

			card = getICardByCUID(new CUID(icardTO.getCuid()));
			return card;

		} catch (final Exception e) {
			LOG.error(e, e);
			try {
				if (card != null && card.isEditMode()) {
					card.cancelUpdates();
				}
			} catch (final Throwable tr) {
			}
			throw new RppsException(e);
		} finally {
		}
	}

	/**
	 * Update card.
	 * 
	 * @param userId
	 *            the id to identify the user.
	 * @param password
	 *            the password to identify the user.
	 * @param icardTO
	 *            the card transfer object.
	 * @return the card transfer object.
	 */
	public ICardUpdateResponseTO[] updateICards(final ICardTO[] icardTOList) throws RppsException {
		ICardUsageManager cardUsageManager = null;
		try {
			final ICardUpdateResponseTO[] result = new ICardUpdateResponseTO[icardTOList.length];
			cardUsageManager = getUserProfileService().getCardUsageManager(getHandler());
			for (int i = 0; i < icardTOList.length; i++) {
				try {
					final ICard card = updateICard(icardTOList[i], cardUsageManager);
					result[i] = new ICardUpdateResponseTO(card.getCUID().toString());
				} catch (final Exception e) {
					LOG.error(e, e);
					result[i] = new ICardUpdateResponseTO(ICardUpdateResponseTO.unknownError, e.getMessage());
				}
			}
			return result;
		} catch (final Exception e) {
			LOG.error(e, e);
			throw new RppsException(e);
		} finally {
			if (cardUsageManager != null) {
				try {
					cardUsageManager.close();
				} catch (final UserProfileException e) {
					LOG.error(e, e);
				}
			}
		}
	}

	/**
	 * Checks availability of user profile by userIdentifier
	 * 
	 * @param userIdentifier
	 * @return Returns <code>true</code> if user identifier exists. Otherwise <code>false</code>
	 * @throws RppsException
	 *             some situation
	 */
	public boolean userIdentifierExists(final String userIdentifier) {
		try {
			return getUserProfileService().userIdentifierExists(userIdentifier);
		} catch (final UserProfileException e) {
		}
		return false;
	}

	protected UserProfile getUserProfileInstance() {
		return userProfileInstance;
	}

	protected void setUserProfileInstance(UserProfile userProfileInstance) {
		this.userProfileInstance = userProfileInstance;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.higgins.rpps.core.IRppsService#getDefautPCard()
	 */
	public ICardTO getDefaultPCard() throws RppsException {
		try {
			String defaultPCardCUID = getUserProfileInstance().getDefaultPCardCUID();
			if (null != defaultPCardCUID && defaultPCardCUID.length() > 0) {
				return convertICard_To_ICardTO(ICardRegistry.getInstance().getICardByCUID(getHandler(),
						defaultPCardCUID), false, null);
			} else {
				return null;
			}
		} catch (final Exception e) {
			LOG.error(e, e);
			throw new RppsException(e);
		}

	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.higgins.rpps.core.IRppsService#setDefautPCard(java.lang.String)
	 */
	public void setDefaultPCard(String pcardId) throws RppsException {
		getUserProfileInstance().setDefaultPCardCUID(pcardId);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.higgins.rpps.core.IRppsService#generatePasswordResetCode(java.lang.String)
	 */
	public void generatePasswordResetCode(String userIdentifier) throws RppsException {
		try {
			UserProfile userProfile = getUserProfileService().getUserProfile(userIdentifier);
			if (null == userProfile.getEmail()) {
				throw new RppsException("Email address is empty, couldn't send email.");
			} else {
				// generate password reset code
				// Random randomGenerator = new Random();
				// SecureRandom secureRandom = SecureRandom();

				String passwordResetCode = PPIDHelper.getUserFriendlyPPID(SecureRandom.getSeed(32));
				userProfile.setPasswordResetCode(passwordResetCode.replaceAll("-", ""));
				getUserProfileService().modifyUserProfile(userProfile);
				new EmailUtils().sendPasswordResetCode(userProfile.getEmail(), passwordResetCode);
			}
		} catch (Exception e) {
			LOG.error(e, e);
			throw new RppsException(e);
		}
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.higgins.rpps.core.IRppsService#modifyPasswordWithPasswordResetCode(java.lang.String,
	 * java.lang.String)
	 */
	public void modifyPasswordWithPasswordResetCode(final String userIdentifier, final String passwordResetCode,
			final String newPassword) throws RppsException {
		try {
			UserProfile userProfile = getUserProfileService().getUserProfile(userIdentifier);
			if (null == userProfile.getPasswordResetCode()
					|| !userProfile.getPasswordResetCode().equals(passwordResetCode.replaceAll("-", ""))) {
				throw new RppsException("Password reset code incorrect");
			} else {
				// clean PasswordResetCode
				userProfile.setPasswordResetCode(null);
				getUserProfileService().modifyUserProfile(userProfile);
				// update password
				getUserProfileService().modifyPassword(userIdentifier, newPassword);
				new EmailUtils().sendPasswordUpdatedEmail(userProfile.getEmail());

			}
		} catch (Exception e) {
			LOG.error(e, e);
			throw new RppsException(e);
		}

	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.higgins.rpps.core.IRppsService#modifyPasswordWithOldPassword(java.lang.String, java.lang.String,
	 * java.lang.String)
	 */
	public void modifyPasswordWithOldPassword(String userIdentifier, String oldPassword, String newPassword)
			throws RppsException {
		try {
			getUserProfile();
			if (null == newPassword) {
				throw new RppsException("Password shouldn't be null");
			} else {
				getUserProfileService().modifyPassword(userIdentifier, newPassword);
				new EmailUtils().sendPasswordUpdatedEmail(getUserProfile().getEmail());
			}
		} catch (Exception e) {
			LOG.error(e, e);
			throw new RppsException(e);
		}
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.higgins.rpps.core.IRppsService#createCaptcha()
	 */
	public Captcha createCaptcha() throws RppsException {
		try {
			Captcha captcha = new Captcha();
			captcha.setDtCreation(new Date());
			// captcha.setKey("" + new Date().getTime() + "_" + Math.random());
			captcha.setKey(UUIDGenerator.getUUID());

			return getUserProfileService().addCaptcha(captcha);

		} catch (Exception e) {
			LOG.error(e, e);
			throw new RppsException(e);
		}

	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.higgins.rpps.core.IRppsService#getCaptcha(java.lang.String)
	 */
	public Captcha getCaptcha(String id) throws RppsException {
		try {
			return getUserProfileService().getCaptcha(id);
		} catch (final Exception e) {
			LOG.error(e, e);
			throw new RppsException(e);
		}
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.higgins.rpps.core.IRppsService#deleteCaptcha(java.lang.String)
	 */
	public void deleteCaptcha(String id) throws RppsException {
		try {
			getUserProfileService().deleteCaptcha(getUserProfileService().getCaptcha(id));
		} catch (final Exception e) {
			LOG.error(e, e);
			throw new RppsException(e);
		}
	}

}
