/*******************************************************************************
 * Copyright (c) 2007 Google
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     Markus Sabadello - Initial API and implementation
 *******************************************************************************/
package org.eclipse.higgins.saml2idp.server.util;

import java.util.Date;
import java.util.HashMap;
import java.util.Map;

import javax.servlet.ServletContext;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

public class StateUtil {

	private static final String SESSION_USERNAME = "_username";
	private static final String SERVLETCONTEXT_FAILUREMAP = "_failures";
	private static final String SERVLETCONTEXT_FAILUREDATEMAP = "_failuredates";

	/**
	 * Make constructor private since we only use static methods in this util class.
	 */
	private StateUtil() {

	}

	/**
	 * Logs in a user in the GIB. 
	 * This is done by storing an attribute in the session.
	 * Next time a SAML request comes in, the user is immediately authenticated without the need
	 * to provide credentials.
	 * @param request The current HttpServletRequest object.
	 * @param response The current HttpServletResponse object.
	 * @param username The name of the user to log in.
	 */
	public static void loginUser(HttpServletRequest request, HttpServletResponse response, String username) {

		request.getSession().setAttribute(SESSION_USERNAME, username);
	}

	/**
	 * Logs out a user from the GIB. 
	 * This is done by removing an attribute from the session.
	 * Next time a SAML request comes in, the user is asked to provide credentials again.
	 * @param request The current HttpServletRequest object.
	 * @param response The current HttpServletResponse object.
	 */
	public static void logout(HttpServletRequest request, HttpServletResponse response) {

		request.getSession().removeAttribute(SESSION_USERNAME);
	}

	/**
	 * Get the currently logged in user.
	 * This is done by checking if a certain key is present in the session.
	 * @param request The current HttpServletRequest object.
	 * @param response The current HttpServletResponse object.
	 * @return The username of the currently logged in user.
	 */
	public static String getUser(HttpServletRequest request, HttpServletResponse response) {

		return((String) request.getSession().getAttribute(SESSION_USERNAME));
	}

	/**
	 * Increment the recorded number of failed login attempts.
	 * This is done by checking a map in the servlet context. This map holds
	 * the number of failures for the remote IP address.
	 * @param request The current HttpServletRequest object.
	 * @param response The current HttpServletResponse object.
	 */
	public static void incFailures(HttpServletRequest request, HttpServletResponse response) {

		ServletContext servletContext = request.getSession().getServletContext();
		String remoteAddr = request.getRemoteAddr();

		// update failure map
		
		Map failureMap = (Map) servletContext.getAttribute(SERVLETCONTEXT_FAILUREMAP);
		if (failureMap == null) {

			failureMap = new HashMap();
			servletContext.setAttribute(SERVLETCONTEXT_FAILUREMAP, failureMap);
		}

		Integer failures = (Integer) failureMap.get(remoteAddr);
		if (failures == null) failures = new Integer(0);

		failures = new Integer(failures.intValue() + 1);
		failureMap.put(remoteAddr, failures);
		
		// update failure date map
		
		Map failureDateMap = (Map) servletContext.getAttribute(SERVLETCONTEXT_FAILUREDATEMAP);
		if (failureDateMap == null) {
			
			failureDateMap = new HashMap();
			servletContext.setAttribute(SERVLETCONTEXT_FAILUREDATEMAP, failureDateMap);
		}
		
		failureDateMap.put(remoteAddr, new Date());
	}

	/**
	 * Get the recorded number of failed login attempts.
	 * This is done by checking a map in the servlet context. This map holds
	 * the number of failures for the remote IP address.
	 * @param request The current HttpServletRequest object.
	 * @param response The current HttpServletResponse object.
	 * @return The number of login failures for the remote IP address of the
	 * current request.
	 */
	public static int getFailures(HttpServletRequest request, HttpServletResponse response) {

		ServletContext servletContext = request.getSession().getServletContext();
		String remoteAddr = request.getRemoteAddr();

		Map failureMap = (Map) servletContext.getAttribute(SERVLETCONTEXT_FAILUREMAP);
		if (failureMap == null) return(0);
		
		Integer failures = (Integer) failureMap.get(remoteAddr);
		if (failures == null) return(0);
		
		return(failures.intValue());
	}

	/**
	 * Resets the recorded number of failed login attempts.
	 * This is done by checking a map in the servlet context. This map holds
	 * the number of failures for the remote IP address.
	 * @param request The current HttpServletRequest object.
	 * @param response The current HttpServletResponse object.
	 */
	public static void resetFailures(HttpServletRequest request, HttpServletResponse response) {

		ServletContext servletContext = request.getSession().getServletContext();
		String remoteAddr = request.getRemoteAddr();

		Map failureMap = (Map) servletContext.getAttribute(SERVLETCONTEXT_FAILUREMAP);
		if (failureMap == null) return;
		
		failureMap.remove(remoteAddr);
	}

	/**
	 * Get the time/date of the last failed login attempt.
	 * This is done by checking a map in the servlet context. This map holds
	 * the last failure date/time for the remote IP address.
	 * @param request The current HttpServletRequest object.
	 * @param response The current HttpServletResponse object.
	 * @return The number of login failures for the remote IP address of the
	 * current request.
	 */
	public static Date getLastFailureDate(HttpServletRequest request, HttpServletResponse response) {

		ServletContext servletContext = request.getSession().getServletContext();
		String remoteAddr = request.getRemoteAddr();

		Map failureDateMap = (Map) servletContext.getAttribute(SERVLETCONTEXT_FAILUREDATEMAP);
		if (failureDateMap == null) return(null);
		
		return((Date) failureDateMap.get(remoteAddr));
	}
}
