/*******************************************************************************
 * Copyright (c) 2007 Google
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     Markus Sabadello - Initial API and implementation
 *******************************************************************************/
package org.eclipse.higgins.saml2idp.server.util;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.StringReader;
import java.security.PrivateKey;
import java.security.PublicKey;
import java.util.Date;
import java.util.HashMap;
import java.util.Map;
import java.util.zip.DataFormatException;
import java.util.zip.Inflater;

import javax.servlet.ServletException;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import org.apache.commons.codec.binary.Base64;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.eclipse.higgins.saml2idp.server.Init;
import org.eclipse.higgins.util.saml.SAML2Assertion;
import org.eclipse.higgins.util.saml.SAML2AuthnRequest;
import org.eclipse.higgins.util.saml.SAML2AuthnStatement;
import org.eclipse.higgins.util.saml.SAML2Conditions;
import org.eclipse.higgins.util.saml.SAML2Response;
import org.eclipse.higgins.util.saml.SAML2Subject;
import org.eclipse.higgins.util.saml.SAMLConstants;
import org.xml.sax.SAXException;

public class SAMLUtil {

	private static final Log log = LogFactory.getLog(SAMLUtil.class);

	private SAMLUtil() {

	}

	public static SAML2AuthnRequest parseAuthnRequest(String samlString) throws DataFormatException, IOException, SAXException {

		// Decode the Base64-encoded string and inflate it.

		byte[] samlBytes = Base64.decodeBase64(samlString.getBytes("UTF-8"));

		Inflater inflater = new Inflater(true);
		inflater.setInput(samlBytes);

		ByteArrayOutputStream stream = new ByteArrayOutputStream();
		byte[] buffer = new byte[1024];

		while (! inflater.finished()) {

			int count = inflater.inflate(buffer);
			if (count == 0) break;
			stream.write(buffer, 0, count);
		}

		stream.close();

		samlBytes = stream.toByteArray();

		// Parse the SAML2 AuthnRequest XML document from the string.

		return(new SAML2AuthnRequest(new ByteArrayInputStream(samlBytes)));
	}

	public static void postResponse(HttpServletRequest request, HttpServletResponse response, String nameid, String destination, String relayState) throws ServletException, IOException {

		log.info("Creating SAML Response for destination " + destination + " with relaystate " + relayState);

		// Create a SAML2 Response message.

		SAML2Response samlResponse = new SAML2Response();
		samlResponse.setStatusCodeValue(SAMLConstants.STATUSCCODE_SUCCESS);
		samlResponse.setDestination(destination);

		SAML2Assertion samlAssertion = new SAML2Assertion(samlResponse.getDocument());
		samlAssertion.setIssuer(Init.getSAML2Issuer());
		samlResponse.setSAMLAssertion(samlAssertion);

		SAML2Subject samlSubject = new SAML2Subject(samlAssertion.getDocument());
		samlSubject.setNameIDFormat(SAMLConstants.NAMEIDFORMAT_ENTITY);
		samlSubject.setNameID(nameid);
		samlSubject.setSubjectConfirmationMethod(SAMLConstants.SUBJECTCONFIRMATIONMETHOD_BEARER);
		samlAssertion.setSAMLSubject(samlSubject);

		SAML2Conditions samlConditions = new SAML2Conditions(samlAssertion.getDocument());
		samlConditions.setNotBefore(new Date(samlAssertion.getIssueInstant().getTime() - Init.getSAML2AssertionValidityMillis()));
		samlConditions.setNotOnOrAfter(new Date(samlAssertion.getIssueInstant().getTime() + Init.getSAML2AssertionValidityMillis()));
		samlAssertion.setSAMLConditions(samlConditions);

		SAML2AuthnStatement samlAuthnStatement = new SAML2AuthnStatement(samlAssertion.getDocument());
		samlAuthnStatement.setAuthnContextClassRef(SAMLConstants.AUTHNCONTEXTCLASSREF_PASSWORD);
		samlAssertion.setSAMLAuthnStatement(samlAuthnStatement);

		// Sign it.

		PrivateKey idpPrivateKey = Init.getIdpPrivateKey();
		PublicKey idpPublicKey = Init.getIdpPublicKey();

		if (idpPrivateKey != null && idpPublicKey != null) {

			try {

				samlResponse = new SAML2Response(new StringReader(samlResponse.dump()));
				samlResponse.sign(idpPrivateKey, idpPublicKey);
			} catch (Exception ex) {

				log.error(ex);
				throw new ServletException("Cannot sign SAML2 message.", ex);
			}
		}

		// We need to convert the SAML message to Base64.

		String samlString = samlResponse.dump();
		byte[] samlBytes = samlString.getBytes();

		samlString = new String(Base64.encodeBase64(samlBytes));

		// Initialize hidden form fields.

		Map attributes = new HashMap ();

		attributes.put("SAMLResponse", samlString);
		if (relayState != null) attributes.put("RelayState", relayState);

		// Display the auto-redirect form.

		log.info(samlResponse.getDestination());

		request.setAttribute("action", samlResponse.getDestination());
		request.setAttribute("attributes", attributes);
		request.getRequestDispatcher("FormRedirect.jsp").forward(request, response);
	}
}
