/*******************************************************************************
 * Copyright (c) 2007 Google
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     Markus Sabadello - Initial API and implementation
 *******************************************************************************/
package org.eclipse.higgins.saml2idp.server;

import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.security.PublicKey;
import java.security.cert.X509Certificate;
import java.util.Set;

import javax.servlet.ServletException;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import org.apache.commons.codec.binary.Base64;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.eclipse.higgins.saml2idp.server.util.ExtractUtil;
import org.eclipse.higgins.saml2idp.server.util.SAMLUtil;
import org.eclipse.higgins.saml2idp.server.util.StateUtil;
import org.eclipse.higgins.util.saml.SAML2AuthnRequest;

/**
 * Servlet implementation class for Servlet: SAMLEndpoint
 *
 */
public class SAMLEndpoint extends javax.servlet.http.HttpServlet implements javax.servlet.Servlet {

	private static final long serialVersionUID = 1L;

	private static final String PARAMETER_SAMLREQUEST = "SAMLRequest";
	private static final String PARAMETER_RELAYSTATE = "RelayState";

	private static final Log log = LogFactory.getLog(SAMLEndpoint.class);

	public SAMLEndpoint() {

		super();
	}   	

	protected void doGet(HttpServletRequest request, HttpServletResponse response) throws ServletException, IOException {

		log.trace("doGet()");

		// Read and check parameters.

		String samlString = request.getParameter(PARAMETER_SAMLREQUEST);
		String relayState = request.getParameter(PARAMETER_RELAYSTATE);

		if (samlString == null) {

			response.sendError(HttpServletResponse.SC_BAD_REQUEST, "Required parameter " + PARAMETER_SAMLREQUEST + " not found.");
			return;
		}

		// Try to parse the SAML2 AuthnRequest.

		SAML2AuthnRequest samlAuthnRequest;

		try {

			samlAuthnRequest = SAMLUtil.parseAuthnRequest(samlString);
		} catch (Exception ex) {

			log.error(ex);
			throw new ServletException("Cannot parse SAML2 AuthnRequest.", ex);
		}

		// Process the request.

		this.processRequest(request, response, samlAuthnRequest, relayState);
	}

	protected void doPost(HttpServletRequest request, HttpServletResponse response) throws ServletException, IOException {

		log.trace("doPost()");

		// Read and check parameters.

		String samlString = request.getParameter(PARAMETER_SAMLREQUEST);
		String relayState = request.getParameter(PARAMETER_RELAYSTATE);

		if (samlString == null) {

			response.sendError(HttpServletResponse.SC_BAD_REQUEST, "Required parameter " + PARAMETER_SAMLREQUEST + " not found.");
			return;
		}

		byte[] samlBytes = Base64.decodeBase64(samlString.getBytes());

		// Try to parse the SAML2 AuthnRequest.

		SAML2AuthnRequest samlAuthnRequest;

		try {

			samlAuthnRequest = new SAML2AuthnRequest(new ByteArrayInputStream(samlBytes));
		} catch (Exception ex) {

			log.error(ex);
			throw new ServletException("Cannot parse SAML2 AuthnRequest.", ex);
		}

		// Process the request.

		this.processRequest(request, response, samlAuthnRequest, relayState);
	}

	protected void processRequest(HttpServletRequest request, HttpServletResponse response, SAML2AuthnRequest samlAuthnRequest, String relayState) throws ServletException, IOException {

		log.trace("processRequest()");

		String username;

		// Check the SAML2 AuthnRequest.

		if (samlAuthnRequest.getAssertionConsumerServiceURL() == null) {

			log.error("SAML2 AuthnRequest did not provide a AssertionConsumerServiceURL.");
			response.sendError(HttpServletResponse.SC_BAD_REQUEST, "SAML2 AuthnRequest did not provide a AssertionConsumerServiceURL.");
			return;
		}

		// If the request contains a KeyInfo element, we try to use it to verify the signature
		// (But this is not yet a reason to accept the request).

		if (samlAuthnRequest.hasKeyInfo()) {

			log.info("The SAML2 AuthnRequest's signature has a KeyInfo element. We try to use this to verify the signature.");

			try {

				samlAuthnRequest.verifyFromKeyInfo();
			} catch (Exception ex) {

				log.error("!!! SAML2 AuthnRequest has invalid signature !!!", ex);
				response.sendError(HttpServletResponse.SC_BAD_REQUEST, "Sorry, your signature could not be verified.");
				return;
			}

			log.info("SAML2 AuthnRequest XML Signature successfully verified with KeyInfo element.");
		}

		// We need to find out whether to accept the request or not.
		// First we check if it contains a signature of an RP for which we have the certificate.

		boolean acceptRequest = false;

		if (samlAuthnRequest.hasSignature()) {

			log.info("SAML2 AuthnRequest contains a signature. Checking if we have a matching RP certificate.");

			PublicKey[] rpPublicKeys = Init.getRpPublicKeys();
			X509Certificate[] rpCertificates = Init.getRpCertificates();

			for (int i=0; i<rpPublicKeys.length; i++) {

				PublicKey rpPublicKey = rpPublicKeys[i];
				X509Certificate rpCertificate = rpCertificates[i];

				try {

					samlAuthnRequest.verify(rpPublicKey);
				} catch (Exception ex) {

					continue;
				}

				acceptRequest = true;
				log.info("SAML2 AuthnRequest XML Signature successfully verified with certificate from " + rpCertificate.getSubjectDN().getName());
			}

			if (! acceptRequest) {

				log.info("No matching RP certificate found to verify the signature.");
			}
		}

		// If the signature did not make us happy enough, we check if the ACS URL is acceptable.

		if (! acceptRequest) {

			String acsUrl = samlAuthnRequest.getAssertionConsumerServiceURL();
			Set acceptAcsUrls = Init.getAcceptAcsUrls();

			log.info("The SAML2 AuthnRequest did not contain a signature that makes us happy. Therefore we check if we know the ACS URL: " + acsUrl); 

			if (acceptAcsUrls.contains("*")) {

				log.info("We are configured to allow all ACS URLs.");

				acceptRequest = true;
			}

			if (acceptAcsUrls.contains(samlAuthnRequest.getAssertionConsumerServiceURL())) {

				log.info("The SAML2 AuthnRequest contains an ACS URL which is known to us.");

				acceptRequest = true;
			}
		}

		// If we know neither the signature nor the ACS URL, we refuse to answer the request.

		if (! acceptRequest) {

			log.error("The request contained neither a signature nor an ACS URL that makes us happy. Rejecting the request.");
			response.sendError(HttpServletResponse.SC_FORBIDDEN, "Sorry, this IdP is not configured to answer your request.");
			return;
		}

		log.info("Accepting the SAML2 AuthnRequest.");

		// Check if we can extract a username from the request; if yes, send SAML Response with SAML Assertion.

		username = ExtractUtil.getUser(request, response);

		if (username != null) {

			log.info("Successfully extracted username " + username + " from the request. Sending SAML2 Response to SP.");

			SAMLUtil.postResponse(request, response, username, samlAuthnRequest.getAssertionConsumerServiceURL(), relayState);
			return;
		}

		// Check if the user is logged in already; if yes, send SAML Response with SAML Assertion.

		username = StateUtil.getUser(request, response);

		if (username != null) {

			log.info("User is logged in already. Sending SAML2 Response to SP.");

			SAMLUtil.postResponse(request, response, username, samlAuthnRequest.getAssertionConsumerServiceURL(), relayState);
			return;
		}

		// Since the user is not logged in, they need to provide credentials now.
		// Depending on the Higgins context type, different credentials may be required.
		// We put the SAML relaystate into a hidden field of the form, since we need it later.

		String higginsContextType = Init.getHigginsContextType();

		log.info("User is not logged in. Displaying credentials form for context type " + higginsContextType + ".");

		if (higginsContextType.equals("$context+ldap")) {

			request.setAttribute("authnrequest", samlAuthnRequest);
			request.setAttribute("relaystate", relayState);
			request.setAttribute("destination", samlAuthnRequest.getAssertionConsumerServiceURL());
			request.getRequestDispatcher("LDAPLogin.jsp").forward(request, response);
			return;
			/*		
		TODO: Right now we only support LDAP. Add other ways of authenticating here.

		} else if (higginsContextType.equals("$context+other")) {

			request.getRequestDispatcher("/OtherLogin.jsp").forward(request, response);
			return;
			 */
		} else {

			response.sendError(HttpServletResponse.SC_INTERNAL_SERVER_ERROR, "Unknown context type.");
			return;
		}
	}
}
