/*******************************************************************************
 * Copyright (c) 2007 Google
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     Markus Sabadello - Initial API and implementation
 *******************************************************************************/
package org.eclipse.higgins.saml2idp.server.util;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.security.PrivateKey;
import java.security.cert.Certificate;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.zip.DataFormatException;
import java.util.zip.Inflater;

import javax.servlet.ServletException;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import javax.xml.namespace.QName;

import org.apache.axiom.om.OMElement;
import org.apache.commons.codec.binary.Base64;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.eclipse.higgins.saml2idp.server.Init;
import org.eclipse.higgins.sts.api.IConstants;
import org.eclipse.higgins.sts.api.IElement;
import org.eclipse.higgins.sts.api.IRequestSecurityTokenResponse;
import org.eclipse.higgins.sts.api.ISTSRequest;
import org.eclipse.higgins.sts.api.ISTSResponse;
import org.eclipse.higgins.sts.api.ISecurityTokenService;
import org.eclipse.higgins.sts.client.TokenRequestFactory;
import org.eclipse.higgins.sts.common.Constants;
import org.eclipse.higgins.sts.common.STSResponse;
import org.eclipse.higgins.util.saml.SAML2Assertion;
import org.eclipse.higgins.util.saml.SAML2AuthnRequest;
import org.eclipse.higgins.util.saml.SAML2AuthnStatement;
import org.eclipse.higgins.util.saml.SAML2Response;
import org.eclipse.higgins.util.saml.SAMLConditions;
import org.eclipse.higgins.util.saml.SAMLConstants;
import org.eclipse.higgins.util.saml.SAMLSubject;
import org.eclipse.higgins.util.saml.SAMLSubjectConfirmationData;
import org.eclipse.higgins.util.saml.XMLElement;
import org.xml.sax.SAXException;

public class SAMLUtil {

	private static final Log log = LogFactory.getLog(SAMLUtil.class);

	private SAMLUtil() {

	}

	public static SAML2AuthnRequest parseAuthnRequestFromGet(String samlString) throws DataFormatException, IOException, SAXException {

		// Decode the Base64-encoded string and inflate it.

		byte[] samlBytes = Base64.decodeBase64(samlString.getBytes("UTF-8"));

		Inflater inflater = new Inflater(true);
		inflater.setInput(samlBytes);

		ByteArrayOutputStream stream = new ByteArrayOutputStream();
		byte[] buffer = new byte[1024];

		while (! inflater.finished()) {

			int count = inflater.inflate(buffer);
			if (count == 0) break;
			stream.write(buffer, 0, count);
		}

		stream.close();

		samlBytes = stream.toByteArray();

		// Parse the SAML2 AuthnRequest XML document from the string.

		return(new SAML2AuthnRequest(new ByteArrayInputStream(samlBytes)));
	}

	public static SAML2AuthnRequest parseAuthnRequestFromPost(String samlString) throws DataFormatException, IOException, SAXException {

		// Decode the Base64-encoded string

		byte[] samlBytes = Base64.decodeBase64(samlString.getBytes("UTF-8"));

		// Parse the SAML2 AuthnRequest XML document from the string.

		return(new SAML2AuthnRequest(new ByteArrayInputStream(samlBytes)));
	}

	private static ISTSRequest makeSTSRequest(SAML2AuthnRequest samlAuthnRequest, String username, String password) {

		final TokenRequestFactory factoryRequest = new TokenRequestFactory();

		ISTSRequest stsRequest = factoryRequest.createRequest(
				java.net.URI.create(samlAuthnRequest.getAssertionConsumerServiceURL()),
				java.net.URI.create("http://eclipse.org/higgins/saml2idp"),
				java.net.URI.create("urn:oasis:names:tc:SAML:2.0:protocol"),
				null,
				username,
				password);

		return(stsRequest);
	}

	private static ISTSResponse makeSTSResponse(ISTSRequest stsRequest) throws ServletException {

		ISTSResponse stsResponse = new STSResponse(); 

		ISecurityTokenService securityTokenService = Init.getHigginsStsSecurityTokenService();
		Map configurationSettings = Init.getHigginsConfigurationSettings();
		IConstants constants = new Constants();

		securityTokenService.invoke(
				configurationSettings,
				"SecurityTokenService",
				null,
				null,
				constants,
				stsRequest,
				stsResponse);

		if (stsResponse.getFault() != null) {

			ServletException ex = new ServletException("Cannot invoke the STS: " + stsResponse.getFault().getReason() + " - " + stsResponse.getFault().getDetail());
			log.error(ex);
			throw ex;
		}

		return(stsResponse);
	}

	public static void postResponse(HttpServletRequest request, HttpServletResponse response, SAML2AuthnRequest samlAuthnRequest, String name, String relayState) throws ServletException, IOException {

		String destination = samlAuthnRequest.getAssertionConsumerServiceURL();
		byte[] samlBytes;

		log.info("Sending response for " + name + " to " + destination + " with relaystate " + relayState);

		if (Init.getUseSts()) {

			// Use the STS for creating the SAML Response

			ISTSRequest stsRequest = SAMLUtil.makeSTSRequest(samlAuthnRequest, name, null);
			ISTSResponse stsResponse = SAMLUtil.makeSTSResponse(stsRequest);

			List requestSecurityTokenResponseCollection = stsResponse.getRequestSecurityTokenResponseCollection();
			String strRequestedSecurityToken;

			if (requestSecurityTokenResponseCollection.size() < 1) {

				ServletException ex = new ServletException("STS response contains no token.");
				log.error(ex);
				throw ex;
			}

			try {

				IRequestSecurityTokenResponse requestSecurityTokenResponse = (IRequestSecurityTokenResponse) requestSecurityTokenResponseCollection.get(0);
				IElement requestedSecurityToken = requestSecurityTokenResponse.getRequestedSecurityToken();

				OMElement requestedSecurityTokenElement = (org.apache.axiom.om.OMElement) requestedSecurityToken.getAs(OMElement.class);
				OMElement samlResponseElement = requestedSecurityTokenElement.getFirstChildWithName(new QName("urn:oasis:names:tc:SAML:2.0:protocol", "Response"));

				strRequestedSecurityToken = samlResponseElement.toString();
			} catch (Exception ex) {

				log.error(ex);
				throw new ServletException("Cannot read requested security token from STS response: " + ex.getMessage(), ex);
			}

			samlBytes = strRequestedSecurityToken.getBytes();
		} else {

			// Don't use the STS for creating the SAML Response

			Date now = new Date();

			SAML2Response samlResponse = new SAML2Response();
			SAML2Assertion samlAssertion = new SAML2Assertion(samlResponse.getDocument());
			SAMLSubject samlSubject = new SAMLSubject(samlResponse.getDocument(), SAMLConstants.NS_SAML_ASSERTION);
			SAML2AuthnStatement samlAuthnStatement = new SAML2AuthnStatement(samlResponse.getDocument());
			SAMLConditions samlConditions = new SAMLConditions(samlResponse.getDocument());
			SAMLSubjectConfirmationData samlSubjectConfirmationData = new SAMLSubjectConfirmationData(samlResponse.getDocument());

			samlSubjectConfirmationData.setInResponseTo(samlAuthnRequest.getID());
			samlSubjectConfirmationData.setNotOnOrAfter(new Date(now.getTime() + Init.getSAML2AssertionValidityMillis()));
			samlSubjectConfirmationData.setRecipient(samlAuthnRequest.getAssertionConsumerServiceURL());
			samlSubjectConfirmationData.deleteID();

			samlSubject.setNameIDFormat(SAMLConstants.NAMEIDFORMAT_UNSPECIFIED);
			samlSubject.setNameID(name);
			samlSubject.setSubjectConfirmationMethod(SAMLConstants.SUBJECTCONFIRMATIONMETHOD_BEARER);
			samlSubject.setSAMLSubjectConfirmationData(samlSubjectConfirmationData);
			samlSubject.deleteID();

			samlAuthnStatement.setAuthnInstant(now);
			samlAuthnStatement.setAuthnContextClassRef(SAMLConstants.AUTHNCONTEXTCLASSREF_PASSWORD);
			samlAuthnStatement.setSessionIndex(samlAssertion.getID());
			samlAuthnStatement.deleteID();

			samlConditions.setNotOnOrAfter(new Date(now.getTime() + Init.getSAML2AssertionValidityMillis()));
			samlConditions.setNotBefore(now);
			samlConditions.setAudienceRestrictionAudience(samlAuthnRequest.getAssertionConsumerServiceURL());
			samlConditions.deleteID();

			XMLElement.addNamespaceAttributes(samlAssertion.getElement());
			samlAssertion.setIssueInstant(now);
			samlAssertion.setIssuer(Init.getSAML2Issuer());
			samlAssertion.setSAMLSubject(samlSubject);
			samlAssertion.setSAMLConditions(samlConditions);
			samlAssertion.setSAMLAuthnStatement(samlAuthnStatement);

			samlResponse.setVersion("2.0");
			samlResponse.setIssueInstant(now);
			samlResponse.setIssuer(Init.getSAML2Issuer());
			samlResponse.setStatusCodeValue(SAMLConstants.STATUSCCODE_SUCCESS);
			samlResponse.setInResponseTo(samlAuthnRequest.getID());
			samlResponse.setSAMLAssertion(samlAssertion);

			// Sign it.

			PrivateKey idpPrivateKey = Init.getIdpPrivateKey();
			Certificate idpCertificate = Init.getIdpCertificate();

			if (idpPrivateKey != null && idpCertificate != null) {

				try {

					samlAssertion.sign(idpPrivateKey, idpCertificate, samlSubject.getElement());
				} catch (Exception ex) {

					log.error(ex);
					throw new ServletException("Cannot sign SAML2 message.", ex);
				}
			}

			samlBytes = samlResponse.dump(true).getBytes("UTF-8");
		}

		// Base64 encode the SAML Response

		String samlString = new String(Base64.encodeBase64(samlBytes));

		// XML encode the RelayState

		String encodedRelayState = relayState.replaceAll("&", "&amp;");

		// Initialize hidden form fields.

		Map attributes = new HashMap ();

		attributes.put("SAMLResponse", samlString);
		if (relayState != null) attributes.put("RelayState", encodedRelayState);

		// Display the auto-redirect form.

		log.info(destination);

		request.setAttribute("action", destination);
		request.setAttribute("attributes", attributes);
		request.getRequestDispatcher("FormRedirect.jsp").forward(request, response);
	}
}
