/*******************************************************************************
 * Copyright (c) 2008 Copyright (c) 2007 Parity Communications, Inc.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     Markus Sabadello - Initial API and implementation
 *******************************************************************************/
package org.eclipse.higgins.saml2idp.server.card;

import java.io.IOException;
import java.util.Date;

import javax.servlet.ServletException;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.eclipse.higgins.rp.tokens.RPSecurityToken;
import org.eclipse.higgins.saml2idp.server.Init;
import org.eclipse.higgins.saml2idp.server.util.CardUtil;
import org.eclipse.higgins.saml2idp.server.util.SAMLUtil;
import org.eclipse.higgins.saml2idp.server.util.StateUtil;
import org.eclipse.higgins.util.saml.SAML2AuthnRequest;

/**
 * Servlet implementation class for Servlet: CardLogin
 */
public class CardLogin extends javax.servlet.http.HttpServlet implements javax.servlet.Servlet {

	private static final long serialVersionUID = 1L;

	private static final int MAX_LOGIN_FAILURES = 3;
	private static final long LOGIN_LOCK_MILLIS = 900000;	// 900000 = 15 minutes

	private static final Log log = LogFactory.getLog(CardLogin.class);

	public CardLogin() {

		super();
	}   	

	protected void doPost(HttpServletRequest request, HttpServletResponse response) throws ServletException, IOException {

		log.trace("doPost()");

		// Read and check parameters from the login form.

		String xmlToken = request.getParameter("xmlToken");

		RPSecurityToken token = CardUtil.decodeToken(xmlToken, request.getSession());

		// Security check: Maximal login attempts exceeded?

		if (StateUtil.getFailures(request, response) >= MAX_LOGIN_FAILURES) {

			Date lastFailure = StateUtil.getLastFailureDate(request, response);
			long diff = new Date().getTime() - lastFailure.getTime();

			if (diff < LOGIN_LOCK_MILLIS) {

				log.error("Too many failed login attempts.");
				reportError(request, response, "Too many failed login attempts.");
				return;
			} else {

				StateUtil.resetFailures(request, response);
			}
		}

		// Check if we can handle the card.
		// If yes, send a SAMLResponse with SAMLAssertion using a HTTP POST redirect form.
		// If no, display and log the error, and prompt the user again.

		String username;

		try {

			ICardHandler cardHandler = Init.getCardHandler();
			username = cardHandler.handleCard(token);
			if (username == null || username.trim().equals("")) throw new NullPointerException();
		} catch (Exception ex) {

			log.error("Invalid card.", ex);
			reportError(request, response, "Invalid card or no card received.");
			return;
		}

		// Since we could instantiate and open the context, the user successfully logged in.

		log.info("User " + username + " logged in. Sending SAML2 Response to SP.");

		StateUtil.loginUser(request, response, username);

		SAML2AuthnRequest samlAuthnRequest = StateUtil.getSAMLAuthnRequest(request, response);
		String relayState = StateUtil.getRelayState(request, response);
		
		// Create and send a SAML2 Response over the HTTP POST binding

		SAMLUtil.postResponse(request, response, samlAuthnRequest, username, relayState);
		return;
	}

	/**
	 * Re-display the login form, with an error message.
	 * @param request The current HttpServletRequest object.
	 * @param response The current HttpServletResponse object.
	 * @param error The error message to display.
	 */
	private static void reportError(HttpServletRequest request, HttpServletResponse response, String error) throws ServletException, IOException {

		request.setAttribute("error", error);
		request.getRequestDispatcher("CardLogin.jsp").forward(request, response);
	}
}
