/*******************************************************************************
 * Copyright (c) 2007 Google
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     Markus Sabadello - Initial API and implementation
 *******************************************************************************/
package org.eclipse.higgins.saml2idp.server.context.ldap;

import java.io.IOException;
import java.util.Date;

import javax.naming.AuthenticationException;
import javax.servlet.ServletException;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.eclipse.higgins.idas.api.IContext;
import org.eclipse.higgins.idas.api.IContextFactory;
import org.eclipse.higgins.idas.api.IContextId;
import org.eclipse.higgins.idas.api.IEntity;
import org.eclipse.higgins.idas.api.IdASException;
import org.eclipse.higgins.idas.common.AuthNNamePasswordMaterials;
import org.eclipse.higgins.saml2idp.server.Init;
import org.eclipse.higgins.saml2idp.server.util.SAMLUtil;
import org.eclipse.higgins.saml2idp.server.util.StateUtil;
import org.eclipse.higgins.util.saml.SAML2AuthnRequest;

/**
 * Servlet implementation class for Servlet: LDAPLogin
 */
public class LDAPLogin extends javax.servlet.http.HttpServlet implements javax.servlet.Servlet {

	private static final long serialVersionUID = 1L;

	private static final int MAX_LOGIN_FAILURES = 3;
	private static final long LOGIN_LOCK_MILLIS = 900000;	// 900000 = 15 minutes

	private static final Log log = LogFactory.getLog(LDAPLogin.class);

	public LDAPLogin() {

		super();
	}   	

	protected void doPost(HttpServletRequest request, HttpServletResponse response) throws ServletException, IOException {

		log.trace("doPost()");

		// Read and check parameters from the login form.

		String username = request.getParameter("username");
		String password = request.getParameter("password");

		if (username == null || password == null || username.trim().equals("") || password.trim().equals("")) {

			log.warn("Missing parameter: username=" + username + ", password=" + password);
			reportError(request, response, "Both Username and Password are required fields.");
			return;
		}

		// Security check: Maximal login attempts exceeded?

		if (StateUtil.getFailures(request, response) >= MAX_LOGIN_FAILURES) {

			Date lastFailure = StateUtil.getLastFailureDate(request, response);
			long diff = new Date().getTime() - lastFailure.getTime();

			if (diff < LOGIN_LOCK_MILLIS) {

				log.error("Too many failed login attempts.");
				reportError(request, response, "Too many failed login attempts.");
				return;
			} else {

				StateUtil.resetFailures(request, response);
			}
		}

		// Check if we can instantiate and open our context.
		// If yes, send a SAMLResponse with SAMLAssertion using a HTTP POST redirect form.
		// If no, display and log the error, and prompt the user again.
		// TODO: Add security mechanisms to prevent password guessing, etc.

		IContextId higginsContextId = Init.getHigginsContextId();
		IContextFactory higginsContextFactory = Init.getHigginsContextFactory();
		IContext higginsContext = null;

		try {

			higginsContext = higginsContextFactory.createContext(higginsContextId);
			Object authnMaterials = new AuthNNamePasswordMaterials(higginsContext, username, password);
			String entityId = higginsContext.open(authnMaterials);
			if (entityId == null) throw new NullPointerException("entityId is null");
			IEntity entity = higginsContext.getEntity(entityId);
			if (entity == null) throw new NullPointerException("entity is null");
		} catch (Exception ex) {

			if (ex instanceof IdASException && ex.getCause() instanceof AuthenticationException) {

				StateUtil.incFailures(request, response);
				int fails = StateUtil.getFailures(request, response);

				log.warn("Cannot login user: " + ex.getMessage() + ", Username=" + username + " (fail #" + fails + ").");
				reportError(request, response, "Cannot login user: " + username + " (fail #" + fails + ").");
				return;
			} else {

				log.error("Exception while opening context: Username=" + username + ", Context ID=" + higginsContextId + ".", ex);
				reportError(request, response, "Internal error. Please check the server logs for details.");
				return;
			}
		} finally {

			if (higginsContext != null) {

				try {

					higginsContext.close();
				} catch (IdASException ex) { }
			}
		}

		// Since we could instantiate and open the context, the user successfully logged in.

		log.info("User " + username + " logged in. Sending SAML2 Response to SP.");

		StateUtil.loginUser(request, response, username);

		SAML2AuthnRequest samlAuthnRequest = StateUtil.getSAMLAuthnRequest(request, response);
		String relayState = StateUtil.getRelayState(request, response);

		// Create and send a SAML2 Response over the HTTP POST binding

		SAMLUtil.postResponse(request, response, samlAuthnRequest, username, relayState);
		return;
	}

	/**
	 * Re-display the login form, with an error message.
	 * @param request The current HttpServletRequest object.
	 * @param response The current HttpServletResponse object.
	 * @param error The error message to display.
	 */
	private static void reportError(HttpServletRequest request, HttpServletResponse response, String error) throws ServletException, IOException {

		request.setAttribute("error", error);
		request.getRequestDispatcher("LDAPLogin.jsp").forward(request, response);
	}
}
