/*******************************************************************************
 * Copyright (c) 2007 Google
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     Markus Sabadello - Initial API and implementation
 *******************************************************************************/
package org.eclipse.higgins.saml2idp.test;

import java.io.File;
import java.io.FileInputStream;
import java.security.KeyFactory;
import java.security.PrivateKey;
import java.security.PublicKey;
import java.security.cert.CertificateFactory;
import java.security.cert.X509Certificate;
import java.security.spec.KeySpec;
import java.security.spec.PKCS8EncodedKeySpec;
import java.util.Properties;

import javax.servlet.ServletContext;
import javax.servlet.ServletException;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

/**
 * Servlet implementation class for Servlet: Init
 *
 */
public class Init extends javax.servlet.http.HttpServlet implements javax.servlet.Servlet {

	private static final long serialVersionUID = 1L;

	private static final Log log = LogFactory.getLog(Init.class);

	// Application-wide properties.

	private static Properties properties;

	// The private key we use to sign SAML2 messages.

	private static PrivateKey rpPrivateKey;

	// The public key we use for the KeyInfo element in XML Signatures.

	private static X509Certificate rpCertificate;
	private static PublicKey rpPublicKey;

	// The certificate and public key we use to verify SAML2 messages.

	private static X509Certificate idpCertificate;
	private static PublicKey idpPublicKey;

	public Init() {

		super();
	}   	 	  	  	  

	public void init() throws ServletException {

		log.trace("init()");

		super.init();

		ServletContext servletContext = this.getServletContext();

		// Load application-wide properties.

		try {

			File propertiesFile = new File(servletContext.getRealPath("./conf/application.properties"));
			properties = new Properties();
			properties.load(new FileInputStream(propertiesFile));
		} catch (Exception ex) {

			throw new ServletException("Cannot load application properties.", ex);
		}

		log.info("Loaded application properties.");

		// Load the private key we need to sign SAML2 messages.

		try {

			File keyFile = new File(servletContext.getRealPath("./conf/privkey.der"));
			FileInputStream stream = new FileInputStream(keyFile);
			byte[] keyBytes = new byte[stream.available()];
			stream.read(keyBytes);
			stream.close();

			KeySpec keySpec = new PKCS8EncodedKeySpec(keyBytes);
			KeyFactory keyFactory = KeyFactory.getInstance("DSA");
			rpPrivateKey = keyFactory.generatePrivate(keySpec);

			log.info("Loaded RP private key (algorithm: " + rpPrivateKey.getAlgorithm() + ", format: " + rpPrivateKey.getFormat());
		} catch (Exception ex) {

			rpPrivateKey = null;
			log.warn("No RP private key loaded. SAML2 messages will not be signed.");
		}

		// Load the certificate and public key we put into the KeyInfo element of XML Signatures.

		try {

			File keyFile = new File(servletContext.getRealPath("./conf/cacert.pem"));
			FileInputStream stream = new FileInputStream(keyFile);

			CertificateFactory certificateFactory = CertificateFactory.getInstance("X.509");
			rpCertificate = (X509Certificate) certificateFactory.generateCertificate(stream);
			rpPublicKey = rpCertificate.getPublicKey();
			stream.close();

			log.info("Loaded RP certificate from issuer: " + rpCertificate.getIssuerX500Principal().toString());
			log.info("Loaded RP public key (algorithm: " + rpPublicKey.getAlgorithm() + ", format: " + rpPublicKey.getFormat());
		} catch (Exception ex) {

			rpCertificate = null;
			rpPublicKey = null;
			log.warn("No RP certificate and public key loaded. SAML2 messages will not be signed.");
		}

		// Load the certificate and public key we need to verify SAML2 messages.

		try {

			File keyFile = new File(servletContext.getRealPath("./conf/idp/cacert.pem"));
			FileInputStream stream = new FileInputStream(keyFile);

			CertificateFactory certificateFactory = CertificateFactory.getInstance("X.509");
			idpCertificate = (X509Certificate) certificateFactory.generateCertificate(stream);
			idpPublicKey = idpCertificate.getPublicKey();
			stream.close();

			log.info("Loaded IdP certificate from issuer: " + idpCertificate.getIssuerX500Principal().toString());
			log.info("Loaded IdP public key (algorithm: " + idpPublicKey.getAlgorithm() + ", format: " + idpPublicKey.getFormat() + ")");
		} catch (Exception ex) {

			idpCertificate = null;
			idpPublicKey = null;
			log.warn("No IdP certificate and public key loaded. SAML2 messages will not be verified.");
		}

		// Done.

		log.info("Up and running.");
	}

	/**
	 * Returns the application properties.
	 */
	public static Properties getProperties() {

		log.trace("getProperties()");

		return(properties);
	}

	/**
	 * Returns whether minimal request mode is desired.
	 */
	public static boolean getMinimal() {

		log.trace("getMinimal()");
		
		return Boolean.valueOf(properties.getProperty("minimal", "false")).booleanValue();
	}

	/**
	 * Returns the SAML2 IdP endpoint to be used.
	 */
	public static String getSAML2IdPEndpoint() {

		log.trace("getSAML2IdPEndpoint()");

		return(properties.getProperty("saml2-idp-endpoint"));
	}

	/**
	 * Returns the SAML2 SP endpoint to be used.
	 */
	public static String getSAML2SPEndpoint() {

		log.trace("getSAML2SPEndpoint()");

		return(properties.getProperty("saml2-sp-endpoint"));
	}

	/**
	 * Returns the SAML2 provider name.
	 */
	public static String getSAML2ProviderName() {

		log.trace("getSAML2ProviderName()");

		return(properties.getProperty("saml2-providername"));
	}

	/**
	 * Returns the SAML2 issuer.
	 */
	public static String getSAML2Issuer() {

		log.trace("getSAML2Issuer()");

		return(properties.getProperty("saml2-issuer"));
	}

	/**
	 * Returns the private key used to sign SAML2 messages.
	 */
	public static PrivateKey getRpPrivateKey() {

		return(rpPrivateKey);
	}

	/**
	 * Returns the certificate used for the KeyInfo element in XML Signatures.
	 */
	public static X509Certificate getRpCertificate() {

		return(rpCertificate);
	}

	/**
	 * Returns the public key used for the KeyInfo element in XML Signatures.
	 */
	public static PublicKey getRpPublicKey() {

		return(rpPublicKey);
	}

	/**
	 * Returns the certificate used to validate SAML2 messages.
	 */
	public static X509Certificate getIdpCertificate() {

		return(idpCertificate);
	}

	/**
	 * Returns the public key used to validate SAML2 messages.
	 */
	public static PublicKey getIdpPublicKey() {

		return(idpPublicKey);
	}
}
