/*******************************************************************************
 * Copyright (c) 2007 Google
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     Markus Sabadello - Initial API and implementation
 *******************************************************************************/
package org.eclipse.higgins.saml2idp.test.util;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.StringReader;
import java.net.URLEncoder;
import java.security.PrivateKey;
import java.security.PublicKey;
import java.util.zip.Deflater;
import java.util.zip.DeflaterOutputStream;

import javax.servlet.ServletException;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import org.apache.commons.codec.binary.Base64;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.eclipse.higgins.saml2idp.test.Init;
import org.eclipse.higgins.util.saml.SAML2AuthnRequest;
import org.eclipse.higgins.util.saml.SAML2Response;
import org.eclipse.higgins.util.saml.SAMLConstants;
import org.xml.sax.SAXException;

public class SAMLUtil {

	private static final Log log = LogFactory.getLog(SAMLUtil.class);

	private SAMLUtil() {

	}

	public static SAML2Response parseResponse(String samlString) throws SAXException, IOException {

		// Decode the Base64-encoded string.

		byte[] samlBytes = Base64.decodeBase64(samlString.getBytes("UTF-8"));

		// Parse the SAML2 Response XML document from the string.

		return(new SAML2Response(new ByteArrayInputStream(samlBytes)));
	}

	public static void redirectAuthnRequest(HttpServletRequest request, HttpServletResponse response, String relayState) throws ServletException, IOException {

		// Create a SAML2 AuthnRequest message (normal or minimal can be configured).

		SAML2AuthnRequest samlAuthnRequest = new SAML2AuthnRequest();

		// We need the ACS URL at least.
		
		samlAuthnRequest.setAssertionConsumerServiceURL(Init.getSAML2SPEndpoint());
		
		if (! Init.getMinimal()) {

			// If minimal mode is disabled, we add a few more things. 
			
			samlAuthnRequest.setDestination(Init.getSAML2IdPEndpoint());
			samlAuthnRequest.setProviderName(Init.getSAML2ProviderName());
			samlAuthnRequest.setIssuer(Init.getSAML2Issuer());
			samlAuthnRequest.setProtocolBinding(SAMLConstants.PROTOCOLBINDING_HTTP_REDIRECT);
			samlAuthnRequest.setNameIDPolicyAllowCreate(true);
			samlAuthnRequest.setNameIDPolicyFormat(SAMLConstants.NAMEIDPOLICY_UNSPECIFIED);

			// Sign it.
	
			PrivateKey rpPrivateKey = Init.getRpPrivateKey();
			PublicKey rpPublicKey = Init.getRpPublicKey();
	
			if (rpPrivateKey != null && rpPublicKey != null) {
	
				try {
	
					samlAuthnRequest = new SAML2AuthnRequest(new StringReader(samlAuthnRequest.dump()));
					samlAuthnRequest.sign(rpPrivateKey, rpPublicKey);
				} catch (Exception ex) {
	
					log.error(ex);
					throw new ServletException("Cannot sign SAML2 message.", ex);
				}
			}
		}

		// We need to deflate the SAML2 message and convert it to Base64.

		String samlString = samlAuthnRequest.dump();
		byte[] samlBytes = samlString.getBytes("UTF-8");
		ByteArrayOutputStream outputStream = new ByteArrayOutputStream();

		Deflater deflater = new Deflater(5, true);
		DeflaterOutputStream deflaterOutputStream = new DeflaterOutputStream(outputStream, deflater);
		deflaterOutputStream.write(samlBytes, 0, samlBytes.length);
		deflaterOutputStream.close();

		byte[] encodedBytes = Base64.encodeBase64(outputStream.toByteArray());
		String encodedString = new String(encodedBytes);

		// Construct URL with SAML2 AuthnRequest.

		StringBuffer url = new StringBuffer();
		url.append(Init.getSAML2IdPEndpoint()).append("?");
		url.append("SAMLRequest=").append(URLEncoder.encode(encodedString, "UTF-8"));
		url.append("&");
		url.append("RelayState=").append(URLEncoder.encode(relayState, "UTF-8"));

		// Redirect to SAML2 IdP.

		response.sendRedirect(url.toString());
	}
}
