/*******************************************************************************
 * Copyright (c) 2007 Google
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     Markus Sabadello - Initial API and implementation
 *******************************************************************************/
package org.eclipse.higgins.saml2idp.test;

import java.io.IOException;
import java.security.PublicKey;
import java.security.cert.X509Certificate;

import javax.servlet.ServletException;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.eclipse.higgins.saml2idp.test.util.SAMLUtil;
import org.eclipse.higgins.util.saml.SAML2Assertion;
import org.eclipse.higgins.util.saml.SAML2Response;
import org.eclipse.higgins.util.saml.SAMLSubject;
import org.eclipse.higgins.util.saml.SAMLConstants;
import org.eclipse.higgins.util.saml.XMLElement.VerificationException;

/**
 * Servlet implementation class for Servlet: SAMLEndpoint
 *
 */
public class SAMLEndpoint extends javax.servlet.http.HttpServlet implements javax.servlet.Servlet {

	private static final long serialVersionUID = 1L;

	private static final String PARAMETER_SAMLRESPONSE = "SAMLResponse";
	private static final String PARAMETER_RELAYSTATE = "RelayState";

	private static final Log log = LogFactory.getLog(SAMLEndpoint.class);

	public SAMLEndpoint() {

		super();
	}   	

	protected void doGet(HttpServletRequest request, HttpServletResponse response) throws ServletException, IOException {

		log.trace("doGet()");

		doPost(request, response);
	}  	

	protected void doPost(HttpServletRequest request, HttpServletResponse response) throws ServletException, IOException {

		log.trace("doPost()");

		// Read and check parameters.

		String samlString = request.getParameter(PARAMETER_SAMLRESPONSE);
		String relayState = request.getParameter(PARAMETER_RELAYSTATE);

		if (samlString == null) {

			response.sendError(HttpServletResponse.SC_BAD_REQUEST, "Required parameter " + PARAMETER_SAMLRESPONSE + " not found.");
			return;
		}

		// Try to parse the SAML2 Response.

		SAML2Response samlResponse;

		try {

			samlResponse = SAMLUtil.parseResponse(samlString);
		} catch (Exception ex) {

			log.error(ex);
			throw new ServletException("Cannot parse SAML2 Response.", ex);
		}

		// Verify the XML Signature.

		PublicKey idpPublicKey = Init.getIdpPublicKey();
		X509Certificate idpCertificate = Init.getIdpCertificate();
		
		try {

			samlResponse.verify(idpPublicKey);
		} catch (VerificationException ex) {

			log.error("!!! SAML2 Response has invalid signature !!!", ex);
			throw new ServletException("!!! SAML2 Response has invalid signature !!!", ex);
		} catch (Exception ex) {

			log.error(ex);
			throw new ServletException("Cannot verify XML Signature.", ex);
		}

		log.info("SAML2 Response XML Signature verified with certificate from " + idpCertificate.getSubjectDN().getName());

		// Check the SAML2 Response.

		String statusCodeValue = samlResponse.getStatusCodeValue();
		log.info("SAML2 Response StatusCode: " + statusCodeValue);

		SAML2Assertion samlAssertion = samlResponse.getSAMLAssertion();
		SAMLSubject samlSubject = samlAssertion.getSubject(); 
		String nameId = samlSubject.getNameID();

		log.info("SAML2 Response NameID: " + nameId);

		if (! statusCodeValue.equals(SAMLConstants.STATUSCCODE_SUCCESS)) {

			log.warn("User NOT successfully logged in.");
			response.sendRedirect("/");
			return;
		}

		// Log in.

		log.info("User successfully logged in.");

		request.setAttribute("nameid", nameId);
		request.setAttribute("relaystate", relayState);
		request.getRequestDispatcher("success.jsp").forward(request, response);
	}   	  	    
}
