/*
 * Copyright 2001-2004 The Apache Software Foundation.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 *      http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 * 
 * Changes:
 * Michael McIntosh added changes to deal with Axis header incompatibilites
 */

package org.apache.axis.encoding;

import java.io.IOException;
import java.io.Writer;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.Date;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.Stack;

import javax.xml.namespace.QName;
import javax.xml.rpc.JAXRPCException;
import javax.xml.rpc.holders.QNameHolder;

import org.apache.axis.AxisEngine;
import org.apache.axis.AxisProperties;
import org.apache.axis.Constants;
import org.apache.axis.Message;
import org.apache.axis.MessageContext;
import org.apache.axis.attachments.Attachments;
import org.apache.axis.client.Call;
import org.apache.axis.components.encoding.XMLEncoder;
import org.apache.axis.components.encoding.XMLEncoderFactory;
import org.apache.axis.components.logger.LogFactory;
import org.apache.axis.constants.Use;
import org.apache.axis.description.OperationDesc;
import org.apache.axis.description.TypeDesc;
import org.apache.axis.encoding.ser.ArraySerializer;
import org.apache.axis.encoding.ser.BaseSerializerFactory;
import org.apache.axis.encoding.ser.SimpleListSerializerFactory;
import org.apache.axis.handlers.soap.SOAPService;
import org.apache.axis.schema.SchemaVersion;
import org.apache.axis.soap.SOAPConstants;
import org.apache.axis.types.HexBinary;
import org.apache.axis.utils.IDKey;
import org.apache.axis.utils.JavaUtils;
import org.apache.axis.utils.Mapping;
import org.apache.axis.utils.Messages;
import org.apache.axis.utils.NSStack;
import org.apache.axis.utils.XMLUtils;
import org.apache.axis.utils.cache.MethodCache;
import org.apache.axis.wsdl.symbolTable.SchemaUtils;
import org.apache.axis.wsdl.symbolTable.SymbolTable;
import org.apache.axis.wsdl.symbolTable.Utils;
import org.apache.commons.logging.Log;
import org.w3c.dom.Attr;
import org.w3c.dom.CDATASection;
import org.w3c.dom.CharacterData;
import org.w3c.dom.Comment;
import org.w3c.dom.Element;
import org.w3c.dom.NamedNodeMap;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;
import org.w3c.dom.Text;
import org.xml.sax.Attributes;
import org.xml.sax.helpers.AttributesImpl;

/** Manage a serialization, including keeping track of namespace mappings
 * and element stacks.
 *
 * @author Glen Daniels (gdaniels@apache.org)
 * @author Rich Scheuerle <scheu@us.ibm.com>
 */
public class SerializationContext implements javax.xml.rpc.encoding.SerializationContext
{
    protected static Log log =
            LogFactory.getLog(SerializationContext.class.getName());

    // invariant member variable to track low-level logging requirements
    // we cache this once per instance lifecycle to avoid repeated lookups
    // in heavily used code.
    private final boolean debugEnabled = SerializationContext.log.isDebugEnabled();

    private NSStack nsStack = null;
    private boolean writingStartTag = false;
    private boolean onlyXML = true;
    private int indent=0;
    private final Stack elementStack = new Stack();
    private Writer writer;
    private int lastPrefixIndex = 1;
    private MessageContext msgContext;
    private QName currentXMLType;
    /** The item QName if we're serializing a literal array... */
    private QName itemQName;
    /** The item type if we're serializing a literal array... */
    private QName itemType;

    /** The SOAP context we're using */
    private SOAPConstants soapConstants = SOAPConstants.SOAP11_CONSTANTS;

    private static final QName multirefQName = new QName("","multiRef");
    private static final Class[] SERIALIZER_CLASSES =
            new Class[] {String.class, Class.class, QName.class};
    private static final String SERIALIZER_METHOD = "getSerializer";

    /**
     * Should I write out objects as multi-refs?
     *
     * !!! For now, this is an all-or-nothing flag.  Either ALL objects will
     * be written in-place as hrefs with the full serialization at the end
     * of the body, or we'll write everything inline (potentially repeating
     * serializations of identical objects).
     */
    private boolean doMultiRefs = false;
    
    /**
     * Should I disable the pretty xml completely.
     */ 
    private boolean disablePrettyXML = false;


    /**
     * Should I disable the namespace prefix optimization.
     */ 
    private boolean enableNamespacePrefixOptimization = false;

    /**
     * current setting for pretty
     */ 
    private boolean pretty = false;
    
    /**
     * Should I send an XML declaration?
     */
    private boolean sendXMLDecl = true;

    /**
     * Should I send xsi:type attributes?  By default, yes.
     */
    private boolean sendXSIType = true;

    /**
     * Send an element with an xsi:nil="true" attribute for null
     * variables (if Boolean.TRUE), or nothing (if Boolean.FALSE).
     */
    private Boolean sendNull = Boolean.TRUE;

    /**
     * A place to hold objects we cache for multi-ref serialization, and
     * remember the IDs we assigned them.
     */
    private HashMap multiRefValues = null;
    private int multiRefIndex = -1;
    private boolean noNamespaceMappings = true;
    private QName writeXMLType;
    private XMLEncoder encoder = null;
    
    /** The flag whether the XML decl should be written */
    protected boolean startOfDocument = true;
 
    /** The encoding to serialize */
    private String encoding = XMLEncoderFactory.DEFAULT_ENCODING;

    class MultiRefItem {
        String id;
        QName xmlType;
        Boolean sendType;
        Object value;
        MultiRefItem(final String id,
                     final QName xmlType,
                     final Boolean sendType, final Object value) {
            this.id = id;
            this.xmlType = xmlType;
            this.sendType = sendType;
            this.value = value;
        }

    }
    /**
     * These three variables are necessary to process
     * multi-level object graphs for multi-ref serialization.
     * While writing out nested multi-ref objects (via outputMultiRef), we
     * will fill the secondLevelObjects vector
     * with any new objects encountered.
     * The outputMultiRefsFlag indicates whether we are currently within the
     * outputMultiRef() method (so that serialization() knows to update the
     * secondLevelObjects vector).
     * The forceSer variable is the trigger to force actual serialization of the indicated object.
     */
    private HashSet secondLevelObjects = null;
    private Object forceSer = null;
    private boolean outputMultiRefsFlag = false;

    /**
     * Which schema version are we using?
     */
    SchemaVersion schemaVersion = SchemaVersion.SCHEMA_2001;

    /**
     * A list of particular namespace -> prefix mappings we should prefer.
     * See getPrefixForURI() below.
     */
    HashMap preferredPrefixes = new HashMap();

    /**
     * Construct SerializationContext with associated writer
     * @param writer java.io.Writer
     */
    public SerializationContext(final Writer writer)
    {
        this.writer = writer;
        this.initialize();
    }

    private void initialize() {
        // These are the preferred prefixes we'll use instead of the "ns1"
        // style defaults.  MAKE SURE soapConstants IS SET CORRECTLY FIRST!
        this.preferredPrefixes.put(this.soapConstants.getEncodingURI(),
                              Constants.NS_PREFIX_SOAP_ENC);
        this.preferredPrefixes.put(Constants.NS_URI_XML,
                              Constants.NS_PREFIX_XML);
        this.preferredPrefixes.put(this.schemaVersion.getXsdURI(),
                              Constants.NS_PREFIX_SCHEMA_XSD);
        this.preferredPrefixes.put(this.schemaVersion.getXsiURI(),
                              Constants.NS_PREFIX_SCHEMA_XSI);
        this.preferredPrefixes.put(this.soapConstants.getEnvelopeURI(),
                              Constants.NS_PREFIX_SOAP_ENV);
        this.nsStack = new NSStack(this.enableNamespacePrefixOptimization);
    }


    /**
     * Construct SerializationContext with associated writer and MessageContext
     * @param writer java.io.Writer
     * @param msgContext is the MessageContext
     */
    public SerializationContext(final Writer writer, final MessageContext msgContext)
    {
        this.writer = writer;
        this.msgContext = msgContext;

        if ( msgContext != null ) {
            this.soapConstants = msgContext.getSOAPConstants();

            // Use whatever schema is associated with this MC
            this.schemaVersion = msgContext.getSchemaVersion();

            final Boolean shouldSendDecl = (Boolean)msgContext.getProperty(
                                                  AxisEngine.PROP_XML_DECL);
            if (shouldSendDecl != null) {
				this.sendXMLDecl = shouldSendDecl.booleanValue();
			}

            final Boolean shouldSendMultiRefs =
                  (Boolean)msgContext.getProperty(AxisEngine.PROP_DOMULTIREFS);
            if (shouldSendMultiRefs != null) {
				this.doMultiRefs = shouldSendMultiRefs.booleanValue();
			}
            
            final Boolean shouldDisablePrettyXML =
                  (Boolean)msgContext.getProperty(AxisEngine.PROP_DISABLE_PRETTY_XML);
            if (shouldDisablePrettyXML != null) {
				this.disablePrettyXML = shouldDisablePrettyXML.booleanValue();
			}
            
            final Boolean shouldDisableNamespacePrefixOptimization =
                  (Boolean)msgContext.getProperty(AxisEngine.PROP_ENABLE_NAMESPACE_PREFIX_OPTIMIZATION);
            if (shouldDisableNamespacePrefixOptimization != null) {
                this.enableNamespacePrefixOptimization = shouldDisableNamespacePrefixOptimization.booleanValue();
            } else {
                this.enableNamespacePrefixOptimization = JavaUtils.isTrue(AxisProperties.getProperty(AxisEngine.PROP_ENABLE_NAMESPACE_PREFIX_OPTIMIZATION,
                                "true"));
            }
            boolean sendTypesDefault = this.sendXSIType;

            // A Literal use operation overrides the above settings. Don't
            // send xsi:type, and don't do multiref in that case.
            final OperationDesc operation = msgContext.getOperation();
            if (operation != null) {
                if (operation.getUse() != Use.ENCODED) {
                    this.doMultiRefs = false;
                    sendTypesDefault = false;
                }
            } else {
                // A Literal use service also overrides the above settings. 
                final SOAPService service = msgContext.getService();
                if (service != null) {
                    if (service.getUse() != Use.ENCODED) {
                        this.doMultiRefs = false;
                        sendTypesDefault = false;
                    }
                }
            }

            // The SEND_TYPE_ATTR and PROP_SEND_XSI options indicate
            // whether the elements should have xsi:type attributes.
            // Only turn this off is the user tells us to
            if ( !msgContext.isPropertyTrue(Call.SEND_TYPE_ATTR, sendTypesDefault )) {
				this.sendXSIType = false ;
			}

// Don't need this since the above isPropertyTrue should walk up to the engine's
// properties...?
//            
//            Boolean opt = (Boolean)optionSource.getOption(AxisEngine.PROP_SEND_XSI);
//            if (opt != null) {
//                sendXSIType = opt.booleanValue();
//            }
        } else {
            this.enableNamespacePrefixOptimization = JavaUtils.isTrue(AxisProperties.getProperty(AxisEngine.PROP_ENABLE_NAMESPACE_PREFIX_OPTIMIZATION,
                            "true"));
        }

        // Set up preferred prefixes based on current schema, soap ver, etc.
        this.initialize();
    }

    /**
     * Get whether the serialization should be pretty printed.
     * @return true/false
     */
    public boolean getPretty() {
        return this.pretty;
    }

    /**
     * Indicate whether the serialization should be pretty printed.
     * @param pretty true/false
     */
    public void setPretty(final boolean pretty) {
        if(!this.disablePrettyXML) {
            this.pretty = pretty;
        }
    }

    /**
     * Are we doing multirefs?
     * @return true or false
     */
    public boolean getDoMultiRefs() {
        return this.doMultiRefs;
    }

    /**
     * Set whether we are doing multirefs
     */
    public void setDoMultiRefs (final boolean shouldDo)
    {
        this.doMultiRefs = shouldDo;
    }

    /**
     * Set whether or not we should write XML declarations.
     * @param sendDecl true/false
     */
    public void setSendDecl(final boolean sendDecl)
    {
        this.sendXMLDecl = sendDecl;
    }

    /**
     * Get whether or not to write xsi:type attributes.
     * @return true/false
     */
    public boolean shouldSendXSIType() {
        return this.sendXSIType;
    }

    /**
     * Get the TypeMapping we're using.
     * @return TypeMapping or null
     */
    public TypeMapping getTypeMapping()
    {
        // Always allow the default mappings
        if (this.msgContext == null) {
			return DefaultTypeMappingImpl.getSingletonDelegate();
		}

        String encodingStyle = this.msgContext.getEncodingStyle();
        if (encodingStyle == null) {
			encodingStyle = this.soapConstants.getEncodingURI();
		}
        return (TypeMapping) this.msgContext.
                        getTypeMappingRegistry().getTypeMapping(encodingStyle);
    }

    /**
     * Get the TypeMappingRegistry we're using.
     * @return TypeMapping or null
     */
    public TypeMappingRegistry getTypeMappingRegistry() {
        if (this.msgContext == null) {
			return null;
		}
        return this.msgContext.getTypeMappingRegistry();
    }

    /**
     * Get a prefix for a namespace URI.  This method will ALWAYS
     * return a valid prefix - if the given URI is already mapped in this
     * serialization, we return the previous prefix.  If it is not mapped,
     * we will add a new mapping and return a generated prefix of the form
     * "ns<num>".
     * @param uri is the namespace uri
     * @return prefix
     */
    public String getPrefixForURI(final String uri)
    {
        return this.getPrefixForURI(uri, null, false);
    }

    /**
     * Get a prefix for the given namespace URI.  If one has already been
     * defined in this serialization, use that.  Otherwise, map the passed
     * default prefix to the URI, and return that.  If a null default prefix
     * is passed, use one of the form "ns<num>"
     */
    public String getPrefixForURI(final String uri, final String defaultPrefix)
    {
        return this.getPrefixForURI(uri, defaultPrefix, false);
    }

    /**
     * Get a prefix for the given namespace URI.  If one has already been
     * defined in this serialization, use that.  Otherwise, map the passed
     * default prefix to the URI, and return that.  If a null default prefix
     * is passed, use one of the form "ns<num>"
     */
    public String getPrefixForURI(final String uri, final String defaultPrefix, final boolean attribute)
    {
        if ((uri == null) || (uri.length() == 0)) {
			return null;
		}

        // If we're looking for an attribute prefix, we shouldn't use the
        // "" prefix, but always register/find one.
        String prefix = this.nsStack.getPrefix(uri, attribute);

        if (prefix == null) {
            prefix = (String)this.preferredPrefixes.get(uri);

            if (prefix == null) {
                if (defaultPrefix == null) {
                    prefix = "ns" + this.lastPrefixIndex++;
                    while(this.nsStack.getNamespaceURI(prefix)!=null) {
                        prefix = "ns" + this.lastPrefixIndex++;    
                    }
                } else {
                    prefix = defaultPrefix;
                }
            }

            this.registerPrefixForURI(prefix, uri);
        }

        return prefix;
    }

    /**
     * Register prefix for the indicated uri
     * @param prefix
     * @param uri is the namespace uri
     */
    public void registerPrefixForURI(final String prefix, final String uri)
    {
        if (this.debugEnabled) {
            SerializationContext.log.debug(Messages.getMessage("register00", prefix, uri));
        }

        if ((uri != null) && (prefix != null)) {
            if (this.noNamespaceMappings) {
                this.nsStack.push();
                this.noNamespaceMappings = false;
            }
            final String activePrefix = this.nsStack.getPrefix(uri,true);
            if((activePrefix == null) || !activePrefix.equals(prefix)) {
                this.nsStack.add(uri, prefix);
            }
        }
    }

    /**
     * Return the current message
     */
    public Message getCurrentMessage()
    {
        if (this.msgContext == null) {
			return null;
		}
        return this.msgContext.getCurrentMessage();
    }

    /**
     * Get the MessageContext we're operating with
     */
    public MessageContext getMessageContext() {
        return this.msgContext;
    }

    /**
     * Returns this context's encoding style.  If we've got a message
     * context then we'll get the style from that; otherwise we'll
     * return a default.
     *
     * @return a <code>String</code> value
     */
    public String getEncodingStyle() {
        return this.msgContext == null ? Use.DEFAULT.getEncoding() : this.msgContext.getEncodingStyle();
    }

    /**
     * Returns whether this context should be encoded or not.
     *
     * @return a <code>boolean</code> value
     */
    public boolean isEncoded() {
        return Constants.isSOAP_ENC(this.getEncodingStyle());
    }

    /**
     * Convert QName to a string of the form <prefix>:<localpart>
     * @param qName
     * @return prefixed qname representation for serialization.
     */
    public String qName2String(final QName qName, final boolean writeNS)
    {
        String prefix = null;
        final String namespaceURI = qName.getNamespaceURI();
        String localPart = qName.getLocalPart();
        
        if((localPart != null) && (localPart.length() > 0)) {
            final int index = localPart.indexOf(':');
            if(index!=-1){
                prefix = localPart.substring(0,index);
                if((prefix.length()>0) && !prefix.equals("urn")){
                    this.registerPrefixForURI(prefix, namespaceURI);
                    localPart = localPart.substring(index+1);
                } else {
                    prefix = null;
                }
            }
            localPart = Utils.getLastLocalPart(localPart);            
        }

        if (namespaceURI.length() == 0) {
            if (writeNS) {
                // If this is unqualified (i.e. prefix ""), set the default
                // namespace to ""
                final String defaultNS = this.nsStack.getNamespaceURI("");
                if ((defaultNS != null) && (defaultNS.length() > 0)) {
                    this.registerPrefixForURI("", "");
                }
            }
        } else {
            prefix = this.getPrefixForURI(namespaceURI);
        }

        if ((prefix == null) || (prefix.length() == 0)) {
			return localPart;
		}

        return prefix + ':' + localPart;
    }

    public String qName2String(final QName qName)
    {
        return this.qName2String(qName, false);
    }

    /**
     * Convert attribute QName to a string of the form <prefix>:<localpart>
     * There are slightly different rules for attributes:
     *  - There is no default namespace
     *  - any attribute in a namespace must have a prefix
     *
     * @param qName QName
     * @return prefixed qname representation for serialization.
     */
    public String attributeQName2String(final QName qName) {
        String prefix = null;
        final String uri = qName.getNamespaceURI(); 
        if (uri.length() > 0) {
            prefix = this.getPrefixForURI(uri, null, true);
        }

        if ((prefix == null) || (prefix.length() == 0)) {
			return qName.getLocalPart();
		}
        
        return prefix + ':' + qName.getLocalPart();
    }

    /**
     * Get the QName associated with the specified class.
     * @param cls Class of an object requiring serialization.
     * @return appropriate QName associated with the class.
     */
    public QName getQNameForClass(final Class cls)
    {
        return this.getTypeMapping().getTypeQName(cls);
    }

    /**
     * Indicates whether the object should be interpretted as a primitive
     * for the purposes of multi-ref processing.  A primitive value
     * is serialized directly instead of using id/href pairs.  Thus
     * primitive serialization/deserialization is slightly faster.
     * @param value to be serialized
     * @return true/false
     */
    public boolean isPrimitive(final Object value)
    {
        if (value == null) {
			return true;
		}

        final Class javaType = value.getClass();

        if (javaType.isPrimitive()) {
			return true;
		}

        if (javaType == String.class) {
			return true;
		}
        if (Calendar.class.isAssignableFrom(javaType)) {
			return true;
		}
        if (Date.class.isAssignableFrom(javaType)) {
			return true;
		}
        if (HexBinary.class.isAssignableFrom(javaType)) {
			return true;
		}
        if (Element.class.isAssignableFrom(javaType)) {
			return true;
		}
        if (javaType == byte[].class) {
			return true;
		}

        // There has been discussion as to whether arrays themselves should
        // be regarded as multi-ref.
        // Here are the three options:
        //   1) Arrays are full-fledged Objects and therefore should always be
        //      multi-ref'd  (Pro: This is like java.  Con: Some runtimes don't
        //      support this yet, and it requires more stuff to be passed over the wire.)
        //   2) Arrays are not full-fledged Objects and therefore should
        //      always be passed as single ref (note the elements of the array
        //      may be multi-ref'd.) (Pro:  This seems reasonable, if a user
        //      wants multi-referencing put the array in a container.  Also
        //      is more interop compatible.  Con: Not like java serialization.)
        //   3) Arrays of primitives should be single ref, and arrays of
        //      non-primitives should be multi-ref.  (Pro: Takes care of the
        //      looping case.  Con: Seems like an obtuse rule.)
        //
        // Changing the code from (1) to (2) to see if interop fairs better.
        if (javaType.isArray()) {
			return true;
		}

        // Note that java.lang wrapper classes (i.e. java.lang.Integer) are
        // not primitives unless the corresponding type is an xsd type.
        // (If the wrapper maps to a soap encoded primitive, it can be nillable
        // and multi-ref'd).
        final QName qName = this.getQNameForClass(javaType);
        if ((qName != null) && Constants.isSchemaXSD(qName.getNamespaceURI())) {
            if (SchemaUtils.isSimpleSchemaType(qName)) {
                return true;
            }
        }

        return false;
    }

    /**
     * Serialize the indicated value as an element with the name
     * indicated by elemQName.
     * The attributes are additional attribute to be serialized on the element.
     * The value is the object being serialized.  (It may be serialized
     * directly or serialized as an mult-ref'd item)
     * The value is an Object, which may be a wrapped primitive, the
     * javaType is the actual unwrapped object type.
     * xsi:type is set by using the javaType to
     * find an appopriate xmlType from the TypeMappingRegistry.
     * Null values and the xsi:type flag will be sent or not depending 
     * on previous configuration of this SerializationContext.
     * @param elemQName is the QName of the element
     * @param attributes are additional attributes
     * @param value is the object to serialize
     */
    public void serialize(final QName elemQName,
                          final Attributes attributes,
                          final Object value)
        throws IOException {
        this.serialize(elemQName, attributes, value, null, null, null, null);
    }

    /**
     * Serialize the indicated value as an element with the name
     * indicated by elemQName.
     * The attributes are additional attribute to be serialized on the element.
     * The value is the object being serialized.  (It may be serialized
     * directly or serialized as an mult-ref'd item)
     * The value is an Object, which may be a wrapped primitive, the
     * javaType is the actual unwrapped object type.
     * The xmlType is the QName of the type that is used to set
     * xsi:type.  If not specified, xsi:type is set by using the javaType to
     * find an appopriate xmlType from the TypeMappingRegistry.
     * Null values and the xsi:type flag will be sent or not depending 
     * on previous configuration of this SerializationContext.
     * @param elemQName is the QName of the element
     * @param attributes are additional attributes
     * @param value is the object to serialize
     * @param xmlType is the qname of the type or null.
     * @deprecated use serialize(QName, Attributes, Object, QName, Class) instead
     */
    public void serialize(final QName elemQName,
                          final Attributes attributes,
                          final Object value,
                          final QName xmlType)
        throws IOException {
        this.serialize(elemQName, attributes, value, xmlType, null, null, null);
    }
    
    /**
     * Serialize the indicated value as an element with the name
     * indicated by elemQName.
     * The attributes are additional attribute to be serialized on the element.
     * The value is the object being serialized.  (It may be serialized
     * directly or serialized as an mult-ref'd item)
     * The value is an Object, which may be a wrapped primitive, the
     * javaType is the actual unwrapped object type.
     * The xmlType is the QName of the type that is used to set
     * xsi:type.  If not specified, xsi:type is set by using the javaType to
     * find an appopriate xmlType from the TypeMappingRegistry.
     * Null values and the xsi:type flag will be sent or not depending 
     * on previous configuration of this SerializationContext.
     * @param elemQName is the QName of the element
     * @param attributes are additional attributes
     * @param value is the object to serialize
     * @param xmlType is the qname of the type or null.
     * @param javaType is the java type of the value
     */
    public void serialize(final QName elemQName,
                          final Attributes attributes,
                          final Object value,
                          final QName xmlType, final Class javaType)
        throws IOException {
        this.serialize(elemQName, attributes, value, xmlType, javaType, null, null);
    }

    /**
     * Serialize the indicated value as an element with the name
     * indicated by elemQName.
     * The attributes are additional attribute to be serialized on the element.
     * The value is the object being serialized.  (It may be serialized
     * directly or serialized as an mult-ref'd item)
     * The value is an Object, which may be a wrapped primitive.
     * The xmlType (if specified) is the QName of the type that is used to set
     * xsi:type.
     * The sendNull flag indicates whether null values should be sent over the
     * wire (default is to send such values with xsi:nil="true").
     * The sendType flag indicates whether the xsi:type flag should be sent
     * (default is true).
     * @param elemQName is the QName of the element
     * @param attributes are additional attributes
     * @param value is the object to serialize
     * @param xmlType is the qname of the type or null.
     * @param sendNull determines whether to send null values.
     * @param sendType determines whether to set xsi:type attribute.
     *
     * @deprecated use serialize(QName, Attributes, Object, QName,
     * Boolean, Boolean) instead.
     */
    public void serialize(final QName elemQName,
                          final Attributes attributes,
                          final Object value,
                          final QName xmlType,
                          final boolean sendNull,
                          final Boolean sendType)
        throws IOException
    {
        this.serialize( elemQName, attributes, value, xmlType, null, 
                   (sendNull) ? Boolean.TRUE : Boolean.FALSE, 
                   sendType);
    }

    /**
     * Serialize the indicated value as an element with the name
     * indicated by elemQName.
     * The attributes are additional attribute to be serialized on the element.
     * The value is the object being serialized.  (It may be serialized
     * directly or serialized as an mult-ref'd item)
     * The value is an Object, which may be a wrapped primitive.
     * The xmlType (if specified) is the QName of the type that is used to set
     * xsi:type.
     * The sendNull flag indicates whether to end an element with an xsi:nil="true" attribute for null
     * variables (if Boolean.TRUE), or nothing (if Boolean.FALSE).
     * The sendType flag indicates whether the xsi:type flag should be sent
     * (default is true).
     * @param elemQName is the QName of the element
     * @param attributes are additional attributes
     * @param value is the object to serialize
     * @param xmlType is the qname of the type or null.
     * @param sendNull determines whether to send null values.
     * @param sendType determines whether to set xsi:type attribute.
     */
    public void serialize(final QName elemQName,
                          final Attributes attributes,
                          final Object value,
                          final QName xmlType,
                          final Boolean sendNull,
                          final Boolean sendType)
        throws IOException 
    {
        this.serialize(elemQName, attributes, value, xmlType, null, sendNull, sendType);
        
    }
    
    /**
     * Serialize the indicated value as an element with the name
     * indicated by elemQName.
     * The attributes are additional attribute to be serialized on the element.
     * The value is the object being serialized.  (It may be serialized
     * directly or serialized as an mult-ref'd item)
     * The value is an Object, which may be a wrapped primitive.
     * The xmlType (if specified) is the QName of the type that is used to set
     * xsi:type.
     * The sendNull flag indicates whether to end an element with an xsi:nil="true" attribute for null
     * variables (if Boolean.TRUE), or nothing (if Boolean.FALSE).
     * The sendType flag indicates whether the xsi:type flag should be sent
     * (default is true).
     * @param elemQName is the QName of the element
     * @param attributes are additional attributes
     * @param value is the object to serialize
     * @param xmlType is the qname of the type or null.
     * @param javaType is the java type of the value
     * @param sendNull determines whether to send null values.
     * @param sendType determines whether to set xsi:type attribute.
     */
    public void serialize(final QName elemQName,
                          final Attributes attributes,
                          final Object value,
                          final QName xmlType,
                          final Class javaClass,
                          Boolean sendNull,
                          final Boolean sendType)
        throws IOException
    {
        final boolean sendXSITypeCache = this.sendXSIType;
        if (sendType != null) {
            this.sendXSIType = sendType.booleanValue();
        }
        final boolean shouldSendType = this.shouldSendXSIType();

        try {
            final Boolean sendNullCache = this.sendNull;
            if (sendNull != null) {
                this.sendNull = sendNull;
            } else {
                sendNull = this.sendNull;
            }

            if (value == null) {
                // If the value is null, the element is
                // passed with xsi:nil="true" to indicate that no object is present.
                if (this.sendNull.booleanValue()) {
                    AttributesImpl attrs = new AttributesImpl();
                    if ((attributes != null) && (0 < attributes.getLength())) {
						attrs.setAttributes(attributes);
					}
                    if (shouldSendType) {
						attrs = (AttributesImpl) this.setTypeAttribute(attrs, xmlType);
					}
                    final String nil = this.schemaVersion.getNilQName().getLocalPart();
                    attrs.addAttribute(this.schemaVersion.getXsiURI(), nil, "xsi:" + nil,
                                       "CDATA", "true");
                    this.startElement(elemQName, attrs);
                    this.endElement();
                }
                this.sendNull = sendNullCache;
                return;
            }

            final Message msg= this.getCurrentMessage();
            if(null != msg){
                //Get attachments. returns null if no attachment support.
                final Attachments attachments= this.getCurrentMessage().getAttachmentsImpl();

                if( (null != attachments) && attachments.isAttachment(value)){
                    //Attachment support and this is an object that should be treated as an attachment.

                    //Allow an the attachment to do its own serialization.
                    this.serializeActual(elemQName, attributes, value,
                                    xmlType, javaClass, sendType);

                    //No need to add to mulitRefs. Attachment data stream handled by
                    // the message;
                    this.sendNull = sendNullCache;
                    return;
                }
            }

            // If multi-reference is enabled and this object value is not a primitive
            // and we are not forcing serialization of the object, then generate
            // an element href (and store the object for subsequent outputMultiRef
            // processing).

            // NOTE : you'll notice that everywhere we register objects in the
            // multiRefValues and secondLevelObjects collections, we key them
            // using getIdentityKey(value) instead of the Object reference itself.
            // THIS IS IMPORTANT, and please make sure you understand what's
            // going on if you change any of this code.  It's this way to make
            // sure that individual Objects are serialized separately even if the
            // hashCode() and equals() methods have been overloaded to make two
            // Objects appear equal.

            if (this.doMultiRefs && this.isEncoded() &&
                    (value != this.forceSer) && !this.isPrimitive(value)) {
                if (this.multiRefIndex == -1) {
					this.multiRefValues = new HashMap();
				}

                String id;

                // Look for a multi-ref descriptor for this Object.
                MultiRefItem mri = (MultiRefItem)this.multiRefValues.get(
                        this.getIdentityKey(value));
                if (mri == null) {
                    // Didn't find one, so create one, give it a new ID, and store
                    // it for next time.
                    this.multiRefIndex++;
                    id = "id" + this.multiRefIndex;
                    mri = new MultiRefItem (id, xmlType, sendType, value);
                    this.multiRefValues.put(this.getIdentityKey(value), mri);

                    /**
                     * If we're SOAP 1.2, we can "inline" the serializations,
                     * so put it out now, with it's ID.
                     */
                    if (this.soapConstants == SOAPConstants.SOAP12_CONSTANTS) {
                        final AttributesImpl attrs = new AttributesImpl();
                        if ((attributes != null) && (0 < attributes.getLength())) {
							attrs.setAttributes(attributes);
						}
                        attrs.addAttribute("", Constants.ATTR_ID, "id", "CDATA",
                                           id);
                        this.serializeActual(elemQName, attrs, value, xmlType, javaClass, sendType);
                        this.sendNull = sendNullCache;
                        return;
                    }


                    /** If we're in the middle of writing out
                     * the multi-refs, we've already cloned the list of objects
                     * and so even though we add a new one to multiRefValues,
                     * it won't get serialized this time around.
                     *
                     * To deal with this, we maintain a list of "second level"
                     * Objects - ones that need serializing as a result of
                     * serializing the first level.  When outputMultiRefs() is
                     * nearly finished, it checks to see if secondLevelObjects
                     * is empty, and if not, it goes back and loops over those
                     * Objects.  This can happen N times depending on how deep
                     * the Object graph goes.
                     */
                    if (this.outputMultiRefsFlag) {
                        if (this.secondLevelObjects == null) {
							this.secondLevelObjects = new HashSet();
						}
                        this.secondLevelObjects.add(this.getIdentityKey(value));
                    }
                } else {
                    // Found one, remember it's ID
                    id = mri.id;
                }

                // Serialize an HREF to our object
                final AttributesImpl attrs = new AttributesImpl();
                if ((attributes != null) && (0 < attributes.getLength())) {
					attrs.setAttributes(attributes);
				}
                attrs.addAttribute("", this.soapConstants.getAttrHref(), this.soapConstants.getAttrHref(),
                                   "CDATA", '#' + id);

                this.startElement(elemQName, attrs);
                this.endElement();
                this.sendNull = sendNullCache;
                return;
            }

            // The forceSer variable is set by outputMultiRefs to force
            // serialization of this object via the serialize(...) call
            // below.  However, if the forced object contains a self-reference, we
            // get into an infinite loop..which is why it is set back to null
            // before the actual serialization.
            if (value == this.forceSer) {
				this.forceSer = null;
			}

            // Actually serialize the value.  (i.e. not an href like above)
            this.serializeActual(elemQName, attributes, value, xmlType, javaClass, sendType);
        } finally {
            this.sendXSIType = sendXSITypeCache;
        }
    }

    /**
     * Get an IDKey that represents the unique identity of the object.
     * This is used as a unique key into a HashMap which will
     * not give false hits on other Objects where hashCode() and equals()
     * have been overriden to match.
     *
     * @param value the Object to hash
     * @return a unique IDKey for the identity
     */
    private IDKey getIdentityKey(final Object value) {
        return new IDKey(value);
    }

    /**
     * The serialize method uses hrefs to reference all non-primitive
     * values.  These values are stored and serialized by calling
     * outputMultiRefs after the serialize method completes.
     */
    public void outputMultiRefs() throws IOException
    {
        if (!this.doMultiRefs || (this.multiRefValues == null) ||
                (this.soapConstants == SOAPConstants.SOAP12_CONSTANTS)) {
			return;
		}
        this.outputMultiRefsFlag = true;
        final AttributesImpl attrs = new AttributesImpl();
        attrs.addAttribute("","","","","");

        final String encodingURI = this.soapConstants.getEncodingURI();
        // explicitly state that this attribute is not a root
        final String prefix = this.getPrefixForURI(encodingURI);
        final String root = prefix + ":root";
        attrs.addAttribute(encodingURI, Constants.ATTR_ROOT, root,
                           "CDATA", "0");

        // Make sure we put the encodingStyle on each multiref element we
        // output.
        String encodingStyle;
        if (this.msgContext != null) {
            encodingStyle = this.msgContext.getEncodingStyle();
        } else {
            encodingStyle = this.soapConstants.getEncodingURI();
        }
        final String encStyle = this.getPrefixForURI(this.soapConstants.getEnvelopeURI()) +
                                          ':' + Constants.ATTR_ENCODING_STYLE;
        attrs.addAttribute(this.soapConstants.getEnvelopeURI(),
                           Constants.ATTR_ENCODING_STYLE,
                           encStyle,
                           "CDATA",
                           encodingStyle);

        // Make a copy of the keySet because it could be updated
        // during processing
        final HashSet keys = new HashSet();
        keys.addAll(this.multiRefValues.keySet());
        Iterator i = keys.iterator();
        while (i.hasNext()) {
            while (i.hasNext()) {
                final AttributesImpl attrs2 = new AttributesImpl(attrs);
                final Object val = i.next();
                final MultiRefItem mri = (MultiRefItem) this.multiRefValues.get(val);
                attrs2.setAttribute(0, "", Constants.ATTR_ID, "id", "CDATA",
                                   mri.id);

                this.forceSer = mri.value;

                // Now serialize the value.
                // The sendType parameter is defaulted for interop purposes.
                // Some of the remote services do not know how to
                // ascertain the type in these circumstances (though Axis does).
                this.serialize(SerializationContext.multirefQName, attrs2, mri.value,
                          mri.xmlType,
                          null,
                          this.sendNull,
                          Boolean.TRUE);   // mri.sendType
            }

            // Done processing the iterated values.  During the serialization
            // of the values, we may have run into new nested values.  These
            // were placed in the secondLevelObjects map, which we will now
            // process by changing the iterator to locate these values.
            if (this.secondLevelObjects != null) {
                i = this.secondLevelObjects.iterator();
                this.secondLevelObjects = null;
            }
        }

        // Reset maps and flags
        this.forceSer = null;
        this.outputMultiRefsFlag = false;
        this.multiRefValues = null;
        this.multiRefIndex = -1;
        this.secondLevelObjects = null;
    }

    public void writeXMLDeclaration() throws IOException {
        this.writer.write("<?xml version=\"1.0\" encoding=\"");        
        this.writer.write(this.encoding);
        this.writer.write("\"?>\n");
        this.startOfDocument = false;        
    }
    
    /**
     * Writes (using the Writer) the start tag for element QName along with the
     * indicated attributes and namespace mappings.
     * @param qName is the name of the element
     * @param attributes are the attributes to write
     */
    public void startElement(final QName qName, Attributes attributes)
        throws IOException
    {
        java.util.ArrayList vecQNames = null;
        if (this.debugEnabled) {
            SerializationContext.log.debug(Messages.getMessage("startElem00",
                    "[" + qName.getNamespaceURI() + "]:" + qName.getLocalPart()));
        }

        if (this.startOfDocument && this.sendXMLDecl) {
            this.writeXMLDeclaration();
        }

        if (this.writingStartTag) {
            this.writer.write('>');
            if (this.pretty) {
				this.writer.write('\n');
			}
            this.indent++;
        }

        if (this.pretty) {
			for (int i=0; i<this.indent; i++) {
				this.writer.write(' ');
			}
		}
        final String elementQName = this.qName2String(qName, true);
        this.writer.write('<');

        this.writer.write(elementQName);
        
        if (this.writeXMLType != null) {
            attributes = this.setTypeAttribute(attributes, this.writeXMLType);
            this.writeXMLType = null;
        }

        if (attributes != null) {
            for (int i = 0; i < attributes.getLength(); i++) {
                String qname = attributes.getQName(i);
                this.writer.write(' ');

                String prefix = "";
                final String uri = attributes.getURI(i);
                if ((uri != null) && (uri.length() > 0)) {
                    if (qname.length() == 0) {
                        // If qname isn't set, generate one
                        prefix = this.getPrefixForURI(uri);
                    } else {
                        // If it is, make sure the prefix looks reasonable.
                        final int idx = qname.indexOf(':');
                        if (idx > -1) {
                            prefix = qname.substring(0, idx);
                            prefix = this.getPrefixForURI(uri,
                                                     prefix, true);
                        }
                    }
                    if (prefix.length() > 0) {
                        qname = prefix + ':' + attributes.getLocalName(i);
                    } else {
                        qname = attributes.getLocalName(i);
                    }
                } else {
                   qname = attributes.getQName(i);
                    if(qname.length() == 0) {
						qname = attributes.getLocalName(i);
					}
                }

                if (qname.startsWith("xmlns")) {
                  if (vecQNames == null) {
					vecQNames = new ArrayList();
				}
                  vecQNames.add(qname);
                }
                this.writer.write(qname);
                this.writer.write("=\"");
                
                this.getEncoder().writeEncoded(this.writer, attributes.getValue(i));
                
                this.writer.write('"');
            }
        }

        if (this.noNamespaceMappings) {
            this.nsStack.push();
        } else {
            for (Mapping map=this.nsStack.topOfFrame(); map!=null; map=this.nsStack.next()) {
                if (!(map.getNamespaceURI().equals(Constants.NS_URI_XMLNS) && map.getPrefix().equals("xmlns")) &&
                    !(map.getNamespaceURI().equals(Constants.NS_URI_XML) && map.getPrefix().equals("xml")))
                {                
                    final StringBuffer sb = new StringBuffer("xmlns");
                    if (map.getPrefix().length() > 0) {
                        sb.append(':');
                        sb.append(map.getPrefix());
                    }
                    if ((vecQNames==null) || (vecQNames.indexOf(sb.toString())==-1)) {
                        this.writer.write(' ');
                        sb.append("=\"");
                        sb.append(map.getNamespaceURI());
                        sb.append('"');
                        this.writer.write(sb.toString());
                    }
                }
            }

            this.noNamespaceMappings = true;
        }

        this.writingStartTag = true;

        this.elementStack.push(elementQName);

        this.onlyXML=true;
    }

    /**
     * Writes the end element tag for the open element.
     **/
    public void endElement()
        throws IOException
    {
        final String elementQName = (String)this.elementStack.pop();

        if (this.debugEnabled) {
            SerializationContext.log.debug(Messages.getMessage("endElem00", "" + elementQName));
        }

        this.nsStack.pop();

        if (this.writingStartTag) {
            this.writer.write("/>");
            if (this.pretty) {
				this.writer.write('\n');
			}
            this.writingStartTag = false;
            return;
        }

        if (this.onlyXML) {
            this.indent--;
            if (this.pretty) {
				for (int i=0; i<this.indent; i++) {
					this.writer.write(' ');
				}
			}
        }
        this.writer.write("</");
        this.writer.write(elementQName);
        this.writer.write('>');
        if (this.pretty) {
			if (this.indent>0) {
				this.writer.write('\n');
			}
		}
        this.onlyXML=true;
    }

    /**
     * Convenience operation to write out (to Writer) the characters
     * in p1 starting at index p2 for length p3.
     * @param p1 character array to write
     * @param p2 starting index in array
     * @param p3 length to write
     */
    public void writeChars(final char [] p1, final int p2, final int p3)
        throws IOException
    {
        if (this.startOfDocument && this.sendXMLDecl) {
            this.writeXMLDeclaration();
        }
        
        if (this.writingStartTag) {
            this.writer.write('>');
            this.writingStartTag = false;
        }
        this.writeSafeString(String.valueOf(p1,p2,p3));
        this.onlyXML=false;
    }

    /**
     * Convenience operation to write out (to Writer) the String
     * @param string is the String to write.
     */
    public void writeString(final String string)
        throws IOException
    {
        if (this.startOfDocument && this.sendXMLDecl) {
            this.writeXMLDeclaration();
        }
        
        if (this.writingStartTag) {
            this.writer.write('>');
            this.writingStartTag = false;
        }
        this.writer.write(string);
        this.onlyXML=false;
    }

    /**
     * Convenience operation to write out (to Writer) the String
     * properly encoded with xml entities (like &amp)
     * @param string is the String to write.
     */
    public void writeSafeString(final String string)
        throws IOException
    {
        if (this.startOfDocument && this.sendXMLDecl) {
            this.writeXMLDeclaration();
        }
        
        if (this.writingStartTag) {
            this.writer.write('>');
            this.writingStartTag = false;
        }
        
        this.getEncoder().writeEncoded(this.writer, string);
        this.onlyXML=false;
    }

    /**
     * Output a DOM representation to a SerializationContext
     * @param el is a DOM Element
     */
    public void writeDOMElement(final Element el)
        throws IOException
    {
        if (this.startOfDocument && this.sendXMLDecl) {
            this.writeXMLDeclaration();
        }
        
        // If el is a Text element, write the text and exit
        if (el instanceof org.apache.axis.message.Text) {            
            this.writeSafeString(((Text)el).getData());
            return;
        }   
        
        AttributesImpl attributes = null;
        final NamedNodeMap attrMap = el.getAttributes();

        if (attrMap.getLength() > 0) {
            attributes = new AttributesImpl();
            for (int i = 0; i < attrMap.getLength(); i++) {
                final Attr attr = (Attr)attrMap.item(i);
                final String tmp = attr.getNamespaceURI();
                if ( (tmp != null) && tmp.equals(Constants.NS_URI_XMLNS) ) {
                    String prefix = attr.getLocalName();
                    if (prefix != null) {
                        if (prefix.equals("xmlns")) {
							prefix = "";
						}
                        final String nsURI = attr.getValue();
                        this.registerPrefixForURI(prefix, nsURI);
                    }
                    continue;
                }

                attributes.addAttribute(attr.getNamespaceURI(),
                                        attr.getLocalName(),
                                        attr.getName(),
                                        "CDATA", attr.getValue());
            }
        }

        final String namespaceURI = el.getNamespaceURI();
        String localPart = el.getLocalName();
        if((namespaceURI == null) || (namespaceURI.length()==0)) {
			localPart = el.getNodeName();
		}
        final QName qName = new QName(namespaceURI, localPart);

        this.startElement(qName, attributes);

        final NodeList children = el.getChildNodes();
        for (int i = 0; i < children.getLength(); i++) {
            final Node child = children.item(i);
            if (child instanceof Element) {
                this.writeDOMElement((Element)child);
            } else if (child instanceof CDATASection) {
                this.writeString("<![CDATA[");
                this.writeString(((Text)child).getData());
                this.writeString("]]>");
            } else if (child instanceof Comment) {
                this.writeString("<!--");
                this.writeString(((CharacterData)child).getData());
                this.writeString("-->");
            } else if (child instanceof Text) {
                this.writeSafeString(((Text)child).getData());
            }
        }

        this.endElement();
    }

    /**
     * Convenience method to get the Serializer for a specific
     * java type
     * @param javaType is Class for a type to serialize
     * @return Serializer
     */
    public final Serializer getSerializerForJavaType(final Class javaType) {
        SerializerFactory serF = null;
        Serializer ser = null;
        try {
            serF = (SerializerFactory) this.getTypeMapping().getSerializer(javaType);
            if (serF != null) {
                ser = (Serializer) serF.getSerializerAs(Constants.AXIS_SAX);
            }
        } catch (final JAXRPCException e) {
        }

        return ser;
    }

    /**
     * Obtains the type attribute that should be serialized and returns the new list of Attributes
     * @param attributes of the qname
     * @param type is the qname of the type
     * @return new list of Attributes
     */
    public Attributes setTypeAttribute(final Attributes attributes, final QName type)
    {
        SchemaVersion schema = SchemaVersion.SCHEMA_2001;
        if (this.msgContext != null) {
            schema = this.msgContext.getSchemaVersion();
        }

        if ((type == null) ||
             (type.getLocalPart().indexOf(SymbolTable.ANON_TOKEN) >= 0) ||
            ((attributes != null) &&
             (attributes.getIndex(schema.getXsiURI(),
                                "type") != -1))) {
			return attributes;
		}

        final AttributesImpl attrs = new AttributesImpl();
        if ((attributes != null) && (0 < attributes.getLength()) ) {
			attrs.setAttributes(attributes);
		}

        final String prefix = this.getPrefixForURI(schema.getXsiURI(),
                                           "xsi");

        attrs.addAttribute(schema.getXsiURI(),
                           "type",
                           prefix + ":type",
                           "CDATA", this.attributeQName2String(type));
        return attrs;
    }

    /**
     * Invoked to do the actual serialization of the qName (called by serialize above).
     * additional attributes that will be serialized with the qName.
     * @param elemQName is the QName of the element
     * @param attributes are additional attributes
     * @param value is the object to serialize
     * @param xmlType (optional) is the desired type QName.
     * @param sendType indicates whether the xsi:type attribute should be set.
     */
    private void serializeActual(final QName elemQName,
                                final Attributes attributes,
                                final Object value,
                                QName xmlType,
                                final Class javaClass,
                                final Boolean sendType)
        throws IOException
    {
        boolean shouldSendType = (sendType == null) ? this.shouldSendXSIType() :
            sendType.booleanValue();

        if (value != null) {
            final TypeMapping tm = this.getTypeMapping();

            if (tm == null) {
                throw new IOException(
                        Messages.getMessage("noSerializer00",
                                             value.getClass().getName(),
                                             "" + this));
            }

            // Set currentXMLType to the one desired one.
            // Note for maxOccurs usage this xmlType is the
            // type of the component not the type of the array.
            this.currentXMLType = xmlType;

            // if we're looking for xsd:anyType, accept anything...
            if (Constants.equals(Constants.XSD_ANYTYPE,xmlType)){
                xmlType = null;
                shouldSendType = true;
            }

            // Try getting a serializer for the prefered xmlType
            final QNameHolder actualXMLType = new QNameHolder();
                        
            final Class javaType = this.getActualJavaClass(xmlType, javaClass, value);
                        
            final Serializer ser = this.getSerializer(javaType, xmlType,
                                           actualXMLType);

            if ( ser != null ) {
                // Send the xmlType if indicated or if
                // the actual xmlType is different than the
                // prefered xmlType
                if (shouldSendType ||
                    ((xmlType != null) &&
                     (!xmlType.equals(actualXMLType.value)))) {

                    if(!this.isEncoded()) {
                        if (Constants.isSOAP_ENC(actualXMLType.value.getNamespaceURI())) {
                            // Don't write SOAP_ENC types (i.e. Array) if we're not using encoding
                        } else if (javaType.isPrimitive() && (javaClass != null) && (JavaUtils.getWrapperClass(javaType) == javaClass)) {
                            // Don't write xsi:type when serializing primitive wrapper value as primitive type.
                        }
                        else {
                            if(!(javaType.isArray() && (xmlType != null) && Constants.isSchemaXSD(xmlType.getNamespaceURI())) ) {
                                this.writeXMLType = actualXMLType.value;
                            }
                        }
                    } else {
                        this.writeXMLType = actualXMLType.value;
                    }
                }

                // -----------------
                // NOTE: I have seen doc/lit tests that use
                // the type name as the element name in multi-ref cases
                // (for example <soapenc:Array ... >)
                // In such cases the xsi:type is not passed along.
                // -----------------
                // The multiref QName is our own fake name.
                // It may be beneficial to set the name to the
                // type name, but I didn't see any improvements
                // in the interop tests.
                //if (name.equals(multirefQName) && type != null)
                //    name = type;
                ser.serialize(elemQName, attributes, value, this);
                return;
            }
            throw new IOException(Messages.getMessage("noSerializer00",
                    value.getClass().getName(), "" + tm));
        }
        // !!! Write out a generic null, or get type info from somewhere else?
    }
    
    /**
     * Returns the java class for serialization. 
     * If the xmlType is xsd:anyType or javaType is array or javaType is java.lang.Object
     * the java class for serialization is the class of obj.
     * If the obj is not array and the obj's class does not match with the javaType,
     * the java class for serialization is the javaType.
     * Otherwise, the java class for serialization is the obj's class.
     * 
     * @param xmlType    the qname of xml type
     * @param javaType   the java class from serializer 
     * @param obj        the object to serialize
     * @return the java class for serialization
     */
    private Class getActualJavaClass(final QName xmlType, final Class javaType, final Object obj) {
        final Class cls = obj.getClass();
        
        if (((xmlType != null) 
                    && Constants.isSchemaXSD(xmlType.getNamespaceURI()) && "anyType".equals(xmlType.getLocalPart()))
                || ((javaType != null) 
                        && (javaType.isArray() || (javaType == Object.class)))) {
            return cls;
        }
        
        if ((javaType != null) && (cls != javaType) && !cls.isArray()) {
            return javaType;
        }
        
        return cls;
    }

    private Serializer getSerializerFromClass(final Class javaType, final QName qname) {
        
        Serializer serializer = null;
        try {
            final Method method = 
                MethodCache.getInstance().getMethod(javaType,
                                                    SerializationContext.SERIALIZER_METHOD,
                                                    SerializationContext.SERIALIZER_CLASSES);
            if (method != null) {
                serializer = (Serializer) method.invoke(null,
                    new Object[] {this.getEncodingStyle(), javaType, qname});
            }
       } catch (final NoSuchMethodException e) {
       } catch (final IllegalAccessException e) {
       } catch (final InvocationTargetException e) {
       }
       return serializer;
    }

    /**
     * Get the currently prefered xmlType
     * @return QName of xmlType or null
     */
    public QName getCurrentXMLType() {
        return this.currentXMLType;
    }

    /**
     * Walk the interfaces of a class looking for a serializer for that
     * interface.  Include any parent interfaces in the search also.
     *
     */
    private SerializerFactory getSerializerFactoryFromInterface(final Class javaType,
                                                                final QName xmlType,
                                                                final TypeMapping tm)
    {
        SerializerFactory  serFactory  = null ;
        final Class [] interfaces = javaType.getInterfaces();
        if (interfaces != null) {
            for (int i = 0; i < interfaces.length; i++) {
                final Class iface = interfaces[i];
                serFactory = (SerializerFactory) tm.getSerializer(iface,
                                                                  xmlType);
                if (serFactory == null) {
					serFactory = this.getSerializerFactoryFromInterface(iface, xmlType, tm);
				}
                if (serFactory != null) {
					break;
				}

            }
        }
        return serFactory;
    }

    /**
     * getSerializer
     * Attempts to get a serializer for the indicated javaType and xmlType.
     * @param javaType is the type of the object
     * @param xmlType is the preferred qname type.
     * @param actualXMLType is set to a QNameHolder or null.
     *                     If a QNameHolder, the actual xmlType is returned.
     * @return found class/serializer or null
     **/
    private Serializer getSerializer(Class javaType, final QName xmlType,
                                     final QNameHolder actualXMLType) {
        SerializerFactory  serFactory  = null ;
        final TypeMapping tm = this.getTypeMapping();
        if (actualXMLType != null) {
            actualXMLType.value = null;
        }

        while (javaType != null) {
            // check type mapping
            serFactory = (SerializerFactory) tm.getSerializer(javaType, xmlType);
            if (serFactory != null) {
                break;
            }

            // check the class for serializer
            final Serializer serializer = this.getSerializerFromClass(javaType, xmlType);
            if (serializer != null) {
                if (actualXMLType != null) {
                    final TypeDesc typedesc = TypeDesc.getTypeDescForClass(javaType);
                    if (typedesc != null) {
                        actualXMLType.value = typedesc.getXmlType();
                    }
                }
                return serializer;
            }

            // Walk my interfaces...
            serFactory = this.getSerializerFactoryFromInterface(javaType, xmlType, tm);
            if (serFactory != null) {
                break;
            }

            // Finally, head to my superclass
            javaType = javaType.getSuperclass();
        }

        // Using the serialization factory, create a serializer
        Serializer ser = null;
        if ( serFactory != null ) {
            ser = (Serializer) serFactory.getSerializerAs(Constants.AXIS_SAX);

            if (actualXMLType != null) {
                // Get the actual qname xmlType from the factory.
                // If not found via the factory, fall back to a less
                // performant solution.
                if (serFactory instanceof BaseSerializerFactory) {
                    actualXMLType.value =
                        ((BaseSerializerFactory) serFactory).getXMLType();
                }
                boolean encoded = this.isEncoded();
                if ((actualXMLType.value == null) ||
                        (!encoded &&
                        (actualXMLType.value.equals(Constants.SOAP_ARRAY) ||
                        actualXMLType.value.equals(Constants.SOAP_ARRAY12)))) {
                    actualXMLType.value = tm.getXMLType(javaType,
                                                        xmlType,
                                                        encoded);
                }
            }
        }

        return ser;
    }

    public String getValueAsString(final Object value, final QName xmlType, final Class javaClass) throws IOException {
        Class cls = value.getClass();
        cls = this.getActualJavaClass(xmlType, javaClass, value);
        
        Serializer ser = this.getSerializer(cls, xmlType, null);
        
        // The java type is an array, but we need a simple type.
        if (ser instanceof ArraySerializer)
        {
            final SimpleListSerializerFactory factory =
                new SimpleListSerializerFactory(cls, xmlType);
            ser = (Serializer)
                factory.getSerializerAs(this.getEncodingStyle());
        }

        if (!(ser instanceof SimpleValueSerializer)) {
            throw new IOException(
                    Messages.getMessage("needSimpleValueSer",
                                         ser.getClass().getName()));
        }
        final SimpleValueSerializer simpleSer = (SimpleValueSerializer)ser;
        return simpleSer.getValueAsString(value, this);
    }

    public void setWriteXMLType(final QName type) {
        this.writeXMLType = type;
    }

    public XMLEncoder getEncoder() {
        if(this.encoder == null) {
            this.encoder = XMLUtils.getXMLEncoder(this.encoding);
        }
        return this.encoder;
    }

    /**
     * get the encoding for the serialization
     * @return
     */
    public String getEncoding() {
        return this.encoding;
    }

    /**
     * set the encoding for the serialization
     */
    public void setEncoding(final String encoding) {
        this.encoding = encoding;
    }

    public QName getItemQName() {
        return this.itemQName;
    }

    public void setItemQName(final QName itemQName) {
        this.itemQName = itemQName;
    }

    public QName getItemType() {
        return this.itemType;
    }

    public void setItemType(final QName itemType) {
        this.itemType = itemType;
    }
}