/*******************************************************************************
 * Copyright (c) 2006 IBM Corporation.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *    Bjoern Assmann - initial API and implementation
 *    Sergey Lyakhov - implement v.1.5 functionality
 *******************************************************************************/

package org.eclipse.higgins.sts.client;

import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.security.cert.CertStore;
import java.security.cert.X509Certificate;

import org.apache.axiom.om.util.Base64;
import org.eclipse.higgins.icard.IInformationCard;

/**
 * Implements PPID generation algorithm according to CardSpace Identity Selector Interoperability Profile V1.5
 *
 */
public class PPIDHelper {
	private final static char[] userFriendlyPPIDValuetabe = { 'Q', 'L', '2', '3', '4', '5',
		'6', '7', '8', '9', 'A', 'B', 'C', 'D', 'E', 'F', 'G', 'H', 'J',
		'K', 'M', 'N', 'P', 'R', 'S', 'T', 'U', 'V', 'W', 'X', 'Y', 'Z' };

	/**
	 * @param informationCard
	 * @param rp Certificate (or certificate chain) of relying party
	 * @param certStore 
	 * @param rpURI URL which contain domain name or IP of relying party (used if no RP certificate present)
	 * @return
	 * @throws Exception
	 */
	static public byte[] computeClaimValuePPID(IInformationCard informationCard, X509Certificate [] rp, CertStore certStore, String rpURI) throws Exception {
		byte[] rpId = RelyingPartyIdentifierV15Helper.getRelyingPartyIdentifier(rpURI, rp);
		byte[] ppid;
		if (informationCard.isSelfIssued()) {
			MessageDigest md = MessageDigest.getInstance("SHA-256");
			byte[] cardIdB = informationCard.getID().getBytes("UTF-16LE");
			byte[] canonicalCardId = md.digest(cardIdB);
			md.reset();
			md.update(rpId);
			md.update(canonicalCardId);
			ppid = md.digest();
		}
		else {
			MessageDigest md = MessageDigest.getInstance("SHA-256");
			byte[] masterKey = informationCard.getMasterKey();
			byte[] salt = informationCard.getHashSalt();
			md.update(masterKey);
			md.update(rpId);
			md.update(salt);
			ppid = md.digest();
		}
		return ppid;
	}


	public static String getUserFriendlyPPID(byte[] ppid) throws NoSuchAlgorithmException {
		MessageDigest md = MessageDigest.getInstance("SHA-1");
		byte[] ppidh = md.digest(ppid);
		StringBuffer res = new StringBuffer();
		for (int i = 0; i < 10; i++) {
			int value = ppidh[i];
			value = value % 32;
			if (value < 0) {
				value = value + 32;
			}
			char A_i = userFriendlyPPIDValuetabe[value];
			res.append(A_i);
			if (i == 2 || i == 6) {
				res.append('-');
			}
		}
		return res.toString();
	}

	public static String getUserFriendlyPPID(String ppid) throws Exception {
		byte[] bppid = Base64.decode(ppid);
		return getUserFriendlyPPID(bppid);
	}

}
