/*******************************************************************************
 * Copyright (c) 2006 IBM Corporation.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *    Suresh Chari (IBM Corporation) - initial API and implementation
 *******************************************************************************/ 

package org.eclipse.higgins.sts.utilities;

import java.math.BigInteger;
import java.security.KeyPair;
import java.security.interfaces.RSAPrivateCrtKey;
import java.security.interfaces.RSAPublicKey;
import java.security.spec.RSAKeyGenParameterSpec;

/**
 *
 * Generate keys  for self signed keys in Higgins according to the X9.31 specifications. Requirements  
 * as obtained from the "Identity Selector Interoperability Profile V1.0" (Arun Nanda) and the  X9 specification 
 * "Digital Signatures Using Reversible Public Key Cryptography for the Financial Services Industry (rDSA)".
 * 
 */

public class X931KeyGenerator {

	// Information to store the key structure
	
	// Given inputs to generate the keys
	private BigInteger xQ1,xQ2,xP1,xP2,xP,xQ;
	
	// The private key parts which are generated
	private BigInteger primeP,primeQ,modulus, publicExponent;
	private BigInteger privateExponent, privateExponentP, privateExponentQ;
	private BigInteger crtCoefficient; 
	
	/**
	 * Tests given BigInteger for primality according to the mandate of the X9.31 standard.
	 * The standard BigInteger implementation in Java run the Miller-Rabin test as well
	 * as the Lucas-Lehmer test so we can just plug in the regular primaility test.
	 * 
	 * @param candidate to be tested for primality
	 */
	private static boolean isProbableX931Prime( BigInteger candidate){
		
		
		if (candidate!=null)
			// Note that the BigInteger implementation of isProbablePrime will factor in the efficacy of the
			// Miller Rabin test and only run enough rounds to get a 2^{-100} probability of being wrong.
			// e.g. for 1024 bits it runs Rabin-Miller 27 times
			return candidate.isProbablePrime( 100 );
		else
			return false;
	}
	
	
	/**
	 * Successively searches for the smallest odd number starting from the given parameter
	 * till it finds a probable prime. If the input number is not odd it returns false.
	 * 
	 * @param startFrom the bigInteger to start searching from
	 * @return a probable prime number greater than the given number
	 */
	public static BigInteger findSmallPrimeSequential( BigInteger startFrom){
		
		BigInteger result = startFrom;
		
		if (startFrom.getLowestSetBit() != 0 )
		{
			// The initial candidate is even.
			// start search by adding one
			result = startFrom.add( BigInteger.ONE );
		}
		
		
		BigInteger TWO = new BigInteger("2");
		
		while (!isProbableX931Prime( result )) {
			result = result.add( TWO );
		}
			
		return result;
	}
	
	 //TODO: See if its easy to handle the even exponent case
	/**
	 * Finds prime P which is the next largest prime after startFrom which satisfies the following conditions 
	 * a. P - 1 has the factor factorOfPMinus1
	 * b. P + 1 has th factor factorOfPPlus1
	 * c. e and p - 1 are relatively prime
	 * NOTE: this does NO size checking on either the startFrom or the factors. It also assumes that the factors
	 * are indeed primes.
	 *  
	 * @param startFrom Integer to start search from
	 * @param factorOfPMinus1 requested factor of P - 1
	 * @param factorOfPPlus1  requested factor of P + 1
	 * @param publicExponent the public exponent for the key being generated. Note that this is required so that 
	 *        the generated prime has the property that e and p-1 are relatively prime. This exponent is ASSUMED
	 * 	      to be ODD. If this parameter is even it returns NULL
	 * @return The prime P found with these features.
	 */
	public static BigInteger findX931KeyPrime( BigInteger startFrom, 
											   BigInteger factorOfPMinus1, 
											   BigInteger factorOfPPlus1,
											   BigInteger publicExponent)
	throws UnsupportedOptionException
	{
		// R is a number such that:
		// R=1 mod (factorOfPMinus1) and
		// R=-1 (mod) (factorOfPPlus1)
		BigInteger R ;
		
		
		// Throw exception if the public exponent is not odd
		// Can modify this at a later stage if there is a need for
		//supporting even exponents
		if (publicExponent.getLowestSetBit() !=0 )
				throw new UnsupportedOptionException();
		
		BigInteger product = factorOfPMinus1.multiply( factorOfPPlus1 );
		
		// R is computed using the Chinese Remainder Theorem as
		// R  = p2-inverse (mod p1) * p2 - p1-inverse (mod p2) p1
		// where p1 is factorOfPMinus1
		// and p2 is factorOfPPlus1
		R  = (factorOfPPlus1.modInverse( factorOfPMinus1 )).multiply( factorOfPPlus1 );
		R  = R.subtract ( (factorOfPMinus1.modInverse( factorOfPPlus1 )).multiply( factorOfPMinus1 ));
		
		// Use R to compute the candidate, test for primality, add multiples of p1p2 etc.
		// Add a factor if R is negative
		
		if (R.signum()== -1)
			R = R.add( product );
		
		BigInteger temp = R.subtract( startFrom.mod( product));
		BigInteger candidate = startFrom.add( temp );
		
		if (candidate.compareTo( startFrom ) == -1)
			candidate = candidate.add( product );
		
		// The condition for the while loop in English is 
		// while ( ( gcd(e,p-1)!= 1 ) or ( p is not a prime) ) 
		while ( ( ( ( publicExponent.gcd ( candidate.subtract( BigInteger.ONE ))).compareTo( BigInteger.ONE))!= 0 ) ||
				 (!isProbableX931Prime( candidate )) ){
			
			candidate = candidate.add( product );
			
		}
		
		return candidate ;
	}
	/**
	 * @param xq1
	 * @param xq2
	 * @param xp1
	 * @param xp2
	 * @param xp
	 * @param xq
	 */
	public X931KeyGenerator(byte[] xp1, byte[] xp2, byte[] xq1, byte[] xq2,
			byte[] xp, byte[] xq) throws InvalidRandomParameterException{
		
		// Just try to make bigIntegers out of the given parameters. We do NOT
		// check any of the conditions mandated by the X9.31 standard NOR do
		// we check for size consistencies which are done in the relevant routines
		// Only case this can have an exception is when the byte arrays are null 
		// so we just catch and throw.
		try {
			xQ1 = new BigInteger(1,xq1);
			xQ2 = new BigInteger(1,xq2);
			xP1 = new BigInteger(1,xp1);
			xP2 = new BigInteger(1,xp2);
			xP = new BigInteger(1,xp);
			xQ = new BigInteger(1,xq);
		} catch( Exception e){
			throw new InvalidRandomParameterException( InvalidRandomParameterException.INVALID_INPUT_DATA);
		}
	}
	
	/**
	 * @param params
	 * @return The generated key which is an object implementing the interface
	 * @see java.security.KeyPair . The private key of this keypair is
	 * a @see java.security.interfaces.RSAPrivateCrtKey . Note that the interface
	 * @see java.security.interfaces.RSAPrivateKey doesnt let you specify the
	 * primes P and Q so we use the Crt Key. 
	 *
	 * Throws exceptions if the requested options are not supported(currently only
	 * odd exponents are supported) OR if the initially supplied parameters used
	 * in constructing this generator do not match the requested keygen parameters
	 */
	public KeyPair GenerateKeyPair(RSAKeyGenParameterSpec params) 
		throws UnsupportedOptionException, InvalidRandomParameterException
	{
		
		
		// X9.31 mandates a number of checks on the generated keys which we ensure we can satisfy
		// First key size is 1024 + 256s bits
		int keysize = params.getKeysize();
		
		if ( (keysize <1024) || ((( keysize - 1024) % 256 ) !=0 ))
			throw new UnsupportedOptionException();
		
		
		// checks on the random parameters which were given in the constructor.
		
		// First the small primes should be at least 101 bits long
		if ( ( xP1.bitLength() < 101) || (xP2.bitLength() < 101 ) || (xQ1.bitLength() < 101 ) || (xQ2.bitLength() < 101 ) )
			throw new InvalidRandomParameterException( InvalidRandomParameterException.INVALID_INPUT_DATA);
		
		// for a 1024 + 256s bit key i.e. for the prime size 512 + 128s the parameters xP and xQ are
		// required to be at least sqrt(2)* 2 ^{511 + 128s}. Cheat a bit by enforcing the "stronger"
		// condition that they are at least 1.5 * 2^(511+128s) i.e. the top 2 bits are set
		
		if ( (!xP.testBit( keysize/2 -1)) || (!xP.testBit( keysize/2 -2)) ||
		     (!xQ.testBit( keysize/2 -1)) || (!xQ.testBit( keysize/2 -2)) )
			throw new InvalidRandomParameterException( InvalidRandomParameterException.INVALID_INPUT_DATA);
		
		publicExponent = params.getPublicExponent();
		
		if ( publicExponent.getLowestSetBit() !=0 )
			throw new UnsupportedOptionException();
		
		//First find the small primes
		xP1 = findSmallPrimeSequential( xP1);
		xP2 = findSmallPrimeSequential( xP2);
		xQ1 = findSmallPrimeSequential( xQ1);
		xQ2 = findSmallPrimeSequential( xQ2);
		
		//Find the two primes P and Q
		
		primeP = findX931KeyPrime( xP, xP1, xP2, publicExponent );
		primeQ = findX931KeyPrime( xQ, xQ1, xQ2, publicExponent );
		
		// check size of p-q
		BigInteger diff =  primeP.subtract( primeQ).abs();
		if (diff.bitLength() < keysize/2 -100 )
			throw new InvalidRandomParameterException( InvalidRandomParameterException.INSUFFICIENT_PRIME_DISTANCE);
		
		// First fix p and q so that p > q. This will help in correctly defining the CRT coefficient.
		// NOTE: This WILL cause test which explicitly check the values of P and Q to Fail. 
		if (primeP.compareTo( primeQ ) == -1){
			// swap
			BigInteger temp = primeP;
			primeP = primeQ;
			primeQ = temp;
		}
		
		modulus = primeP.multiply( primeQ );
		
		BigInteger lcmP1Q1 =  primeP.subtract(BigInteger.ONE).multiply( primeQ.subtract(BigInteger.ONE ));
		lcmP1Q1 = lcmP1Q1.divide( primeP.subtract(BigInteger.ONE).gcd( primeQ.subtract(BigInteger.ONE) ));
		
		// Find the private exponent
		privateExponent = publicExponent.modInverse( lcmP1Q1 );
		
		if ( privateExponent.bitLength() < keysize/2 )
			throw new InvalidRandomParameterException( InvalidRandomParameterException.INVALID_PRIVATE_EXPONENT );
		
		//find dp,dq,crtcoeffienct
		
		privateExponentP = privateExponent.mod( primeP.subtract( BigInteger.ONE ));
		privateExponentQ = privateExponent.mod( primeQ.subtract( BigInteger.ONE ));
		
		crtCoefficient = primeQ.modInverse( primeP );
		
		// run all the checks in the text to ensure sanity of the key being generated.
		X931RSAPrivateCrtKey privateKey = new X931RSAPrivateCrtKey
			(primeP, primeQ, modulus, publicExponent, privateExponent, privateExponentP, privateExponentQ, crtCoefficient);
		
		X931RSAPublicKey publicKey = new X931RSAPublicKey
			(primeP, primeQ, modulus, publicExponent, privateExponent, privateExponentP, privateExponentQ, crtCoefficient);
		
		// The PrivateCrtKey object also implements the interface for a RSAPublicKey so we just use the same object
		return new KeyPair ( (RSAPublicKey)publicKey, privateKey ); 
		
	}
	
	private class X931RSAPrivateCrtKey implements RSAPrivateCrtKey
	{
		/**
		 * 
		 */
		private static final long serialVersionUID = 1L;
		BigInteger p,q,n,e,d, dp,dq,c;
		
				
		
		/**
		 * @param p the prime p
		 * @param q	prime q
		 * @param n modulus
		 * @param e public exponent
		 * @param d private exponent
		 * @param dp private exponent ( mod p)
		 * @param dq private exponet ( mod q )
		 * @param c the crt coefficient
		 */
		public X931RSAPrivateCrtKey(BigInteger p, BigInteger q, BigInteger n,
				BigInteger e, BigInteger d, BigInteger dp, BigInteger dq,
				BigInteger c) {
			super();
			this.p = p;
			this.q = q;
			this.n = n;
			this.e = e;
			this.d = d;
			this.dp = dp;
			this.dq = dq;
			this.c = c;
		}
		
		
		/* (non-Javadoc)
		 * @see java.security.interfaces.RSAPrivateCrtKey#getCrtCoefficient()
		 */
		public BigInteger getCrtCoefficient() {
			return c;
		}
		/* (non-Javadoc)
		 * @see java.security.interfaces.RSAPrivateCrtKey#getPrimeExponentP()
		 */
		public BigInteger getPrimeExponentP() {
			return dp;
		}
		/* (non-Javadoc)
		 * @see java.security.interfaces.RSAPrivateCrtKey#getPrimeExponentQ()
		 */
		public BigInteger getPrimeExponentQ() {
			return dq; 
		}
		/* (non-Javadoc)
		 * @see java.security.interfaces.RSAPrivateCrtKey#getPrimeP()
		 */
		public BigInteger getPrimeP() {
			
			return p;
		}
		/* (non-Javadoc)
		 * @see java.security.interfaces.RSAPrivateCrtKey#getPrimeQ()
		 */
		public BigInteger getPrimeQ() {
			return q;
		}
		/* (non-Javadoc)
		 * @see java.security.interfaces.RSAPrivateCrtKey#getPublicExponent()
		 */
		public BigInteger getPublicExponent() {
			
			return e;
		}
		/* (non-Javadoc)
		 * @see java.security.interfaces.RSAPrivateKey#getPrivateExponent()
		 */
		public BigInteger getPrivateExponent() {
			
			return d;
		}
		/* (non-Javadoc)
		 * @see java.security.interfaces.RSAKey#getModulus()
		 */
		public BigInteger getModulus() {
			// TODO Auto-generated method stub
			return n;
		}
		
		
		/* (non-Javadoc)
		 * @see java.security.Key#getAlgorithm()
		 */
		public String getAlgorithm() {
			return "RSA";
		}
		/* (non-Javadoc)
		 * @see java.security.Key#getEncoded()
		 */
		public byte[] getEncoded() {
			return null;
		}
		/* (non-Javadoc)
		 * @see java.security.Key#getFormat()
		 */
		public String getFormat() {
			return null;
		}
	}

	private class X931RSAPublicKey implements RSAPublicKey
	{
		/**
		 * 
		 */
		private static final long serialVersionUID = 1L;
		BigInteger p,q,n,e,d, dp,dq,c;		
		
		/**
		 * @param p the prime p
		 * @param q	prime q
		 * @param n modulus
		 * @param e public exponent
		 * @param d private exponent
		 * @param dp private exponent ( mod p)
		 * @param dq private exponet ( mod q )
		 * @param c the crt coefficient
		 */
		public X931RSAPublicKey(BigInteger p, BigInteger q, BigInteger n,
				BigInteger e, BigInteger d, BigInteger dp, BigInteger dq,
				BigInteger c) {
			super();
			this.p = p;
			this.q = q;
			this.n = n;
			this.e = e;
			this.d = d;
			this.dp = dp;
			this.dq = dq;
			this.c = c;
		}
		
		
		/* (non-Javadoc)
		 * @see java.security.interfaces.RSAPrivateCrtKey#getCrtCoefficient()
		 */
		public BigInteger getCrtCoefficient() {
			return c;
		}
		/* (non-Javadoc)
		 * @see java.security.interfaces.RSAPrivateCrtKey#getPrimeExponentP()
		 */
		public BigInteger getPrimeExponentP() {
			return dp;
		}
		/* (non-Javadoc)
		 * @see java.security.interfaces.RSAPrivateCrtKey#getPrimeExponentQ()
		 */
		public BigInteger getPrimeExponentQ() {
			return dq; 
		}
		/* (non-Javadoc)
		 * @see java.security.interfaces.RSAPrivateCrtKey#getPrimeP()
		 */
		public BigInteger getPrimeP() {
			
			return p;
		}
		/* (non-Javadoc)
		 * @see java.security.interfaces.RSAPrivateCrtKey#getPrimeQ()
		 */
		public BigInteger getPrimeQ() {
			return q;
		}
		/* (non-Javadoc)
		 * @see java.security.interfaces.RSAPrivateCrtKey#getPublicExponent()
		 */
		public BigInteger getPublicExponent() {
			
			return e;
		}
		/* (non-Javadoc)
		 * @see java.security.interfaces.RSAPrivateKey#getPrivateExponent()
		 */
		public BigInteger getPrivateExponent() {
			
			return d;
		}
		/* (non-Javadoc)
		 * @see java.security.interfaces.RSAKey#getModulus()
		 */
		public BigInteger getModulus() {
			// TODO Auto-generated method stub
			return n;
		}
		
		
		/* (non-Javadoc)
		 * @see java.security.Key#getAlgorithm()
		 */
		public String getAlgorithm() {
			return "RSA";
		}
		/* (non-Javadoc)
		 * @see java.security.Key#getEncoded()
		 */
		public byte[] getEncoded() {
			return null;
		}
		/* (non-Javadoc)
		 * @see java.security.Key#getFormat()
		 */
		public String getFormat() {
			return null;
		}
	}

	public static void main(String[] args){
		
		/*
		// This is the test case from the X9.31. Also see unit tests which test out more
		// known test vectors as well as random tests
		byte[] xp1 = { 
					// The first test vector from the x9.31 spec.
					//(byte)0x1A, (byte)0x19, (byte)0x16, (byte)0xdd, (byte)0xb2, (byte)0x9b, (byte)0x4e, (byte)0xb7, 
					//(byte)0xeb, (byte)0x67, (byte)0x32, (byte)0xe1, (byte)0x28
				
					// 2048 bit keys with e=3 from the spec: SAME AS ABOVE
					(byte)0x1A, (byte)0x19, (byte)0x16, (byte)0xdd, (byte)0xb2, (byte)0x9b, (byte)0x4e, (byte)0xb7, 
					(byte)0xeb, (byte)0x67, (byte)0x32, (byte)0xe1, (byte)0x28
				};
	
			
			
		
		byte[] xp2 = {
				//case 1 and 3 from x9.31 spec
				(byte)0x19, (byte)0x2e, (byte)0x8a, (byte)0xac, (byte)0x41, (byte)0xc5, (byte)0x76, (byte)0xc8, 
				(byte)0x22, (byte)0xd9, (byte)0x3e, (byte)0xa4, (byte)0x33};
					
			
			

		byte[] xp = {	
					//Case 1 From x9.31 spec
					// 
					(byte)0xd8, (byte)0xcd, (byte)0x81, (byte)0xf0, (byte)0x35, (byte)0xec, (byte)0x57, (byte)0xef, 
					(byte)0xe8, (byte)0x22, (byte)0x95, (byte)0x51, (byte)0x49, (byte)0xd3, (byte)0xbf, (byte)0xf7, 
					(byte)0x0c, (byte)0x53, (byte)0x52, (byte)0x0d,	(byte)0x76, (byte)0x9d, (byte)0x6d, (byte)0x76, 
					(byte)0x64, (byte)0x6c, (byte)0x7a, (byte)0x79, (byte)0x2e, (byte)0x16, (byte)0xeb, (byte)0xd8, 
					(byte)0x9f, (byte)0xe6, (byte)0xfc, (byte)0x5b, (byte)0x60, (byte)0x5A, (byte)0x64, (byte)0x93,
					(byte)0x39, (byte)0xdf, (byte)0xc9, (byte)0x25, (byte)0xa8, (byte)0x6a, (byte)0x4c, (byte)0x6d, 
					(byte)0x15, (byte)0x0b, (byte)0x71, (byte)0xb9, (byte)0xee, (byte)0xa0, (byte)0x2d, (byte)0x68, 
					(byte)0x88, (byte)0x5f, (byte)0x50, (byte)0x09, (byte)0xb9, (byte)0x8b, (byte)0xd9, (byte)0x84,
								
				};
			
			
		byte[] xq1 = { 
					(byte)0x1a, (byte)0x5c, (byte)0xf7, (byte)0x2e, (byte)0xe7, (byte)0x70, (byte)0xde, (byte)0x50, 
					(byte)0xcb, (byte)0x09, (byte)0xac, (byte)0xce, (byte)0xa9
				};
				
			
			
			
		byte[] xq2 ={  
					(byte)0x13, (byte)0x4e, (byte)0x4c, (byte)0xaa, (byte)0x16, (byte)0xd2, (byte)0x35, (byte)0x0a,
					(byte)0x21, (byte)0xd7, (byte)0x75, (byte)0xc4, (byte)0x9f
					};
			
		byte[] xq = {	
				(byte)0xcc, (byte)0x10, (byte)0x92, (byte)0x49, (byte)0x5d, (byte)0x86, (byte)0x7e, (byte)0x64, 
				(byte)0x06, (byte)0x5d, (byte)0xee, (byte)0x3e, (byte)0x79, (byte)0x55, (byte)0xf2, (byte)0xeb, 
				(byte)0xc7, (byte)0xd4, (byte)0x7a, (byte)0x2d, (byte)0x7c, (byte)0x99, (byte)0x53, (byte)0x38,
				(byte)0x8f, (byte)0x97, (byte)0xdd, (byte)0xdc, (byte)0x3e, (byte)0x1c, (byte)0xa1, (byte)0x9c, 
				(byte)0x35, (byte)0xca, (byte)0x65, (byte)0x9e, (byte)0xdc, (byte)0x2f, (byte)0xc3, (byte)0x25,
				(byte)0x6d, (byte)0x29, (byte)0xc2, (byte)0x62, (byte)0x74, (byte)0x79, (byte)0xc0, (byte)0x86, 
				(byte)0xa6, (byte)0x99, (byte)0xa4, (byte)0x9c, (byte)0x4c, (byte)0x9c, (byte)0xee, (byte)0x7e, 
				(byte)0xf7, (byte)0xbd, (byte)0x1b, (byte)0x34, (byte)0x32, (byte)0x1d, (byte)0xe3, (byte)0x4a
			};
		
		/*
		byte[] xq = {	
				(byte)0xd2, (byte)0xdd, (byte)0xb9, (byte)0x27, (byte)0x57, (byte)0x60, (byte)0xc8, (byte)0xaa, 
				(byte)0xcf, (byte)0xc4, (byte)0x6a, (byte)0xd6, (byte)0xc3, (byte)0xd7, (byte)0x8f, (byte)0x07, 
				(byte)0x88, (byte)0xd1, (byte)0x87, (byte)0x33, (byte)0xd6, (byte)0xb0, (byte)0x24, (byte)0x04,
				(byte)0x3e, (byte)0x04, (byte)0xae, (byte)0x70, (byte)0xe7, (byte)0x17, (byte)0x94, (byte)0xc0, 
				(byte)0x23, (byte)0x1a, (byte)0xcb, (byte)0xad, (byte)0x28, (byte)0xfc, (byte)0x0f, (byte)0xc1,
				(byte)0x37, (byte)0xd8, (byte)0xcf, (byte)0xa8, (byte)0x48, (byte)0x74, (byte)0x75, (byte)0x94, 
				(byte)0xf6, (byte)0xee, (byte)0x8d, (byte)0x5d, (byte)0x31, (byte)0x8c, (byte)0x69, (byte)0x62, 
				(byte)0xa7, (byte)0x0a, (byte)0xb4, (byte)0x71, (byte)0xba, (byte)0x43, (byte)0x9b, (byte)0x12,
				(byte)0x27, (byte)0xee, (byte)0xea, (byte)0xfd, (byte)0xea, (byte)0xb7, (byte)0x5a, (byte)0xc1, 
				(byte)0x19, (byte)0xd7, (byte)0x21, (byte)0x78, (byte)0xba, (byte)0xf4, (byte)0x68, (byte)0x5b, 
				(byte)0x93, (byte)0x1d, (byte)0xf1, (byte)0xc8, (byte)0x7e, (byte)0x23, (byte)0x0f, (byte)0x33,
				(byte)0xeb, (byte)0x12, (byte)0x0c, (byte)0xdf, (byte)0xe7, (byte)0xa6, (byte)0x18, (byte)0xa4, 
				(byte)0x8c, (byte)0x63, (byte)0xb2, (byte)0x06, (byte)0x9e, (byte)0x62, (byte)0x61, (byte)0x01,
				(byte)0x52, (byte)0x75, (byte)0x2c, (byte)0x11, (byte)0x39, (byte)0x90, (byte)0xa2, (byte)0x1a, 
				(byte)0x55, (byte)0x1f, (byte)0xe1, (byte)0x7f, (byte)0x6f, (byte)0x71, (byte)0x1c, (byte)0xe8, 
				(byte)0x12, (byte)0x19, (byte)0x7e, (byte)0x1c, (byte)0x9c, (byte)0xe5, (byte)0xde, (byte)0xa6
			};
		*/
		byte[] xp1 = { 
				(byte)0x1A, (byte)0x19, (byte)0x16, (byte)0xdd, (byte)0xb2, (byte)0x9b, (byte)0x4e, (byte)0xb7, 
				(byte)0xeb, (byte)0x67, (byte)0x32, (byte)0xe1, (byte)0x28
			};
	
	byte[] xp2 = {
			//case 1 and 3 from x9.31 spec
			(byte)0x19, (byte)0x2e, (byte)0x8a, (byte)0xac, (byte)0x41, (byte)0xc5, (byte)0x76, (byte)0xc8, 
			(byte)0x22, (byte)0xd9, (byte)0x3e, (byte)0xa4, (byte)0x33};
				
	byte[] xp = {	
			//Case 3 From x9.31 spec
			// 
			(byte)0xe5, (byte)0x32, (byte)0xcf, (byte)0x6e, (byte)0x17, (byte)0xe1, (byte)0x92, (byte)0xb6, 
			(byte)0x94, (byte)0xe1, (byte)0x31, (byte)0x3a, (byte)0x3c, (byte)0xe0, (byte)0x43, (byte)0x53, 
			(byte)0x4c, (byte)0xd2, (byte)0x4e, (byte)0x26, (byte)0x4e, (byte)0x77, (byte)0x3f, (byte)0x7b,
			(byte)0xde, (byte)0x8d, (byte)0x9a, (byte)0x19, (byte)0x0e, (byte)0xf1, (byte)0xa5, (byte)0xab, 
			(byte)0x84, (byte)0x1c, (byte)0x4d, (byte)0x39, (byte)0x32, (byte)0xa8, (byte)0x5b, (byte)0x48,
			(byte)0xd4, (byte)0x28, (byte)0x87, (byte)0x5d, (byte)0xd7, (byte)0x10, (byte)0x17, (byte)0x64, 
			(byte)0x89, (byte)0xe6, (byte)0x97, (byte)0x3d, (byte)0xcb, (byte)0xb9, (byte)0xdc, (byte)0x37, 
			(byte)0x94, (byte)0x32, (byte)0xc2, (byte)0x6b, (byte)0x33, (byte)0xaa, (byte)0x6f, (byte)0xb3,
			(byte)0xb1, (byte)0x9b, (byte)0x03, (byte)0xe0, (byte)0x6b, (byte)0xd8, (byte)0xaf, (byte)0xdf, 
			(byte)0xd0, (byte)0x45, (byte)0x2a, (byte)0xa2, (byte)0x93, (byte)0xa7, (byte)0x7f, (byte)0xee, 
			(byte)0x18, (byte)0xa6, (byte)0xa4, (byte)0x1a, (byte)0xed, (byte)0x3a, (byte)0xda, (byte)0xec,
			(byte)0xc8, (byte)0xd8, (byte)0xf4, (byte)0xf2, (byte)0x5c, (byte)0x2d, (byte)0xd7, (byte)0xb1, 
			(byte)0x3f, (byte)0xcd, (byte)0xf8, (byte)0xb6, (byte)0xc0, (byte)0xb4, (byte)0xc9, (byte)0x26,
			(byte)0x73, (byte)0xbf, (byte)0x92, (byte)0x4b, (byte)0x50, (byte)0xd4, (byte)0x98, (byte)0xda, 
			(byte)0x2a, (byte)0x16, (byte)0xe3, (byte)0x73, (byte)0xdd, (byte)0x40, (byte)0x5a, (byte)0xd1, 
			(byte)0x10, (byte)0xbe, (byte)0xa0, (byte)0x13, (byte)0x73, (byte)0x17, (byte)0xb3, (byte)0x08
			
			
		};
	
	byte[] xq1 = { 
			(byte)0x1f, (byte)0x26, (byte)0x54, (byte)0xce, (byte)0x23, (byte)0xe5, (byte)0xf7, (byte)0x77, 
			(byte)0x0f, (byte)0x87, (byte)0x28, (byte)0x67, (byte)0x0c
		};
			
	
	
	byte[] xq2 ={  
				(byte)0x1d, (byte)0x86, (byte)0x7e, (byte)0x64, (byte)0x06, (byte)0x5d, (byte)0xee, (byte)0x3e,
				(byte)0x79, (byte)0x55, (byte)0xf2, (byte)0xeb, (byte)0xc7
				};
		
	
	byte[] xq = {	
			(byte)0xd2, (byte)0xdd, (byte)0xb9, (byte)0x27, (byte)0x57, (byte)0x60, (byte)0xc8, (byte)0xaa, 
			(byte)0xcf, (byte)0xc4, (byte)0x6a, (byte)0xd6, (byte)0xc3, (byte)0xd7, (byte)0x8f, (byte)0x07, 
			(byte)0x88, (byte)0xd1, (byte)0x87, (byte)0x33, (byte)0xd6, (byte)0xb0, (byte)0x24, (byte)0x04,
			(byte)0x3e, (byte)0x04, (byte)0xae, (byte)0x70, (byte)0xe7, (byte)0x17, (byte)0x94, (byte)0xc0, 
			(byte)0x23, (byte)0x1a, (byte)0xcb, (byte)0xad, (byte)0x28, (byte)0xfc, (byte)0x0f, (byte)0xc1,
			(byte)0x37, (byte)0xd8, (byte)0xcf, (byte)0xa8, (byte)0x48, (byte)0x74, (byte)0x75, (byte)0x94, 
			(byte)0xf6, (byte)0xee, (byte)0x8d, (byte)0x5d, (byte)0x31, (byte)0x8c, (byte)0x69, (byte)0x62, 
			(byte)0xa7, (byte)0x0a, (byte)0xb4, (byte)0x71, (byte)0xba, (byte)0x43, (byte)0x9b, (byte)0x12,
			(byte)0x27, (byte)0xee, (byte)0xea, (byte)0xfd, (byte)0xea, (byte)0xb7, (byte)0x5a, (byte)0xc1, 
			(byte)0x19, (byte)0xd7, (byte)0x21, (byte)0x78, (byte)0xba, (byte)0xf4, (byte)0x68, (byte)0x5b, 
			(byte)0x93, (byte)0x1d, (byte)0xf1, (byte)0xc8, (byte)0x7e, (byte)0x23, (byte)0x0f, (byte)0x33,
			(byte)0xeb, (byte)0x12, (byte)0x0c, (byte)0xdf, (byte)0xe7, (byte)0xa6, (byte)0x18, (byte)0xa4, 
			(byte)0x8c, (byte)0x63, (byte)0xb2, (byte)0x06, (byte)0x9e, (byte)0x62, (byte)0x61, (byte)0x01,
			(byte)0x52, (byte)0x75, (byte)0x2c, (byte)0x11, (byte)0x39, (byte)0x90, (byte)0xa2, (byte)0x1a, 
			(byte)0x55, (byte)0x1f, (byte)0xe1, (byte)0x7f, (byte)0x6f, (byte)0x71, (byte)0x1c, (byte)0xe8, 
			(byte)0x12, (byte)0x19, (byte)0x7e, (byte)0x1c, (byte)0x9c, (byte)0xe5, (byte)0xde, (byte)0xa6
		};
		
		RSAPrivateCrtKey mykey = null;
		
		try{
			X931KeyGenerator mykeygen = new X931KeyGenerator( xp1 ,xp2, xq1, xq2, xp, xq);
			mykey = (RSAPrivateCrtKey)mykeygen.GenerateKeyPair( new RSAKeyGenParameterSpec( 2048, RSAKeyGenParameterSpec.F0)).getPrivate();
		} catch ( UnsupportedOptionException e1){
			System.out.println(" option is not supported ");
		}
		catch ( InvalidRandomParameterException e2){
			System.out.println(" input random parameter is not valid: reason code is " +e2.getStatusCode());
		}		
	}
}
