/**
 * Copyright (c) 2006 Novell, Inc.
 * All Rights Reserved.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; version 2.1 of the license.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; if not, contact Novell, Inc.
 *
 * To contact Novell about this file by physical or electronic mail,
 * you may find current contact information at www.novell.com
 */

package org.eclipse.higgins.sts.informationcardgenerator;

import java.awt.Image;
import java.awt.Toolkit;
import java.io.*;
import java.net.URI;
import java.util.Date;
import java.text.SimpleDateFormat;
import java.util.TimeZone;
import java.awt.image.*;
import java.awt.Graphics;
import java.security.KeyStore;
import java.security.PrivateKey;
import java.security.cert.X509Certificate;
import javax.imageio.*;
import javax.swing.ImageIcon;
import javax.xml.parsers.*;
import org.apache.xml.security.signature.ObjectContainer;
import org.apache.xml.security.signature.XMLSignature;
import org.apache.xml.security.transforms.Transforms;
import org.apache.xml.security.utils.Constants;

/******************************************************************************
* Desc:
******************************************************************************/
public class InfoCard
{
	private X509Certificate				m_certificateSigner = null;
	private X509Certificate				m_certificateMEX = null;
	private String							m_strCertificateMEX = null;
	private String							m_strCertificateSigner = null;
	private PrivateKey 					m_privateKeySigner = null;
	private DocumentBuilder 			m_documentBuilder = null;
	private org.w3c.dom.Document		m_docCard = null;
	private org.w3c.dom.Element		m_elemCard = null;
	private org.w3c.dom.Element		m_elemCardRef = null;
	private org.w3c.dom.Element		m_elemCardTokenServiceList = null;
	private org.w3c.dom.Element		m_elemCardSupportedTokenTypeList = null;
	private org.w3c.dom.Element		m_elemCardSupportedClaimTypeList = null;
	
	private static final String		strCardIssuerTag = "Issuer";
	private static final String		strTimeIssuedTag = "TimeIssued";
	private static final String		strTimeExpiresTag = "TimeExpires";
	private static final String		strPrivacyNoticeTag = "PrivacyNotice";
	private static final String		strRequireAppliesToTag = "RequireAppliesTo";
	private static final String		strCardRefTag = "InformationCardReference";
	private static final String		strCardNameTag = "CardName";
	private static final String		strCardVersionTag = "CardVersion";
	private static final String		strCardIdTag = "CardId";
	private static final String		strCardImageTag = "CardImage";
	private static final String		strUserCredentialTag = "UserCredential";
	private static final String		strTokenTypeTag = "TokenType";
	
	/***************************************************************************
	* Desc:
	***************************************************************************/
	private void _init(
		PrivateKey					privateKeySigner,
		X509Certificate			certificateSigner,
		X509Certificate			certificateMEX) throws InfoCardException
	{
		DocumentBuilderFactory 	documentBuilderFactory;

		documentBuilderFactory = javax.xml.parsers.DocumentBuilderFactory.newInstance();
		documentBuilderFactory.setNamespaceAware( true);

		try
		{
			m_documentBuilder = documentBuilderFactory.newDocumentBuilder();
		}
		catch( Exception e)
		{
			throw new InfoCardException( "Unable to allocate a document builder");
		}
		
		m_docCard = m_documentBuilder.newDocument();
		
		// Crypto
		
		try
		{
			org.apache.xml.security.Init.init();
			org.apache.xml.security.utils.Constants.setSignatureSpecNSprefix( "");
		}
		catch( Exception e)
		{
			throw new InfoCardException( "Unable to initialize security subsystem");
		}
		
		m_privateKeySigner = privateKeySigner;
		m_certificateSigner = certificateSigner;
		m_certificateMEX = certificateMEX;
		
		try
		{		
			m_strCertificateSigner = org.apache.xml.security.utils.Base64.encode( 
											m_certificateSigner.getEncoded(), 0);
											
			m_strCertificateSigner = stripNewLinesFromString( 
											m_strCertificateSigner);
		}
		catch( Exception e)
		{
			throw new InfoCardException( "Unable to encode certificate");
		}
								
		try
		{		
			m_strCertificateMEX = org.apache.xml.security.utils.Base64.encode( 
											m_certificateMEX.getEncoded(), 0);
			m_strCertificateMEX = stripNewLinesFromString( 
											m_strCertificateMEX); 
		}
		catch( Exception e)
		{
			throw new InfoCardException( "Unable to encode certificate");
		}
		
		// Build an empty card
		
		m_elemCard = m_docCard.createElement( "ic:InformationCard");
		m_elemCard.setAttribute( "xmlns:ic", 
			"http://schemas.xmlsoap.org/ws/2005/05/identity");
		m_elemCard.setAttribute( "xmlns:ds", 
			"http://www.w3.org/2000/09/xmldsig#");
		m_elemCard.setAttribute( "xmlns:wsse", 
			"http://docs.oasis-open.org/wss/2004/01/oasis-200401-wss-wssecurity-secext-1.0.xsd");
		m_elemCard.setAttribute( "xml:lang", "en-us");
		
		// Card reference
		
		m_elemCardRef = m_docCard.createElement( "ic:" + strCardRefTag);
		m_elemCard.appendChild( m_elemCardRef);
		
		// Card name
		
		m_elemCard.appendChild( m_docCard.createElement( "ic:" + strCardNameTag));
		
		// Card image
		
		m_elemCard.appendChild( m_docCard.createElement( "ic:" + strCardImageTag));
		
		// Card issuer
		
		m_elemCard.appendChild( m_docCard.createElement( "ic:" + strCardIssuerTag));
		
		// Time issued / expires
		
		m_elemCard.appendChild( m_docCard.createElement( "ic:" + strTimeIssuedTag));
		m_elemCard.appendChild( m_docCard.createElement( "ic:" + strTimeExpiresTag));
		
		// Token service list
		
		m_elemCardTokenServiceList = m_docCard.createElement( 
			"ic:TokenServiceList");
		m_elemCard.appendChild( m_elemCardTokenServiceList);
		
		// Supported token list
		
		m_elemCardSupportedTokenTypeList = m_docCard.createElement( 
			"ic:SupportedTokenTypeList");
		m_elemCard.appendChild( m_elemCardSupportedTokenTypeList);
		
		// Supported claim list
		
		m_elemCardSupportedClaimTypeList = m_docCard.createElement( 
			"ic:SupportedClaimTypeList");
		m_elemCard.appendChild( m_elemCardSupportedClaimTypeList);
	}
	
	/***************************************************************************
	* Desc:
	***************************************************************************/
	public InfoCard(
		PrivateKey					privateKeySigner,
		X509Certificate			certificateSigner,
		X509Certificate			certificateMEX) throws InfoCardException
	{
		_init( privateKeySigner, certificateSigner, certificateMEX);
	}
	
	/***************************************************************************
	* Desc:
	***************************************************************************/
	public InfoCard(
		String				strSignerKeystoreType,
		String				strSignerKeystoreFile,
		String				strSignerKeystorePassword,
		String				strSignerCertificateAlias,
		String				strSignerKeyAlias,
		String				strSignerKeyPassword,
		String				strMEXKeystoreType,
		String				strMEXKeystoreFile,
		String				strMEXKeystorePassword,
		String				strMEXCertificateAlias) throws InfoCardException
	{
		KeyStore				ksSigner;
		KeyStore				ksMEX;
		PrivateKey			privateKeySigner;
		X509Certificate	certificateSigner;
		X509Certificate	certificateMEX;

		ksSigner = loadKeyStore( strSignerKeystoreType, 
							strSignerKeystoreFile, strSignerKeystorePassword);
							
		try
		{
			privateKeySigner = (PrivateKey)ksSigner.getKey( strSignerKeyAlias, 
											strSignerKeyPassword.toCharArray());
		}
		catch( Exception e)
		{
			throw new InfoCardException( "Unable to retrieve private key");
		}

		try
		{		
			certificateSigner = (X509Certificate)ksSigner.getCertificate( 
											strSignerCertificateAlias);
		}
		catch( Exception e)
		{
			throw new InfoCardException( "Unable to retrieve certificate");
		}
								
		ksMEX = loadKeyStore( strMEXKeystoreType, 
						strMEXKeystoreFile, strMEXKeystorePassword);

		try
		{		
			certificateMEX = (X509Certificate)ksMEX.getCertificate( 
							strMEXCertificateAlias);
		}
		catch( Exception e)
		{
			throw new InfoCardException( "Unable to retrieve certificate");
		}
		
		_init( privateKeySigner, certificateSigner, certificateMEX);
	}
	
	/***************************************************************************
	* Desc:
	***************************************************************************/
	static public void createInfoCard(
		String				strSignerKeystoreType,
		String				strSignerKeystoreFile,
		String				strSignerKeystorePassword,
		String				strSignerCertificateAlias,
		String				strSignerKeyAlias,
		String				strSignerKeyPassword,
		String				strMEXKeystoreType,
		String				strMEXKeystoreFile,
		String				strMEXKeystorePassword,
		String				strMEXCertificateAlias,
		String				strCardId,
		String				strCardVersion,
		String				strCardName,
		String				strCardExpiration,
		String				strCardImageFileName,
		String				strIssuerURI,
		String				strSTSEndpointReference,
		String				strMEXEndpointReference,
		String				strUsername,
		String[]				strClaims,
		String				strCardFileName) throws InfoCardException
	{
		InfoCard					infoCard;
		org.w3c.dom.Element	elemUserCredential;
		
		infoCard = new InfoCard( 
				strSignerKeystoreType, strSignerKeystoreFile, 
				strSignerKeystorePassword, strSignerCertificateAlias,
				strSignerKeyAlias, strSignerKeyPassword, 
				strMEXKeystoreType, strMEXKeystoreFile,
				strMEXKeystorePassword, strMEXCertificateAlias);
				
		infoCard.setCardReference( strCardId, strCardVersion);
		infoCard.setIssuerURI( strIssuerURI);
		infoCard.setCardName( strCardName);
		infoCard.setCardExpiration( new Integer( strCardExpiration).intValue());
		
		elemUserCredential = infoCard.buildUsernamePasswordCredential( 
			strUsername, null);				
		infoCard.addTokenServiceRef( strSTSEndpointReference, 
			strMEXEndpointReference, elemUserCredential);
			
		if( strCardImageFileName != null)
		{
			infoCard.setCardImage( strCardImageFileName);
		}
		
		if( strClaims != null)
		{
			for(int idx= 0; idx < strClaims.length; idx++)
			{				
				infoCard.addSupportedClaim( strClaims[idx], strClaims[idx], strClaims[idx]);
			}
		}
		
		infoCard.toFile( strCardFileName);
	}
	
	/***************************************************************************
	* Desc:
	***************************************************************************/
	public void setCardReference(
		String							strCardId,
		String							strCardVersion)
	{
		org.w3c.dom.Element			elemCardRef;
		org.w3c.dom.Element			elemCardId;
		org.w3c.dom.Element			elemCardVer;
		org.w3c.dom.Element			elemCardName;
		
		elemCardRef = m_docCard.createElement( "ic:" + strCardRefTag);

		elemCardId = m_docCard.createElement( "ic:" + strCardIdTag);
		elemCardId.appendChild( m_docCard.createTextNode( strCardId));
		elemCardRef.appendChild( elemCardId);
		
		elemCardVer = m_docCard.createElement( "ic:" + strCardVersionTag);
		elemCardVer.appendChild( m_docCard.createTextNode( strCardVersion));
		elemCardRef.appendChild( elemCardVer);
		
		if( m_elemCardRef != null)
		{
			m_elemCard.replaceChild( elemCardRef, m_elemCardRef);
			m_elemCardRef = elemCardRef;
		}
		else
		{
			m_elemCardRef = elemCardRef;
			m_elemCard.appendChild( m_elemCardRef);
		}
	}
	
	/***************************************************************************
	* Desc:
	***************************************************************************/
	public void setCardName(
		String		strCardName)
	{
		updateChildElement( m_elemCard, "ic", strCardNameTag, strCardName);
	}
		
	/***************************************************************************
	* Desc:
	***************************************************************************/
	public void setCardExpiration(
		long			lExpireGMTTime)
	{
		if( lExpireGMTTime == 0)
		{
			updateChildElement( m_elemCard, "ic", strTimeExpiresTag, 
				"9999-12-31T23:59:59.9999999Z");
		}
		else
		{
			updateChildElement( m_elemCard, "ic", strTimeExpiresTag, 
				getDateString( lExpireGMTTime));
		}
	}
		
	/***************************************************************************
	* Desc:
	***************************************************************************/
	public void setCardExpiration(
		String		sExpireTime)
	{
		if( sExpireTime == null || sExpireTime == "")
		{
			updateChildElement( m_elemCard, "ic", strTimeExpiresTag, 
				"9999-12-31T23:59:59.9999999Z");
		}
		else
		{
			updateChildElement( m_elemCard, "ic", strTimeExpiresTag, sExpireTime); 
		}
	}
		
	/***************************************************************************
	* Desc:
	***************************************************************************/
	public void addTokenServiceRef(
		String						stsURI,
		String						mexURI,
		org.w3c.dom.Element		elemUserCredential) throws InfoCardException
	{
		org.w3c.dom.Element		elemTokenService;
		org.w3c.dom.Element		elemEndpointRef;
		org.w3c.dom.Element		elemAddress;
		org.w3c.dom.Element		elemWsaMetadata;
		org.w3c.dom.Element		elemMexMetadata;
		org.w3c.dom.Element		elemMetadataSection;
		org.w3c.dom.Element		elemMetadataReference;
		org.w3c.dom.Element		elemIdentity;
		org.w3c.dom.Element		elemMexAddress;
		org.w3c.dom.Element		elemKeyInfo;
		org.w3c.dom.Element		elemX509Data;
		org.w3c.dom.Element		elemX509Certificate;
		
		elemTokenService = m_docCard.createElement( "ic:TokenService");
		m_elemCardTokenServiceList.appendChild( elemTokenService);
		
		elemEndpointRef = m_docCard.createElement( "wsa:EndpointReference");
		elemEndpointRef.setAttribute( "xmlns:wsa", 
			"http://www.w3.org/2005/08/addressing");
		elemTokenService.appendChild( elemEndpointRef);
		
		elemAddress = m_docCard.createElement( "wsa:Address");
		elemAddress.appendChild( m_docCard.createTextNode( stsURI)); 
		elemEndpointRef.appendChild( elemAddress);
		
		elemWsaMetadata = m_docCard.createElement( "wsa:Metadata");
		elemEndpointRef.appendChild( elemWsaMetadata);
		
		elemMexMetadata = m_docCard.createElement( "mex:Metadata");
		elemMexMetadata.setAttribute( "xmlns:mex",
			"http://schemas.xmlsoap.org/ws/2004/09/mex");
		elemWsaMetadata.appendChild( elemMexMetadata);
		
		elemMetadataSection = m_docCard.createElement( "mex:MetadataSection");
		elemMexMetadata.appendChild( elemMetadataSection);
		
		elemMetadataReference = m_docCard.createElement( "mex:MetadataReference");
		elemMetadataSection.appendChild( elemMetadataReference);
		
		elemMexAddress = m_docCard.createElement( "wsa:Address");
		elemMexAddress.appendChild( m_docCard.createTextNode( mexURI));
		elemMetadataReference.appendChild( elemMexAddress);
		
		elemIdentity = m_docCard.createElement( "wsai:Identity");
		elemIdentity.setAttribute( "xmlns:wsai", 
			"http://schemas.xmlsoap.org/ws/2006/02/addressingidentity");
		elemEndpointRef.appendChild( elemIdentity);
		
		elemKeyInfo = m_docCard.createElement( "dsig:KeyInfo");
		elemKeyInfo.setAttribute( "xmlns:dsig", 
			"http://www.w3.org/2000/09/xmldsig#");
		elemIdentity.appendChild( elemKeyInfo);
		
		elemX509Data = m_docCard.createElement( "dsig:X509Data");
		elemKeyInfo.appendChild( elemX509Data);

		elemX509Certificate = m_docCard.createElement( "dsig:X509Certificate");
		elemX509Certificate.appendChild( m_docCard.createTextNode( m_strCertificateMEX));
		elemX509Data.appendChild( elemX509Certificate);
		
		elemTokenService.appendChild( elemUserCredential);
	}
	
	/***************************************************************************
	* Desc:
	***************************************************************************/
	public void addSupportedClaim(
		String							strClaimURI,
		String							strClaimDisplayTag,
		String							strClaimDescription)
	{
		org.w3c.dom.Element			elemClaimType;
		org.w3c.dom.Element			elemDisplayTag;
		org.w3c.dom.Element			elemDescription;
		
		elemClaimType = m_docCard.createElement( "ic:SupportedClaimType");
		elemClaimType.setAttribute( "Uri", strClaimURI);
		m_elemCardSupportedClaimTypeList.appendChild( elemClaimType);
		
		if( strClaimDisplayTag != null)
		{
			elemDisplayTag = m_docCard.createElement( "ic:DisplayTag");
			elemDisplayTag.appendChild( m_docCard.createTextNode( strClaimDisplayTag));
			elemClaimType.appendChild( elemDisplayTag);
		}

		if( strClaimDescription != null)
		{
			elemDescription = m_docCard.createElement( "ic:Description");
			elemDescription.appendChild( m_docCard.createTextNode( strClaimDescription));
			elemClaimType.appendChild( elemDescription);
		}
	}
	
	/***************************************************************************
	* Desc:
	***************************************************************************/
	public void addSupportedTokenType(
		String							strTokenTypeURI)
	{
		org.w3c.dom.Element			elemTokenType;
		
		elemTokenType = m_docCard.createElement( "trust:" + strTokenTypeTag);
		elemTokenType.setAttribute( "xmlns:trust", 
					"http://schemas.xmlsoap.org/ws/2005/02/trust");
		elemTokenType.appendChild( m_docCard.createTextNode( strTokenTypeURI));
		m_elemCardSupportedTokenTypeList.appendChild( elemTokenType);
	}
	
	/***************************************************************************
	* Desc:
	***************************************************************************/
	public org.w3c.dom.Element buildUsernamePasswordCredential(
		String							strUserName,
		String							strHint)
	{
		org.w3c.dom.Element			elemUserCredential;
		org.w3c.dom.Element			elemCredentialHint;
		org.w3c.dom.Element			elemUsernamePasswordCredential;
		org.w3c.dom.Element			elemUsername;
		
		if( strHint == null)
		{
			strHint = "Enter your username and password";
		}
		
		elemUserCredential = m_docCard.createElement( "ic:" + strUserCredentialTag);
		
		elemCredentialHint = m_docCard.createElement( "ic:DisplayCredentialHint");
		elemCredentialHint.appendChild( m_docCard.createTextNode( strHint));
		elemUserCredential.appendChild( elemCredentialHint);
		
		elemUsernamePasswordCredential = m_docCard.createElement( 
			"ic:UsernamePasswordCredential");
		elemUserCredential.appendChild( elemUsernamePasswordCredential);
		
		elemUsername = m_docCard.createElement( "ic:Username");
		elemUsername.appendChild( m_docCard.createTextNode( strUserName));
		elemUsernamePasswordCredential.appendChild( elemUsername);
		
		return( elemUserCredential);
	}
	
	/***************************************************************************
	* Desc:
	***************************************************************************/
	public org.w3c.dom.Element buildSelfIssuedCredential(
		String							strPPID)
	{
		org.w3c.dom.Element			elemUserCredential;
		org.w3c.dom.Element			elemSelfIssuedCredential;
		org.w3c.dom.Element			elemPrivatePersonalIdentifier;
		
		elemUserCredential = m_docCard.createElement( "ic:" + strUserCredentialTag);
		
		elemSelfIssuedCredential = m_docCard.createElement( "ic:SelfIssuedCredential");
		elemUserCredential.appendChild( elemSelfIssuedCredential);
		elemPrivatePersonalIdentifier = m_docCard.createElement( "ic:PrivatePersonalIdentifier");
		elemPrivatePersonalIdentifier.appendChild( m_docCard.createTextNode( strPPID));
		elemSelfIssuedCredential.appendChild( elemPrivatePersonalIdentifier);
		
		return( elemUserCredential);
	}
	
	/***************************************************************************
	* Desc:
	***************************************************************************/
	public org.w3c.dom.Element buildX509CertificateCredential(
		String							strCertThumbprint,
		String							strHint)
	{
		org.w3c.dom.Element			elemUserCredential;
		org.w3c.dom.Element			elemCredentialHint;
		org.w3c.dom.Element			elemX509V3Credential;
		org.w3c.dom.Element			elemX509Data;
		org.w3c.dom.Element			elemKeyIdentifier;
		
		elemUserCredential = m_docCard.createElement( "ic:" + strUserCredentialTag);
		
		if (strHint != null && !strHint.trim().equals( ""))
		{
			elemCredentialHint = m_docCard.createElement( "ic:DisplayCredentialHint");
			elemCredentialHint.appendChild( m_docCard.createTextNode( strHint));
			elemUserCredential.appendChild( elemCredentialHint);
		}
		
		elemX509V3Credential = m_docCard.createElement( "ic:X509V3Credential");
		elemUserCredential.appendChild( elemX509V3Credential);
		
		elemX509Data = m_docCard.createElement( "ds:X509Data");
		elemX509V3Credential.appendChild( elemX509Data);
		
		elemKeyIdentifier = m_docCard.createElement( "wsse:KeyIdentifier");
		elemX509Data.appendChild( elemKeyIdentifier);
		
		elemKeyIdentifier.setAttribute( "EncodingType", "http://docs.oasis-open.org/wss/2004/01/oasis200401-wss-soap-message-security-1.0#Base64Binary");
//		elemKeyIdentifier.setAttribute( "ValueType", "http://docs.oasis-open.org/wss/oasis-wss-soap-message-security-1.1#ThumbPrintSHA1");
		elemKeyIdentifier.setAttribute( "ValueType", "http://docs.oasis-open.org/wss/2004/xx/oasis-2004xx-wss-soap-message-security-1.1#ThumbprintSHA1");
		
		elemKeyIdentifier.appendChild( m_docCard.createTextNode( strCertThumbprint));
		
		return( elemUserCredential);
	}
	
	/***************************************************************************
	* Desc:
	***************************************************************************/
	public void setCardImage(
		String							strImageFileName) throws InfoCardException
	{
		try
		{
			Toolkit						toolkit;
			Image							tmpImage;
			ByteArrayOutputStream	imageBufferStream;
			BufferedImage 				bi;
			Graphics 					bg;
			
			toolkit = Toolkit.getDefaultToolkit();
			tmpImage = toolkit.getImage( strImageFileName);
			tmpImage = new ImageIcon( tmpImage).getImage();
			
			setCardImage( tmpImage);
		}
		catch( Exception e)
		{
			throw new InfoCardException( "Unable to render card image");
		}
	}
	
	/***************************************************************************
	* Desc:
	***************************************************************************/
	public void setCardImage(
		Image								image) throws InfoCardException
	{
		org.w3c.dom.Element			elemCardImage;
		org.w3c.dom.Element			elemOldCardImage;
		int								iCardImageBytes;
		byte[]							byteCardImage;
		String							strCardImage;
		
		try
		{
			Toolkit						toolkit;
			Image							cardPicture;
			ByteArrayOutputStream	imageBufferStream;
			BufferedImage 				bi;
			Graphics 					bg;
			
			toolkit = Toolkit.getDefaultToolkit();
			cardPicture = image.getScaledInstance( 200, 150, Image.SCALE_DEFAULT);
			cardPicture = new ImageIcon( cardPicture).getImage();
			imageBufferStream = new ByteArrayOutputStream();
			
			bi = new BufferedImage( cardPicture.getWidth( null), 
						cardPicture.getHeight( null), BufferedImage.TYPE_INT_RGB);			
			bg = bi.getGraphics();
 
			bg.drawImage( cardPicture, 0, 0, null);
			bg.dispose();
			
			ImageIO.write( bi, "PNG", imageBufferStream);
			strCardImage = org.apache.xml.security.utils.Base64.encode( 
									imageBufferStream.toByteArray(), 0);
		}
		catch( Exception e)
		{
			throw new InfoCardException( "Unable to render card image");
		}
		
		try
		{
			elemCardImage = m_docCard.createElement( "ic:" + strCardImageTag);
			elemCardImage.setAttribute( "MimeType", "image/png");
			elemCardImage.appendChild( m_docCard.createTextNode( strCardImage));
			
			elemOldCardImage = getFirstChildElement( m_elemCard, 
											"ic", strCardImageTag);
			if( elemOldCardImage != null)
			{
				m_elemCard.replaceChild( elemCardImage, elemOldCardImage);
			}
			else
			{
				m_elemCard.appendChild( elemCardImage);
			}
		}
		catch( Exception e)
		{
			throw new InfoCardException( "Unable to set card image");
		}
	}
	
	/***************************************************************************
	* Desc:
	***************************************************************************/
	public void setCardImage(
		String							strBase64EncodedImage,
		String							strMimeType) throws InfoCardException
	{
		org.w3c.dom.Element			elemCardImage;
		org.w3c.dom.Element			elemOldCardImage;
		
		try
		{
			elemCardImage = m_docCard.createElement( "ic:" + strCardImageTag);
			elemCardImage.setAttribute( "MimeType", strMimeType);
			elemCardImage.appendChild( m_docCard.createTextNode( strBase64EncodedImage));
			
			elemOldCardImage = getFirstChildElement( m_elemCard, 
										"ic", strCardImageTag);
										
			if( elemOldCardImage != null)
			{
				m_elemCard.replaceChild( elemCardImage, elemOldCardImage);
			}
			else
			{
				m_elemCard.appendChild( elemCardImage);
			}
		}
		catch( Exception e)
		{
			throw new InfoCardException( "Unable to set card image");
		}
	}
	
	/***************************************************************************
	* Desc:
	***************************************************************************/
	public void setIssuerURI(
		String		strValue)
	{
		updateChildElement( m_elemCard, "ic", strCardIssuerTag, strValue);
	}

	/***************************************************************************
	* Desc:
	***************************************************************************/
	public void setPrivacyNoticeURI(
		String		strValue)
	{
		updateChildElement( m_elemCard, "ic", strPrivacyNoticeTag, strValue);
	}
	
	/***************************************************************************
	* Desc:
	***************************************************************************/
	public void setRequiresAppliesTo()
	{
		updateChildElement( m_elemCard, "ic", strRequireAppliesToTag, null);
	}
	
	/***************************************************************************
	* Desc:
	***************************************************************************/
	public void setCardID(
		String		strValue)
	{
		updateChildElement( m_elemCardRef, "ic", strCardIdTag, strValue);
	}
	
	/***************************************************************************
	* Desc:
	***************************************************************************/
	public void setCardVersion(
		int		iValue)
	{
		updateChildElement( m_elemCardRef, "ic", strCardVersionTag, 
			new Integer( iValue).toString());
	}
	
	/***************************************************************************
	* Desc:
	***************************************************************************/
	public void setCardVersion(
		String		strValue)
	{
		setCardVersion( new Integer( strValue).intValue());
	}
	
	/***************************************************************************
	* Desc:
	***************************************************************************/
	public String toXML() throws InfoCardException
	{
		String						strSignedCard = null;
		org.w3c.dom.Document		docSignedCard;
		org.w3c.dom.Element		elemSignedCard;
		String						strId = "IC01";
		int							iIndex;
		
		// Set the card issue time
		
		updateChildElement( m_elemCard, "ic", strTimeIssuedTag, getDateString( 0));
		
		// Make sure the card is valid
		
		validateCard();
		
		// Sign the card

		try
		{
			XMLSignature 				xmlSignature;
			ObjectContainer 			objectContainer;
			Transforms 					transforms;
			org.w3c.dom.Element		reparsedCard;
			org.w3c.dom.Element		elemCardImage;
			
			// Add standard claim types if no claim types were specified
			
			if( getFirstChildElement( m_elemCardSupportedTokenTypeList, "ic",
				strTokenTypeTag) == null) 
			{
				addSupportedTokenType(  
					"http://docs.oasis-open.org/wss/oasis-wss-saml-token-profile-1.1#SAMLV1.1");
				addSupportedTokenType(  
					"urn:oasis:names:tc:SAML:1.0:assertion");
			}
			
			docSignedCard = m_documentBuilder.newDocument();
			reparsedCard = reparseElement( m_elemCard);
			
			// Remove card image tag if empty
			
			if( (elemCardImage = getFirstChildElement( 
					reparsedCard, "ic", strCardImageTag)) != null)
			{
				if( getFirstChildTextNode( elemCardImage) == null)
				{
					reparsedCard.removeChild( elemCardImage);
				}
			}
			
			// Build a document for signing

			try
			{			
				xmlSignature = new XMLSignature( docSignedCard, "",
										XMLSignature.ALGO_ID_SIGNATURE_RSA_SHA1,
										Transforms.TRANSFORM_C14N_EXCL_OMIT_COMMENTS);
			}
			catch( Exception e)
			{
				throw new InfoCardException( "Error creating XMLSignature: " + e.getMessage());
			}

			objectContainer = new ObjectContainer( docSignedCard);
			objectContainer.appendChild( docSignedCard.importNode( reparsedCard, true));
			objectContainer.setId( strId);
			
			xmlSignature.appendObject( objectContainer);

			transforms = new Transforms( docSignedCard);
			transforms.addTransform( Transforms.TRANSFORM_C14N_EXCL_OMIT_COMMENTS);
			
			xmlSignature.addDocument( "#" + strId, transforms, 
				Constants.ALGO_ID_DIGEST_SHA1);
			xmlSignature.addKeyInfo( m_certificateSigner);
			
			xmlSignature.sign( m_privateKeySigner);
			elemSignedCard = xmlSignature.getElement();
			
//			try
//			{
//				if( !xmlSignature.checkSignatureValue( m_certificateSigner))
//				{
//					throw new InfoCardException( "Signature verification failed");
//				}
//			}
//			catch( Exception e)
//			{
//				throw new InfoCardException( "Error verifying signature: " + e.getMessage());
//			}
		}
		catch( Exception e)
		{
			throw new InfoCardException( "Error signing card: " + e.getMessage());
		}
		
		// Serialize the card
		
		try
		{
			strSignedCard = serializeToString( elemSignedCard);
			
			// Work around a strange problem in the RC1 CardSpace client that
			// causes the import of the card to fail if the closing tag of the
			// signature element is on its own line
			
			if( (iIndex = strSignedCard.indexOf( "</Signature>")) != -1)
			{
				StringBuffer		sbValue = new StringBuffer( strSignedCard);

				iIndex--;
				
				while( sbValue.charAt( iIndex) == 10 ||
					 sbValue.charAt( iIndex) == 13)
				{
					sbValue.deleteCharAt( iIndex);
					iIndex--;
				}
				
				strSignedCard = sbValue.toString();
			}
		}
		catch( Exception e)
		{
			throw new InfoCardException( "Unable to serialize card");
		}
		
		return( strSignedCard);
	}
	
	/***************************************************************************
	* Desc:
	***************************************************************************/
	public org.w3c.dom.Element reparseElement(
		org.w3c.dom.Element		domElement) throws InfoCardException
	{
		String						strElement;
		
		if( domElement == null)
		{
			return( null);
		}
		
		strElement = serializeToString( domElement);
		return( toDOM( strElement));
	}
	
	/***************************************************************************
	* Desc:
	***************************************************************************/
	public org.w3c.dom.Element toDOM(
		String			strElement) throws InfoCardException
	{
		org.w3c.dom.Document		docElement = null;
		
		if( strElement == null)
		{
			return( null);
		}
		
		try
		{
			docElement = m_documentBuilder.parse( 
									new ByteArrayInputStream( strElement.getBytes()));
		}
		catch( Exception e)
		{	
			throw new InfoCardException( "Unable to create document");
		}
		
		return( docElement.getDocumentElement());
	}
	
	/***************************************************************************
	* Desc:
	***************************************************************************/
	public void toFile(
		String		strFileName) throws InfoCardException
	{
		String				strOutCard;
		FileOutputStream	crdOutFile;
		
		try
		{
			strOutCard = toXML();
			crdOutFile = new FileOutputStream( strFileName);
			
			crdOutFile.write( strOutCard.getBytes(), 0, strOutCard.length());
			crdOutFile.flush();
			crdOutFile.close();
			
//			verifyFile( strFileName);
		}
		catch( Exception e)
		{
			throw new InfoCardException( "Error writing card to file: " + e.getMessage());
		}
	}
	
	/***************************************************************************
	* Desc:
	***************************************************************************/
	private void verifyFile(
		String		strFileName) throws Exception
	{
		String						strCard = null;
		FileInputStream			crdInFile;
		byte[]						crdBuffer;
		org.w3c.dom.Element		domParsed;
		XMLSignature 				xmlSignature;
		Transforms					transforms;
		String						strId = "IC01";

		crdInFile = new FileInputStream( strFileName);
		
		crdBuffer = new byte[ crdInFile.available()];
		crdInFile.read( crdBuffer);
		crdInFile.close();
		
		strCard = new String( crdBuffer, 0, crdBuffer.length, "UTF-8");
		domParsed = toDOM( strCard);
		xmlSignature = new XMLSignature( domParsed, "");
		
		if( !xmlSignature.checkSignatureValue( m_certificateSigner))
		{
			throw new InfoCardException( 
				"Signature verification failed for " + strFileName);
		}
	}
	
	/***************************************************************************
	* Desc:
	***************************************************************************/
	private org.w3c.dom.Element getFirstChildElement(
		org.w3c.dom.Element			elemParent,
		String							strPrefix,
		String							strChildTag)
	{
		org.w3c.dom.Node			nodeChild;
		org.w3c.dom.Element		elemChild = null;
		
		nodeChild = elemParent.getFirstChild();
		while( nodeChild != null)
		{
			if( nodeChild instanceof org.w3c.dom.Element)
			{
				elemChild = (org.w3c.dom.Element)nodeChild;
				
				if( strChildTag == null ||
					 elemChild.getNodeName().equals( strPrefix + ":" + strChildTag))
				{
					break;
				}
			}
			
			nodeChild = nodeChild.getNextSibling();
			elemChild = null;
		}
		
		return( elemChild);
	}
	
	/***************************************************************************
	* Desc:
	***************************************************************************/
	private org.w3c.dom.Text getFirstChildTextNode(
		org.w3c.dom.Element			elemParent)
	{
		org.w3c.dom.Node			nodeChild;
		org.w3c.dom.Text			textChild = null;
		
		nodeChild = elemParent.getFirstChild();
		while( nodeChild != null)
		{
			if( nodeChild instanceof org.w3c.dom.Text)
			{
				textChild = (org.w3c.dom.Text)nodeChild;
				break;
			}
			
			nodeChild = nodeChild.getNextSibling();
		}
		
		return( textChild);
	}
	
	/***************************************************************************
	* Desc:
	***************************************************************************/
	private void updateChildElement(
		org.w3c.dom.Element			elemParent,
		String							strPrefix,
		String							strChildTag,
		String							strChildValue)
	{
		org.w3c.dom.Element			elemChild;
		org.w3c.dom.Text				textChild;
		
		if( (elemChild = getFirstChildElement( elemParent, strPrefix, 
			strChildTag)) == null)
		{
			elemChild = m_docCard.createElement( strPrefix + ":" + strChildTag);
			elemParent.appendChild( elemChild);
		}
		else if( (textChild = getFirstChildTextNode( elemChild)) != null)
		{
			elemChild.removeChild( textChild);
		}
		
		if (strChildValue != null)
		{
			elemChild.appendChild( m_docCard.createTextNode( strChildValue));
		}
	}
	
	/***************************************************************************
	* Desc:
	***************************************************************************/
	private String getDateString(
		long						lGMTDate)
	{
		Date						date;
		SimpleDateFormat		dateFormat;
		SimpleDateFormat		timeFormat;
		StringBuffer			strDateTime = new StringBuffer();
		
		if( lGMTDate == 0)
		{
			date = new Date();
		}
		else
		{
			date = new Date( lGMTDate);
		}
		
		dateFormat = new SimpleDateFormat( "yyyy-MM-dd");
		timeFormat = new SimpleDateFormat( "HH:mm:ss");
		
		dateFormat.setTimeZone( TimeZone.getTimeZone( "GMT"));
		timeFormat.setTimeZone( TimeZone.getTimeZone( "GMT"));
		
		strDateTime.append( dateFormat.format( date));
		strDateTime.append( "T");
		strDateTime.append( timeFormat.format( date));
		strDateTime.append( "Z");
		
		return( strDateTime.toString());
	}
	
	/***************************************************************************
	* Desc:
	***************************************************************************/
	public KeyStore loadKeyStore(
		String				strStoreType,
		String				strFile,
		String				strPassword) throws InfoCardException
	{
		FileInputStream 	storeFile;
		KeyStore 			keyStore;

		try
		{
			storeFile = new FileInputStream( strFile);
			keyStore = KeyStore.getInstance( strStoreType);
			keyStore.load( storeFile, strPassword.toCharArray());
		}
		catch( Exception e)
		{
			throw new InfoCardException( "Unable to load key store");
		}
		
		return( keyStore);
	}
	
	/***************************************************************************
	* Desc:
	***************************************************************************/
	public void validateCard() throws InfoCardException
	{
		org.w3c.dom.Element			elemTmp;
		
		if( m_elemCardRef == null)
		{
			throw new InfoCardException( "Missing InformationCardReference");
		}
		
		// Card ID
		
		elemTmp = getFirstChildElement( m_elemCardRef, "ic", strCardIdTag);
		if( elemTmp == null)
		{
			throw new InfoCardException( "Missing CardId");
		}

		try
		{
			new URI( getTextValue( elemTmp));
		}
		catch( Exception e)
		{
			throw new InfoCardException( 
				"Invalid CardId: " + getTextValue( elemTmp));
		}
		
		// Card version
				
		elemTmp = getFirstChildElement( m_elemCardRef, "ic", strCardVersionTag);
		if( elemTmp == null)
		{
			throw new InfoCardException( "Missing CardVersion");
		}

		try
		{
			new Integer( getTextValue( elemTmp));
		}
		catch( Exception e)
		{
			throw new InfoCardException( "Invalid CardVersion");
		}
		
		// Card name
		
		elemTmp = getFirstChildElement( m_elemCard, "ic", strCardNameTag);
		if( elemTmp == null)
		{
			throw new InfoCardException( "Missing CardName");
		}
		
		if( getTextValue( elemTmp) == null || 
			 getTextValue( elemTmp).equals( ""))
		{
			throw new InfoCardException( "Invalid CardName");
		}
		
		// Card issuer
		
		elemTmp = getFirstChildElement( m_elemCard, "ic", strCardIssuerTag);
		if( elemTmp == null)
		{
			throw new InfoCardException( "Missing Issuer URI");
		}

		try
		{
			new URI( getTextValue( elemTmp));
		}
		catch( Exception e)
		{
			throw new InfoCardException( "Invalid Issuer");
		}
		
		// Time issued
		
		elemTmp = getFirstChildElement( m_elemCard, "ic", strTimeIssuedTag);
		if( elemTmp == null)
		{
			throw new InfoCardException( "Missing TimeIssued");
		}
		
		// Time expires
		
		elemTmp = getFirstChildElement( m_elemCard, "ic", strTimeExpiresTag);
		if( elemTmp == null)
		{
			throw new InfoCardException( "Missing TimeExpires");
		}
	}
	
	/***************************************************************************
	* Desc:
	***************************************************************************/
	private String getTextValue(
		org.w3c.dom.Element		elemParent)
	{
		org.w3c.dom.Node			nodeChild;
		
		nodeChild = elemParent.getFirstChild();
		while( nodeChild != null)
		{
			if( nodeChild instanceof org.w3c.dom.Text)
			{
				break;
			}
			
			nodeChild = nodeChild.getNextSibling();
		}
		
		return( nodeChild == null ? null : nodeChild.getNodeValue());
	}
	
	/***************************************************************************
	* Desc:
	***************************************************************************/
	private String serializeToString(
		org.w3c.dom.Element			elemRoot) throws InfoCardException
	{
		ByteArrayOutputStream							outputStream;
		org.apache.xml.serialize.XMLSerializer 	xmlSerializer;
		org.apache.xml.serialize.OutputFormat		outputFormat;
		String												strOutput;
		
		try
		{
			outputStream = new ByteArrayOutputStream();
			
			outputFormat = new org.apache.xml.serialize.OutputFormat();
			outputFormat.setLineWidth( 0);
			outputFormat.setIndenting( false);
			outputFormat.setPreserveSpace( true);
			outputFormat.setOmitXMLDeclaration( true);
			
			xmlSerializer = new org.apache.xml.serialize.XMLSerializer( 
										outputStream, outputFormat);
			xmlSerializer.setNamespaces( true);
			xmlSerializer.serialize( elemRoot);
			strOutput = outputStream.toString( "UTF-8");
		}
		catch( Exception e)
		{
			throw new InfoCardException( "Unable to serialize XML");
		}
		
		return( strOutput);
	}
	
	/***************************************************************************
	* Desc:
	***************************************************************************/
	public static String stripNewLinesFromString(
		String		strValue)
	{
		int			iIndex;

		iIndex = strValue.indexOf( 13);		
		while( iIndex != -1)
		{
			StringBuffer		sbValue = new StringBuffer( strValue);
			
			sbValue.deleteCharAt( iIndex);
			strValue = sbValue.toString();
			iIndex = strValue.indexOf( 13);
		}
		
		iIndex = strValue.indexOf( 10);
		while( iIndex != -1)
		{
			StringBuffer		sbValue = new StringBuffer( strValue);
			
			sbValue.deleteCharAt( iIndex);
			strValue = sbValue.toString();
			iIndex = strValue.indexOf( 10);
		}
		
		return( strValue);
	}
}
