/*******************************************************************************
 * Copyright (c) 2006 IBM Corporation.
 * Copyright (c) 2007 Novell, Inc.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *    Michael McIntosh (IBM Corporation) - initial API and implementation
 *    Jim Sermersheim (Novell)
 *******************************************************************************/ 

package org.eclipse.higgins.sts.server.token.saml;

import java.security.PrivateKey;
import java.security.PublicKey;
import java.security.cert.X509Certificate;
import java.util.Map;
import org.eclipse.higgins.sts.api.IConstants;
import org.eclipse.higgins.sts.api.IRequestSecurityToken;
import org.eclipse.higgins.sts.api.ISTSRequest;
import org.eclipse.higgins.sts.api.ISTSResponse;
import org.eclipse.higgins.sts.common.Fault;

/**
 * Handle RSTs and generate RSTRs as SAML Assertions. 
 * Compatible with www.identityblog.com
 * 
 * @author mikemci at us dot ibm dot com
 */
public class TokenGeneratorHandler
	extends org.eclipse.higgins.sts.server.token.handler.TokenHandler
{
	private final org.eclipse.higgins.sts.utilities.LogHelper log = new org.eclipse.higgins.sts.utilities.LogHelper
		(TokenGeneratorHandler.class.getName());
	
	javax.xml.namespace.QName qnameIdentityClaimType = new javax.xml.namespace.QName
		(null,
		"ClaimType");
	javax.xml.namespace.QName qnameIdentityClaimURI = new javax.xml.namespace.QName
		(null,
		"Uri");
	
    private boolean bConfigured = false;
	
	/**
	 * Protected constructor, must use TokenGeneratorHandlerFactory
	 */
	protected TokenGeneratorHandler()
	{
		this.log.trace("TokenGeneratorHandler::TokenGeneratorHandler");
	}
	
    /* (non-Javadoc)
	 * @see org.eclipse.higgins.sts.IExtension#configure(java.util.Hashtable)
	 */
	public void configure
		(final Map mapGlobalSettings,
		final String strComponentName,
		final Map mapComponentSettings)
	{
		this.log.trace("TokenGeneratorHandler::initialize");

		this.bConfigured = true;
	}

	/**
	 * Invoked to add each claim to the AttributeStatement
	 * 
	 * @param constants  an IConstants containing the URIs appropriate for the request
	 * @param elemAttributeStatement the AttrobuteStatement element that will contain the added claim
     * @param uriClaim the URI for the claim to be added
	 * @param strClaimName the name for the claim to be added
	 * @param strClaimValue the value for the claim to be added
	 */
  	private void AddIdentityClaim
  		(final org.eclipse.higgins.sts.api.IConstants constants,
  		final IRequestSecurityToken RST,
  		final org.apache.axiom.om.OMElement elemAttributeStatement,
  		final org.apache.axiom.om.OMElement elemDisplayToken,
  		final org.eclipse.higgins.sts.api.IClaim claim,
  		final java.util.Map mapAttributeClaim)
  	{
  		final org.eclipse.higgins.sts.api.IClaimType claimType = claim.getType();
  		final java.net.URI uriClaimName = claimType.getName();
  		final String strClaimName = uriClaimName.toString();

  		this.log.trace("Adding Identity Claim URI: " + strClaimName);

  		final java.util.Map mapClaim = (java.util.Map)mapAttributeClaim.get(strClaimName);

  		final String strAttributeName = (String)mapClaim.get("ClaimName");
  		final java.net.URI uriAttributeNamespace = (java.net.URI)mapClaim.get("AttributeNamespace");
  		final String strDisplayName = (String)mapClaim.get("DisplayName");

  		this.log.trace("Adding Identity Claim Name: " + strAttributeName);
  		if (null == strAttributeName)
  		{
  			this.log.error("Unsupported claim");
  			return;
  		}

  		final org.apache.axiom.om.OMFactory omFactory = elemAttributeStatement.getOMFactory();
		final org.apache.axiom.om.OMNamespace omSAMLNamespace = omFactory.createOMNamespace
			(constants.getSAML10Namespace().toString(),
			"saml");
		final org.apache.axiom.om.OMNamespace omIdentityNamespace = omFactory.createOMNamespace
			(constants.getIdentityNamespace().toString(),
			"ic");
  		final org.apache.axiom.om.OMElement elemAttribute = omFactory.createOMElement
  			("Attribute",
  			omSAMLNamespace,
  			elemAttributeStatement);
  		final org.apache.axiom.om.OMElement elemDisplayClaim = omFactory.createOMElement
			("DisplayClaim",
			omIdentityNamespace,
			elemDisplayToken);
  		elemDisplayClaim.addAttribute
  			("Uri",
  			claimType.getName().toString(),
  			null);
  		final org.apache.axiom.om.OMElement elemDisplayTag = omFactory.createOMElement
			("DisplayTag",
			omIdentityNamespace,
			elemDisplayClaim);
  		elemDisplayTag.setText(strDisplayName);
		elemAttribute.addAttribute
			("AttributeName",
			strAttributeName,
			null);
		elemAttribute.addAttribute
			("AttributeNamespace",
			uriAttributeNamespace.toString(),
			null);
  		try
  		{
 			final java.util.Iterator iterValue = claim.getValues();
  			while (iterValue.hasNext())
  			{
				final String strValue = (String)iterValue.next();
				this.log.trace("Adding Claim Value: " + strValue);
				final org.apache.axiom.om.OMElement elemDisplayValue = omFactory.createOMElement
					("DisplayValue",
					omIdentityNamespace,
					elemDisplayClaim);  
		  		elemDisplayValue.setText(strValue);
		  		final org.apache.axiom.om.OMElement elemAttributeValue = omFactory.createOMElement
					("AttributeValue",
					omSAMLNamespace,
					elemAttribute);
		  		elemAttributeValue.setText(strValue);
 			}
		}
 		catch (final Exception e)
  		{
    		org.eclipse.higgins.sts.utilities.ExceptionHelper.Log
				(this.log,
				e);	
  		}
  	}
  	
	/**
	 * Invoked to add a PPID claim to the AttributeStatement
	 * 
	 * @param constants  an IConstants containing the URIs appropriate for the request
	 * @param elemAttributeStatement the AttrobuteStatement element that will contain the added claim
     * @param uriClaim the URI for the claim to be added
	 * @param strClaimName the name for the claim to be added
	 * @param strClaimValue the value for the claim to be added
	 */
  	private void AddPrivatePersonalIdentityClaim
  		(final org.eclipse.higgins.sts.api.IConstants constants,
  		final IRequestSecurityToken RST,
  		final org.apache.axiom.om.OMElement elemAttributeStatement,
  		final org.apache.axiom.om.OMElement elemDisplayToken,
  		final org.eclipse.higgins.sts.api.IClaim claim,
  		final java.util.Map mapAttributeClaim)
  	{
  		final org.eclipse.higgins.sts.api.IClaimType claimType = claim.getType();
  		final java.net.URI uriClaimName = claimType.getName();
  		final String strClaimName = uriClaimName.toString();

  		this.log.trace("Adding Identity Claim URI: " + strClaimName);

  		final java.util.Map mapClaim = (java.util.Map)mapAttributeClaim.get(strClaimName);

  		final String strAttributeName = (String)mapClaim.get("ClaimName");
  		final java.net.URI uriAttributeNamespace = (java.net.URI)mapClaim.get("AttributeNamespace");
  		final String strDisplayName = (String)mapClaim.get("DisplayName");

  		this.log.trace("Adding Identity Claim Name: " + strAttributeName);
  		if (null == strAttributeName)
  		{
  			this.log.error("Unsupported claim");
  			return;
  		}

  		final org.apache.axiom.om.OMFactory omFactory = elemAttributeStatement.getOMFactory();
		final org.apache.axiom.om.OMNamespace omSAMLNamespace = omFactory.createOMNamespace
			(constants.getSAML10Namespace().toString(),
			"saml");
		final org.apache.axiom.om.OMNamespace omIdentityNamespace = omFactory.createOMNamespace
			(constants.getIdentityNamespace().toString(),
			"ic");
  		final org.apache.axiom.om.OMElement elemAttribute = omFactory.createOMElement
  			("Attribute",
  			omSAMLNamespace,
  			elemAttributeStatement);
  		final org.apache.axiom.om.OMElement elemDisplayClaim = omFactory.createOMElement
			("DisplayClaim",
			omIdentityNamespace,
			elemDisplayToken);
  		elemDisplayClaim.addAttribute
  			("Uri",
  			claimType.getName().toString(),
  			null);
  		final org.apache.axiom.om.OMElement elemDisplayTag = omFactory.createOMElement
			("DisplayTag",
			omIdentityNamespace,
			elemDisplayClaim);
  		elemDisplayTag.setText(strDisplayName);
		elemAttribute.addAttribute
			("AttributeName",
			strAttributeName,
			null);
		elemAttribute.addAttribute
			("AttributeNamespace",
			uriAttributeNamespace.toString(),
			null);
  		try
  		{
  			final org.apache.axiom.om.OMElement elemDisplayValue = omFactory.createOMElement
				("DisplayValue",
				omIdentityNamespace,
				elemDisplayClaim);
	   		final org.eclipse.higgins.sts.api.IElement elemClientPseudonym = RST.getClientPseudonym();
	   		String strPPID = "BOGUS";
	   		if (null != elemClientPseudonym)
	   		{
		   		final org.apache.axiom.om.OMElement omClientPseudonym = (org.apache.axiom.om.OMElement)elemClientPseudonym.getAs
	   				(org.apache.axiom.om.OMElement.class);
		   		final java.util.Iterator iterPPID = omClientPseudonym.getChildElements();
		   		final org.apache.axiom.om.OMElement omPPID = (org.apache.axiom.om.OMElement)iterPPID.next();
		   		strPPID = omPPID.getText();
	   		}
			this.log.trace("Adding Claim Value: " + strPPID);
	  		elemDisplayValue.setText(strPPID);
	  		final org.apache.axiom.om.OMElement elemAttributeValue = omFactory.createOMElement
				("AttributeValue",
				omSAMLNamespace,
				elemAttribute);
	  		elemAttributeValue.setText(strPPID);
		}
 		catch (final Exception e)
  		{
    		org.eclipse.higgins.sts.utilities.ExceptionHelper.Log
				(this.log,
				e);	
  		}
  	}

	/* (non-Javadoc)
	 * @see org.eclipse.higgins.sts.IExtension#invoke
     */
	public void invoke
		(final java.util.Map mapGlobalSettings,
		final String strComponentName,
		final java.util.Map mapComponentSettings,
		final java.util.Map mapInvocationSettings,
		final IConstants constants,
		final ISTSRequest request,
		final ISTSResponse response)
	{
		this.log.trace("TokenGeneratorHandler::invoke: " + strComponentName);
		
		if (!this.bConfigured)
		{
			final Fault fault = new Fault
				(constants.getWSTrustNamespace(),
				"wst",
				constants.getRequestFailedFaultCode(),
				"The specified request failed",
				"Issue handler not configured");
			response.setFault(fault);
			return;		
		}
		
		final java.net.URI uriDefaultKeyType = (java.net.URI)mapComponentSettings.get
    		("DefaultKeyType");
		this.log.trace("DefaultKeyType: " + uriDefaultKeyType.toString());
		
		final java.lang.Boolean bIncludeBearerSubjectName = (java.lang.Boolean)mapComponentSettings.get
    		("IncludeBearerSubjectName");
		this.log.trace("IncludeBearerSubjectName: " + bIncludeBearerSubjectName.toString());
		
		final java.net.URI uriTokenIssuer = (java.net.URI)mapComponentSettings.get
    		("TokenIssuer");
		this.log.trace("TokenIssuer: " + uriTokenIssuer.toString());
		if (null == uriTokenIssuer)
		{
			final Fault fault = new Fault
				(constants.getWSTrustNamespace(),
				"wst",
				constants.getRequestFailedFaultCode(),
				"The specified request failed",
				"TokenIssuer not set.");
			response.setFault(fault);
			return;
		}

		final java.net.URI uriSubjectNameIdentifier = (java.net.URI)mapComponentSettings.get
    		("SubjectNameIdentifierAttribute");
		if (null != uriSubjectNameIdentifier)
			this.log.trace("SubjectNameIdentifier: " + uriSubjectNameIdentifier.toString());
		
		final java.net.URI uriSubjectNameIdentifierFormat = (java.net.URI)mapComponentSettings.get
			("SubjectNameIdentifierFormat");
		if (null != uriSubjectNameIdentifierFormat)
			this.log.trace("SubjectNameIdentifierFormat: " + uriSubjectNameIdentifierFormat.toString());

		final java.lang.Boolean bEncryptToken = (java.lang.Boolean)mapComponentSettings.get
    		("EncryptToken");
		this.log.trace("EncryptToken: " + bEncryptToken.toString());
	
		final java.util.Map mapAttributeClaim = (java.util.Map)mapGlobalSettings.get
			("AttributeClaimMap");
		
		if (null == mapAttributeClaim)
		{
			final Fault fault = new Fault
				(constants.getWSTrustNamespace(),
				"wst",
				constants.getRequestFailedFaultCode(),
				"The specified request failed",
				"AttributeClaimMap not set.");
			response.setFault(fault);
	   		return;
		}
		
		final org.eclipse.higgins.sts.spi.IBase64Extension base64Extension = (org.eclipse.higgins.sts.spi.IBase64Extension)mapGlobalSettings.get
			("Base64Extension");
		
		// TODO: Properly namespace qualify LocalNames throughout
		final java.util.List listRST = request.getRequestSecurityTokenCollection();
		final IRequestSecurityToken RST = (IRequestSecurityToken)listRST.get(0);
		// TODO: Deal with collections?
		
		final java.net.URI uriTokenType = RST.getTokenType();
		final org.eclipse.higgins.sts.api.IAppliesTo appliesToRequest = RST.getAppliesTo();
		final org.eclipse.higgins.sts.api.ILifetime ltLifetime = RST.getLifetime();

		java.net.URI uriKeyType = RST.getKeyType();
		if (null == uriKeyType)
		{
			if (null == uriDefaultKeyType) {
				uriKeyType = constants.getNoProofKeyKeyType();
			} else {
				uriKeyType = uriDefaultKeyType;
			}
		}
		String strSubjectNameIdentifier = null;
		boolean bBearerToken = uriKeyType.equals(constants.getNoProofKeyKeyType());
 		
		org.eclipse.higgins.sts.api.IEndpointReference eprAppliesTo = null;
		java.net.URI uriAppliesTo = null;
		this.log.trace("Checking for AppliesTo");
		if (appliesToRequest != null)
		{
			this.log.trace("Found AppliesTo");
			eprAppliesTo = appliesToRequest.getEndpointReference();
			uriAppliesTo = eprAppliesTo.getAddress();
		}
		
		final X509Certificate certificateIssuer = (X509Certificate)mapGlobalSettings.get
			("IssuerCertificate");
		PublicKey publicKeyIssuer = null;
		PrivateKey privateKeyIssuer = null;
		if (uriTokenIssuer.equals(constants.getIssuerSelf()))
		{
			java.security.KeyPair kpSelf = request.getSelfSigningKeyPair();
			this.log.trace("KeyPair Class: " + kpSelf.getClass().getName());
			if (null == kpSelf)
			{
				final Fault fault = new Fault
					(constants.getWSTrustNamespace(),
					"wst",
					constants.getRequestFailedFaultCode(),
					"The specified request failed",
					"Self Signing KeyPair not set.");
				response.setFault(fault);
		   		return;
			}
			publicKeyIssuer = kpSelf.getPublic();
			this.log.trace("PublicKey Class: " + publicKeyIssuer.getClass().getName());
			if (null == publicKeyIssuer)
			{
				final Fault fault = new Fault
					(constants.getWSTrustNamespace(),
					"wst",
					constants.getRequestFailedFaultCode(),
					"The specified request failed",
					"Self Signing PublicKey not set.");
				response.setFault(fault);
		   		return;
			}
			privateKeyIssuer = kpSelf.getPrivate();
			this.log.trace("PrivateKey Class: " + privateKeyIssuer.getClass().getName());
			if (null == privateKeyIssuer)
			{
				final Fault fault = new Fault
					(constants.getWSTrustNamespace(),
					"wst",
					constants.getRequestFailedFaultCode(),
					"The specified request failed",
					"Self Signing PrivateKey not set.");
				response.setFault(fault);
		   		return;
			}	
		}
		else
		{
			publicKeyIssuer = certificateIssuer.getPublicKey();
			privateKeyIssuer = (PrivateKey)mapGlobalSettings.get
				("IssuerPrivateKey");
			if (null == certificateIssuer)
			{
				final Fault fault = new Fault
					(constants.getWSTrustNamespace(),
					"wst",
					constants.getRequestFailedFaultCode(),
					"The specified request failed",
					"IssuerCertificate not set.");
				response.setFault(fault);
		   		return;
			}
			if (null == privateKeyIssuer)
			{
				final Fault fault = new Fault
					(constants.getWSTrustNamespace(),
					"wst",
					constants.getRequestFailedFaultCode(),
					"The specified request failed",
					"IssuePrivateKey not set.");
				response.setFault(fault);
		   		return;
			}
		}
	
		final java.net.URI uriAssertionID = org.eclipse.higgins.sts.utilities.UUIDHelper.getAsURI();
		final java.util.Date dateNow = new java.util.Date();
		final java.util.Date dateLater = new java.util.Date(dateNow.getTime() + (1000*60*60*24*7));
		final String strNow = org.eclipse.higgins.sts.utilities.DateHelper.getIso8601Date(dateNow);
		final String strLater = org.eclipse.higgins.sts.utilities.DateHelper.getIso8601Date(dateLater);
		final org.apache.axiom.om.OMFactory omFactory = org.apache.axiom.om.OMAbstractFactory.getOMFactory();
		
		final org.apache.axiom.om.OMNamespace omIdentityNamespace = omFactory.createOMNamespace
			(constants.getIdentityNamespace().toString(),
			"ic");
		final org.apache.axiom.om.OMNamespace omWSTrustNamespace = omFactory.createOMNamespace
			(constants.getWSTrustNamespace().toString(),
			"wst");
		final org.apache.axiom.om.OMElement omRequestedAttachedReference = omFactory.createOMElement
			("RequestedAttachedReference",
			omWSTrustNamespace);
		final org.apache.axiom.om.OMElement omRequestedUnattachedReference = omFactory.createOMElement
			("RequestedUnattachedReference",
			omWSTrustNamespace);
		final org.apache.axiom.om.OMElement omRequestedDisplayToken = omFactory.createOMElement
			("RequestedDisplayToken",
			omIdentityNamespace);
		final org.apache.axiom.om.OMElement omDisplayToken = omFactory.createOMElement
			("DisplayToken",
			omIdentityNamespace,
			omRequestedDisplayToken);
		final org.apache.axiom.om.OMNamespace omWSSNamespace = omFactory.createOMNamespace
			(constants.getWSSecurityNamespace().toString(),
			"wsse");
		final org.apache.axiom.om.OMElement omSecurityTokenReference1 = omFactory.createOMElement
			("SecurityTokenReference",
			omWSSNamespace,
			omRequestedAttachedReference);
		final org.apache.axiom.om.OMElement omSecurityTokenReference2 = omFactory.createOMElement
			("SecurityTokenReference",
			omWSSNamespace,
			omRequestedUnattachedReference);
		final org.apache.axiom.om.OMElement omKeyIdentifier1 = omFactory.createOMElement
			("KeyIdentifier",
			omWSSNamespace,
			omSecurityTokenReference1);
		final org.apache.axiom.om.OMElement omKeyIdentifier2 = omFactory.createOMElement
			("KeyIdentifier",
			omWSSNamespace,
			omSecurityTokenReference2);
		omKeyIdentifier1.addAttribute
			("ValueType", 
			"http://docs.oasis-open.org/wss/oasis-wss-saml-token-profile-1.0#SAMLAssertionID",
			null);
		omKeyIdentifier1.setText(uriAssertionID.toASCIIString());
		omKeyIdentifier2.addAttribute
			("ValueType", 
			"http://docs.oasis-open.org/wss/oasis-wss-saml-token-profile-1.0#SAMLAssertionID",
			null);
		omKeyIdentifier2.setText(uriAssertionID.toASCIIString());
		org.apache.axiom.om.OMElement omRequestedSecurityToken = omFactory.createOMElement
			("RequestedSecurityToken",
			omWSTrustNamespace);
		final org.apache.axiom.om.OMNamespace omSAMLNamespace = omFactory.createOMNamespace
			(constants.getSAML10Namespace().toString(),
			"saml");
		final org.apache.axiom.om.OMElement omAssertion = omFactory.createOMElement
			("Assertion",
			omSAMLNamespace,
			omRequestedSecurityToken);
		omAssertion.addAttribute
			("MajorVersion",
			"1",
			null);
		omAssertion.addAttribute
			("MinorVersion",
			"1",
			null);
		omAssertion.addAttribute
			("Issuer",
			uriTokenIssuer.toString(),
			null);
		omAssertion.addAttribute
			("AssertionID",
			uriAssertionID.toASCIIString(),
			null);
		omAssertion.addAttribute
			("IssueInstant",
			strNow.toString(),
			null);
		final org.apache.axiom.om.OMElement omConditions = omFactory.createOMElement
			("Conditions",
			omSAMLNamespace,
			omAssertion);
		omConditions.addAttribute("NotBefore", strNow.toString(), null);
		omConditions.addAttribute("NotOnOrAfter", strLater.toString(), null);
		
		if (null != uriAppliesTo)
		{
			final org.apache.axiom.om.OMElement omAudienceRestrictionCondition = omFactory.createOMElement
				("AudienceRestrictionCondition",
				omSAMLNamespace,
				omConditions);
			final org.apache.axiom.om.OMElement omAudience = omFactory.createOMElement
				("Audience",
				omSAMLNamespace,
				omAudienceRestrictionCondition);
			omAudience.setText(uriAppliesTo.toString());
		}
		
		// TODO: Deal with complete content of request (rather than assuming all SAML per Kim's Blog)
		final org.apache.axiom.om.OMElement omAttributeStatement = omFactory.createOMElement
			("AttributeStatement",
			omSAMLNamespace,
			omAssertion);
		final org.apache.axiom.om.OMElement omSubject = omFactory.createOMElement
			("Subject",
			omSAMLNamespace,
			omAttributeStatement);
		
		final org.eclipse.higgins.sts.api.IDigitalIdentity digitalIdentity = RST.getDigitalIdentity();
		if (null == digitalIdentity)
	   	{
			final Fault fault = new Fault
				(constants.getWSTrustNamespace(),
				"wst",
				constants.getRequestFailedFaultCode(),
				"The specified request failed",
				"Digital Subject was not found");
			response.setFault(fault);
			return;
	   	}
		
		if (digitalIdentity.getIncludePPIClaim())
		{
			final org.eclipse.higgins.sts.api.IClaimType claimType = new org.eclipse.higgins.sts.common.ClaimType();
			claimType.setName(constants.getIdentityClaimPrivatePersonalIdentifier());
			final org.eclipse.higgins.sts.api.IClaim claim = new org.eclipse.higgins.sts.common.Claim();
			claim.setType(claimType);
	  		this.AddPrivatePersonalIdentityClaim
	  			(constants,
	  			RST,
	  			omAttributeStatement,
	  			omDisplayToken,
	  			claim,
	  			mapAttributeClaim);
		}
		
		final java.util.List listClaims = digitalIdentity.getClaims();

		for (int i = 0; i < listClaims.size(); ++i)
		{
			final org.eclipse.higgins.sts.api.IClaim claim = (org.eclipse.higgins.sts.api.IClaim)listClaims.get(i);
	  		this.AddIdentityClaim
	  			(constants,
	  			RST,
	  			omAttributeStatement,
	  			omDisplayToken,
	  			claim,
	  			mapAttributeClaim);
		}
		
		if ((!bBearerToken) || (bIncludeBearerSubjectName.booleanValue()))
		{
			final org.eclipse.higgins.sts.api.IClaimType claimType = new org.eclipse.higgins.sts.common.ClaimType();
			claimType.setName(uriSubjectNameIdentifier);
			final org.eclipse.higgins.sts.api.IClaim claim = digitalIdentity.getClaim(uriSubjectNameIdentifier);
			if (null != claim)
			{
				final java.util.Iterator iterClaimValue = claim.getValues();
				while (iterClaimValue.hasNext())
				{
					strSubjectNameIdentifier = (String)iterClaimValue.next();
					break;
	  			}
	  		}
			if (null == strSubjectNameIdentifier)
			{
				this.log.error("Digital Subject does not contain Subject NameIdentifier Claim: " + uriSubjectNameIdentifier.toString());
				final Fault fault = new Fault
					(constants.getWSTrustNamespace(),
					"wst",
					constants.getRequestFailedFaultCode(),
					"The specified request failed",
					"Digital Subject does not contain Subject NameIdentifier Claim: " + uriSubjectNameIdentifier.toString());
				response.setFault(fault);
				return;	
			}
			final org.apache.axiom.om.OMElement omNameIdentifier = omFactory.createOMElement
				("NameIdentifier",
				omSAMLNamespace,
				omSubject);
			omNameIdentifier.addAttribute
				("Format",
				uriSubjectNameIdentifierFormat.toString(),
				null);
			omNameIdentifier.setText
				(strSubjectNameIdentifier);
		}
		final org.apache.axiom.om.OMElement omSubjectConfirmation = omFactory.createOMElement
	   		("SubjectConfirmation",
			omSAMLNamespace,
			omSubject);
	   	final org.apache.axiom.om.OMElement omConfirmationMethod = omFactory.createOMElement
	   		("ConfirmationMethod",
			omSAMLNamespace,
			omSubjectConfirmation);
		final org.apache.axiom.om.OMNamespace omDSNamespace = omFactory.createOMNamespace
			(constants.getXMLSignatureNamespace().toString(),
			"ds");
    	final org.eclipse.higgins.sts.spi.IXMLSecurityExtension xmlSecurity = (org.eclipse.higgins.sts.spi.IXMLSecurityExtension)mapGlobalSettings.get
			("XMLSecurityExtension");
		if (bBearerToken)
	   	{
	   		omConfirmationMethod.setText(constants.getSAML10ConfirmationMethodBearer().toString());
	   	}
	   	else
	   	{
	   		try
	   		{
		   		omConfirmationMethod.setText(constants.getSAML10ConfirmationMethodHolderOfKey().toString());	   		
			   	final org.apache.axiom.om.OMElement omKeyInfo = omFactory.createOMElement
		   			("KeyInfo",
		   			omDSNamespace,
		   			omSubjectConfirmation);
			   	final org.apache.axiom.om.OMElement omKeyValue = omFactory.createOMElement
					("KeyValue",
					omDSNamespace,
					omKeyInfo);
			   	final org.apache.axiom.om.OMElement omRSAKeyValue = omFactory.createOMElement
					("RSAKeyValue",
					omDSNamespace,
					omKeyValue);
			    final java.security.interfaces.RSAPublicKey rsaPublicKey = (java.security.interfaces.RSAPublicKey)publicKeyIssuer;
			    final java.math.BigInteger biModulus = rsaPublicKey.getModulus();
			   	String strModulus = base64Extension.encode
			   		(biModulus);
			   	strModulus = org.eclipse.higgins.sts.utilities.XMLHelper.stripNewLinesFromString
			   		(strModulus);
			    final java.math.BigInteger biExponent = rsaPublicKey.getPublicExponent();
			   	String strExponent = base64Extension.encode
			   		(biExponent);
			   	strExponent = org.eclipse.higgins.sts.utilities.XMLHelper.stripNewLinesFromString
			   		(strExponent);
			   	final org.apache.axiom.om.OMElement omModulus = omFactory.createOMElement
					("Modulus",
					omDSNamespace,
					omRSAKeyValue);
			   	omModulus.setText
			   		(strModulus);
			   	final org.apache.axiom.om.OMElement omExponent = omFactory.createOMElement
					("Exponent",
					omDSNamespace,
					omRSAKeyValue);
			   	omExponent.setText
			   		(strExponent);
	   		}
		    catch (final Exception e)
		    {
	    		org.eclipse.higgins.sts.utilities.ExceptionHelper.Log
					(this.log,
					e);
				final Fault fault = new Fault
					(constants.getWSTrustNamespace(),
					"wst",
					constants.getRequestFailedFaultCode(),
					"The specified request failed",
					"XML Signature operation failed.");
				response.setFault(fault);
	    		return;	
		    }
	   	}
		
		org.eclipse.higgins.sts.api.IElement elemSignedRequestedSecurityToken = null;
		try
		{
			org.eclipse.higgins.sts.api.IElement elemRequestedSecurityToken = new org.eclipse.higgins.sts.common.Element();
			elemRequestedSecurityToken.set(omRequestedSecurityToken);
			elemSignedRequestedSecurityToken = xmlSecurity.SignEnveloped
	    		("saml:Assertion",
	    		elemRequestedSecurityToken,
	    		uriAssertionID.toASCIIString(),
	    		privateKeyIssuer,
	    		publicKeyIssuer,
	    		constants);
		}
	    catch (final Exception e)
	    {
    		org.eclipse.higgins.sts.utilities.ExceptionHelper.Log
				(this.log,
				e);
			final Fault fault = new Fault
				(constants.getWSTrustNamespace(),
				"wst",
				constants.getRequestFailedFaultCode(),
				"The specified request failed",
				"XML Signature operation failed.");
			response.setFault(fault);
    		return;	
	    }
	    
		final java.util.List listRSTR = response.getRequestSecurityTokenResponseCollection();
		if (0 == listRSTR.size())
		{
			listRSTR.add(new org.eclipse.higgins.sts.common.RequestSecurityTokenResponse());
		}
		final org.eclipse.higgins.sts.api.IRequestSecurityTokenResponse RSTR = (org.eclipse.higgins.sts.common.RequestSecurityTokenResponse)listRSTR.get(0);
		try
		{
			RSTR.setTokenType
				(uriTokenType);
			RSTR.setLifetime
				(ltLifetime);
			RSTR.setRequestedSecurityToken
				(elemSignedRequestedSecurityToken);
			RSTR.setRequestedAttachedReference
				(org.eclipse.higgins.sts.utilities.XMLHelper.toElement(omRequestedAttachedReference));
			RSTR.setRequestedUnattachedReference
				(org.eclipse.higgins.sts.utilities.XMLHelper.toElement(omRequestedUnattachedReference));
			RSTR.setRequestedDisplayToken
				(org.eclipse.higgins.sts.utilities.XMLHelper.toElement(omRequestedDisplayToken));
		}
		catch (final Exception e)
		{
    		org.eclipse.higgins.sts.utilities.ExceptionHelper.Log
    			(this.log,
    			e);
			final Fault fault = new Fault
				(constants.getWSTrustNamespace(),
				"wst",
				constants.getRequestFailedFaultCode(),
				"The specified request failed",
				"Failed to set RequestSecurityToken elements.");
			response.setFault(fault);
			return;
		}
	}
}